/******************************************************************************

  File:        ift.c
  Description: Functions for basic processing of IFT data structure.

  Copyright (c) 2004,2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2004-08-15 Vesa Oikonen
    First version
  2004-09-20 VO
    Doxygen style comments are corrected.
  2004-10-15 VO
    Included iftFindNthKey().
  2004-11-23 VO
    iftWriteItem() writes also comment character if available.
    iftWriteItem() does not write key and equal sign if key is empty.
    Included iftReplaceNthValue().
  2005-01-05 VO
    Moved from libpet to its own library libift.
    Contents of ift.c is split into several .c files.


******************************************************************************/

/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
/*****************************************************************************/
#include "include/ift.h"
/*****************************************************************************/
static char *ift_status[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot open file",
  /*  4 */ "cannot write file",
  /*  5 */ "unsupported file type",
  /*  6 */ "key not found",
  /*  7 */ "file contains no data",
  /*  8 */ "value not found",
  /*  9 */ "key not found"
  };
/*****************************************************************************/

/*****************************************************************************/
void iftSetStatus(IFT *ift, int status)
{
  if(status<0 || status>9) ift->status=ift_status[1];
  else ift->status=ift_status[status];
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate IFT structure. This should be called once before first use. */
void iftInit(IFT *ift)
{
  if(IFT_TEST) printf("iftInit()\n");
  memset(ift, 0, sizeof(IFT));
  ift->_memNr=ift->keyNr=0; ift->item=NULL;
  ift->data=NULL; ift->datasize=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for IFT. All contents are destroyed. */
void iftEmpty(IFT *ift)
{
  int i;

  if(IFT_TEST) printf("iftEmpty()\n");
  for(i=0; i<ift->_memNr; i++) {
    if(ift->item[i].key!=NULL) free(ift->item[i].key);
    if(ift->item[i].value!=NULL) free(ift->item[i].value);
  }
  free(ift->item); ift->item=NULL; ift->_memNr=ift->keyNr=0;
  ift->data=NULL; ift->datasize=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified key and its value to the IFT.
    Also comment type (first character pointed to) can be added.
    Either key or value can be empty, but not both of them.
\return Returns 0 if ok. Sets ift->status.
 */
int iftPut(
  /** Pointer to initiated IFT */
  IFT *ift,
  /** Key string; can be empty ("" or NULL) */
  char *key,
  /** Value string; can be empty ("" or NULL) */
  char *value,
  /** Pointer to comment character, e.g. '#' or ';' or '!'
      can be empty ("" or NULL) */
  char *cmt_type
) {
  int i, add_nr=1000;

  if(ift==NULL) {iftSetStatus(ift, 1); return(1);}
  if((key==NULL || strlen(key)<1) && (value==NULL || strlen(value)<1)) {
    iftSetStatus(ift, 1); return(2);}
  if(IFT_TEST) {
    printf("iftPut(ift, ");
    if(key!=NULL) printf("\"%s\", ", key); else printf("NULL, ");
    if(value!=NULL) printf("\"%s\", ", value); else printf("NULL, ");
    if(cmt_type!=NULL) printf("\"%1.1s\")\n", cmt_type); else printf("NULL)\n");
  }

  /* If necessary, allocate more memory for items */
  if(ift->_memNr<=ift->keyNr) {
    ift->item=realloc(ift->item, (ift->_memNr+add_nr)*sizeof(IFT_KEY_AND_VALUE));
    if(ift->item==NULL) {iftSetStatus(ift, 2); return(5);}
    ift->_memNr+=add_nr;
    for(i=ift->keyNr; i<ift->_memNr; i++) {
      ift->item[i].type=(char)0;
      ift->item[i].sw=(short int)0;
      ift->item[i].key=NULL;
      ift->item[i].value=NULL;
    }
  }
  
  /* Set the contents */
  /* type */
  if(cmt_type!=NULL) ift->item[ift->keyNr].type=cmt_type[0];
  /* key */
  if(key!=NULL) ift->item[ift->keyNr].key=strdup(key);
  else ift->item[ift->keyNr].key=strdup("");
  if(ift->item[ift->keyNr].key==NULL) {iftSetStatus(ift, 2); return(7);}
  /* value */
  if(value!=NULL) ift->item[ift->keyNr].value=strdup(value);
  else ift->item[ift->keyNr].value=strdup("");
  if(ift->item[ift->keyNr].value==NULL) {iftSetStatus(ift, 2); return(8);}

  ift->keyNr++;
  iftSetStatus(ift, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove the specified item from IFT.
\return Returns 0 if ok.
 */
int iftDeleteItem(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Index [0..keyNr-1] of key and value to delete */
  int item
) {
  int i;

  if(IFT_TEST) printf("iftDeleteItem(*ift, %d)\n", item);
  iftSetStatus(ift, 1);
  if(ift==NULL) {return(1);}
  if(item<0 || item>=ift->keyNr) {return(2);}

  if(ift->item[item].key!=NULL) free(ift->item[item].key);
  if(ift->item[item].value!=NULL) free(ift->item[item].value);
  ift->item[item].key=ift->item[item].value=NULL;
  for(i=item+1; i<ift->keyNr; i++) {
    ift->item[i-1].type=ift->item[i].type;
    ift->item[i-1].sw=ift->item[i].sw;
    ift->item[i-1].key=ift->item[i].key;
    ift->item[i-1].value=ift->item[i].value;
    ift->item[i].key=ift->item[i].value=NULL;
  }
  ift->keyNr--;
  iftSetStatus(ift, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Replaces specified value in IFT with a new value.
\return Returns 0 if ok.
 */
int iftReplaceNthValue(
  /** Pointer to initiated IFT */
  IFT *ift,
  /** Index [0..keyNr-1] of key and value */
  int item,
  /** Value string; can be empty ("" or NULL) */
  char *value
) {
  if(ift==NULL) {iftSetStatus(ift, 1); return(1);}
  if(item>=ift->keyNr) {iftSetStatus(ift, 1); return(2);}
  if(IFT_TEST) printf("iftReplaceNthValue(ift, %d, %s)\n", item, value);
  /* Delete old value */
  if(ift->item[item].value!=NULL) free(ift->item[item].value);
  /* Set new value */
  if(value!=NULL) ift->item[item].value=strdup(value);
  else ift->item[item].value=strdup("");
  if(ift->item[item].value==NULL) {iftSetStatus(ift, 2); return(8);}
  iftSetStatus(ift, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

