/*****************************************************************************

  aic.c  (c) 2003,2004 Turku PET Centre

  This file contains the routines for model selection and weighting using
  Akaike's information criteria.
  
  For reference, see Sederholm K. TPCMOD0016 in
  http://pet.utu.fi/staff/vesoik/reports/tpcmod0000.html
  
  Version:
  2003-03-20 Vesa Oikonen & Kaisa Sederholm
  2004-09-17 VO
    Doxygen style comments.


*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
/****************************************************************************/
#include "include/aic.h"
/****************************************************************************/

/****************************************************************************/
/** Computation of AIC in the special case of sum-of-squares optimization
  from the SS, nr of fitted points and nr of fitted parameters.
  If variance is different between the data points, weighted SS must be given.
\return Returns the AIC value.
*/
double aicSS(
  /** Sum-of-Squares of the fit */
  double ss,
  /** Sample size, i.e. nr of fitted data points */
  int n,
  /** Number of fitted model parameters */
  int k
) {
  double aic=0.0, bias_adj, css;
  int dr, dv;
  
  dr=n-k-1; dv=2*k*(k+1);
  if(dr!=0) bias_adj=((double)dv)/((double)dr); else bias_adj=0.0;
  if(ss<1.0e-50) css=1.0e-50; else css=ss; /* Because log(0) is an error */
  if(n>0) aic= n*log(css/(double)n) + 2.0*(double)k + bias_adj;
  return(aic);
} 
/****************************************************************************/

/****************************************************************************/
/** Computation of the Akaike weights for model averaging.
  Requires an array of AIC values, and an output array for weights.
\return Returns 0, if OK.
*/
int aicWeights(
  /** Array of AICs */
  double *aic,
  /** Array of weights (output) */
  double *w,
  /** Lengths of arrays */
  int n
) {
  int i, mini;
  double minaic, sume;

  /* Check data */
  if(n<1 || aic==NULL || w==NULL) return(1);
  if(n==1) {w[0]=1.0; return(0);}
  /* Find out which model gave the smallest AIC */
  mini=0; minaic=aic[0];
  for(i=1; i<n; i++) if(aic[i]<minaic) {mini=i; minaic=aic[i];}
  /* Compute relative weights for each model */
  for(i=0, sume=0.0; i<n; i++) {
    w[i]=exp(-0.5*(aic[i]-minaic));
    sume+=w[i];
  }
  if(sume==0.0) return(2);
  for(i=0; i<n; i++) w[i]/=sume;
  return(0);
}
/****************************************************************************/

/****************************************************************************/
/** Computation of the Akaike weighted model parameter average.
    Requires arrays of AIC weight values, and corresponding parameter values.
\return Returns the weighted average.
*/
double aicWeightedAvg(
  /** Array of weights */
  double *w,
  /** Array of parameters */
  double *p,
  /** Lengths of arrays */
  int n
) {
  int i;
  double avg;

  /* Check data */
  if(n<1 || w==NULL || p==NULL) return(1);
  for(i=0, avg=0.0; i<n; i++) avg+=w[i]*p[i];
  return(avg);
}
/****************************************************************************/

/****************************************************************************/
/** Calculates a value describing the relative goodness of models, based on
    an array of model weights.
\return Returns the weighted average of model number.
*/
double aicModel(
  /** Array of weights */
  double *w,
  /** Length of array */
  int n
) {
  int i;
  double avg;

  if(n<1 || w==NULL) avg=0.0;
  else for(i=0, avg=0.0; i<n; i++) avg+=w[i]*(double)(i+1);
  return(avg);
}
/****************************************************************************/

/****************************************************************************/

