/******************************************************************************

  ecat63ml.c (c) 2003,2004 by Turku PET Centre

  Procedures for reading and writing ECAT 6.3 matrix list.

  Assumptions:
  1. Assumes that matrix list data is in VAX little endian format
  2. Data is automatically converted to big endian when read, if necessary
     according to the current platform.

  Version: Contents were previously in ecat63.c
  2003-07-21 Vesa Oikonen
  2003-08-05 VO
    Included functions ecat63SortMatlistByPlane(), ecat63SortMatlistByFrame()
    and ecat63CheckMatlist().
  2004-06-20 VO
    ecat63PrintMatlist(): blkNr is printed correctly (+1).
  2004-06-27 VO
    Included ecat63DeleteLateFrames().
  2004-09-20 VO
    Doxygen style comments.

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "include/swap.h"
#include "include/ecat63.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate ECAT matrix list. Call this once before first use. */
void ecat63InitMatlist(MATRIXLIST *mlist)
{
  mlist->matrixSpace=mlist->matrixNr=0; mlist->matdir=NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for ECAT matrix list */
void ecat63EmptyMatlist(MATRIXLIST *mlist)
{
  if(mlist->matrixSpace>0) free((char*)(mlist->matdir));
  mlist->matrixSpace=mlist->matrixNr=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ECAT matrix list.
    Matrix list must be initiated (once) before calling this.
\return Returns 0 if ok, >0 in case of an error.
 */
int ecat63ReadMatlist(FILE *fp, MATRIXLIST *ml)
{
  int i, err=0, little;
  int blk=MatFirstDirBlk, next_blk=0, nr_free, prev_blk, nr_used;
  size_t sn;
  unsigned int dirbuf[MatBLKSIZE/4];


  if(ECAT63_TEST) printf("ecat63ReadMatlist(fp, mlist)\n");
  if(fp==NULL) return(1);
  little=little_endian();
  /* Make sure that matrix list is empty */
  ecat63EmptyMatlist(ml);
  /* Seek the first list block */
  fseek(fp, (blk-1)*MatBLKSIZE, SEEK_SET); if(ftell(fp)!=(blk-1)*MatBLKSIZE) return(2);
  do {
    /* Read the data block */
    if(ECAT63_TEST) printf("  reading dirblock %d\n", blk);
    sn=fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp); if(sn<MatBLKSIZE/4) return(3);
    /* Allocate (more) memory for one block */
    if(ml->matrixSpace==0) {
      ml->matrixSpace=MatBLKSIZE/4;
      ml->matdir=(MatDir*)malloc(ml->matrixSpace*sizeof(MatDir));
    } else if(ml->matrixSpace<(ml->matrixNr+MatBLKSIZE/4)) {
      ml->matrixSpace+=MatBLKSIZE/4;
      ml->matdir=(MatDir*)realloc(ml->matdir, sizeof(MatDir)*ml->matrixSpace);
    }
    if(ml->matdir==NULL) return(4);
    /* Byte order conversion for ints in big endian platforms */
    if(!little) swawbip(dirbuf, MatBLKSIZE);
    /* Read "header" integers */
    nr_free  = dirbuf[0];
    next_blk = dirbuf[1];
    prev_blk = dirbuf[2];
    nr_used  = dirbuf[3];
    /*printf("nr_free=%d next_blk=%d prev_blk=%d nr_used=%d\n", nr_free, next_blk, prev_blk, nr_used);*/
    for(i=4; i<MatBLKSIZE/4; i+=4) if(dirbuf[i]>0) {
      ml->matdir[ml->matrixNr].matnum=dirbuf[i];
      ml->matdir[ml->matrixNr].strtblk=dirbuf[i+1];
      ml->matdir[ml->matrixNr].endblk=dirbuf[i+2];
      ml->matdir[ml->matrixNr].matstat=dirbuf[i+3];
      /*
      printf("matnum=%d strtblk=%d endblk=%d matstat=%d matrixNr=%d\n",
          ml->matdir[ml->matrixNr].matnum, ml->matdir[ml->matrixNr].strtblk,
          ml->matdir[ml->matrixNr].endblk, ml->matdir[ml->matrixNr].matstat,
          ml->matrixNr);
      */
      ml->matrixNr++;
    }
    blk=next_blk;
    /* Seek the next list block */
    fseek(fp, (blk-1)*MatBLKSIZE, SEEK_SET); if(ftell(fp)!=(blk-1)*MatBLKSIZE) err=1;
  } while(err==0 && feof(fp)==0 && blk!=MatFirstDirBlk);
  if(err) {ecat63EmptyMatlist(ml); return(5);}
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print ECAT matrix list on stdout. */
void ecat63PrintMatlist(MATRIXLIST *ml)
{
  int i;
  Matval matval;

  printf("     matrix   pl  fr gate bed startblk blknr\n");
  for(i=0; i<ml->matrixNr; i++) {
    mat_numdoc(ml->matdir[i].matnum, &matval);
    printf("%4d %8d %3d %3d %3d %3d %8d %3d\n", i+1, ml->matdir[i].matnum,
      matval.plane, matval.frame, matval.gate, matval.bed,
      ml->matdir[i].strtblk, 1+ml->matdir[i].endblk-ml->matdir[i].strtblk);
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Prepare matrix list for additional matrix data and return block number
 *  for matrix header, or <0 in case of an error.
 *  Directory records are written in big endian byte order.
 *  Set block_nr to the number of data blocks excluding header;
 */
int ecat63Matenter(FILE *fp, int matnum, int blkNr)
{
  unsigned int i=0, dirblk, little, busy=1, nxtblk=0, oldsize;
  unsigned int dirbuf[MatBLKSIZE/4];

  if(ECAT63_TEST) printf("ecat63Matenter(fp, %d, %d)\n", matnum, blkNr);
  if(fp==NULL || matnum<1 || blkNr<1) return(0);
  little=little_endian(); memset(dirbuf, 0, MatBLKSIZE);
  /* Read first matrix list block */
  dirblk=MatFirstDirBlk;
  fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
  if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(0);
  if(fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(0);
  /* Byte order conversion for ints in big endian platforms */
  if(!little) swawbip(dirbuf, MatBLKSIZE);

  while(busy) {
    nxtblk=dirblk+1;
    for(i=4; i<MatBLKSIZE/4; i+=4) {
      if(dirbuf[i]==0) {  /* Check for end of matrix list */
        busy=0; break;
      } else if(dirbuf[i]==matnum) {  /* Check if this matrix already exists */
        oldsize=dirbuf[i+2]-dirbuf[i+1]+1;
        if(oldsize<blkNr) {  /* If old matrix is smaller */
          dirbuf[i] = 0xFFFFFFFF;
          if(!little) swawbip(dirbuf, MatBLKSIZE);
          fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
          if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(0);
          if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(0);
          if(!little) swawbip(dirbuf, MatBLKSIZE);
          nxtblk=dirbuf[i+2]+1;
        } else { /* old matrix size is ok */
          nxtblk=dirbuf[i+1]; dirbuf[0]++; dirbuf[3]--; busy=0;
          break;
        }
      } else  /* not this one */
        nxtblk=dirbuf[i+2]+1;
    }
    if(!busy) break;
    if(dirbuf[1]!=MatFirstDirBlk) {
      dirblk=dirbuf[1];
      fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(0);
      if(fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(0);
      if(!little) swawbip(dirbuf, MatBLKSIZE);
    } else {
      dirbuf[1]=nxtblk;
      if(!little) swawbip(dirbuf, MatBLKSIZE);
      fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(0);
      if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(0);
      dirbuf[0]=31; dirbuf[1]=MatFirstDirBlk; dirbuf[2]=dirblk;
      dirbuf[3]=0; dirblk=nxtblk;
      for(i=4; i<MatBLKSIZE/4; i++) dirbuf[i]=0;
    }
  }
  dirbuf[i]=matnum;
  dirbuf[i+1]=nxtblk;
  dirbuf[i+2]=nxtblk+blkNr;
  dirbuf[i+3]=1;
  dirbuf[0]--;
  dirbuf[3]++;
  if(!little) swawbip(dirbuf, MatBLKSIZE);
  fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET); if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(0);
  if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(0);
  if(ECAT63_TEST) printf("returning %d from ecat63Matenter()\n", nxtblk);
  return(nxtblk);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the matrix identifier */
int mat_numcod(int frame, int plane, int gate, int data, int bed)
{
  return((frame&0xFFF)|((bed&0xF)<<12)|((plane&0xFF)<<16)|
         ((gate&0x3F)<<24)|((data&0x3)<<30));
}
/** Conversion of matrix identifier to numerical values */
void mat_numdoc(int matnum, Matval *matval)
{
  matval->frame = matnum&0xFFF;
  matval->plane = (matnum>>16)&0xFF;
  matval->gate  = (matnum>>24)&0x3F;
  matval->data  = (matnum>>30)&0x3;
  matval->bed   = (matnum>>12)&0xF;
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort matrixlist by plane and frame */
void ecat63SortMatlistByPlane(MATRIXLIST *ml)
{
  int i, j;
  Matval mv1, mv2;
  MatDir tmpMatdir;

  for(i=0; i<ml->matrixNr-1; i++) {
    mat_numdoc(ml->matdir[i].matnum, &mv1);
    for(j=i+1; j<ml->matrixNr; j++) {
      mat_numdoc(ml->matdir[j].matnum, &mv2);
      if(mv2.plane<mv1.plane||(mv2.plane==mv1.plane&&mv2.frame<mv1.frame)) {
        tmpMatdir=ml->matdir[i];
        ml->matdir[i]=ml->matdir[j]; ml->matdir[j]=tmpMatdir;
        mat_numdoc(ml->matdir[i].matnum, &mv1);
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort matrixlist by frame and plane */
void ecat63SortMatlistByFrame(MATRIXLIST *ml)
{
  int i, j;
  Matval mv1, mv2;
  MatDir tmpMatdir;

  for(i=0; i<ml->matrixNr-1; i++) {
    mat_numdoc(ml->matdir[i].matnum, &mv1);
    for(j=i+1; j<ml->matrixNr; j++) {
      mat_numdoc(ml->matdir[j].matnum, &mv2);
      if(mv2.frame<mv1.frame||(mv2.frame==mv1.frame&&mv2.plane<mv1.plane)) {
        tmpMatdir=ml->matdir[i];
        ml->matdir[i]=ml->matdir[j]; ml->matdir[j]=tmpMatdir;
        mat_numdoc(ml->matdir[i].matnum, &mv1);
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Checks that all matrixlist entries have read/write status.
\return Returns 0 if ok, or 1 if an entry is marked as deleted or unfinished.
 */
int ecat63CheckMatlist(MATRIXLIST *ml)
{
  int i;

  if(ml==NULL) return(1);
  for(i=0; i<ml->matrixNr; i++) if(ml->matdir[i].matstat!=1) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Mark deleted the frames after the specified frame number.
    This can be used to delete sum images from the end of dynamic ECAT images.
\return Returns the number of deleted matrices.
 */
int ecat63DeleteLateFrames(MATRIXLIST *ml, int frame_nr)
{
  int i, del_nr=0;
  Matval matval;

  for(i=0; i<ml->matrixNr; i++) {
    mat_numdoc(ml->matdir[i].matnum, &matval);
    if(matval.frame>frame_nr) {del_nr++; ml->matdir[i].matstat=-1;}
  }
  return(del_nr);
}
/*****************************************************************************/

/*****************************************************************************/

