/******************************************************************************

  ecat7ml.c  (c) 2003,2004 by Turku PET Centre

  Procedures for reading and writing ECAT 7.x matrix list.

  Version:
  2003-07-21 Vesa Oikonen
  2004-06-20 VO
    ecat7PrintMatlist(): blkNr is printed correctly (+1).
  2004-06-27 VO
    Included ecat7DeleteLateFrames().

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
/*****************************************************************************/
#include "include/swap.h"
#include "include/ecat7.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate ECAT matrix list. Call this once before first use. */
void ecat7InitMatlist(ECAT7_MATRIXLIST *mlist)
{
  mlist->matrixSpace=mlist->matrixNr=0; mlist->matdir=NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for ECAT matrix list. */
void ecat7EmptyMatlist(ECAT7_MATRIXLIST *mlist)
{
  if(mlist->matrixSpace>0) free((char*)(mlist->matdir));
  mlist->matrixSpace=mlist->matrixNr=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ECAT matrix list.
    Matrix list must be initiated (once) before calling this.
\return Returns 0 if ok, or >0 in case of an error.
 */
int ecat7ReadMatlist(FILE *fp, ECAT7_MATRIXLIST *ml)
{
  int i, err=0, little;
  int blk=MatFirstDirBlk, next_blk=0, nr_free, prev_blk, nr_used;
  size_t sn;
  unsigned int dirbuf[MatBLKSIZE/4];


  if(ECAT7_TEST) printf("ecat7ReadMatlist(fp, mlist)\n");
  if(fp==NULL) return(1);
  little=little_endian();
  /* Make sure that matrix list is empty */
  ecat7EmptyMatlist(ml);
  /* Seek the first list block */
  fseek(fp, (blk-1)*MatBLKSIZE, SEEK_SET); if(ftell(fp)!=(blk-1)*MatBLKSIZE) return(2);
  do {
    /* Read the data block */
    if(ECAT7_TEST) printf("  reading dirblock %d\n", blk);
    sn=fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp); if(sn<MatBLKSIZE/4) return(3);
    /* Allocate (more) memory for one block */
    if(ml->matrixSpace==0) {
      ml->matrixSpace=MatBLKSIZE/4;
      ml->matdir=(ECAT7_MatDir*)malloc(ml->matrixSpace*sizeof(ECAT7_MatDir));
    } else if(ml->matrixSpace<(ml->matrixNr+MatBLKSIZE/4)) {
      ml->matrixSpace+=MatBLKSIZE/4;
      ml->matdir=(ECAT7_MatDir*)realloc(ml->matdir, sizeof(ECAT7_MatDir)*ml->matrixSpace);
    }
    if(ml->matdir==NULL) return(4);
    /* Byte order conversion for ints in little endian platforms */
    if(little) swawbip(dirbuf, MatBLKSIZE);
    /* Read "header" integers */
    nr_free  = dirbuf[0];
    next_blk = dirbuf[1];
    prev_blk = dirbuf[2];
    nr_used  = dirbuf[3];
    if(ECAT7_TEST>1) printf("nr_free=%d next_blk=%d prev_blk=%d nr_used=%d\n", nr_free, next_blk, prev_blk, nr_used);
    for(i=4; i<MatBLKSIZE/4; i+=4) if(dirbuf[i]>0) {
      ml->matdir[ml->matrixNr].id=dirbuf[i];
      ml->matdir[ml->matrixNr].strtblk=dirbuf[i+1];
      ml->matdir[ml->matrixNr].endblk=dirbuf[i+2];
      ml->matdir[ml->matrixNr].status=dirbuf[i+3];
      if(ECAT7_TEST>3) {
        printf("matnum=%d strtblk=%d endblk=%d matstat=%d matrixNr=%d\n",
          ml->matdir[ml->matrixNr].id, ml->matdir[ml->matrixNr].strtblk,
          ml->matdir[ml->matrixNr].endblk, ml->matdir[ml->matrixNr].status,
          ml->matrixNr);
      }
      ml->matrixNr++;
    }
    blk=next_blk;
    /* Seek the next list block */
    fseek(fp, (blk-1)*MatBLKSIZE, SEEK_SET); if(ftell(fp)!=(blk-1)*MatBLKSIZE) err=1;
  } while(err==0 && feof(fp)==0 && blk!=MatFirstDirBlk);
  if(err) {ecat7EmptyMatlist(ml); return(5);}
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print ECAT matrix list on stdout. */
void ecat7PrintMatlist(ECAT7_MATRIXLIST *ml)
{
  int i;
  ECAT7_Matval matval;

  printf("     matrix   pl  fr gate bed startblk blknr  status\n");
  for(i=0; i<ml->matrixNr; i++) {
    ecat7_id_to_val(ml->matdir[i].id, &matval);
    printf("%4d %8d %3d %3d %3d %3d %8d %5d  ", i+1, ml->matdir[i].id,
      matval.plane, matval.frame, matval.gate, matval.bed,
      ml->matdir[i].strtblk, 1+ml->matdir[i].endblk-ml->matdir[i].strtblk);
    if(ml->matdir[i].status==1) printf("read/write\n");
    else if(ml->matdir[i].status==0) printf("not ready\n");
    else if(ml->matdir[i].status==-1) printf("deleted\n");
    else printf("%d\n", ml->matdir[i].status);
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Prepare matrix list for additional matrix data and return block number
    for matrix header.
    Directory records are written in big endian byte order.
    Set block_nr to the number of data blocks + (nr of header blocks - 1)
\return Returns the block number for matrix header, or <0 in case of an error.
 */
int ecat7EnterMatrix(FILE *fp, int matrix_id, int block_nr)
{
  unsigned int i=0, dirblk, little, busy=1, nxtblk=0, oldsize;
  /*unsigned*/ int dirbuf[MatBLKSIZE/4];

  if(ECAT7_TEST) printf("ecat7EnterMatrix(fp, %d, %d)\n", matrix_id, block_nr);
  /* Check the input */
  if(fp==NULL || matrix_id<1 || block_nr<1) return(-1);
  /* Is this a little endian machine? */
  little=little_endian();
  /* Read first directory record block */
  dirblk=MatFirstDirBlk;
  fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
  if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(-2);
  if(fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(-3);
  /* Byte order conversion for ints in little endian platforms */
  if(little) swawbip(dirbuf, MatBLKSIZE);
  /* Read through the existing directory records */
  while(busy) {
    /* Go through the directory entries in this record */
    for(i=4, nxtblk=dirblk+1; i<MatBLKSIZE/4; i+=4) {
      oldsize=dirbuf[i+2]-dirbuf[i+1]+1;
      if(dirbuf[i]==0) {  /* Check for end of matrix list */
        busy=0; break;
      } else if(dirbuf[i]==matrix_id) {  /* Maybe this matrix already exists? */
        /* yes it does; is old data smaller? */
        if(oldsize<block_nr) {
          /* it was smaller, so do not use it, but mark it deleted */
          dirbuf[i] = 0xFFFFFFFF; dirbuf[i+3]=-1;
          if(little) swawbip(dirbuf, MatBLKSIZE);
          fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
          if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(-6);
          if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(-7);
          if(little) swawbip(dirbuf, MatBLKSIZE);
          nxtblk=dirbuf[i+2]+1;
        } else { /* old matrix size is ok */
          nxtblk=dirbuf[i+1]; dirbuf[0]++; dirbuf[3]--; busy=0;
          break;
        }
      } else { /* this is not the same matrix */
        /* But is deleted and of same or smaller size? */
        if(dirbuf[i+3]==-1 && block_nr<=oldsize) {
          /* yes it was, so lets recycle it */
          dirbuf[i]=matrix_id;
          nxtblk=dirbuf[i+1]; dirbuf[0]++; dirbuf[3]--; busy=0;
          break;
        }
        /* nothing to be done with this entry */
        nxtblk=dirbuf[i+2]+1;
      }
    } /* next entry in this record */
    if(!busy) break; /* stop reading existing records */
    /* Read the next directory record */
    if(dirbuf[1]!=MatFirstDirBlk) {
      /* There are more records left to read */
      dirblk=dirbuf[1];
      fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(-9);
      if(fread(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(-10);
      if(little) swawbip(dirbuf, MatBLKSIZE);
    } else {
      /* No more records to read, so lets write a new empty one */
      dirbuf[1]=nxtblk; /* write a pointer to the new one */
      if(little) swawbip(dirbuf, MatBLKSIZE);
      fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
      if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(-11);
      if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(-12);
      /* and then initiate the contents of the next one, but do not write it */
      dirbuf[0]=31; dirbuf[1]=MatFirstDirBlk; dirbuf[2]=dirblk;
      dirbuf[3]=0; dirblk=nxtblk;
      for(i=4; i<MatBLKSIZE/4; i++) dirbuf[i]=0;
    }
  } /* next directory record */
  dirbuf[i]=matrix_id;
  dirbuf[i+1]=nxtblk;
  dirbuf[i+2]=nxtblk+block_nr;
  dirbuf[i+3]=1; /* mark the entry as read/write */
  dirbuf[0]--;
  dirbuf[3]++;
  if(little) swawbip(dirbuf, MatBLKSIZE);
  fseek(fp, (dirblk-1)*MatBLKSIZE, SEEK_SET);
  if(ftell(fp)!=(dirblk-1)*MatBLKSIZE) return(-15);
  if(fwrite(dirbuf, sizeof(int), MatBLKSIZE/4, fp) != MatBLKSIZE/4) return(-16);
  if(ECAT7_TEST) printf("returning %d from ecat7EnterMatrix()\n", nxtblk);
  return(nxtblk);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the matrix identifier */
int ecat7_val_to_id(int frame, int plane, int gate, int data, int bed)
{
  return(
    ((bed & 0xF) << 12) |    /* bed */
    (frame & 0x1FF) |        /* frame */
    ((gate & 0x3F) << 24) |  /* gate */
    ((plane & 0xFF) << 16) | /* plane low */
    ((plane & 0x300) << 1) | /* plane high */
    ((data & 0x3) << 30) |   /* data low */
    ((data & 0x4) << 9)      /* data high */
  );
}
/** Conversion of matrix identifier to numerical values */
void ecat7_id_to_val(int matrix_id, ECAT7_Matval *matval)
{
  matval->frame = matrix_id & 0x1FF;
  matval->plane = ((matrix_id >> 16) & 0xFF) + ((matrix_id >> 1) & 0x300);
  matval->gate  = (matrix_id >> 24) & 0x3F;
  matval->data  = ((matrix_id >> 30) & 0x3) + ((matrix_id >> 9) & 0x4);
  matval->bed   = (matrix_id >> 12) & 0xF;
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort matrixlist by plane and frame */
void ecat7SortMatlistByPlane(ECAT7_MATRIXLIST *ml)
{
  int i, j;
  ECAT7_Matval mv1, mv2;
  ECAT7_MatDir tmpMatdir;

  for(i=0; i<ml->matrixNr-1; i++) {
    ecat7_id_to_val(ml->matdir[i].id, &mv1);
    for(j=i+1; j<ml->matrixNr; j++) {
      ecat7_id_to_val(ml->matdir[j].id, &mv2);
      if(mv2.plane<mv1.plane||(mv2.plane==mv1.plane&&mv2.frame<mv1.frame)) {
        tmpMatdir=ml->matdir[i];
        ml->matdir[i]=ml->matdir[j]; ml->matdir[j]=tmpMatdir;
        ecat7_id_to_val(ml->matdir[i].id, &mv1);
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort matrixlist by frame and plane */
void ecat7SortMatlistByFrame(ECAT7_MATRIXLIST *ml)
{
  int i, j;
  ECAT7_Matval mv1, mv2;
  ECAT7_MatDir tmpMatdir;

  for(i=0; i<ml->matrixNr-1; i++) {
    ecat7_id_to_val(ml->matdir[i].id, &mv1);
    for(j=i+1; j<ml->matrixNr; j++) {
      ecat7_id_to_val(ml->matdir[j].id, &mv2);
      if(mv2.frame<mv1.frame||(mv2.frame==mv1.frame&&mv2.plane<mv1.plane)) {
        tmpMatdir=ml->matdir[i];
        ml->matdir[i]=ml->matdir[j]; ml->matdir[j]=tmpMatdir;
        ecat7_id_to_val(ml->matdir[i].id, &mv1);
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Checks that all matrixlist entries have read/write status.
\return Returns 0 if ok, or 1 if an entry is marked as deleted or unfinished.
 */
int ecat7CheckMatlist(ECAT7_MATRIXLIST *ml)
{
  int i;

  if(ml==NULL) return(1);
  for(i=0; i<ml->matrixNr; i++) if(ml->matdir[i].status!=1) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Mark deleted the frames after the specified frame number.
\return Returns the number of deleted matrices.
 */
int ecat7DeleteLateFrames(ECAT7_MATRIXLIST *ml, int frame_nr)
{
  int i, del_nr=0;
  ECAT7_Matval matval;

  for(i=0; i<ml->matrixNr; i++) {
    ecat7_id_to_val(ml->matdir[i].id, &matval);
    if(matval.frame>frame_nr) {del_nr++; ml->matdir[i].status=-1;}
  }
  return(del_nr);
}
/*****************************************************************************/

/*****************************************************************************/

