/*****************************************************************************

  Copyright (c) 2003-2005 by Turku PET Centre

  File:        halflife.c
  Description: Contains functions for processing isotope half-life and
               decay correction.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2003-07-07 Vesa Oikonen
  2004-01-28 VO
    hlFromIsotope() identifies Cu, Br and Rb.
    Function info text format is changed.
  2004-09-17 VO
    Doxygen style comments.
  2005-01-21 VO
    Added isotope code list.
    Added function hlCorrectIsotopeCode().


*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
/****************************************************************************/
#include "include/halflife.h"
/****************************************************************************/

/****************************************************************************/
/** Valid isotope codes. Note: when adding isotopes, make sure that all
    isotopes with one letter are AFTER all two letter isotopes with the
    same initial letter.
 */
static char *isotope_code[] = {
  "Br-75", "Br-76", "Cu-62", "Cu-64", "Fe-52",
  "Ga-68", "Ge-68", "Na-22", "Rb-82", "Zn-62",
  "F-18",  "C-11", "N-13",  "O-15", "O-14",
0};
/****************************************************************************/

/****************************************************************************/
/** Identify the isotope from the specified isotope code string and
    return the halflife (min).
\return A negative value is returned in case of error.
*/
double hlFromIsotope(
  /** Pointer to string "C-11", "18f" etc. */
  char *isocode
) {
  char *cptr;
  double halflife=-1.0;
  
  /* Convert the isotope code to uppercase letters */
  cptr=isocode; while(*cptr!=(char)0) {*cptr=toupper(*cptr); cptr++;}

  /* Check if code starts with number */
  cptr=isocode; while(isdigit((int)cptr[0])) cptr++;
  
  /* and now that we hopefully are at the beginning of the atom */
  if(     strncmp(cptr, "BR", 2)==0) halflife=HL_Br76;
  else if(strncmp(cptr, "CU", 2)==0) halflife=HL_Cu62;
  else if(strncmp(cptr, "C",  1)==0) halflife=HL_C11;
  else if(strncmp(cptr, "GA", 2)==0) halflife=HL_Ga68;
  else if(strncmp(cptr, "GE", 2)==0) halflife=HL_Ge68;
  else if(strncmp(cptr, "F",  1)==0) halflife=HL_F18;
  else if(strncmp(cptr, "N",  1)==0) halflife=HL_N13;
  else if(strncmp(cptr, "O",  1)==0) halflife=HL_O15;
  else if(strncmp(cptr, "RB", 2)==0) halflife=HL_Rb82;

  return(halflife);
} 
/****************************************************************************/

/****************************************************************************/
/** Calculates the isotope lambda from specified halflife.
\return A negative value is returned in case of error.
  */
double hl2lambda(double halflife)
{
//  if(halflife>0.0) return(log(2.0)/halflife); else return(-1.0);
  if(halflife>0.0) return(M_LN2/halflife); else return(-1.0);
}
/****************************************************************************/

/****************************************************************************/
/** Calculate the decay correction factor for specified isotope lambda.
\return A negative value is returned in case of error.
*/
double lambda2factor(
  /** Negative lambda removes decay correction */
  double lambda,
  /** Frame start time, or mid time if framedur<=0 */
  double frametime,
  /** If unknown, set <0 and give mid time for frametime */
  double framedur
) {
  double cf;

  if(frametime<0) return(-1.0);
  cf=exp(lambda*frametime);
  if(framedur>1.0E-5) cf*=lambda*framedur/(1.0-exp(-lambda*framedur));
  return(cf);
}  
/****************************************************************************/

/****************************************************************************/
/** Check that isotope code, e.g. F-18, is in valid format, containing '-'
    and in this order. Returns the correct isotope code.
\return Returns pointer to correct isotope code, and NULL if it was not
    valid and could not be corrected.
 */
char *hlCorrectIsotopeCode(
  /** Pointer to string "C-11", "11c" etc; contents of this string is not
      changed, and this is not returned in any case */
  char *isocode
) {
  int i, ok=0, n;
  char *cptr;

  /* Check, if isocode can be found in the list */
  i=ok=0;
  while(isotope_code[i]!=0) {
    if(strcmp(isotope_code[i], isocode)==0) {ok=1; break;}
    i++;
  }
  if(ok==1) return(isotope_code[i]);

  /* Try to figure out what it is */
  /* check if code starts with number */
  cptr=isocode; while(isdigit((int)cptr[0])) cptr++;
  /* and now that we hopefully are at the beginning of the atom */
  i=ok=0;
  while(isotope_code[i]!=0) {
    n=strcspn(isotope_code[i], "-");
    if(strncasecmp(isotope_code[i], cptr, n)==0) {ok=1; break;}
    i++;
  }
  if(ok==1) return(isotope_code[i]); else return(NULL);
}
/****************************************************************************/

/****************************************************************************/

