/***************************************************************************
  Copyright (c) 2003,2004 by Turku PET Centre
 
  hholder.c
 
  This file containes the routines needed for the implementation and
  use of Householder transform.
 
  These routines are based on the code 
  provided in the GSL library (http://sources.redhat.com/gsl/)

  Version:
  2003-08-28 Kaisa Sederholm
  2003-09-11 KS
    Removed memory allocations 
  2004-09-17 Vesa Oikonen
    Doxygen style comments.

 ***************************************************************************/
#include <stdlib.h>
#include <math.h>
#include "include/hholder.h"
/****************************************************************************/


/** This function prepares a Householder transformation
  P = I - tau h h^T 
  which can be used to zero all the elements of the input vector 
  except the first one that will get value beta. 
  On output the elements 1 - size-1 of the vector h are stored 
  in locations vector[1] - vector[size-1] of the input vector
  and value of beta is stored in location vector[0]. 
\return The scalar tau is returned.
*/
double
householder_transform(
  /** The N-vector to be transformed */
  double *vector,
  /** size of the vector */
  int N,
  /** N-array of working space */
  double *help
) {
  double vnorm, alpha, beta, tau;
  double *subvector;
  int n;

  if (N == 1) {
    return 0.0; /* tau = 0 */
  } else {

    /* Local variable */
    subvector=help;

    for(n=1; n<N; n++) {
      subvector[n-1]=vector[n];
    }

    /* Euclidean norm of the subvector */
    vnorm = householder_norm(subvector, N-1);

    if (vnorm == 0) {
      return 0.0; /* tau = 0 */
    }

    /* Computing the coefficient tau */
    alpha = vector[0];
    beta = - (alpha >= 0.0 ? +1.0 : -1.0) * hypot(alpha, vnorm) ;
    tau = (beta - alpha) / beta ;

    /* Scale the Householder vector so that the first
     * element will be 1. (Scaling is also affecting the 
     * coefficient tau.) Without scaling, the first element
     * would have value (alpha - beta). 
     */

    for(n=1; n<N; n++) {
      vector[n]=(1.0 / (alpha - beta))*vector[n];
    }

    vector[0]=beta;

    return tau;
  }
}



/** applies a householder transformation defined by vector "vector"
 * and scalar tau to the left-hand side of the matrix. 
 * (I - tau vector vector^T)*matrix
 * The result of the transform is stored in matrix. 
\return Returns 0 if ok.
 */

int
householder_hm (
  /** Coefficient defining householder transform */
  double tau,
  /** Vector defining householder transform */
  double *vector,
  /** the matrix that is to be transformed */
  double **matrix,
  /** Nr of rows in matrix */
  int rowNr,
  /** Nr of columns in matrix */
  int columnNr
) {
  int i, j;
  double wj, A0j, Aij, vi;

  if (tau == 0.0) {
    return 0;
  }

  for (j = 0; j < columnNr; j++) {
    /* Compute wj = vk Akj */

    wj = matrix[0][j];

    for (i = 1; i < rowNr; i++)  /* note, computed for v(0) = 1 above */
    {
      wj += vector[i] * matrix[i][j];
    }

    /* Aij = Aij - tau vi wj */

    /* i = 0 */

    A0j = matrix[0][j];
    matrix[0][j]= A0j - tau *  wj;


    /* i = 1 .. M-1 */

    for (i = 1; i < rowNr; i++) {
      Aij = matrix[i][j];
      vi = vector[i];
      matrix[i][j]= Aij - tau * vi * wj;
    }
  }

  return 0;
}



/** applies a householder transformation defined by vector v
 * and coeffiecient tau to vector w 
 * w = (I - tau v v^T) w 
\return Returns 0 if ok.
 */

int
householder_hv(
  double tau,
  int size,
  double *v,
  double *w
) {
  int i;
  double d = 0, w0, wi, vi;

  if (tau == 0)
    return 0;

  /* d = v'w */

  d = w[0];

  for (i = 1 ; i < size ; i++) {
    d += v[i] * w[i];
  }

  /* w = w - tau (v) (v'w) */

  {
    w0 = w[0];
    w[0]= w0 - tau * d;
  }

  for (i = 1; i < size ; i++) {
    wi = w[i];
    vi = v[i];
    w[i] = wi - tau * vi * d;
  }

  return 0;
}


/** Returns the euclidean norm of a vector */
double householder_norm(double *v, int size) {

  double help;
  int i;

  for(i=0, help=0; i<size; i++) {
    help+=v[i]*v[i];
  }
  return sqrt(help);
}

