/******************************************************************************
  Copyright (c) 2002-2004 by Turku PET Centre

  img.c
  
  Procedures for
  - storing and processing 4D PET image data
  
  Known bugs:

  Version:
  2002-01-20 Vesa Oikonen
  2002-03-28 VO
    sampleDistance included in IMG structure.
    Included function imgDecayCorrection().
  2002-07-30 VO
    memset() added to imgInit().
  2002-08-23 VO
    _dataType included in IMG structure in img.h.
  2002-12-01 VO
    imgDecayCorrection() can be used also to remove decay correction.
  2002-12-03 VO
    Included function imgCopyhdr().
  2002-12-23 VO
    patientName included in IMG structure.
    imgIsotope() included.
    Decay correction factors are saved and used in imgDecayCorrection().
  2003-09-04 VO
    _fileFormat included in IMG structure in img.h.
  2003-11-04 VO
    Added unit nCi/mL.
  2003-12-14 VO
    Memory for all pixels is allocated in one chunk, because:
    -faster when disk swapping is necessary
    -pixel data can be easily saved in tmpfile
    -whole data set can be easily processed with one pointer.
    Unnecessary includes were removed.
    (Patient) orientation included in IMG structure in img.h.
    Scanner (type) included in IMG structure in img.h.
  2003-12-18 VO
    Included function imgExtractRange().
  2004-05-23 VO
    Added unit MBq/ml in img_unit().
    Added a few comments.
  2004-06-21 VO
    Previous addition should now work.
  2004-08-23 VO
    New library function studynr_from_fname() is applied.
    MAX_STUDYNR_LEN applied where appropriate.
  2004-09-20 VO
    Doxygen style comments are corrected.
  2004-09-24 VO
    Added units Bq/cc and uCi/cc in img_unit().


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "include/halflife.h"
#include "include/studynr.h"
#include "include/img.h"
/*****************************************************************************/
/** Status (error) messages from image processing */
char *_imgStatusMessage[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory"
};
/*****************************************************************************/

/*****************************************************************************/
/** Call this once before any use of IMG data. */
void imgInit(IMG *image)
{
  if(IMG_TEST) printf("imgInit()\n");
  memset(image, 0, sizeof(IMG));
  /*if(image->status!=IMG_STATUS_UNINITIALIZED) return;*/
  image->status=IMG_STATUS_INITIALIZED;
  image->statmsg=_imgStatusMessage[0];
  image->type=0;
  image->unit=0;
  image->zoom=0.0;
  image->radiopharmaceutical[0]=(char)0;
  image->isotopeHalflife=0.0;
  image->decayCorrected=(char)0;
  image->unit=0;
  image->scanStart=0;
  image->orientation=0;
  image->axialFOV=image->transaxialFOV=image->sampleDistance=0.0;
  image->studyNr[0]=image->patientName[0]=(char)0;
  image->sizex=image->sizey=image->sizez=0;
  image->_dataType=0;
  image->_fileFormat=0;
  image->scanner=0;
  image->dimt=image->dimx=image->dimy=image->dimz=0;
  image->m=(float****)NULL;
  image->_header=(float*)NULL;
  image->pixel=(float*)NULL;
  image->column=(float**)NULL;
  image->row=(float***)NULL;
  image->plane=(float****)NULL;
  image->planeNumber=(int*)NULL;
  image->start=image->end=image->mid=(float*)NULL;
  image->isWeight=0;
  image->weight=image->sd=(float*)NULL;
  image->decayCorrFactor=(float*)NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory that is allocated for IMG. */
void imgEmpty(IMG *image)
{
  if(IMG_TEST) printf("imgEmpty()\n");
  if(image->status<IMG_STATUS_OCCUPIED) return;
  /* Free up memory */
  if(image->_pxl!=NULL) free(image->_pxl);
  if(image->_col!=NULL) free(image->_col);
  if(image->_row!=NULL) free(image->_row);
  if(image->_pln!=NULL) free(image->_pln);
  if(image->dimz>0) free(image->planeNumber);
  if(image->dimt>0) free(image->_header);
  /* Set variables */
  image->statmsg=_imgStatusMessage[0];
  image->type=0;
  image->unit=0;
  image->zoom=0.0;
  image->radiopharmaceutical[0]=(char)0;
  image->isotopeHalflife=0.0;
  image->decayCorrected=(char)0;
  image->unit=0;
  image->scanStart=0;
  image->orientation=0;
  image->axialFOV=image->transaxialFOV=image->sampleDistance=0.0;
  image->studyNr[0]=image->patientName[0]=(char)0;
  image->sizex=image->sizey=image->sizez=0;
  image->_dataType=0;
  image->_fileFormat=0;
  image->scanner=0;
  image->dimt=image->dimx=image->dimy=image->dimz=0;
  image->m=(float****)NULL;
  image->_header=(float*)NULL;
  image->pixel=(float*)NULL;
  image->column=(float**)NULL;
  image->row=(float***)NULL;
  image->plane=(float****)NULL;
  image->planeNumber=(int*)NULL;
  image->start=image->end=image->mid=(float*)NULL;
  image->isWeight=0;
  image->weight=image->sd=(float*)NULL;
  image->decayCorrFactor=(float*)NULL;
  /* Set status */
  image->status=IMG_STATUS_INITIALIZED;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocates memory for img data. Old contents are not saved.
\return Returns 0 if ok.
 */
int imgAllocate(IMG *image, int planes, int rows, int columns, int frames)
{
  unsigned short int zi, ri, ci;
  float ***rptr, **cptr, *pptr;

  if(IMG_TEST) printf("imgAllocate(*image, %d, %d, %d, %d)\n", planes, rows, columns, frames);
  /* Check arguments */
  image->statmsg=_imgStatusMessage[1];
  if(image->status==IMG_STATUS_UNINITIALIZED) return(1);
  if(planes<1 || rows<1 || columns<1 || frames<1) return(2);
  if(image->status>=IMG_STATUS_OCCUPIED) imgEmpty(image);
  /* Allocate memory for header data */
  image->statmsg=_imgStatusMessage[2];
  image->_header=(float*)calloc(6*frames, sizeof(float));
  if(image->_header==NULL)
    return(3);
  image->planeNumber=(int*)calloc(planes, sizeof(int));
  if(image->planeNumber==NULL) {
    free(image->_header); return(4);}
  /* Allocate memory for image data */
  image->_pln=(float****)malloc(planes*sizeof(float***));
  if(image->_pln==NULL) {
    free(image->_header); free(image->planeNumber); return(5);}
  image->_row=(float***)malloc(planes*rows*sizeof(float**));
  if(image->_row==NULL) {
    free(image->_header); free(image->planeNumber); free(image->_pln); return(6);}
  image->_col=(float**)malloc(planes*rows*columns*sizeof(float*));
  if(image->_col==NULL) {
    free(image->_header); free(image->planeNumber);
    free(image->_pln); free(image->_row); return(7);
  }
  image->_pxl=(float*)calloc(planes*rows*columns*frames, sizeof(float));
  if(image->_pxl==NULL) {
    free(image->_header); free(image->planeNumber);
    free(image->_pln); free(image->_row); free(image->_col); return(8);
  }
  /* Set data pointers */
  rptr=image->_row; cptr=image->_col; pptr=image->_pxl;
  for(zi=0; zi<planes; zi++) {
    image->_pln[zi]=rptr;
    for(ri=0; ri<rows; ri++) {
      *rptr++=cptr;
      for(ci=0; ci<columns; ci++) {
        *cptr++=pptr; pptr+=frames;
      }
    }
  }
  image->m=image->_pln;
  image->plane=image->_pln;
  image->column=image->_col;
  image->row=image->_row;
  image->pixel=image->_pxl;
  /* Set header pointers */
  image->start=          image->_header+0*frames;
  image->end=            image->_header+1*frames;
  image->mid=            image->_header+2*frames;
  image->weight=         image->_header+3*frames;
  image->sd=             image->_header+4*frames;
  image->decayCorrFactor=image->_header+5*frames;
  /* Ok */
  image->dimz=planes; image->dimy=rows; image->dimx=columns; image->dimt=frames;
  image->statmsg=_imgStatusMessage[0];
  image->status=IMG_STATUS_OCCUPIED;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns pointer to string describing the calibrated image data unit */
char *imgUnit(
  /** 0="unknown", 1="cnts/sec", 2="counts", 3="kBq/mL", 4="sec*kBq/mL",
      5="1/sec", 6="1/min", 7="mL/mL", 8="mL/dL", 9="mL/(mL*min)",
      10="mL/(dL*min)", 11="unitless", 12="nCi/mL", 13="MBq/mL", 14="Bq/cc",
      15="uCi/cc" */
  int dunit
) {
  static char *img_unit[]={
  /*  0 */  "unknown",
  /*  1 */  "cnts/sec",
  /*  2 */  "counts",
  /*  3 */  "kBq/mL",
  /*  4 */  "sec*kBq/mL",
  /*  5 */  "1/sec",
  /*  6 */  "1/min",
  /*  7 */  "mL/mL",
  /*  8 */  "mL/dL",
  /*  9 */  "mL/(mL*min)",
  /* 10 */  "mL/(dL*min)",
  /* 11 */  "unitless",
  /* 12 */  "nCi/mL",
  /* 13 */  "MBq/mL",
  /* 14 */  "Bq/cc",
  /* 15 */  "uCi/cc"
  };
  if(dunit>=0 && dunit<16) return(img_unit[dunit]);
  else return(img_unit[0]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns pointer to string describing the isotope in image data */
char *imgIsotope(IMG *img)
{
  float hlmin;
  int icode=0;

  static char *img_isotope[]={
  /*  0 */  "unknown",
  /*  1 */  "O-15",
  /*  2 */  "N-13",
  /*  3 */  "C-11",
  /*  4 */  "F-18",
  /*  5 */  "Ge-68",
  /*  6 */  "Ga-68",
  /*  7 */  "Br-76",
  /*  8 */  "Rb-82",
  /*  9 */  "Cu-62"
  };
  hlmin=img->isotopeHalflife/60.;
  if(hlmin<=0.01) icode=0;
  else if( fabs(hlmin/HL_O15-1.0)<0.05 ) icode=1;
  else if( fabs(hlmin/HL_N13-1.0)<0.05 ) icode=2;
  else if( fabs(hlmin/HL_C11-1.0)<0.05 ) icode=3;
  else if( fabs(hlmin/HL_F18-1.0)<0.05 ) icode=4;
  else if( fabs(hlmin/HL_Ge68-1.0)<0.05 ) icode=5;
  else if( fabs(hlmin/HL_Ga68-1.0)<0.05 ) icode=6;
  else if( fabs(hlmin/HL_Br76-1.0)<0.05 ) icode=7;
  else if( fabs(hlmin/HL_Rb82-1.0)<0.05 ) icode=8;
  else if( fabs(hlmin/HL_Cu62-1.0)<0.05 ) icode=9;
  else icode=0;
  return(img_isotope[icode]);
}
/*****************************************************************************/

/*****************************************************************************/
/** Prints img information to stdout; mainly for testing purposes. */
void imgInfo(IMG *image)
{
  int i;
  char buf[64];
  struct tm *st;

  if(IMG_TEST) printf("imgInfo()\n");
  if(image->status<=IMG_STATUS_UNINITIALIZED) {
    fprintf(stdout, "Image data is not initialized.\n"); return;}
  if(image->status==IMG_STATUS_INITIALIZED) {
    fprintf(stdout, "Image data is initialized but empty.\n"); return;}
  if(image->status==IMG_STATUS_ERROR) fprintf(stdout, "Image data has errors.\n");
  fprintf(stdout, "Image status: %s\n", image->statmsg);
  fprintf(stdout, "Image type: %d  Saved data type: %d file format: %d Scanner: %d\n",
    image->type, image->_dataType, image->_fileFormat, image->scanner);
  fprintf(stdout, "Identification code: %.*s\n", MAX_STUDYNR_LEN, image->studyNr);
  fprintf(stdout, "Data unit: %s\n", imgUnit((int)image->unit));
  fprintf(stdout, "Image zoom: %g\n", image->zoom);
  fprintf(stdout, "Radiopharmaceutical: %.32s\n", image->radiopharmaceutical);
  fprintf(stdout, "Isotope half-life %e sec\n", image->isotopeHalflife);
  st=localtime(&image->scanStart); strftime(buf, 64, "%Y-%m-%d %T", st);
  fprintf(stdout, "Scan start: %s\n", buf);
  fprintf(stdout, "Patient orientation: %d\n", image->orientation);
  fprintf(stdout, "FOV axial: %g mm transaxial: %g Sample distance: %g\n",
    image->axialFOV, image->transaxialFOV, image->sampleDistance);
  fprintf(stdout, "Pixel sizes (x, y, z): %g %g %g mm\n",
    image->sizex, image->sizey, image->sizez);
  fprintf(stdout, "Dimensions (x, y, z, t): %d  %d %d %d\n",
    image->dimx, image->dimy, image->dimz, image->dimt);
  fprintf(stdout, "Actual plane number(s): %d", image->planeNumber[0]);
  for(i=1; i<image->dimz; i++) fprintf(stdout, ", %d", image->planeNumber[i]);
  fprintf(stdout, "\n");
  fprintf(stdout, "Frame times (sec)        Weights and SD:\n");
  for(i=0; i<image->dimt; i++) fprintf(stdout, "  %e %e %e    %e %e\n",
    image->start[i], image->end[i], image->mid[i],
    image->weight[i], image->sd[i]);
  if(image->isWeight) fprintf(stdout, "Frames are weighted.\n");
  else fprintf(stdout, "Frames are not weighted.\n");
  if(image->decayCorrected==1) {
    fprintf(stdout, "Decay correction factors for each frame:\n");
    for(i=0; i<image->dimt; i++)
      fprintf(stdout, "%03i  %e\n", i+1, image->decayCorrFactor[i]);
  } else
    fprintf(stdout, "Image is not decay corrected.\n");
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Corrects (mode=1) or removes correction (mode=0) for physical decay.
 *  Removal is based on existing decay correction factors, if possible.
\return Returns 0, if ok.
 */
int imgDecayCorrection(
  /** Pointer to IMG data */
  IMG *image,
  /** 0=Remove decay correction; 1=Correct for decay */
  int mode
) {
  int pi, fi, i, j;
  float lambda, cf, dur;

  /* Check for arguments */
  if(image->status!=IMG_STATUS_OCCUPIED) return(1);
  if(image->isotopeHalflife<=0.0) return(1);
  /* Existing/nonexisting decay correction is an error */
  if(mode==1 && image->decayCorrected!=0) return(2);
  if(mode==0 && image->decayCorrected==0) return(2);

  /* All time frames */
  for(fi=0; fi<image->dimt; fi++) {
    dur=image->end[fi]-image->start[fi];
    if(image->end[fi]>0.0) {
      lambda=M_LN2/image->isotopeHalflife;
      cf=1.0;
      if(image->start[fi]>1.0e-3) cf*=exp(lambda*image->start[fi]);
      if(dur>1.0e-3) cf*=lambda*dur/(1.0-exp(-lambda*dur));
      if(IMG_TEST) printf("Frame %03d %.0f-%.0f Decay correction: %g\n",
        fi+1, image->start[fi], image->end[fi], cf);
      if(mode==0) {
        if(image->decayCorrFactor[fi]>1.000001) cf=image->decayCorrFactor[fi];
        cf=1.0/cf;
        image->decayCorrFactor[fi]=1.0;
      } else {
        image->decayCorrFactor[fi]=cf;
      }
      /* All planes, all matrices */
      for(pi=0; pi<image->dimz; pi++)
        for(i=0; i<image->dimy; i++)
          for(j=0; j<image->dimx; j++)
            image->m[pi][i][j][fi]*=cf;
      image->decayCorrected=mode; /* in some cases left unchanged! */
    }
  } /* next frame */
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Copies the header fields from one image struct to another.
 *  Does not copy memory addresses or sizes.
 *  Frame times and plane numbers are copied, if possible.
\return Returns 0 if ok.
 */
int imgCopyhdr(
  /** Pointer to input IMG data */
  IMG *image1,
  /** Pointer to output IMG data */
  IMG *image2
) {
  int i;

  if(IMG_TEST) printf("imgCopyhdr()\n");
  /* check */
  if(image1==NULL || image2==NULL) return(1);
  if(image1==image2) return(2);
  /* copy */
  image2->type=image1->type;
  image2->unit=image1->unit;
  strcpy(image2->studyNr, image1->studyNr);
  strcpy(image2->patientName, image1->patientName);
  image2->zoom=image1->zoom;
  strcpy(image2->radiopharmaceutical, image1->radiopharmaceutical);
  image2->isotopeHalflife=image1->isotopeHalflife;
  image2->decayCorrected=image1->decayCorrected;
  image2->scanStart=image1->scanStart;
  image2->axialFOV=image1->axialFOV;
  image2->transaxialFOV=image1->transaxialFOV;
  image2->sampleDistance=image1->sampleDistance;
  image2->sizex=image1->sizex;
  image2->sizey=image1->sizey;
  image2->sizez=image1->sizez;
  image2->_dataType=image1->_dataType;
  image2->_fileFormat=image1->_fileFormat;
  image2->orientation=image1->orientation;
  image2->scanner=image1->scanner;
  if(image1->dimz==image2->dimz) for(i=0; i<image1->dimz; i++)
    image2->planeNumber[i]=image1->planeNumber[i];
  if(image1->dimt==image2->dimt) for(i=0; i<image1->dimt; i++) {
    image2->start[i]=image1->start[i]; image2->end[i]=image1->end[i];
    image2->mid[i]=image1->mid[i];
    image2->weight[i]=image1->weight[i]; image2->sd[i]=image1->sd[i];
    image2->decayCorrFactor[i]=image1->decayCorrFactor[i];
  }
  image2->isWeight=image1->isWeight;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Extract a smaller 4D image from inside an IMG.
    Any existing data is overwritten.
\return Returns 0 if ok.
 */
int imgExtractRange(IMG *img1, IMG_RANGE r, IMG *img2)
{
  int zi, yi, xi, fi, zj, yj, xj, fj;

  if(IMG_TEST) {
    printf("imgExtractRange(*img1, r, *img2)\n");
    printf("  z=[%d,%d] y=[%d,%d] x=[%d,%d] f=[%d,%d]\n",
      r.z1, r.z2, r.y1, r.y2, r.x1, r.x2, r.f1, r.f2);
  }
  /* Check arguments */
  if(img2==NULL) return(1); else img2->statmsg=_imgStatusMessage[1];
  if(img1->status!=IMG_STATUS_OCCUPIED) return(1);
  if(img2->status==IMG_STATUS_UNINITIALIZED) return(1);
  if(r.z1<1 || r.z2>img1->dimz || r.z1>r.z2) return(1);
  if(r.y1<1 || r.y2>img1->dimy || r.y1>r.y2) return(1);
  if(r.x1<1 || r.x2>img1->dimx || r.x1>r.x2) return(1);
  if(r.f1<1 || r.f2>img1->dimt || r.f1>r.f2) return(1);

  /* Allocate memory unless the same size was previously allocated */
  img2->statmsg=_imgStatusMessage[2];
  zi=r.z2-r.z1+1; yi=r.y2-r.y1+1; xi=r.x2-r.x1+1; fi=r.f2-r.f1+1;
  if(img2->status>=IMG_STATUS_OCCUPIED)
    if(img2->dimz!=zi || img2->dimy!=yi || img2->dimx!=xi || img2->dimt!=fi)
      imgEmpty(img2);
  if(img2->status!=IMG_STATUS_OCCUPIED) {
    if(imgAllocate(img2, zi, yi, xi, fi)!=0) return(2);
  }

  /* Copy data */
  imgCopyhdr(img1, img2);
  for(fi=r.f1-1, fj=0; fi<r.f2; fi++, fj++) {
    img2->start[fj]=img1->start[fi];
    img2->end[fj]=img1->end[fi];
    img2->mid[fj]=img1->mid[fi];
    img2->weight[fj]=img1->weight[fi];
    img2->sd[fj]=img1->sd[fi];
    img2->decayCorrFactor[fj]=img1->decayCorrFactor[fi];
  }
  for(zi=r.z1-1, zj=0; zi<r.z2; zi++, zj++)
    img2->planeNumber[zj]=img1->planeNumber[zi];
  for(zi=r.z1-1, zj=0; zi<r.z2; zi++, zj++)
    for(yi=r.y1-1, yj=0; yi<r.y2; yi++, yj++)
      for(xi=r.x1-1, xj=0; xi<r.x2; xi++, xj++)
        for(fi=r.f1-1, fj=0; fi<r.f2; fi++, fj++)
          img2->m[zj][yj][xj][fj]=img1->m[zi][yi][xi][fi];

  img2->statmsg=_imgStatusMessage[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

