/******************************************************************************
  Copyright (c) 2003,2004 by Turku PET Centre

  imgfile.c
  
  I/O routines for IMG data.
  Currently supported file formats:
    ECAT 6.3 images and sinograms
    ECAT 7.x 2D and 3D images (volumes) and sinograms
    Analyze 7.5 images

  Version:
  2003-07-27 Vesa Oikonen
  2003-08-11 VO
    Corrected a bug in test print in imgRead().
  2003-09-08 VO
    Can now read&write 3D sinograms.
    sampleDistance is now read/written from/to bin_size in ECAT7 mainheader.
    imgWriteEcat7() renamed to imgWrite2DEcat7().
    "New" function imgWriteEcat7() makes 3D format instead of 2D as before.
    Applies the new IMG field _fileFormat in read/write.
  2003-10-05 VO
    Added support for Analyze 7.5 image format.
  2003-10-07 VO
    Setting Analyze db_name is corrected.
  2003-10-08 VO
    Only 7 first letters of image magic number is checked.
  2003-10-21 VO
    imgRead() and imgReadAnalyze() may work even if filename is given
    with extension.
  2003-11-04 VO
    If ECAT 6.3 image mainheader and subheader contain the same calibration
    factor, it is used only once.
    Works with unit nCi/mL.
  2003-11-10 VO
    ECAT 7 3D scan subheader field x_resolution is used as sample distance.
  2003-11-12 VO
    Reading ECAT 6.3 image: pixels x size is read from subheader slice_width
    if it is not found in main header plane_separation field.
  2003-11-30 VO
    For now, calls temp_roundf() instead of roundf().
  2003-12-05 VO
    Analyze files may be read&write without flipping in x,y,z-directions.
    Function anaFlipping() is used to determine whether to flip or not.
  2003-12-10 VO
    Changes in 2003-11-10 and 2003-11-12 had been accidentally deleted
    and are now returned.
  2003-12-14 VO
    Patient orientation is read&written in ECAT7 format.
    Scanner (system_type) is read&written in ECAT formats.
  2004-02-05 VO
    anaFlipping() determines whether image is flipped in z-direction;
    image is always flipped in x,y-directions.
  2004-02-08 VO
    Changes in imgSetEcat7MHeader():
    -sets different magic number for sinograms and image volumes
    -sets sw_version=72.
    When writing ECAT 6.3 format: sets sw_version=2.
  2004-02-22 VO
    Analyze format: zoom factor is not written into/read from funused2,
    because that field is used by SPM2 to another purpose.
  2004-05-23 VO
    ECAT7_3DSCANFIT is now supported as ECAT7_3DSCAN.
    imgReadEcat7() reports that axial compression is not supported.
    ECAT unit 1 is assumed to represent MBq/mL (IMG unit 13).
  2004-05-24 VO
    Pixel sizes are correctly converted from mm to cm when writing ECAT7.
    Changes in unit conversions between ECAT and IMG.
  2004-06-27 VO
    Additional error message.
    ecat63ReadAllToImg() and ecat63ReadPlaneToImg() do not even try to read
    extra frames.
    ECAT63_TEST changed to IMG_TEST.
    Patient name is copied with strncpy in ecat63ReadPlaneToImg() and
    ecat63ReadAllToImg().
  2004-07-10 VO
    Not so many test prints in reading ECAT 6.3 files.
  2004-08-23 VO
    MAX_STUDYNR_LEN applied where appropriate.
  2004-09-20 VO
    Doxygen style comments are corrected.
  2004-09-24 VO
    Better handling of ECAT7 calibration units.
    E.g. imgUnitToEcat7() divided into imgUnitToEcat6() and imgUnitToEcat7().
  2004-10-10 VO
    imgSetEcat7MHeader() sets ECAT7 mainheader plane number to dimz, not 1
    as before.
  2004-10-13 VO
    tm_isdst=-1 (unknown Daylight saving time) when appropriate.
  2004-11-07
    imgReadEcat7() reads files with axial compression, each slice as one plane.
  2004-12-22 VO
    Correction in imgGetEcat7MHeader(): calls imgUnitFromEcat7() instead of
    imgUnitFromEcat().
  2004-12-27 VO
    imgUnitFromEcat(): IMG unit is set to unknown, not to MBq/mL, when ECAT
    unit is unknown.

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "include/img.h"
#include "include/petc99.h"
#include "include/ecat63.h"
#include "include/ecat7.h"
#include "include/analyze.h"
#include "include/imgproc.h"
#include "include/swap.h"
#include "include/imgfile.h"
/*****************************************************************************/
#include <sif.h>
/*****************************************************************************/

/*****************************************************************************/
/** Read an image or sinogram file in ECAT 6.3, ECAT 7.x or Analyze 7.5 format.
\return Returns >0 in case of an error and sets IMG->statmsg
 */
int imgRead(
  /** Input filename */
  char *fname,
  /** Pointer to initialized IMG structure */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot open file",
  /*  4 */ "unknown file format",
  /*  5 */ "unsupported file type",
  /*  6 */ "missing matrix/matrices"
  };
  FILE *fp;
  int ret;
  ECAT7_mainheader ecat7_main_header;
  ECAT63_mainheader ecat63_main_header;
  char temp[FILENAME_MAX], *cptr;


  if(IMG_TEST) printf("imgRead(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) {
    img->statmsg=imgmsg[1]; return(2);}

  /* Check if we have Analyze file */
  /* Check if filename was given accidentally with extension */
  strcpy(temp, fname); cptr=strrchr(temp, '.');
  if(cptr!=NULL && (strcmp(cptr, ".img")==0 || strcmp(cptr, ".hdr")==0))
    *cptr=(char)0;
  /* Are there the Analyze .img and .hdr file? */
  if(anaExists(temp)!=0) {
    /* Read Analyze image */
    ret=imgReadAnalyze(temp, img);
    if(ret==3 || ret==4 || ret==5 || ret==6) {
      img->statmsg=imgmsg[4]; return(4);
    } else if(ret>0) {
      img->statmsg=imgmsg[3]; return(4);
    }
    if(IMG_TEST) printf("%s identified as supported Analyze 7.5 format.\n",
      fname);
    img->statmsg=imgmsg[0];
    return(0);
  }


  /* Check if we have an ECAT file */
  /* Open file for read */
  if((fp=fopen(fname, "rb")) == NULL) {
    img->statmsg=imgmsg[3]; return(4);
  }
  /* Try to read ECAT 7.x main header */
  ret=ecat7ReadMainheader(fp, &ecat7_main_header);
  if(ret) {fclose(fp); img->statmsg=imgmsg[4]; return(4);}
  /* If header could be read, check for magic number */
  if(strncmp(ecat7_main_header.magic_number, ECAT7V_MAGICNR, 7)==0) {
    /* This is ECAT 7.x file */
    /* Check if file type is supported */
    if(ecat7_main_header.file_type!=ECAT7_VOLUME8 &&
       ecat7_main_header.file_type!=ECAT7_VOLUME16 &&
       ecat7_main_header.file_type!=ECAT7_IMAGE8 &&
       ecat7_main_header.file_type!=ECAT7_IMAGE16 &&
       ecat7_main_header.file_type!=ECAT7_2DSCAN &&
       ecat7_main_header.file_type!=ECAT7_3DSCAN8 &&
       ecat7_main_header.file_type!=ECAT7_3DSCAN &&
       ecat7_main_header.file_type!=ECAT7_3DSCANFIT
      ) {
      fclose(fp); img->statmsg=imgmsg[5]; return(5);
    }
    fclose(fp);
    /* Read file */
    if(IMG_TEST) printf("%s identified as supported ECAT 7.x %s format\n",
      fname, ecat7filetype(ecat7_main_header.file_type));
    ret=imgReadEcat7(fname, img);
    if(ret) {return(6);}
  } else {
    /* Check if file is in ECAT 6.3 format */
    ret=ecat63ReadMainheader(fp, &ecat63_main_header);
    fclose(fp);
    if(ret==0) {
      /* It seems to be ECAT 6.3, so read it */
      if(IMG_TEST) printf("%s identified as supported ECAT 6.3 %s format\n",
        fname, ecat7filetype(ecat63_main_header.file_type));
      ret=ecat63ReadAllToImg(fname, img);
      if(ret) {
        if(IMG_TEST) fprintf(stderr, "ecat63ReaddAllToImg: %s\n", ecat63errmsg);
        if(ret==6) img->statmsg=imgmsg[6]; else img->statmsg=imgmsg[5];
        return(6);
      }
    } else {img->statmsg=imgmsg[4]; return(4);}
  }
  img->statmsg=imgmsg[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write an image or sinogram file.
    Format depends on _fileFormat or filename extension.
\return Returns >0 in case of an error and sets IMG->statmsg
 */
int imgWrite(
  /** Output filename */
  char *fname,
  /** Pointer to IMG data */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "no permission to write",
  /*  4 */ "disk full?"
  };
  int ret;
  char *cptr;

  if(IMG_TEST) printf("imgWrite(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    img->statmsg=imgmsg[1]; return(2);}
  if(img->type!=IMG_TYPE_RAW && img->type!=IMG_TYPE_IMAGE) {
    img->statmsg=imgmsg[1]; return(2);}

  /* If _fileFormat is not defined, then determine it from the filename */
  if(img->_fileFormat==IMG_UNKNOWN) {
    img->_fileFormat=IMG_E7;
    cptr=strrchr(fname, '.'); if(cptr!=NULL) cptr++;
    if(cptr!=NULL) {
      if(strcmp(cptr, "img")==0 || strcmp(cptr, "scn")==0 ||
         strcmp(cptr, "IMG")==0 || strcmp(cptr, "SCN")==0)
        img->_fileFormat=IMG_E63;
    } else img->_fileFormat=IMG_ANA;
    if(IMG_TEST && img->_fileFormat==IMG_E63)
      printf("  saving in ECAT 6.3 format\n");
  }

  /* Write */
  if(img->_fileFormat==IMG_E63) {
    ret=ecat63WriteAllImg(fname, img);
    switch(ret) {
      case 0: break;
      case 4: img->statmsg=imgmsg[2]; break;
      case 3: img->statmsg=imgmsg[3]; break;
      case 9: img->statmsg=imgmsg[4]; break;
      default: img->statmsg=imgmsg[1];
    }
    if(ret) return(7);
  } else if(img->_fileFormat==IMG_ANA) {
    ret=imgWriteAnalyze(fname, img); if(ret) return(5);
  } else if(img->_fileFormat==IMG_E7_2D) {
    ret=imgWrite2DEcat7(fname, img); if(ret) return(5);
  } else {
    ret=imgWriteEcat7(fname, img); if(ret) return(5);
  }
  img->statmsg=imgmsg[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ECAT 7 image, volume or 2D sinogram.
\return Returns >0 in case of an error and sets IMG->statmsg
 */
int imgReadEcat7(
  /** Input filename */
  char *fname,
  /** Initialized IMG structure */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot open file",
  /*  4 */ "unknown file format",
  /*  5 */ "unsupported file type",
  /*  6 */ "invalid matrix list",
  /*  7 */ "missing matrix",
  /*  8 */ "variable matrix size",
  /*  9 */ "cannot read subheader",
  /* 10 */ "cannot read matrix",
  /* 11 */ "axial compression is not supported"
  };
  FILE *fp;
  int ret, m, i, fi, pi, xi, yi, frame, plane, prev_frame, prev_plane;
  int dimx, dimy, dimz, planeNr, frameNr, blkNr=0, pxlNr;
  ECAT7_mainheader main_header;
  ECAT7_imageheader image_header;
  ECAT7_2Dscanheader scan2d_header;
  ECAT7_scanheader scan_header;
  ECAT7_MATRIXLIST mlist;
  ECAT7_Matval matval;
  float *fdata=NULL, *fptr;


  if(IMG_TEST) printf("imgReadEcat7(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) {
    img->statmsg=imgmsg[1]; return(2);}

  /* Open file for read */
  if((fp=fopen(fname, "rb")) == NULL) {img->statmsg=imgmsg[3]; return(3);}
  
  /* Read main header */
  ret=ecat7ReadMainheader(fp, &main_header);
  if(ret) {fclose(fp); img->statmsg=imgmsg[4]; return(4);}
  /* Check for magic number */
  if(strncmp(main_header.magic_number, ECAT7V_MAGICNR, 7)!=0) {
    fclose(fp); img->statmsg=imgmsg[4]; return(4);
  }
  /* Check if file_type is supported */
  if(main_header.file_type!=ECAT7_VOLUME8 &&
     main_header.file_type!=ECAT7_VOLUME16 &&
     main_header.file_type!=ECAT7_IMAGE8 &&
     main_header.file_type!=ECAT7_IMAGE16 &&
     main_header.file_type!=ECAT7_2DSCAN &&
     main_header.file_type!=ECAT7_3DSCAN8 &&
     main_header.file_type!=ECAT7_3DSCAN &&
     main_header.file_type!=ECAT7_3DSCANFIT
    ) {
    fclose(fp); img->statmsg=imgmsg[5]; return(5);
  }

  /* Read matrix list */
  ecat7InitMatlist(&mlist);
  ret=ecat7ReadMatlist(fp, &mlist);
  if(ret || mlist.matrixNr<1 || ecat7CheckMatlist(&mlist)) {
    fclose(fp); img->statmsg=imgmsg[6]; return(6);}
  ecat7SortMatlistByPlane(&mlist);
  if(IMG_TEST>2) ecat7PrintMatlist(&mlist);

  /* Calculate the number of planes, frames and gates */
  /* Check that all planes have equal nr of frames (gates) */
  /* and check that frames (gates) are consequentally numbered */
  prev_plane=plane=-1; prev_frame=frame=-1; frameNr=planeNr=0; ret=0;
  for(m=0; m<mlist.matrixNr; m++) {
    ecat7_id_to_val(mlist.matdir[m].id, &matval); plane=matval.plane;
    if(main_header.num_frames>=main_header.num_gates) frame=matval.frame;
    else frame=matval.gate;
    if(plane!=prev_plane) {frameNr=1; planeNr++;}
    else {frameNr++; if(prev_frame>0 && frame!=prev_frame+1) {ret=1; break;}}
    prev_plane=plane; prev_frame=frame;
    /* Calculate and check the size of one data matrix */
    if(m==0) {
      blkNr=mlist.matdir[m].endblk-mlist.matdir[m].strtblk;
    } else if(blkNr!=mlist.matdir[m].endblk-mlist.matdir[m].strtblk) {
      ret=2; break;
    }
  } /* next matrix */
  if(IMG_TEST>2) printf("frameNr=%d planeNr=%d\n", frameNr, planeNr);
  if(ret==1 || (frameNr*planeNr != mlist.matrixNr)) {
    fclose(fp); img->statmsg=imgmsg[7]; ecat7EmptyMatlist(&mlist); return(7);}
  if(ret==2) {
    fclose(fp); img->statmsg=imgmsg[8]; ecat7EmptyMatlist(&mlist); return(8);}

  /* Read the first subheader to get planeNr from volumes and to get x&y dim */
  m=0; dimz=1; img->statmsg=imgmsg[9];
  switch(main_header.file_type) {
    case ECAT7_IMAGE8:
    case ECAT7_IMAGE16:
    case ECAT7_VOLUME8:
    case ECAT7_VOLUME16:
      img->type=IMG_TYPE_IMAGE;
      ret=ecat7ReadImageheader(fp, mlist.matdir[m].strtblk, &image_header);
      dimx=image_header.x_dimension; dimy=image_header.y_dimension;
      if(image_header.num_dimensions>2 && image_header.z_dimension>1)
        planeNr=dimz=image_header.z_dimension;
      break;
    case ECAT7_2DSCAN:
      img->type=IMG_TYPE_RAW;
      ret=ecat7Read2DScanheader(fp, mlist.matdir[m].strtblk, &scan2d_header);
      dimx=scan2d_header.num_r_elements; dimy=scan2d_header.num_angles;
      if(scan2d_header.num_dimensions>2 && scan2d_header.num_z_elements>1)
        planeNr=dimz=scan2d_header.num_z_elements;
      break;
    case ECAT7_3DSCAN:
    case ECAT7_3DSCAN8:
    case ECAT7_3DSCANFIT:
      img->type=IMG_TYPE_RAW;
      ret=ecat7ReadScanheader(fp, mlist.matdir[m].strtblk, &scan_header);
      dimx=scan_header.num_r_elements; dimy=scan_header.num_angles;
      for(i=dimz=0; i<64; i++) dimz+=scan_header.num_z_elements[i];
      planeNr=dimz;
      /*if(scan_header.axial_compression!=0) {img->statmsg=imgmsg[11]; ret=-1;}*/
      break;
    default: dimx=dimy=dimz=planeNr=0; ret=-1;
  }
  pxlNr=dimx*dimy;
  if(ret || pxlNr<1 || planeNr<1) {
    fclose(fp);  ecat7EmptyMatlist(&mlist); return(9);}
  img->statmsg=imgmsg[0];

  /* Allocate memory for IMG data */
  ret=imgAllocate(img, planeNr, dimy, dimx, frameNr);
  if(ret) {
    fclose(fp); img->statmsg=imgmsg[2]; ecat7EmptyMatlist(&mlist); return(11);}
  /* Copy information from mainheader */
  imgGetEcat7MHeader(img, &main_header);
  /* Set fileFormat */
  switch(main_header.file_type) {
    case ECAT7_IMAGE8:
    case ECAT7_IMAGE16:
      img->_fileFormat=IMG_E7_2D; break;
    case ECAT7_VOLUME8:
    case ECAT7_VOLUME16:
      img->_fileFormat=IMG_E7; break;
    case ECAT7_2DSCAN:
      img->_fileFormat=IMG_E7_2D; break;
    case ECAT7_3DSCAN:
    case ECAT7_3DSCAN8:
    case ECAT7_3DSCANFIT:
      img->_fileFormat=IMG_E7; break;
    default:
      img->_fileFormat=IMG_UNKNOWN; break;
  }

  if(dimz>1) {
    /* Read ECAT volume matrices */
    fi=0;
    for(m=0; m<mlist.matrixNr; m++) {
      /* Get matrix values */
      ecat7_id_to_val(mlist.matdir[m].id, &matval);
      /* Read subheader and data */
      if(img->type==IMG_TYPE_IMAGE)
        ret=ecat7ReadImageMatrix(fp, mlist.matdir[m].strtblk,
              mlist.matdir[m].endblk, &image_header, &fdata);
      else
        ret=ecat7ReadScanMatrix(fp, mlist.matdir[m].strtblk,
              mlist.matdir[m].endblk, &scan_header, &fdata);
      if(ret || fdata==NULL) {
        fclose(fp); img->statmsg=imgmsg[10]; ecat7EmptyMatlist(&mlist); return(13);}
      /* Copy subheader information */
      if(img->type==IMG_TYPE_IMAGE) {
        img->_dataType=image_header.data_type;
        img->start[fi]=image_header.frame_start_time/1000.;
        img->end[fi]=img->start[fi]+image_header.frame_duration/1000.;
        img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
        if(image_header.decay_corr_fctr>1.0) {
          img->decayCorrFactor[fi]=image_header.decay_corr_fctr;
          img->decayCorrected=1;
        }
        img->zoom=image_header.recon_zoom;
        img->sizex=10.*image_header.x_pixel_size;
        img->sizey=10.*image_header.y_pixel_size;
        img->sizez=10.*image_header.z_pixel_size;
      } else {
        img->_dataType=scan_header.data_type;
        img->start[fi]=scan_header.frame_start_time/1000.;
        img->end[fi]=img->start[fi]+scan2d_header.frame_duration/1000.;
        img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
        if(scan_header.x_resolution>0.0)
          img->sampleDistance=10.0*scan_header.x_resolution;
        else
          img->sampleDistance=10.0*main_header.bin_size;
        /* also, correct for dead-time */
        if(scan_header.deadtime_correction_factor>0.0)
          for(i=0, fptr=fdata; i<pxlNr; i++, fptr++)
            *fptr*=scan_header.deadtime_correction_factor;
      }
      /* Copy matrix data through volume planes */
      for(pi=0; pi<dimz; pi++) {
        for(yi=0, fptr=fdata+pi*pxlNr; yi<dimy; yi++) for(xi=0; xi<dimx; xi++)
          img->m[pi][yi][xi][fi]= *fptr++;
      }
      free(fdata); fi++;
    } /* next matrix */
    /* Set plane numbers */
    for(pi=0; pi<dimz; pi++) img->planeNumber[pi]=pi+1;
  } else {
    /* Read separate matrices */
    prev_plane=plane=-1; prev_frame=frame=-1; pi=fi=-1;
    for(m=0; m<mlist.matrixNr; m++) {
      ecat7_id_to_val(mlist.matdir[m].id, &matval); plane=matval.plane;
      if(main_header.num_frames>=main_header.num_gates) frame=matval.frame;
      else frame=matval.gate;
      if(plane!=prev_plane) {fi=0; pi++;} else fi++;
      /* Read subheader and data */
      if(img->type==IMG_TYPE_IMAGE)
        ret=ecat7ReadImageMatrix(fp, mlist.matdir[m].strtblk,
              mlist.matdir[m].endblk, &image_header, &fdata);
      else
        ret=ecat7Read2DScanMatrix(fp, mlist.matdir[m].strtblk,
              mlist.matdir[m].endblk, &scan2d_header, &fdata);
      if(ret || fdata==NULL) {
        fclose(fp); img->statmsg=imgmsg[10]; ecat7EmptyMatlist(&mlist); return(13);}
      /* Copy subheader information */
      if(fi==0) img->planeNumber[pi]=plane;
      if(img->type==IMG_TYPE_IMAGE) {
        img->_dataType=image_header.data_type;
        img->start[fi]=image_header.frame_start_time/1000.;
        img->end[fi]=img->start[fi]+image_header.frame_duration/1000.;
        img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
        if(image_header.decay_corr_fctr>1.0) {
          img->decayCorrFactor[fi]=image_header.decay_corr_fctr;
          img->decayCorrected=1;
        }
        img->zoom=image_header.recon_zoom;
        img->sizex=10.*image_header.x_pixel_size;
        img->sizey=10.*image_header.y_pixel_size;
        img->sizez=10.*image_header.z_pixel_size;
      } else {
        img->_dataType=scan2d_header.data_type;
        img->start[fi]=scan2d_header.frame_start_time/1000.;
        img->end[fi]=img->start[fi]+scan2d_header.frame_duration/1000.;
        img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
        if(scan_header.x_resolution>0.0)
          img->sampleDistance=10.0*scan_header.x_resolution;
        else
          img->sampleDistance=10.0*main_header.bin_size;
        /* also, correct for dead-time */
        if(scan2d_header.deadtime_correction_factor>0.0)
          for(i=0, fptr=fdata; i<pxlNr; i++, fptr++)
            *fptr*=scan2d_header.deadtime_correction_factor;
      }
      /* Copy matrix data */
      for(yi=0, fptr=fdata; yi<dimy; yi++) for(xi=0; xi<dimx; xi++)
        img->m[pi][yi][xi][fi]= *fptr++;
      free(fdata);
      /* prepare for the next matrix */
      prev_plane=plane; prev_frame=frame;
    } /* next matrix */
  }
  fclose(fp); ecat7EmptyMatlist(&mlist);

  /* Calibrate */
  if(main_header.ecat_calibration_factor>0.0)
    for(pi=0; pi<img->dimz; pi++)
      for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++)
        for(fi=0; fi<img->dimt; fi++)
          img->m[pi][yi][xi][fi]*=main_header.ecat_calibration_factor;

  img->statmsg=imgmsg[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write ECAT 7 3D image volume or 3D sinogram.
\return Returns >0 in case of an error and sets IMG->statmsg
 */
int imgWriteEcat7(
  /** Output filename */
  char *fname,
  /** Pointer to IMG data */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot open file",
  /*  4 */ "unknown file format",
  /*  5 */ "unsupported file type",
  /*  6 */ "no permission",
  /*  7 */ "disk full?"
  };
  ECAT7_mainheader main_header;
  ECAT7_imageheader image_header;
  ECAT7_scanheader scan_header;
  FILE *fp;
  int fi, pi, xi, yi, pxlNr, matrixId, ret;
  float *fdata, *fptr;


  if(IMG_TEST) printf("imgWriteEcat7(%s, *img)\n", fname);
  if(IMG_TEST>1 && ECAT7_TEST==0) ECAT7_TEST=1;
  /* Check the arguments */
  if(fname==NULL) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    img->statmsg=imgmsg[1]; return(2);}
  if(img->type!=IMG_TYPE_RAW && img->type!=IMG_TYPE_IMAGE) {
    img->statmsg=imgmsg[1]; return(2);}

  /* Initiate headers */
  memset(&main_header, 0, sizeof(ECAT7_mainheader));
  memset(&image_header,0, sizeof(ECAT7_imageheader));
  memset(&scan_header, 0, sizeof(ECAT7_scanheader));

  /* Set main header */
  imgSetEcat7MHeader(img, &main_header);
  main_header.bin_size=img->sampleDistance/10.0;

  /* Allocate memory for matrix float data */
  pxlNr=img->dimx*img->dimy*img->dimz;
  fdata=(float*)malloc(pxlNr*sizeof(float));
  if(fdata==NULL) {img->statmsg=imgmsg[2]; return(3);}

  /* Open file, write main header and initiate matrix list */
  fp=ecat7Create(fname, &main_header);
  if(fp==NULL) {free(fdata); img->statmsg=imgmsg[6]; return(6);}

  /* Set (most of) subheader contents */
  if(img->type==IMG_TYPE_RAW) {
    scan_header.x_resolution=img->sampleDistance/10.0;
    scan_header.num_dimensions=4;
    if(img->dimz==239) {
      scan_header.num_z_elements[0]=63;
      scan_header.num_z_elements[1]=106;
      scan_header.num_z_elements[2]=70;
    } else {
      scan_header.num_z_elements[0]=img->dimz;
    }
    scan_header.storage_order=1;
    scan_header.data_type=ECAT7_SUNI2;
    scan_header.num_r_elements=img->dimx;
    scan_header.num_angles=img->dimy;
  } else if(img->type==IMG_TYPE_IMAGE) {
    image_header.num_dimensions=3;
    image_header.z_dimension=img->dimz;
    image_header.data_type=ECAT7_SUNI2;
    image_header.x_dimension=img->dimx;
    image_header.y_dimension=img->dimy;
    image_header.recon_zoom=img->zoom;
    image_header.x_pixel_size=0.1*img->sizex;
    image_header.y_pixel_size=0.1*img->sizey;
    image_header.z_pixel_size=0.1*img->sizez;
  }

  /* Write each matrix */
  for(fi=0; fi<img->dimt; fi++) {

    /* Create new matrix id (i.e. matnum) */
    matrixId=ecat7_val_to_id(fi+1, 1, 1, 0, 0);

    /* Copy matrix pixel values to fdata */
    fptr=fdata;
    for(pi=0; pi<img->dimz; pi++)
      for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++)
        *fptr++=img->m[pi][yi][xi][fi];

    /* Write subheader and data */
    fptr=fdata;
    if(img->type==IMG_TYPE_RAW) {
      scan_header.frame_start_time=(int)temp_roundf(1000.*img->start[fi]);
      scan_header.frame_duration=(int)temp_roundf(1000.*(img->end[fi]-img->start[fi]));
      /*ecat7PrintScanheader(&scan_header, stdout);*/
      ret=ecat7WriteScanMatrix(fp, matrixId, &scan_header, fptr);
    } else if(img->type==IMG_TYPE_IMAGE) {
      image_header.frame_start_time=(int)temp_roundf(1000.*img->start[fi]);
      image_header.frame_duration=(int)temp_roundf(1000.*(img->end[fi]-img->start[fi]));
      image_header.decay_corr_fctr=img->decayCorrFactor[fi];
      /*ecat7PrintImageheader(&image_header, stdout);*/
      ret=ecat7WriteImageMatrix(fp, matrixId, &image_header, fptr);
    } else {free(fdata); fclose(fp); img->statmsg=imgmsg[5]; return(8);}
    if(ret) {
      if(IMG_TEST) {printf("matrixId=%d ret=%d\n", matrixId, ret);}
      free(fdata); fclose(fp); img->statmsg=imgmsg[7]; return(7);
    }

  } /* next matrix */
  free(fdata); fclose(fp);

  img->statmsg=imgmsg[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write ECAT 7 2D image or 2D sinogram.
\return Returns >0 in case of an error and sets IMG->statmsg
 */
int imgWrite2DEcat7(
  /** Output filename */
  char *fname,
  /** Pointer to IMG data */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot open file",
  /*  4 */ "unknown file format",
  /*  5 */ "unsupported file type",
  /*  6 */ "no permission",
  /*  7 */ "disk full?"
  };
  ECAT7_mainheader main_header;
  ECAT7_imageheader image_header;
  ECAT7_2Dscanheader scan2d_header;
  FILE *fp;
  int fi, pi, xi, yi, pxlNr, matrixId, ret;
  float *fdata, *fptr;


  if(IMG_TEST) printf("imgWrite2DEcat7(%s, *img)\n", fname);
  if(IMG_TEST>1 && ECAT7_TEST==0) ECAT7_TEST=1;
  /* Check the arguments */
  if(fname==NULL) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    img->statmsg=imgmsg[1]; return(2);}

  /* Initiate headers */
  memset(&main_header, 0, sizeof(ECAT7_mainheader));
  memset(&image_header,0, sizeof(ECAT7_imageheader));
  memset(&scan2d_header, 0, sizeof(ECAT7_2Dscanheader));

  /* Set main header */
  imgSetEcat7MHeader(img, &main_header);
  main_header.bin_size=img->sampleDistance/10.0;
  if(img->type==IMG_TYPE_RAW) main_header.file_type=ECAT7_2DSCAN;
  else main_header.file_type=ECAT7_IMAGE16;
  main_header.num_planes=img->dimz;

  /* Allocate memory for matrix float data */
  pxlNr=img->dimx*img->dimy;
  fdata=(float*)malloc(pxlNr*sizeof(float));
  if(fdata==NULL) {img->statmsg=imgmsg[2]; return(3);}

  /* Open file, write main header and initiate matrix list */
  fp=ecat7Create(fname, &main_header);
  if(fp==NULL) {free(fdata); img->statmsg=imgmsg[6]; return(6);}

  /* Set (most of) subheader contents */
  if(img->type==IMG_TYPE_RAW) {
    scan2d_header.num_dimensions=2;
    scan2d_header.num_z_elements=1;
    scan2d_header.data_type=ECAT7_SUNI2;
    scan2d_header.num_r_elements=img->dimx;
    scan2d_header.num_angles=img->dimy;
  } else if(img->type==IMG_TYPE_IMAGE) {
    image_header.num_dimensions=2;
    image_header.z_dimension=1;
    image_header.data_type=ECAT7_SUNI2;
    image_header.x_dimension=img->dimx;
    image_header.y_dimension=img->dimy;
    image_header.recon_zoom=img->zoom;
    image_header.x_pixel_size=0.1*img->sizex;
    image_header.y_pixel_size=0.1*img->sizey;
    image_header.z_pixel_size=0.1*img->sizez;
  }

  /* Write each matrix */
  for(fi=0; fi<img->dimt; fi++) for(pi=0; pi<img->dimz; pi++) {

    /* Create new matrix id (i.e. matnum) */
    matrixId=ecat7_val_to_id(fi+1, img->planeNumber[pi], 1, 0, 0);

    /* Copy matrix pixel values to fdata */
    fptr=fdata;
    for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++)
      *fptr++=img->m[pi][yi][xi][fi];

    /* Write subheader and data */
    fptr=fdata;
    if(img->type==IMG_TYPE_RAW) {
      scan2d_header.frame_start_time=(int)temp_roundf(1000.*img->start[fi]);
      scan2d_header.frame_duration=(int)temp_roundf(1000.*(img->end[fi]-img->start[fi]));
      ret=ecat7Write2DScanMatrix(fp, matrixId, &scan2d_header, fptr);
    } else if(img->type==IMG_TYPE_IMAGE) {
      image_header.frame_start_time=(int)temp_roundf(1000.*img->start[fi]);
      image_header.frame_duration=(int)temp_roundf(1000.*(img->end[fi]-img->start[fi]));
      image_header.decay_corr_fctr=img->decayCorrFactor[fi];
      ret=ecat7WriteImageMatrix(fp, matrixId, &image_header, fptr);
    } else {free(fdata); fclose(fp); img->statmsg=imgmsg[5]; return(8);}
    if(ret) {
      if(IMG_TEST) {printf("matrixId=%d ret=%d\n", matrixId, ret);}
      free(fdata); fclose(fp); img->statmsg=imgmsg[7]; return(7);
    }

  } /* next matrix */
  free(fdata); fclose(fp);

  img->statmsg=imgmsg[0];
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read all matrices in ECAT file to memory.
    Img data must be initialized before this procedure.
\return If ok, 0 is returned.
 */
int ecat63ReadAllToImg(
  /** Name of the input ECAT 6.3 file */
  char *fname,
  /** Data structure in which the file is read */
  IMG *img
) {
  int                 i, j, m, ret, blkNr=0, dim_x=0, dim_y=0, pxlNr=0;
  int                 frameNr, planeNr, del_nr=0;
  int                 frame, plane, prev_frame, prev_plane, seqplane;
  FILE               *fp;
  ECAT63_mainheader   main_header;
  MATRIXLIST          mlist;
  MatDir              tmpMatdir;
  Matval              matval;
  ECAT63_imageheader  image_header;
  ECAT63_scanheader   scan_header;
  ECAT63_attnheader   attn_header;
  ECAT63_normheader   norm_header;
  float               scale=1.0;
  short int          *sptr;
  char               *mdata=NULL, *mptr;
  int                *iptr;
  struct tm           scanStart;


  if(IMG_TEST) printf("ecat63ReadAllToImg(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {strcpy(ecat63errmsg, "invalid ECAT filename"); return(1);}
  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) {
    strcpy(ecat63errmsg, "image data not initialized"); return(2);}

  /* Open input CTI file for read */
  if((fp=fopen(fname, "rb")) == NULL) {
    strcpy(ecat63errmsg, "cannot open ECAT file"); return(3);}

  /* Read main header */
  if((ret=ecat63ReadMainheader(fp, &main_header))) {
    sprintf(ecat63errmsg, "cannot read main header (%d)", ret);
    fclose(fp); return(4);
  }
  if(IMG_TEST>2) ecat63PrintMainheader(&main_header);

  /* Read matrix list and nr */
  ecat63InitMatlist(&mlist);
  ret=ecat63ReadMatlist(fp, &mlist);
  if(ret) {
    sprintf(ecat63errmsg, "cannot read matrix list (%d)", ret);
    fclose(fp); return(5);
  }
  if(mlist.matrixNr<=0) {
    strcpy(ecat63errmsg, "matrix list is empty"); fclose(fp); return(5);}
  /* Sort matrix list */
  for(i=0; i<mlist.matrixNr-1; i++) for(j=i+1; j<mlist.matrixNr; j++) {
    if(mlist.matdir[i].matnum>mlist.matdir[j].matnum) {
      tmpMatdir=mlist.matdir[i];
      mlist.matdir[i]=mlist.matdir[j]; mlist.matdir[j]=tmpMatdir;
    }
  }
  if(IMG_TEST>2) {
    printf("matrix list after sorting:\n"); ecat63PrintMatlist(&mlist);}
  /* Trim extra frames */
  if(main_header.num_frames>0) {
    del_nr=ecat63DeleteLateFrames(&mlist, main_header.num_frames);
    if(IMG_TEST && del_nr>0)
      printf("  %d entries in matrix list will not be used.\n", del_nr);
  }
  /* Calculate the number of planes, frames and gates */
  /* Check that all planes have equal nr of frames (gates) */
  /* and check that frames (gates) are consequentally numbered */
  prev_plane=plane=-1; prev_frame=frame=-1; frameNr=planeNr=0; ret=0;
  for(m=0; m<mlist.matrixNr; m++) if(mlist.matdir[m].matstat==1) {
    /* get frame and plane */
    mat_numdoc(mlist.matdir[m].matnum, &matval);
    plane=matval.plane;
    if(main_header.num_frames>=main_header.num_gates) frame=matval.frame;
    else frame=matval.gate;
    if(IMG_TEST>2) printf("frame=%d plane=%d\n", frame, plane);
    /* did plane change? */
    if(plane!=prev_plane) {
      frameNr=1; planeNr++;
    } else {
      frameNr++;
      /* In each plane, frame (gate) numbers must be continuous */
      if(prev_frame>0 && frame!=prev_frame+1) {ret=1; break;}
    }
    prev_plane=plane; prev_frame=frame;
    /* Calculate and check the size of one data matrix */
    if(m==0) {
      blkNr=mlist.matdir[m].endblk-mlist.matdir[m].strtblk;
    } else if(blkNr!=mlist.matdir[m].endblk-mlist.matdir[m].strtblk) {
      ret=2; break;
    }
  } /* next matrix */
  if(IMG_TEST) printf("frameNr=%d planeNr=%d\n", frameNr, planeNr);
  if(ret==1 || (frameNr*planeNr != mlist.matrixNr-del_nr)) {
    strcpy(ecat63errmsg, "missing matrix");
    ecat63EmptyMatlist(&mlist); fclose(fp);
    return(6); /* this number is used in imgRead() */
  }
  if(ret==2) {
    strcpy(ecat63errmsg, "matrix sizes are different");
    ecat63EmptyMatlist(&mlist); fclose(fp); return(7);
  }
  /* Read x,y-dimensions from 1st matrix subheader */
  m=0; if(main_header.file_type==IMAGE_DATA) {
    ret=ecat63ReadImageheader(fp, mlist.matdir[m].strtblk, &image_header);
    dim_x=image_header.dimension_1; dim_y=image_header.dimension_2;
  } else if(main_header.file_type==RAW_DATA) {
    ret=ecat63ReadScanheader(fp, mlist.matdir[m].strtblk, &scan_header);
    dim_x=scan_header.dimension_1; dim_y=scan_header.dimension_2;
  } else if(main_header.file_type==ATTN_DATA) {
    ret=ecat63ReadAttnheader(fp, mlist.matdir[m].strtblk, &attn_header);
    dim_x=attn_header.dimension_1; dim_y=attn_header.dimension_2;
  } else if(main_header.file_type==NORM_DATA) {
    ret=ecat63ReadNormheader(fp, mlist.matdir[m].strtblk, &norm_header);
    dim_x=norm_header.dimension_1; dim_y=norm_header.dimension_2;
  }
  pxlNr=dim_x*dim_y;
  if(ret) {
    sprintf(ecat63errmsg, "cannot read matrix %u subheader", mlist.matdir[m].matnum);
    ecat63EmptyMatlist(&mlist); fclose(fp); return(8);
  }

  /* Allocate memory for ECAT data matrix */
  if(IMG_TEST>1) printf("allocating memory for %d blocks\n", blkNr);
  mdata=(char*)malloc(blkNr*MatBLKSIZE); if(mdata==NULL) {
    strcpy(ecat63errmsg, "out of memory");
    fclose(fp); ecat63EmptyMatlist(&mlist); return(8);
  }
  /* Allocate memory for whole img data */
  ret=imgAllocate(img, planeNr, dim_y, dim_x, frameNr);
  if(ret) {
    sprintf(ecat63errmsg, "out of memory (%d)", ret);
    fclose(fp); ecat63EmptyMatlist(&mlist); free(mdata); return(9);
  }

  /* Fill img info with ECAT main and sub header information */
  img->scanner=main_header.system_type;
  img->unit=main_header.calibration_units; /* this continues below */
  strncpy(img->radiopharmaceutical, main_header.radiopharmaceutical, 32);
  img->isotopeHalflife=main_header.isotope_halflife;
  memset(&scanStart, 0, sizeof(struct tm));
  scanStart.tm_year=main_header.scan_start_year-1900;
  scanStart.tm_mon=main_header.scan_start_month-1;
  scanStart.tm_mday=main_header.scan_start_day; scanStart.tm_yday=0;
  scanStart.tm_hour=main_header.scan_start_hour;
  scanStart.tm_min=main_header.scan_start_minute;
  scanStart.tm_sec=main_header.scan_start_second;
  scanStart.tm_isdst=-1;
  img->scanStart=mktime(&scanStart);
  img->axialFOV=10.*main_header.axial_fov;
  img->transaxialFOV=10.*main_header.transaxial_fov;
  strncpy(img->studyNr, main_header.study_name, MAX_STUDYNR_LEN);
  strncpy(img->patientName, main_header.patient_name, 31);
  img->sizez=10.*main_header.plane_separation;
  if(main_header.file_type==IMAGE_DATA) {
    img->type=IMG_TYPE_IMAGE;
    if(img->unit<1) img->unit=image_header.quant_units;
    img->zoom=image_header.recon_scale;
    if(image_header.decay_corr_fctr>1.0) img->decayCorrected=1;
    img->sizex=img->sizey=10.*image_header.pixel_size;
    if(img->sizez<10.*image_header.slice_width)
      img->sizez=10.*image_header.slice_width;
  } else if(main_header.file_type==RAW_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  } else if(main_header.file_type==ATTN_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  } else if(main_header.file_type==NORM_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  }
  /* Set _fileFormat */
  img->_fileFormat=IMG_E63;

  /* Read one ECAT matrix at a time and put them to img */
  prev_plane=plane=-1; seqplane=-1;
  for(m=0; m<mlist.matrixNr; m++) if(mlist.matdir[m].matstat==1) {
    /* get plane */
    mat_numdoc(mlist.matdir[m].matnum, &matval);
    plane=matval.plane;
    /* did plane change? */
    if(plane!=prev_plane) {seqplane++; frame=1;} else frame++;
    prev_plane=plane;
    img->planeNumber[seqplane]=plane;
    /* Read subheader */
    if(main_header.file_type==IMAGE_DATA) {
      ret=ecat63ReadImageheader(fp, mlist.matdir[m].strtblk, &image_header);
      if(dim_x!=image_header.dimension_1 || dim_y!=image_header.dimension_2) ret=1;
      scale=image_header.quant_scale;
      if(image_header.ecat_calibration_fctr>0.0
         && fabs(main_header.calibration_factor/image_header.ecat_calibration_fctr-1.0)>0.0001)
        scale*=image_header.ecat_calibration_fctr;
      if(img->unit==0 && image_header.quant_units>0) img->unit=image_header.quant_units;
      img->_dataType=image_header.data_type;
      img->start[frame-1]=image_header.frame_start_time/1000.;
      img->end[frame-1]=img->start[frame-1]+image_header.frame_duration/1000.;
      img->mid[frame-1]=0.5*(img->start[frame-1]+img->end[frame-1]);
      if(image_header.decay_corr_fctr>1.0)
        img->decayCorrFactor[frame-1]=image_header.decay_corr_fctr;
    } else if(main_header.file_type==RAW_DATA) {
      ret=ecat63ReadScanheader(fp, mlist.matdir[m].strtblk, &scan_header);
      if(dim_x!=scan_header.dimension_1 || dim_y!=scan_header.dimension_2) ret=1;
      scale=scan_header.scale_factor;
      if(scan_header.loss_correction_fctr>0.0) scale*=scan_header.loss_correction_fctr;
      img->_dataType=scan_header.data_type;
      img->start[frame-1]=scan_header.frame_start_time/1000.;
      img->end[frame-1]=img->start[frame-1]+scan_header.frame_duration/1000.;
      img->mid[frame-1]=0.5*(img->start[frame-1]+img->end[frame-1]);
      img->sampleDistance=10.0*scan_header.sample_distance;
    } else if(main_header.file_type==ATTN_DATA) {
      ret=ecat63ReadAttnheader(fp, mlist.matdir[m].strtblk, &attn_header);
      if(dim_x!=attn_header.dimension_1 || dim_y!=attn_header.dimension_2) ret=1;
      scale=attn_header.scale_factor;
      img->_dataType=attn_header.data_type;
    } else if(main_header.file_type==NORM_DATA) {
      ret=ecat63ReadNormheader(fp, mlist.matdir[m].strtblk, &norm_header);
      if(dim_x!=norm_header.dimension_1 || dim_y!=norm_header.dimension_2) ret=1;
      scale=norm_header.scale_factor;
      img->_dataType=norm_header.data_type;
    } else img->_dataType=-1;
    if(ret) {
      sprintf(ecat63errmsg, "cannot read matrix %u subheader", mlist.matdir[m].matnum);
      ecat63EmptyMatlist(&mlist); free(mdata); fclose(fp); return(10);
    }
    if(main_header.calibration_factor>0.0) scale*=main_header.calibration_factor;
    if(IMG_TEST>2) printf("scale=%e datatype=%d\n", scale, img->_dataType);
    /* Read the pixel data */
    ret=ecat63ReadMatdata(fp, mlist.matdir[m].strtblk+1,
         mlist.matdir[m].endblk-mlist.matdir[m].strtblk,
         mdata, img->_dataType);
    if(ret) {
      strcpy(ecat63errmsg, "cannot read matrix data");
      ecat63EmptyMatlist(&mlist); free(mdata); fclose(fp); return(11);
    }
    if(img->_dataType==BYTE_TYPE) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr++) {
        img->m[seqplane][i][j][frame-1]=scale*(float)(*mptr);
      }
    } else if(img->_dataType==VAX_I2 || img->_dataType==SUN_I2) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=2) {
        sptr=(short int*)mptr;
        img->m[seqplane][i][j][frame-1]=scale*(float)(*sptr);
      }
    } else if(img->_dataType==VAX_I4 || img->_dataType==SUN_I4) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=4) {
        iptr=(int*)mptr;
        img->m[seqplane][i][j][frame-1]=1.0; //scale*(float)(*iptr);
      }
    } else if(img->_dataType==VAX_R4 || img->_dataType==IEEE_R4) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=4) {
        memcpy(&img->m[seqplane][i][j][frame-1], mptr, 4);
        img->m[seqplane][i][j][frame-1]*=scale;
      }
    }
  } /* next matrix */

  /* Set the unit */
  imgUnitFromEcat(img, img->unit);

  /* Free memory and close file */
  free(mdata);
  ecat63EmptyMatlist(&mlist);
  fclose(fp);

  /* For saving, only 2-byte VAX data types are allowed */
  if(img->_dataType==VAX_I4 || img->_dataType==VAX_R4) img->_dataType=VAX_I2;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write all matrices in memory to the ECAT file.
 *  If ECAT file exists, it is renamed as fname%
\return If ok, 0 is returned.
 */
int ecat63WriteAllImg(
  /** Name of the output ECAT 6.3 file */
  char *fname,
  /** Data structure from which the data is written */
  IMG *img
) {
  int                 frame, plane, n, i, j, m, ret=0;
  float               f, fmax, fmin, g, scale;
  short int          *sdata, *sptr, smin, smax;
  FILE               *fp;
  ECAT63_mainheader   main_header;
  ECAT63_imageheader  image_header;
  ECAT63_scanheader   scan_header;
  struct tm          *scanStart;


  if(IMG_TEST) printf("ecat63WriteAllImg(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {strcpy(ecat63errmsg, "invalid ECAT filename"); return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    strcpy(ecat63errmsg, "image data is empty"); return(2);}
  if(img->_dataType<1) img->_dataType=VAX_I2;

  /* Initiate headers */
  memset(&main_header, 0, sizeof(ECAT63_mainheader));
  memset(&image_header,0, sizeof(ECAT63_imageheader));
  memset(&scan_header, 0, sizeof(ECAT63_scanheader));

  /* Make a main header */
  main_header.sw_version=2;
  main_header.num_planes=img->dimz;  
  main_header.num_frames=img->dimt;  
  main_header.num_gates=1;
  main_header.num_bed_pos=1;
  if(img->type==IMG_TYPE_IMAGE) main_header.file_type=IMAGE_DATA;
  else main_header.file_type=RAW_DATA;
  main_header.data_type=img->_dataType;
  if(img->scanner>0) main_header.system_type=img->scanner;
  else main_header.system_type=ECAT63_SYSTEM_TYPE_DEFAULT;
  main_header.calibration_factor=1.0;
  main_header.axial_fov=img->axialFOV/10.0;
  main_header.transaxial_fov=img->transaxialFOV/10.0;
  main_header.plane_separation=img->sizez/10.0;
  main_header.calibration_units=imgUnitToEcat6(img);
  strncpy(main_header.radiopharmaceutical, img->radiopharmaceutical, 32);
  scanStart=localtime(&img->scanStart);
  main_header.scan_start_year=scanStart->tm_year+1900;
  main_header.scan_start_month=scanStart->tm_mon+1;
  main_header.scan_start_day=scanStart->tm_mday;
  main_header.scan_start_hour=scanStart->tm_hour;
  main_header.scan_start_minute=scanStart->tm_min;
  main_header.scan_start_second=scanStart->tm_sec;
  main_header.isotope_halflife=img->isotopeHalflife;
  strcpy(main_header.isotope_code, imgIsotope(img));
  strcpy(main_header.study_name, img->studyNr);
  strcpy(main_header.patient_name, img->patientName);
  if(IMG_TEST) ecat63PrintMainheader(&main_header);

  /* Allocate memory for matrix data */
  sdata=(short int*)malloc(img->dimx*img->dimy*sizeof(short int));
  if(sdata==NULL) {strcpy(ecat63errmsg, "out of memory"); return(4);}

  /* Open output ECAT file */
  fp=ecat63Create(fname, &main_header);
  if(fp==NULL) {strcpy(ecat63errmsg, "cannot write ECAT file"); return(3);}

  /* Set the subheader contents, as far as possible */
  switch(main_header.file_type) {
    case RAW_DATA:
      scan_header.data_type=main_header.data_type;
      scan_header.dimension_1=img->dimx;
      scan_header.dimension_2=img->dimy;
      scan_header.frame_duration_sec=1;
      scan_header.scale_factor=1.0;
      scan_header.frame_start_time=0;
      scan_header.frame_duration=1000;
      scan_header.loss_correction_fctr=1.0;
      //if(IMG_TEST) ecat63PrintScanheader(&scan_header);
      break;
    case IMAGE_DATA:
      image_header.data_type=main_header.data_type;
      image_header.num_dimensions=2;
      image_header.dimension_1=img->dimx;
      image_header.dimension_2=img->dimy;
      image_header.recon_scale=img->zoom;
      image_header.quant_scale=1.0;
      image_header.slice_width=img->sizez/10.;
      image_header.pixel_size=img->sizex/10.;
      image_header.frame_start_time=0;
      image_header.frame_duration=1000;
      image_header.plane_eff_corr_fctr=1.0;
      image_header.decay_corr_fctr=1.0;
      image_header.loss_corr_fctr=1.0;
      image_header.ecat_calibration_fctr=1.0;
      image_header.well_counter_cal_fctr=1.0;
      image_header.quant_units=main_header.calibration_units;
      //if(IMG_TEST) ecat63PrintImageheader(&image_header);
      break;
  }

  /* Write one matrix at a time */
  n=0;
  for(plane=1; plane<=img->dimz; plane++) for(frame=1; frame<=img->dimt; frame++) {
    /* Scale data to short ints */
    /* Search min and max */
    fmin=fmax=f=img->m[plane-1][0][0][frame-1];
    for(i=0; i<img->dimy; i++) for(j=0; j<img->dimx; j++) {
      f=img->m[plane-1][i][j][frame-1];
      if(f>fmax) fmax=f; else if(f<fmin) fmin=f;
    }
    /* Calculate scaling factor */
    if(fabs(fmin)>fabs(fmax)) g=fabs(fmin); else g=fabs(fmax);
    if(g!=0) scale=32766./g; else scale=1.0;
    //printf("fmin=%e fmax=%e g=%e scale=%e\n", fmin, fmax, g, scale);
    /* Scale matrix data to shorts */
    sptr=sdata;
    for(i=0; i<img->dimy; i++) for(j=0; j<img->dimx; j++) {
      *sptr=(short int)temp_roundf(scale*img->m[plane-1][i][j][frame-1]);
      sptr++;
    }
    /* Calculate and set subheader min&max and scale */
    smin=(short int)temp_roundf(scale*fmin); smax=(short int)temp_roundf(scale*fmax);
    if(main_header.file_type==RAW_DATA) {
      scan_header.scan_min=smin; scan_header.scan_max=smax;
      scan_header.scale_factor=1.0/scale;
      scan_header.frame_start_time=(int)temp_roundf(1000.*img->start[frame-1]);
      scan_header.frame_duration=
        (int)temp_roundf(1000.*(img->end[frame-1]-img->start[frame-1]));
      scan_header.sample_distance=(img->sampleDistance)/10.0;
    } else if(main_header.file_type==IMAGE_DATA) {
      image_header.image_min=smin; image_header.image_max=smax;
      image_header.quant_scale=1.0/scale;
      image_header.frame_start_time=(int)temp_roundf(1000.*img->start[frame-1]);
      image_header.frame_duration=
        (int)temp_roundf(1000.*(img->end[frame-1]-img->start[frame-1]));
      /* Set decay correction factor */
      if(img->decayCorrected)
        image_header.decay_corr_fctr=img->decayCorrFactor[frame-1];
    } 
    /* Write matrix data */
    m=mat_numcod(frame, img->planeNumber[plane-1], 1, 0, 0);
    /*m=mat_numcod(frame, plane, 1, 0, 0);*/
    sptr=sdata;
    if(IMG_TEST) printf("  writing matnum=%d\n", m);
    if(main_header.file_type==RAW_DATA) {
      if(IMG_TEST) ecat63PrintScanheader(&scan_header);
      ret=ecat63WriteScan(fp, m, &scan_header, sptr);
    } else if(main_header.file_type==IMAGE_DATA) {
      if(IMG_TEST) ecat63PrintImageheader(&image_header);
      ret=ecat63WriteImage(fp, m, &image_header, sptr);
    }
    if(ret) {
      sprintf(ecat63errmsg, "cannot write data on pl%02d fr%02d (%d).",
        plane, frame, ret);
      fclose(fp); remove(fname); free(sdata);
      return(9);
    }
    n++;
  } /* next matrix */
  if(IMG_TEST) printf("  %d matrices written in %s\n", n, fname);

  /* Close file and free memory */
  fclose(fp); free(sdata);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/**
    Reads one CTI ECAT 6.3 plane (all frames or gates) at a time to memory.
    Img data must be initialized before this procedure.
    Existing img->_dataType is not changed.
    If img data structure is empty, reads the first plane.
    If img data structure contains data, reads the next plane.
    Any existing data in img is cleared and replaced by the new plane.
\return If ok, 0 or 1 is returned.
    Return value is 1, if next plane was requested but not found anymore.
 */
int ecat63ReadPlaneToImg(
  /** Name of the input ECAT 6.3 file */
  char *fname,
  /** Data structure in which the file is read */
  IMG *img
) {
  int                 i, j, m, ret, blkNr=0, dim_x=0, dim_y=0, pxlNr=0, del_nr=0;
  int                 frameNr, planeNr, datatype=0, firstm=0, isFirst=0;
  int                 frame, plane, prev_frame, prev_plane, prev_frameNr=0;
  FILE               *fp;
  ECAT63_mainheader   main_header;
  MATRIXLIST          mlist;
  MatDir              tmpMatdir;
  Matval              matval;
  ECAT63_imageheader  image_header;
  ECAT63_scanheader   scan_header;
  ECAT63_attnheader   attn_header;
  ECAT63_normheader   norm_header;
  float               scale=1.0;
  short int          *sptr;
  char               *mdata=NULL, *mptr;
  int                *iptr;
  struct tm           scanStart;


  if(IMG_TEST) printf("ecat63ReadPlaneToImg(%s, *img)\n", fname);
  /* Check the arguments */
  if(fname==NULL) {strcpy(ecat63errmsg, "invalid ECAT filename"); return(2);}
  if(img==NULL || img->status==IMG_STATUS_UNINITIALIZED) {
    strcpy(ecat63errmsg, "image data not initialized"); return(2);}

  /* Open input CTI file for read */
  if((fp=fopen(fname, "rb")) == NULL) {
    strcpy(ecat63errmsg, "cannot open ECAT file"); return(3);}

  /* Read main header */
  if((ret=ecat63ReadMainheader(fp, &main_header))) {
    sprintf(ecat63errmsg, "cannot read main header (%d)", ret);
    fclose(fp); return(4);
  }
  if(IMG_TEST) ecat63PrintMainheader(&main_header);

  /* Read matrix list and nr */
  ecat63InitMatlist(&mlist);
  ret=ecat63ReadMatlist(fp, &mlist);
  if(ret) {
    sprintf(ecat63errmsg, "cannot read matrix list (%d)", ret);
    fclose(fp); return(5);
  }
  if(mlist.matrixNr<=0) {
    strcpy(ecat63errmsg, "matrix list is empty"); fclose(fp); return(5);}
  /* Sort matrix list */
  for(i=0; i<mlist.matrixNr-1; i++) for(j=i+1; j<mlist.matrixNr; j++) {
    if(mlist.matdir[i].matnum>mlist.matdir[j].matnum) {
      tmpMatdir=mlist.matdir[i];
      mlist.matdir[i]=mlist.matdir[j]; mlist.matdir[j]=tmpMatdir;
    }
  }
  /* Trim extra frames */
  if(main_header.num_frames>0) {
    del_nr=ecat63DeleteLateFrames(&mlist, main_header.num_frames);
    if(IMG_TEST && del_nr>0)
      printf("  %d entries in matrix list will not be used.\n", del_nr);
  }

  /* Decide the plane to read */
  if(img->status==IMG_STATUS_OCCUPIED) {
    /* img contains data */
    isFirst=0; prev_frameNr=img->dimt;
    /* get the current plane in there */
    prev_plane=img->planeNumber[img->dimz-1];
    /* Clear all data in img: not here but only after finding new data */
  } else {
    /* img does not contain any data */
    isFirst=1;
    /* set "current plane" to -1 */
    prev_plane=-1;
  }
  /* from sorted matrix list, find the first plane larger than the current one */
  for(m=0, plane=-1; m<mlist.matrixNr; m++) if(mlist.matdir[m].matstat==1) {
    mat_numdoc(mlist.matdir[m].matnum, &matval);
    if(matval.plane>prev_plane) {plane=matval.plane; break;}
  }
  /* If not found, return an error or 1 if this was not the first one */
  if(plane<0) {
    fclose(fp); ecat63EmptyMatlist(&mlist);
    if(isFirst) {
      sprintf(ecat63errmsg, "ECAT file contains no matrices");
      return(7);
    } else {
      sprintf(ecat63errmsg, "ECAT file contains no more planes");
      if(IMG_TEST) printf("%s\n", ecat63errmsg);
      return(1);
    }
  }
  if(IMG_TEST) printf("Next plane to read: %d\n", plane);
  /* Clear all data in img */
  imgEmpty(img);

  /* Calculate the number of frames and gates */
  prev_frame=frame=-1; frameNr=0; ret=0; planeNr=1;
  for(m=0; m<mlist.matrixNr; m++) if(mlist.matdir[m].matstat==1) {
    /* get frame and plane; work only with current plane */
    mat_numdoc(mlist.matdir[m].matnum, &matval);
    if(matval.plane<plane) continue; else if(matval.plane>plane) break;
    if(main_header.num_frames>=main_header.num_gates) frame=matval.frame;
    else frame=matval.gate;
    frameNr++;
    /* frame (gate) numbers must be continuous */
    if(prev_frame>0 && frame!=prev_frame+1) {ret=1; break;}
    prev_frame=frame;
    /* Calculate and check the size of one data matrix */
    if(frameNr==1) {
      firstm=m;
      blkNr=mlist.matdir[m].endblk-mlist.matdir[m].strtblk;
    } else if(blkNr!=mlist.matdir[m].endblk-mlist.matdir[m].strtblk) {
      ret=2; break;
    }
    prev_frame=frame;
  } /* next matrix */
  if(ret==1) {
    strcpy(ecat63errmsg, "missing matrix");
    ecat63EmptyMatlist(&mlist); fclose(fp); return(6);
  }
  if(ret==2) {
    strcpy(ecat63errmsg, "matrix sizes are different");
    ecat63EmptyMatlist(&mlist); fclose(fp); return(6);
  }
  /* Check that frameNr is equal to the dimt in IMG */
  if(!isFirst && frameNr!=prev_frameNr) {
    strcpy(ecat63errmsg, "different frame nr in different planes");
    ecat63EmptyMatlist(&mlist); fclose(fp); return(6);
  }

  /* Read x,y-dimensions from 1st matrix subheader on current plane */
  m=firstm; if(main_header.file_type==IMAGE_DATA) {
    ret=ecat63ReadImageheader(fp, mlist.matdir[m].strtblk, &image_header);
    dim_x=image_header.dimension_1; dim_y=image_header.dimension_2;
  } else if(main_header.file_type==RAW_DATA) {
    ret=ecat63ReadScanheader(fp, mlist.matdir[m].strtblk, &scan_header);
    dim_x=scan_header.dimension_1; dim_y=scan_header.dimension_2;
  } else if(main_header.file_type==ATTN_DATA) {
    ret=ecat63ReadAttnheader(fp, mlist.matdir[m].strtblk, &attn_header);
    dim_x=attn_header.dimension_1; dim_y=attn_header.dimension_2;
  } else if(main_header.file_type==NORM_DATA) {
    ret=ecat63ReadNormheader(fp, mlist.matdir[m].strtblk, &norm_header);
    dim_x=norm_header.dimension_1; dim_y=norm_header.dimension_2;
  }
  pxlNr=dim_x*dim_y;
  if(ret) {
    sprintf(ecat63errmsg, "cannot read matrix %u subheader", mlist.matdir[m].matnum);
    ecat63EmptyMatlist(&mlist); fclose(fp); return(7);
  }

  /* Allocate memory for ECAT data matrix */
  if(IMG_TEST) printf("allocating memory for %d blocks\n", blkNr);
  mdata=(char*)malloc(blkNr*MatBLKSIZE); if(mdata==NULL) {
    strcpy(ecat63errmsg, "out of memory");
    fclose(fp); ecat63EmptyMatlist(&mlist); return(8);
  }
  /* Allocate memory for whole img data */
  ret=imgAllocate(img, planeNr, dim_y, dim_x, frameNr);
  if(ret) {
    sprintf(ecat63errmsg, "out of memory (%d)", ret);
    fclose(fp); ecat63EmptyMatlist(&mlist); free(mdata); return(8);
  }

  /* Fill img info with ECAT main and sub header information */
  img->scanner=main_header.system_type;
  img->unit=main_header.calibration_units; /* this continues below */
  strncpy(img->radiopharmaceutical, main_header.radiopharmaceutical, 32);
  img->isotopeHalflife=main_header.isotope_halflife;
  memset(&scanStart, 0, sizeof(struct tm));
  scanStart.tm_year=main_header.scan_start_year-1900;
  scanStart.tm_mon=main_header.scan_start_month-1;
  scanStart.tm_mday=main_header.scan_start_day; scanStart.tm_yday=0;
  scanStart.tm_hour=main_header.scan_start_hour;
  scanStart.tm_min=main_header.scan_start_minute;
  scanStart.tm_sec=main_header.scan_start_second;
  scanStart.tm_isdst=-1;
  img->scanStart=mktime(&scanStart); /*printf("img->scanStart=%d\n", img->scanStart);*/
  img->axialFOV=10.*main_header.axial_fov;
  img->transaxialFOV=10.*main_header.transaxial_fov;
  strncpy(img->studyNr, main_header.study_name, MAX_STUDYNR_LEN);
  strncpy(img->patientName, main_header.patient_name, 31);
  img->sizez=10.*main_header.plane_separation;
  if(main_header.file_type==IMAGE_DATA) {
    img->type=IMG_TYPE_IMAGE;
    if(img->unit<1) img->unit=image_header.quant_units;
    img->zoom=image_header.recon_scale;
    if(image_header.decay_corr_fctr>1.0) img->decayCorrected=1;
    img->sizex=img->sizey=10.*image_header.pixel_size;
    if(img->sizez<10.*image_header.slice_width)
      img->sizez=10.*image_header.slice_width;
  } else if(main_header.file_type==RAW_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  } else if(main_header.file_type==ATTN_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  } else if(main_header.file_type==NORM_DATA) {
    img->type=IMG_TYPE_RAW;
    img->decayCorrected=0;
  }
  img->planeNumber[0]=plane;
  /* Set _fileFormat */
  img->_fileFormat=IMG_E63;

  /* Read one ECAT matrix at a time and put them to img */
  for(m=firstm, frame=1; m<mlist.matrixNr; m++) if(mlist.matdir[m].matstat==1) {
    /* get plane */
    mat_numdoc(mlist.matdir[m].matnum, &matval);
    if(matval.plane>plane) break; /* Quit when current plane is read */
    /* Read subheader */
    if(main_header.file_type==IMAGE_DATA) {
      ret=ecat63ReadImageheader(fp, mlist.matdir[m].strtblk, &image_header);
      if(dim_x!=image_header.dimension_1 || dim_y!=image_header.dimension_2) ret=1;
      scale=image_header.quant_scale;
      if(image_header.ecat_calibration_fctr>0.0
         && fabs(main_header.calibration_factor/image_header.ecat_calibration_fctr-1.0)>0.0001)
        scale*=image_header.ecat_calibration_fctr;
      if(img->unit==0 && image_header.quant_units>0) img->unit=image_header.quant_units;
      datatype=image_header.data_type;
      img->start[frame-1]=image_header.frame_start_time/1000.;
      img->end[frame-1]=img->start[frame-1]+image_header.frame_duration/1000.;
      img->mid[frame-1]=0.5*(img->start[frame-1]+img->end[frame-1]);
      if(image_header.decay_corr_fctr>1.0)
        img->decayCorrFactor[frame-1]=image_header.decay_corr_fctr;
    } else if(main_header.file_type==RAW_DATA) {
      ret=ecat63ReadScanheader(fp, mlist.matdir[m].strtblk, &scan_header);
      if(dim_x!=scan_header.dimension_1 || dim_y!=scan_header.dimension_2) ret=1;
      scale=scan_header.scale_factor;
      if(scan_header.loss_correction_fctr>0.0) scale*=scan_header.loss_correction_fctr;
      datatype=scan_header.data_type;
      img->start[frame-1]=scan_header.frame_start_time/1000.;
      img->end[frame-1]=img->start[frame-1]+scan_header.frame_duration/1000.;
      img->mid[frame-1]=0.5*(img->start[frame-1]+img->end[frame-1]);
      img->sampleDistance=10.0*scan_header.sample_distance;
    } else if(main_header.file_type==ATTN_DATA) {
      ret=ecat63ReadAttnheader(fp, mlist.matdir[m].strtblk, &attn_header);
      if(dim_x!=attn_header.dimension_1 || dim_y!=attn_header.dimension_2) ret=1;
      scale=attn_header.scale_factor;
      datatype=attn_header.data_type;
    } else if(main_header.file_type==NORM_DATA) {
      ret=ecat63ReadNormheader(fp, mlist.matdir[m].strtblk, &norm_header);
      if(dim_x!=norm_header.dimension_1 || dim_y!=norm_header.dimension_2) ret=1;
      scale=norm_header.scale_factor;
      datatype=norm_header.data_type;
    } else datatype=-1;
    if(ret) {
      sprintf(ecat63errmsg, "cannot read matrix %u subheader", mlist.matdir[m].matnum);
      ecat63EmptyMatlist(&mlist); free(mdata); fclose(fp); return(7);
    }
    if(main_header.calibration_factor>0.0) scale*=main_header.calibration_factor;
    if(IMG_TEST>2) printf("scale=%e datatype=%d\n", scale, datatype);
    /* Read the pixel data */
    ret=ecat63ReadMatdata(fp, mlist.matdir[m].strtblk+1,
         mlist.matdir[m].endblk-mlist.matdir[m].strtblk,
         mdata, datatype);
    if(ret) {
      strcpy(ecat63errmsg, "cannot read matrix data");
      ecat63EmptyMatlist(&mlist); free(mdata); fclose(fp); return(9);
    }
    if(datatype==BYTE_TYPE) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr++) {
        img->m[0][i][j][frame-1]=scale*(float)(*mptr);
      }
    } else if(datatype==VAX_I2 || datatype==SUN_I2) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=2) {
        sptr=(short int*)mptr;
        img->m[0][i][j][frame-1]=scale*(float)(*sptr);
      }
    } else if(datatype==VAX_I4 || datatype==SUN_I4) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=4) {
        iptr=(int*)mptr;
        img->m[0][i][j][frame-1]=scale*(float)(*iptr);
      }
    } else if(datatype==VAX_R4 || datatype==IEEE_R4) {
      for(i=0, mptr=mdata; i<dim_y; i++) for(j=0; j<dim_x; j++, mptr+=4) {
        memcpy(&img->m[0][i][j][frame-1], mptr, 4);
        img->m[0][i][j][frame-1]*=scale;
      }
    }
    frame++;
  } /* next matrix (frame) */
  /* Set the unit */
  imgUnitFromEcat(img, img->unit);

  /* Free memory and close file */
  free(mdata);
  ecat63EmptyMatlist(&mlist);
  fclose(fp);

  /* Set _dataType if it has not been set before */
  if(img->_dataType<1) img->_dataType=datatype;
  /* For saving, only 2-byte VAX data types are allowed */
  if(img->_dataType==VAX_I4 || img->_dataType==VAX_R4) img->_dataType=VAX_I2;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/**
    Adds all matrices in memory to the ECAT file.
    If ECAT file does not exist, it is created.
    Please note that existing ECAT file is NOT saved as fname%
\return If ok, 0 is returned.
 */
int ecat63AddImg(
  /** Name of the output ECAT 6.3 file */
  char *fname,
  /** Data structure from which the data is written */
  IMG *img
) {
  int                 n, i, j, m, ret=0, add=0;
  int                 frameNr, planeNr;
  int                 frame, plane, prev_frame, prev_plane;
  float               f, fmax, fmin, g, scale;
  short int          *sdata, *sptr, smin, smax;
  FILE               *fp;
  ECAT63_mainheader   main_header;
  ECAT63_imageheader  image_header;
  ECAT63_scanheader   scan_header;
  MATRIXLIST          mlist;
  MatDir              tmpMatdir;
  Matval              matval;
  struct tm          *scanStart;


  if(IMG_TEST) printf("ecat63AddImg(%s, *img)\n", fname);
  if(IMG_TEST>1) imgInfo(img);
  /* Check the arguments */
  if(fname==NULL) {strcpy(ecat63errmsg, "invalid ECAT filename"); return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    strcpy(ecat63errmsg, "image data is empty"); return(2);}
  if(img->_dataType<1) img->_dataType=VAX_I2;

  /* Initiate headers */
  memset(&main_header, 0, sizeof(ECAT63_mainheader));
  memset(&image_header,0, sizeof(ECAT63_imageheader));
  memset(&scan_header, 0, sizeof(ECAT63_scanheader));

  /* Make a main header */
  main_header.sw_version=2;
  main_header.num_planes=img->dimz;  
  main_header.num_frames=img->dimt;  
  main_header.num_gates=1;
  main_header.num_bed_pos=1;
  if(img->type==IMG_TYPE_IMAGE) main_header.file_type=IMAGE_DATA;
  else main_header.file_type=RAW_DATA;
  main_header.data_type=img->_dataType;
  if(img->scanner>0) main_header.system_type=img->scanner;
  else main_header.system_type=ECAT63_SYSTEM_TYPE_DEFAULT;
  main_header.calibration_factor=1.0;
  main_header.axial_fov=img->axialFOV/10.0;
  main_header.transaxial_fov=img->transaxialFOV/10.0;
  main_header.plane_separation=img->sizez/10.0;
  main_header.calibration_units=imgUnitToEcat6(img);
  strncpy(main_header.radiopharmaceutical, img->radiopharmaceutical, 32);
  scanStart=localtime(&img->scanStart);
  main_header.scan_start_year=scanStart->tm_year+1900;
  main_header.scan_start_month=scanStart->tm_mon+1;
  main_header.scan_start_day=scanStart->tm_mday;
  main_header.scan_start_hour=scanStart->tm_hour;
  main_header.scan_start_minute=scanStart->tm_min;
  main_header.scan_start_second=scanStart->tm_sec;
  main_header.isotope_halflife=img->isotopeHalflife;
  strcpy(main_header.isotope_code, imgIsotope(img));
  strcpy(main_header.study_name, img->studyNr);
  strcpy(main_header.patient_name, img->patientName);

  /* Check if the ECAT file exists */
  if(access(fname, 0) != -1) {
    if(IMG_TEST) printf("Opening existing ECAT file.\n");
    add=1;
    /* Open the ECAT file */
    if((fp=fopen(fname, "rb+")) == NULL) {
      strcpy(ecat63errmsg, "cannot create ECAT file"); return(3);}
    /* Read main header */
    if((ret=ecat63ReadMainheader(fp, &main_header))) {
      sprintf(ecat63errmsg, "cannot read main header (%d)", ret);
      fclose(fp); return(3);
    }
    fflush(fp);
    /* Check that filetypes are matching */
    if((main_header.file_type==IMAGE_DATA && img->type==IMG_TYPE_IMAGE) ||
       (main_header.file_type==RAW_DATA && img->type==IMG_TYPE_RAW)) {
      /* ok */
    } else {
      sprintf(ecat63errmsg, "cannot add different filetype");
      fclose(fp); return(3);
    }
  } else {
    if(IMG_TEST) printf("ECAT file does not exist.\n");
    add=0;
    /* Create output ECAT file */
    fp=ecat63Create(fname, &main_header);
    if(fp==NULL) {strcpy(ecat63errmsg, "cannot create ECAT file"); return(3);}
  }
  if(IMG_TEST) ecat63PrintMainheader(&main_header);

  /* Allocate memory for matrix data */
  sdata=(short int*)malloc(img->dimx*img->dimy*sizeof(short int));
  if(sdata==NULL) {strcpy(ecat63errmsg, "out of memory"); return(4);}

  /* Set the subheader contents, as far as possible */
  switch(main_header.file_type) {
    case RAW_DATA:
      scan_header.data_type=main_header.data_type;
      scan_header.dimension_1=img->dimx;
      scan_header.dimension_2=img->dimy;
      scan_header.frame_duration_sec=1;
      scan_header.scale_factor=1.0;
      scan_header.frame_start_time=0;
      scan_header.frame_duration=1000;
      scan_header.loss_correction_fctr=1.0;
      scan_header.sample_distance=(img->sampleDistance)/10.0;
      //if(IMG_TEST) ecat63PrintScanheader(&scan_header);
      break;
    case IMAGE_DATA:
      image_header.data_type=main_header.data_type;
      image_header.num_dimensions=2;
      image_header.dimension_1=img->dimx;
      image_header.dimension_2=img->dimy;
      image_header.recon_scale=img->zoom;
      image_header.quant_scale=1.0;
      image_header.slice_width=img->sizez/10.;
      image_header.pixel_size=img->sizex/10.;
      image_header.frame_start_time=0;
      image_header.frame_duration=1000;
      image_header.plane_eff_corr_fctr=1.0;
      image_header.decay_corr_fctr=0.0;
      image_header.loss_corr_fctr=1.0;
      image_header.ecat_calibration_fctr=1.0;
      image_header.well_counter_cal_fctr=1.0;
      image_header.quant_units=main_header.calibration_units;
      //if(IMG_TEST) ecat63PrintImageheader(&image_header);
      break;
  }

  /* Write one matrix at a time */
  n=0;
  for(plane=1; plane<=img->dimz; plane++) for(frame=1; frame<=img->dimt; frame++) {
    /* Scale data to short ints */
    /* Search min and max */
    fmin=fmax=f=img->m[plane-1][0][0][frame-1];
    for(i=0; i<img->dimy; i++) for(j=0; j<img->dimx; j++) {
      f=img->m[plane-1][i][j][frame-1];
      if(f>fmax) fmax=f; else if(f<fmin) fmin=f;
    }
    /* Calculate scaling factor */
    if(fabs(fmin)>fabs(fmax)) g=fabs(fmin); else g=fabs(fmax);
    if(g!=0) scale=32766./g; else scale=1.0;
    //printf("fmin=%e fmax=%e g=%e scale=%e\n", fmin, fmax, g, scale);
    /* Scale matrix data to shorts */
    sptr=sdata;
    for(i=0; i<img->dimy; i++) for(j=0; j<img->dimx; j++) {
      *sptr=(short int)temp_roundf(scale*img->m[plane-1][i][j][frame-1]);
      sptr++;
    }
    /* Calculate and set subheader min&max and scale */
    smin=(short int)temp_roundf(scale*fmin); smax=(short int)temp_roundf(scale*fmax);
    if(main_header.file_type==RAW_DATA) {
      scan_header.scan_min=smin; scan_header.scan_max=smax;
      scan_header.scale_factor=1.0/scale;
      scan_header.frame_start_time=(int)temp_roundf(1000.*img->start[frame-1]);
      scan_header.frame_duration=
        (int)temp_roundf(1000.*(img->end[frame-1]-img->start[frame-1]));
    } else if(main_header.file_type==IMAGE_DATA) {
      image_header.image_min=smin; image_header.image_max=smax;
      image_header.quant_scale=1.0/scale;
      image_header.frame_start_time=(int)temp_roundf(1000.*img->start[frame-1]);
      image_header.frame_duration=
        (int)temp_roundf(1000.*(img->end[frame-1]-img->start[frame-1]));
      /* Set decay correction factor */
      if(img->decayCorrected)
        image_header.decay_corr_fctr=img->decayCorrFactor[frame-1];
    } 
    /* Write matrix data */
    m=mat_numcod(frame, img->planeNumber[plane-1], 1, 0, 0);
    sptr=sdata;
    if(IMG_TEST) printf("  writing matnum=%d\n", m);
    if(main_header.file_type==RAW_DATA) {
      //if(IMG_TEST) ecat63PrintScanheader(&scan_header);
      ret=ecat63WriteScan(fp, m, &scan_header, sptr);
    } else if(main_header.file_type==IMAGE_DATA) {
      //if(IMG_TEST) ecat63PrintImageheader(&image_header);
      ret=ecat63WriteImage(fp, m, &image_header, sptr);
    }
    if(ret) {
      sprintf(ecat63errmsg, "cannot write data on pl%02d fr%02d (%d).",
        plane, frame, ret);
      fclose(fp); remove(fname); free(sdata);
      return(9);
    }
    n++;
  } /* next matrix */
  if(IMG_TEST) printf("  %d matrices written in %s\n", n, fname);

  /* Free matrix memory */
  free(sdata);

  /* If matrices were added, set main header */
  if(add==1) {
    fflush(fp);
    /* Read matrix list */
    ecat63InitMatlist(&mlist);
    ret=ecat63ReadMatlist(fp, &mlist);
    if(ret) {
      sprintf(ecat63errmsg, "cannot read matrix list (%d)", ret);
      fclose(fp); return(21);
    }
    if(mlist.matrixNr<=0) {
      strcpy(ecat63errmsg, "matrix list is empty"); fclose(fp); return(21);}
    /* Sort matrix list */
    for(i=0; i<mlist.matrixNr-1; i++) for(j=i+1; j<mlist.matrixNr; j++) {
      if(mlist.matdir[i].matnum>mlist.matdir[j].matnum) {
        tmpMatdir=mlist.matdir[i];
        mlist.matdir[i]=mlist.matdir[j]; mlist.matdir[j]=tmpMatdir;
      }
    }
    /* Calculate the number of planes and frames */
    prev_plane=plane=-1; prev_frame=frame=-1; frameNr=planeNr=0;
    for(m=0; m<mlist.matrixNr; m++) {
      mat_numdoc(mlist.matdir[m].matnum, &matval);
      plane=matval.plane; frame=matval.frame;
      if(plane!=prev_plane) {frameNr=1; planeNr++;} else {frameNr++;}
      prev_plane=plane; prev_frame=frame;
    } /* next matrix */
    ecat63EmptyMatlist(&mlist);
    main_header.num_planes=planeNr; main_header.num_frames=frameNr;
    /* Write main header */
    ret=ecat63WriteMainheader(fp, &main_header);
    if(ret) {
      sprintf(ecat63errmsg, "cannot write mainheader (%d)", ret);
      fclose(fp); return(21);
    }
  }

  /* Close file and free memory */
  fclose(fp);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy ECAT 7 main header information into IMG */
void imgGetEcat7MHeader(IMG *img, ECAT7_mainheader *h)
{
  img->scanner=h->system_type;
  imgUnitFromEcat7(img, h);
  strncpy(img->radiopharmaceutical, h->radiopharmaceutical, 32);
  img->isotopeHalflife=h->isotope_halflife;
  img->scanStart=h->scan_start_time;
  img->axialFOV=10.0*h->distance_scanned;
  img->transaxialFOV=10.0*h->transaxial_fov;
  strncpy(img->studyNr, h->study_type, MAX_STUDYNR_LEN);
  strcpy(img->patientName, h->patient_name);
  img->sizez=10.0*h->plane_separation;
  switch(h->file_type) {
    case ECAT7_IMAGE8:
    case ECAT7_IMAGE16:
    case ECAT7_VOLUME8:
    case ECAT7_VOLUME16: img->type=IMG_TYPE_IMAGE; break;
    default: img->type=IMG_TYPE_RAW;
  }
  img->orientation=h->patient_orientation;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy information from IMG to ECAT 7 main header */
void imgSetEcat7MHeader(IMG *img, ECAT7_mainheader *h)
{
  h->sw_version=72;
  if(img->type==IMG_TYPE_RAW) {
    strcpy(h->magic_number, ECAT7S_MAGICNR);
    h->file_type=ECAT7_3DSCAN;
  } else {
    strcpy(h->magic_number, ECAT7V_MAGICNR);
    h->file_type=ECAT7_VOLUME16;
  }
  h->system_type=img->scanner;
  h->scan_start_time=img->scanStart;
  h->isotope_halflife=img->isotopeHalflife;
  imgUnitToEcat7(img, h);
  h->ecat_calibration_factor=1.0;
  h->transaxial_fov=img->transaxialFOV/10.0;
  h->num_planes=img->dimz; //h->num_planes=1; 
  h->num_frames=img->dimt;  
  h->num_gates=1;
  h->num_bed_pos=0;
  h->distance_scanned=img->axialFOV/10.0;
  h->plane_separation=img->sizez/10.0;
  strncpy(h->radiopharmaceutical, img->radiopharmaceutical, 32);
  strcpy(h->isotope_name, imgIsotope(img));
  strcpy(h->study_type, img->studyNr);
  strcpy(h->patient_name, img->patientName);
  h->patient_orientation=img->orientation;
}
/*****************************************************************************/

/*****************************************************************************/
/** Set IMG calibration unit based on ECAT 6.3 unit */
void imgUnitFromEcat(IMG *img, int ecat_unit)
{
  switch(ecat_unit) {
    case  0: /* Unknown               */
      img->unit=0; break;
    case  1: /* MBq/mL                */
      img->unit=13; break;
    case  2: /* ECAT counts           */
      img->unit=2; break;
    case  3: /* uCi/ml                */
      img->unit=0; break;
    case  4: /* LMRGlu                */
    case  5: /* LMRUGlu umol/min/100g */
    case  6: /* LMRUGlu mg/min/100g   */
      img->unit=0; break;
    case  7: /* nCi/mL                */
      img->unit=12; break;
    case  8: /* Well counts           */
      img->unit=1; break;
    case  9: /* Becquerels            */
      img->unit=14; break;
    case 10: /* kBq/mL                */
      img->unit=3; break;
    case 11: /* 1/min                 */
      img->unit=6; break;
    case 12: /* mL/min/100g           */
      img->unit=10; break;
    case 13: /* sec*kBq/mL            */
      img->unit=4; break;
    case 14: /* sec*nCi/mL            */
      img->unit=0; break;
    case 15: /* 1/sec                 */
      img->unit=5; break;
    case 16: /* Unitless              */
    case 17: /* Unknown               */
    default:
      img->unit=0; break;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Set IMG calibration unit based on ECAT7 main header */
void imgUnitFromEcat7(
  /** Pointer to IMG data where unit will be set */
  IMG *img,
  /** Pointer to ECAT7 main header */
  ECAT7_mainheader *h
) {
  if(h->calibration_units==0) { /* Not calibrated */
    img->unit=1;
  } else if(h->calibration_units==1) {
    if(h->calibration_units_label==0)
      img->unit=14;
    else if(strcasecmp(h->data_units, "MBq/cc")==0)
      img->unit=13;
    else if(strcasecmp(h->data_units, "MBq/mL")==0)
      img->unit=13;
    else if(strcasecmp(h->data_units, "uCi/cc")==0)
      img->unit=15;
    else if(strcasecmp(h->data_units, "uCi/mL")==0)
      img->unit=15;
    else if(strcasecmp(h->data_units, "nCi/cc")==0)
      img->unit=12;
    else if(strcasecmp(h->data_units, "nCi/mL")==0)
      img->unit=12;
    else if(strcasecmp(h->data_units, "Bq/cc")==0)
      img->unit=14;
    else if(strcasecmp(h->data_units, "Bq/mL")==0)
      img->unit=14;
    else if(strcasecmp(h->data_units, "kBq/cc")==0)
      img->unit=3;
    else if(strcasecmp(h->data_units, "kBq/mL")==0)
      img->unit=3;
    else
      img->unit=0;
  } else {
    img->unit=0;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Return ECAT calibration unit based on IMG unit */
int imgUnitToEcat6(IMG *img)
{
  int ecat_unit;

  switch(img->unit) {
    case 1: ecat_unit=9; break;
    case 2: ecat_unit=2; break;
    case 3: ecat_unit=10; break;
    case 4: ecat_unit=13; break;
    case 5: ecat_unit=15; break;
    case 6: ecat_unit=11; break;
    case 7: ecat_unit=16; break;
    case 8: ecat_unit=16; break;
    case 9: ecat_unit=11; break;
    case 10: ecat_unit=12; break;
    case 11: ecat_unit=16; break;
    case 12: ecat_unit=7; break;
    case 13: ecat_unit=1; break;
    case 14: ecat_unit=9; break;
    case 15: ecat_unit=3; break;
    default: ecat_unit=0; break;
  }
  return(ecat_unit);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set ECAT 7 mainheader calibration units based on IMG unit */
void imgUnitToEcat7(IMG *img, ECAT7_mainheader *h)
{
  switch(img->unit) {
    case 1:
    case 2:
      h->calibration_units=0;
      h->calibration_units_label=1;
      strcpy(h->data_units, "ECAT counts/sec");
      break;
    case 3:
      h->calibration_units=1;
      h->calibration_units_label=1;
      strcpy(h->data_units, "kBq/cc");
      break;
    case 4:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "sec*kBq/mL");
      break;
    case 5:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "1/sec");
      break;
    case 6:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "1/min");
      break;
    case 7:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "mL/mL");
      break;
    case 8:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "mL/dL");
      break;
    case 9:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "mL/(mL*min)");
      break;
    case 10:
      h->calibration_units=2;
      h->calibration_units_label=1;
      strcpy(h->data_units, "mL/(dL*min)");
      break;
    case 11:
      h->calibration_units=1;
      h->calibration_units_label=0;
      strcpy(h->data_units, "");
      break;
    case 12:
      h->calibration_units=1;
      h->calibration_units_label=1;
      strcpy(h->data_units, "nCi/cc");
      break;
    case 13:
      h->calibration_units=1;
      h->calibration_units_label=1;
      strcpy(h->data_units, "MBq/cc");
      break;
    case 14:
      h->calibration_units=1;
      h->calibration_units_label=1;
      strcpy(h->data_units, "Bq/cc");
      break;
    case 15:
      h->calibration_units=1;
      h->calibration_units_label=1;
      strcpy(h->data_units, "uCi/cc");
      break;
    default:
      h->calibration_units=2;
      h->calibration_units_label=0;
      strcpy(h->data_units, "");
      break;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Read Analyze 7.5 image.
    Analyze database name must be given with path. Image and header files
    with .img and .hdr extensions must exist. Also SIF file with .sif
    extension is used, if it exists.
    anaFlipping() determines whether image is flipped in z-direction;
    image is always flipped in x,y-directions.
\return Returns >0 in case of an error and sets IMG->statmsg.
*/
int imgReadAnalyze(
  /** Analyze database name with path, with or without extension */
  char *dbname,
  /** Pointer to initialized IMG strucure */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "image datafile does not exist",
  /*  4 */ "header file does not exist",
  /*  5 */ "unsupported file format",
  /*  6 */ "invalid header contents",
  /*  7 */ "cannot read image data"
  /*  8 */ "cannot read sif data"
  /*  9 */ "wrong sif data"
  };
  FILE *fp;
  int ret, fi, pi, xi, yi;
  float *fdata=NULL, *fptr;
  ANALYZE_DSR dsr;
  char datfile[FILENAME_MAX], hdrfile[FILENAME_MAX], siffile[FILENAME_MAX];
  char buf[128], *cptr;
  int dimNr, dimx, dimy, dimz=1, dimt=1, pxlNr=0;
  SIF sif;
  struct tm *st;


  if(IMG_TEST) printf("imgReadAnalyze(%s, *img)\n", dbname);

  /* Check the arguments */
  img->statmsg=imgmsg[0];
  if(dbname==NULL || !dbname[0]) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_INITIALIZED) {
    img->statmsg=imgmsg[1]; return(2);}
  
  /* Make the image and header filenames */
  if(anaExists(dbname)==0) {
    /* Check if filename was given accidentally with extension */
    strcpy(datfile, dbname); cptr=strrchr(datfile, '.');
    if(cptr!=NULL && (strcmp(cptr, ".img")==0 || strcmp(cptr, ".hdr")==0)) {
      *cptr=(char)0; strcpy(hdrfile, datfile);
      if(anaExists(datfile)==0) { /* still not found */
        img->statmsg=imgmsg[4]; return(3);}
      strcat(datfile, ".img"); strcat(hdrfile, ".hdr");
    } else {
      img->statmsg=imgmsg[4]; return(3);
    }
  } else {
    /* Database name was given and img and hdr files were found */
    strcpy(datfile, dbname); strcat(datfile, ".img");
    strcpy(hdrfile, dbname); strcat(hdrfile, ".hdr");
  }

  /* Read Analyze header file */
  ret=anaReadHeader(hdrfile, &dsr);
  if(ret) {
    if(ret==1) img->statmsg=imgmsg[1];
    else if(ret==2) img->statmsg=imgmsg[4];
    else img->statmsg=imgmsg[5];
    return(3);
  }
  if(IMG_TEST) anaPrintHeader(&dsr, stdout);

  /* Open image datafile */
  if(IMG_TEST) fprintf(stdout, "reading image data %s\n", datfile);
  if((fp=fopen(datfile, "rb")) == NULL) {img->statmsg=imgmsg[7]; return(5);}

  /* Prepare IMG for Analyze image */
  /* Get the image dimensions from header */
  dimNr=dsr.dime.dim[0];
  if(dimNr<2) {fclose(fp); img->statmsg=imgmsg[6]; return(4);}
  dimx=dsr.dime.dim[1]; dimy=dsr.dime.dim[2];
  if(dimNr>2) {dimz=dsr.dime.dim[3]; if(dimNr>3) dimt=dsr.dime.dim[4];}
  pxlNr=dimx*dimy*dimz;
  if(pxlNr<1) {fclose(fp); img->statmsg=imgmsg[6]; return(4);}
  /* Allocate memory for IMG */
  ret=imgAllocate(img, dimz, dimy, dimx, dimt);
  if(ret) {fclose(fp); img->statmsg=imgmsg[2]; return(11);}
  /* Copy information from Analyze header */
  img->type=IMG_TYPE_IMAGE;
  strncpy(img->studyNr, dsr.hist.patient_id, MAX_STUDYNR_LEN);
  strcpy(img->patientName, dsr.hist.patient_id);
  img->sizex=dsr.dime.pixdim[1];
  img->sizey=dsr.dime.pixdim[2];
  img->sizez=dsr.dime.pixdim[3];
  /*if(dsr.dime.funused2>1.E-5) img->zoom=dsr.dime.funused2;*/
  if(dsr.dime.funused3>1.E-5) img->isotopeHalflife=dsr.dime.funused3;
  for(pi=0; pi<dimz; pi++) img->planeNumber[pi]=pi+1;
  if(dsr.little) img->_fileFormat=IMG_ANA_L; else img->_fileFormat=IMG_ANA;
  /* Decay correction */
  if(strstr(dsr.hist.descrip, "Decay corrected.")!=NULL)
    img->decayCorrected=1;
  else if(strstr(dsr.hist.descrip, "No decay correction.")!=NULL)
    img->decayCorrected=0;
  else
    img->decayCorrected=0;

  /* Allocate memory for one image frame */
  fdata=malloc(pxlNr*sizeof(float));
  if(fdata==NULL) {fclose(fp); img->statmsg=imgmsg[2]; return(12);}

  /* Read one image frame at a time */
  for(fi=0; fi<dimt; fi++) {
    fptr=fdata;
    ret=anaReadImagedata(fp, &dsr, fi+1, fptr);
    if(ret) {free(fdata); fclose(fp); img->statmsg=imgmsg[7]; return(7);}
    /* Copy pixel values to IMG */
    fptr=fdata;
    if(anaFlipping()==0) { /* no flipping in z-direction */
      for(pi=0; pi<img->dimz; pi++)
        for(yi=dimy-1; yi>=0; yi--)
          for(xi=dimx-1; xi>=0; xi--)
            img->m[pi][yi][xi][fi]=*fptr++;
    } else {
      for(pi=dimz-1; pi>=0; pi--)
        for(yi=dimy-1; yi>=0; yi--)
          for(xi=dimx-1; xi>=0; xi--)
            img->m[pi][yi][xi][fi]=*fptr++;
    }
  } /* next frame */
  free(fdata);
  fclose(fp);
  
  /* Try to read frame time information from SIF file */
  /* Make filename from database or image data file */
  strcpy(siffile, dbname); strcat(siffile, ".sif");
  if(access(siffile, 0) == -1) {
    strcpy(siffile, datfile); strcat(siffile, ".sif");
  }
  /* Check if SIF file is found */
  if(IMG_TEST) printf("reading SIF file %s\n", siffile);
  if(access(siffile, 0) == -1) {
    if(IMG_TEST) printf(" No SIF file; therefore unknown frame times.\n");
    return(0);
  }
  /* If found, then read it */
  initSIF(&sif); ret=readSIF(siffile, &sif);
  if(ret) {img->statmsg=imgmsg[8]; return(21);}
  /* Copy scan start time */
  if(sif.scantime>0) {
    st=localtime(&sif.scantime); strftime(buf, 128, "%Y-%m-%d %T", st);
    img->scanStart=sif.scantime;
  }
  /* Copy frame times */
  if(sif.frameNr!=img->dimt) {
    img->statmsg=imgmsg[9]; emptySIF(&sif); return(22);}
  for(fi=0; fi<sif.frameNr; fi++) {
    img->start[fi]=sif.x1[fi]; img->end[fi]=sif.x2[fi];
    img->mid[fi]=0.5*(img->start[fi]+img->end[fi]);
  }
  emptySIF(&sif);
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write Analyze 7.5 image.
    Analyze database name must be given with path. Path must exist.
    Image and header files with .img and .hdr extensions are created.
    Existing files are overwritten.
    anaFlipping() determines whether image is flipped in z-direction;
    image is always flipped in x,y-directions.
    Byte order is determined based on _fileFormat field.
\return Returns >0 in case of an error and sets IMG->statmsg.
*/
int imgWriteAnalyze(
  /** Analyze database name with path, without extension */
  char *dbname,
  /** Pointer to IMG data */
  IMG *img
) {
  static char *imgmsg[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory",
  /*  3 */ "cannot write image datafile",
  /*  4 */ "cannot write header file",
  };
  FILE *fp;
  int ret, fi, pi, xi, yi, little;
  float g;
  ANALYZE_DSR dsr;
  char datfile[FILENAME_MAX], hdrfile[FILENAME_MAX], *cptr;
  int pxlNr=0;
  struct tm *st;
  short int *sdata, *sptr, smin, smax;


  if(IMG_TEST) printf("imgWriteAnalyze(%s, *img)\n", dbname);

  /* Check the arguments */
  img->statmsg=imgmsg[0];
  if(dbname==NULL || !dbname[0]) {img->statmsg=imgmsg[1]; return(1);}
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) {
    img->statmsg=imgmsg[1]; return(2);}
  
  /* Make the image and header filenames */  
  strcpy(datfile, dbname); strcat(datfile, ".img");
  strcpy(hdrfile, dbname); strcat(hdrfile, ".hdr");


  /*
   *  Fill Analyze header
   */
  if(img->_fileFormat==IMG_ANA_L) dsr.little=1; else dsr.little=0;
  /* Header key */
  memset(&dsr.hk, 0, sizeof(ANALYZE_HEADER_KEY));
  memset(&dsr.dime, 0, sizeof(ANALYZE_HEADER_IMGDIM));
  memset(&dsr.hist, 0, sizeof(ANALYZE_HEADER_HISTORY));
  dsr.hk.sizeof_hdr=348;
  strcpy(dsr.hk.data_type, "");
  cptr=strrchr(dbname, '/'); if(cptr==NULL) cptr=strrchr(dbname, '\\');
  if(cptr!=NULL) cptr++; if(cptr==NULL) cptr=dbname;
  strncpy(dsr.hk.db_name, cptr, 17);
  dsr.hk.extents=16384;
  dsr.hk.regular='r';
  /* Image dimension */
  dsr.dime.dim[0]=4;
  dsr.dime.dim[1]=img->dimx;
  dsr.dime.dim[2]=img->dimy;
  dsr.dime.dim[3]=img->dimz;
  dsr.dime.dim[4]=img->dimt;
  dsr.dime.datatype=ANALYZE_DT_SIGNED_SHORT;
  dsr.dime.bitpix=16;
  dsr.dime.pixdim[0]=0.0;
  dsr.dime.pixdim[1]=img->sizex;
  dsr.dime.pixdim[2]=img->sizey;
  dsr.dime.pixdim[3]=img->sizez;
  dsr.dime.pixdim[4]=0.0;
  dsr.dime.funused1=0.0; /* Scale factor is set later */
  /* dsr.dime.funused2=img->zoom; */ /* Reconstruction zoom */
  dsr.dime.funused3=img->isotopeHalflife;
  /* Data history */
  if(img->decayCorrected==1) strcpy(dsr.hist.descrip, "Decay corrected.");
  else strcpy(dsr.hist.descrip, "No decay correction.");
  strncpy(dsr.hist.scannum, img->studyNr, 10);
  st=localtime(&img->scanStart);
  strftime(dsr.hist.exp_date, 10, "%Y-%m-%d", st);
  strftime(dsr.hist.exp_time, 10, "%T", st);
  

  /*
   *  Scale data to short int range
   *  Determine and set scale factor and cal_min & cal_max
   */
  if(IMG_TEST) printf("scaling data to short ints\n");
  ret=imgMinMax(img, &dsr.dime.cal_min, &dsr.dime.cal_max);
  if(ret) {img->statmsg=imgmsg[1]; return(3);}
  if(fabs(dsr.dime.cal_min)>fabs(dsr.dime.cal_max)) g=fabs(dsr.dime.cal_min);
  else g=fabs(dsr.dime.cal_max);
  if(g<1E-20) g=1.0; else g=32767./g; dsr.dime.funused1=1.0/g;
  if(IMG_TEST) printf("min=%g max=%g scale_factor=%g\n",
    dsr.dime.cal_min, dsr.dime.cal_max, dsr.dime.funused1);
  /* Allocate memory for short int array */
  pxlNr=(img->dimx)*(img->dimy)*(img->dimz)*(img->dimt);
  sdata=malloc(pxlNr*sizeof(short int));
  if(sdata==NULL) {img->statmsg=imgmsg[2]; return(12);}
  /* Copy image matrix data to short int array */
  sptr=sdata; smin=smax=temp_roundf(g*img->m[0][0][0][0]);
  if(anaFlipping()==0) { /* no flipping in z-direction */
    for(fi=0; fi<img->dimt; fi++)
      for(pi=0; pi<img->dimz; pi++)
        for(yi=img->dimy-1; yi>=0; yi--)
          for(xi=img->dimx-1; xi>=0; xi--) {
            *sptr=temp_roundf(g*img->m[pi][yi][xi][fi]);
            if(*sptr>smax) smax=*sptr; else if(*sptr<smin) smin=*sptr;
            sptr++;
          }
  } else {
    for(fi=0; fi<img->dimt; fi++)
      for(pi=img->dimz-1; pi>=0; pi--)
        for(yi=img->dimy-1; yi>=0; yi--)
          for(xi=img->dimx-1; xi>=0; xi--) {
            *sptr=temp_roundf(g*img->m[pi][yi][xi][fi]);
            if(*sptr>smax) smax=*sptr; else if(*sptr<smin) smin=*sptr;
            sptr++;
          }
  }
  if(IMG_TEST) printf("smin=%d smax=%d\n", smin, smax);
  /* Set header glmin & glmax */
  dsr.dime.glmin=smin; dsr.dime.glmax=smax;
  
  /* Change byte order if necessary */
  little=little_endian();
  if(little!=dsr.little) {sptr=sdata; swabip(sptr, pxlNr*sizeof(short int));}
  /* Open image data file for write */
  if((fp=fopen(datfile, "wb")) == NULL) {
    free(sdata); img->statmsg=imgmsg[3]; return(14);}
  /* Write image data */
  if(fwrite((char*)sdata, 2, pxlNr, fp) != pxlNr) {
    free(sdata); img->statmsg=imgmsg[3]; return(15);}
  fclose(fp); free(sdata);

  /* Write Analyze header */
  ret=anaWriteHeader(hdrfile, &dsr);
  if(ret) {img->statmsg=imgmsg[4]; return(21);}
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

