/******************************************************************************
  Copyright (c) 1996-2004 by Turku PET Centre

  mathfunc.c
  
  Procedures for
  - reading and writing FIT files
  - calculating functions

  When you include new functions, remember to change the following functions:
    functionnameFIT()
    functionformatFIT()
    evalFIT()
    ievalFIT()
    devalFIT()


  Version:
  1.0 1996-01-26 Vesa Oikonen
  1.1 1996-02-03 VO
      Rational function of 3rd degree polynomials included.
  1.2 1996-02-05 VO
      Rational function of 1st degree polynomials included.
      And also polynomial functions.
  1.3 1996-02-12 VO
      Rational function 221 included.
  1.4 1997-02-07 VO
      Lunqvist function 321 included.
  1.5 1999-02-15 VO
      nci.c included instead of strip.c
  1.6 1999-05-01 VO
      Hill type function 842 included.
  1.7 1999-07-17 VO
      Rational function 232 included.
      isnormal() functions removed.
  2.0 2001-04-09 VO
      Name changed from fitfunc.c to mathfunc.c
      Included in the libpet.
      Fit data structure (in mathfunc.h) is somewhat changed.
      Fit datafile format is changed.
      Included new functions.
      2001-04-11 VO
      Function 1232 evaluation corrected.
  2.1 2001-10-04 VO
      Feng function 1313 included.
      Test prints included.
      2001-10-05 VO
      Integral for 1313 included.
      2002-05-31 VO
      More test prints.
      2002-07-30 VO
      memset() added to initFIT().
  2.2 2003-02-19 VO
      Gamma variate function included (not integral).
      Identifies Graham's input functions, but does not yet process them.
      Added functions for computation of derivatives,
      although applied only to Gamma variate and Lundqvist functions for now.
      Added integratal calculation for Lundqvist function.
      Added some test prints.
      Floating point values are written with uppercase E.
      2003-02-19 VO
      Identifies Huang's plasma metabolite model, but does not process it.
      2003-03-17 VO
      Identifies more plasma metabolite models, but does not process those.
      2003-03-19 VO
      Identifies Graham's input function with metabolite, but does not
      process it.
  2.3 2003-08-15 VO
      Included "timetypes" um and mm for measuring distances.
      Identifies also empty 'time' unit.
      Image profile function 2111 included.
      2004-04-06 VO
      Correction of function 1313 in functionformatFIT().
      Removed unnecessary includes.
      Function information format is changed.
  2.1 2004-04-07 VO
      Extended Feng function 1314 included.
      2004-09-20 VO
      Doxygen style comments.
      2004-10-13 VO
      Daylight saving time set to unknown (-1) for Sun/Unix.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "include/mathfunc.h"
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for FIT. All contents are cleared. */
void emptyFIT(FIT *fit)
{
  if(fit->_voidataNr>0) {
    free((char*)(fit->voi));
    fit->_voidataNr=0;
  }
  fit->voiNr=0;
  fit->datafile[0]=fit->unit[0]=fit->program[0]=(char)0;
  fit->timeunit=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate FIT structure. Call this once before first use. */
void initFIT(FIT *fit)
{
  memset(fit, 0, sizeof(FIT));
  fit->_voidataNr=0; fit->voiNr=0;
  fit->datafile[0]=fit->unit[0]=fit->program[0]=(char)0;
  fit->timeunit=0;
  fit->time=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Write function parameters in FIT into specified file.
    If necessary, a backup file (*%) is created.
    In case of an error, >0 is returned, and a description is written in
    fiterrmsg.
*/
int writeFIT(FIT *fit, char *filename)
{
  int i, j, n, savedNr=0;
  char tmp[1024], is_stdout=0;
  struct tm *st;
  FILE *fp;


  if(MATHFUNC_TEST) printf("in writeFIT()\n");
  /* Check that there is some data to write */
  if(fit->voiNr<1) {strcpy(fiterrmsg, "no data"); return 1;}
  for(i=0; i<fit->voiNr; i++)
    if(fit->voi[i].wss!=NA && fit->voi[i].type>0) savedNr++;
  if(savedNr<1) {strcpy(fiterrmsg, "no fitted data"); return 1;}

  /* Check if writing to stdout */
  if(!strcmp(filename, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(filename, 0) != -1) {
    strcpy(tmp, filename); strcat(tmp, "%"); rename(filename, tmp);}

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp = fopen(filename, "w")) == NULL) {
    strcpy(fiterrmsg, "cannot open file"); return 2;}

  /* Fit file format */
  n=fprintf(fp, "%-11.11s %s\n", FIT_VER, fit->program);
  if(n==0) {
    strcpy(fiterrmsg, "disk full");
    if(!is_stdout) fclose(fp); return 3;
  }
  /* Write fit date and time */
  st=localtime(&fit->time); strftime(tmp, 256, "%Y-%m-%d %T", st);
  fprintf(fp, "Date:       %s\n", tmp);
  /* Write the name of the original datafile */
  fprintf(fp, "Data file:  %s\n", fit->datafile);
  /* Write the 'activity' unit */
  fprintf(fp, "Data unit:  %s\n", fit->unit);
  /* Write the time unit */
  if(fit->timeunit==0) strcpy(tmp, "min");
  else if(fit->timeunit==1) strcpy(tmp, "sec");
  else if(fit->timeunit==11) strcpy(tmp, "um");
  else if(fit->timeunit==12) strcpy(tmp, "mm");
  else strcpy(tmp, ".");
  if(fit->timeunit<11) fprintf(fp, "Time unit:  %s\n", tmp);
  else fprintf(fp, "Distance unit:  %s\n", tmp);
  /* Write the voiNr to be saved */
  fprintf(fp, "Nr of VOIs: %d\n", savedNr);
  /* Write the Fit title */
  fprintf(fp, "%-13.13s %-6.6s %-7.7s %-7.7s %-6.6s %-9.9s %-5.5s %4.4s %s\n",
    "Region", "Plane", "Start", "End", "dataNr", "WSS", "parNr", "Type", "Parameters");
  /* Write regional fits */
  for(i=0; i<fit->voiNr; i++) {
    if(fit->voi[i].wss==NA || fit->voi[i].type<=0) continue;
    if(fit->voi[i].voiname[0]) strcpy(tmp, fit->voi[i].voiname); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    if(fit->voi[i].hemisphere[0]) strcpy(tmp, fit->voi[i].hemisphere); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    if(fit->voi[i].place[0]) strcpy(tmp, fit->voi[i].place); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    fprintf(fp, "%7.3f %7.3f %6d %9.2E %5d %04d",
       fit->voi[i].start, fit->voi[i].end, fit->voi[i].dataNr,
       fit->voi[i].wss, fit->voi[i].parNr, fit->voi[i].type );
    for(j=0; j<fit->voi[i].parNr; j++) fprintf(fp, " %10.4E", fit->voi[i].p[j]);
    fprintf(fp, "\n");
  }

  /* Close file */
  if(!is_stdout) fclose(fp);
  strcpy(fiterrmsg, "");

  if(MATHFUNC_TEST) printf("done writeFIT()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for FIT data. Any previous contents are destroyed. */
int setmemFIT(FIT *fit, int voiNr)
{
  /* Check that there is something to do */
  if(voiNr<1) return 1;

  /* Clear previous data, but only if necessary */
  if(fit->_voidataNr>0 || fit->voiNr>0) emptyFIT(fit);

  /* Allocate memory for regional curves */
  fit->voi=(FitVOI*)calloc(voiNr, sizeof(FitVOI));
  if(fit->voi==NULL) return 2;
  fit->_voidataNr=voiNr;

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Print to stdout the contents of FIT data structure.
    Mainly for testing purposes.
*/
void printFIT(FIT *fit)
{
  if(MATHFUNC_TEST) printf("Number of curves: %d\n", fit->voiNr);
  if(MATHFUNC_TEST) printf("_voidataNr = %d\n", fit->_voidataNr);
  writeFIT(fit, "stdout");
}
/*****************************************************************************/

/*****************************************************************************/
/** Read FIT file contents to the specified data structure,
    emptying its old contents.
\return In case of an error, >0 is returned, and a description is written in fiterrmsg.
*/
int readFIT(char *filename, FIT *fit)
{
  FILE *fp;
  char *cptr, line[1024], *lptr;
  int i, ri, n, type=0;
  int yy, mm, dd, h, m, s;
  struct tm *st;
  time_t timet;


  if(MATHFUNC_TEST) printf("in readFIT()\n");
  /* Empty data */
  emptyFIT(fit);
  
  /* Open file */
  fp=fopen(filename, "r");
  if(fp==NULL) {strcpy(fiterrmsg, "cannot open file"); return 1;}

  /* Read data */
  strcpy(fiterrmsg, "wrong format");
  /* Read filetype and program name */
  while(fgets(line, 1024, fp)!=NULL) {
    /* Ignore empty and comment lines */
    if(strlen(line)<4 || line[0]=='#') continue;
    /* Check for id string */
    if(!strncmp(line, FIT_VER, strlen(FIT_VER))) type=1; else type=0;
    break;
  }
  if(type!=1) {fclose(fp); return 3;}
  lptr=line; cptr=strtok(lptr, " \t\n\r");
  if(cptr!=NULL) cptr=strtok(NULL, " \t\n\r");
  if(cptr!=NULL && strlen(cptr)<1024) strcpy(fit->program, cptr);
  /* Read fit date and time */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Date:", 5)) {fclose(fp); return 4;}
  lptr=&line[5]; cptr=strtok(lptr, " \t\n\r");
  if(cptr!=NULL) {
    timet=time(NULL); st=localtime(&timet);
    sscanf(cptr, "%d-%d-%d", &yy, &mm, &dd);
    st->tm_mday=dd; st->tm_mon=mm-1; st->tm_year=yy-1900;
    cptr=strtok(NULL, " \t\n\r");
    if(cptr!=NULL) sscanf(cptr, "%d:%d:%d", &h, &m, &s);
    st->tm_hour=h; st->tm_min=m; st->tm_sec=s; st->tm_isdst=-1;
    fit->time=mktime(st);
  }
  /* Read the name of the original datafile */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Data file:", 10)) {fclose(fp); return 5;}
  lptr=&line[10]; cptr=strtok(lptr, " \t\n\r");
  if(cptr!=NULL && strlen(cptr)<1024) strcpy(fit->datafile, cptr);
  /* Read the activity unit */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Data unit:", 10)) {fclose(fp); return 6;}
  lptr=&line[10]; cptr=strtok(lptr, " \t\n\r");
  if(cptr!=NULL && strlen(cptr)<1024) strcpy(fit->unit, cptr);
  /* Read the time unit */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Time unit:", 10)!=0 &&
     strncmp(line, "Distance unit:", 14)!=0) {fclose(fp); return 7;}
  lptr=&line[10]; cptr=strtok(lptr, " \t\n\r");
  if(!strncmp(cptr, "min", 3)) fit->timeunit=0;
  else if(!strncmp(cptr, "sec", 3)) fit->timeunit=1;
  else if(!strncmp(cptr, "um", 2)) fit->timeunit=11;
  else if(!strncmp(cptr, "mm", 2)) fit->timeunit=12;
  else if(!strncmp(cptr, ".", 1)) fit->timeunit=-1;
  else fit->timeunit=0;
  /* Read the nr of regions */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Nr of VOIs:", 11)) {fclose(fp); return 8;}
  lptr=&line[11]; cptr=strtok(lptr, " \t\n\r");
  n=atoi(cptr); if(n<1 || n>32000) {fclose(fp); return 8;}
  /* Allocate memory for regions */
  if(setmemFIT(fit, n)) {strcpy(fiterrmsg, "out of memory"); fclose(fp); return 9;}
  fit->voiNr=n;
  /* Read (and ignore) title line */
  strcpy(fiterrmsg, "wrong format");
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Region", 6)) {fclose(fp); return 10;}
  /* Read regional data */
  for(ri=0; ri<fit->voiNr; ri++) {
    /* Read dataline */
    while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
    if(strlen(line)<10) {fclose(fp); return 11;}
    /* Read region names */
    lptr=line;
    cptr=strtok(lptr, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    if(strlen(cptr)<7) strcpy(fit->voi[ri].voiname, cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    if(strlen(cptr)<7) strcpy(fit->voi[ri].hemisphere, cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    if(strlen(cptr)<7) strcpy(fit->voi[ri].place, cptr);
    /* Fit start and end times, and original data nr */
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].start=atof(cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].end=atof(cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].dataNr=atoi(cptr);
    /* Fit error, parameter nr and function number (type) */
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].wss=atof(cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].parNr=atoi(cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
    fit->voi[ri].type=atoi(cptr);
    if(fit->voi[ri].type<1) {fclose(fp); return 11;}
    /* Parameters */
    for(i=0; i<fit->voi[ri].parNr; i++) {
      cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return 11;}
      fit->voi[ri].p[i]=atof(cptr);
    }
  }

  /* Close file */
  fclose(fp);
  strcpy(fiterrmsg, "");

  if(MATHFUNC_TEST) printf("done readFIT()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copies the description of a function type to the specified string
    which must have space for >=128 characters.
*/
int functionformatFIT(
  /** The number of function */
  int type,
  /** Representation of the format of the function */
  char *str
) {
  strcpy(str, "");
  switch(type) {
    /* Polynomials, including line */
    case 100: strcpy(str, "f(x)=A"); break;
    case 101: strcpy(str, "f(x)=A+B*x"); break;
    case 102: strcpy(str, "f(x)=A+B*x+C*x^2"); break;
    case 103: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3"); break;
    case 104: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4"); break;
    case 105: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4+F*x^5"); break;
    case 106: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4+F*x^5+G*x^6"); break;
    case 107: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4+F*x^5+G*x^6+H*x^7"); break;
    case 108: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4+F*x^5+G*x^6+H*x^7+I*x^8"); break;
    case 109: strcpy(str, "f(x)=A+B*x+C*x^2+D*x^3+E*x^4+F*x^5+G*x^6+H*x^7+I*x^8+J*x^9"); break;
    /* Rational functions */
    case 211: strcpy(str, "f(x)=(A+C*x)/(B+D*x)"); break;
    case 221: strcpy(str, "f(x)=(A+C*x+E*x^2)/(B+D*x)"); break;
    case 222: strcpy(str, "f(x)=(A+C*x+E*x^2)/(B+D*x+F*x^2)"); break;
    case 232: strcpy(str, "f(x)=(A+C*x+E*x^2+G*x^3)/(B+D*x+F*x^2)"); break;
    case 233: strcpy(str, "f(x)=(A+C*x+E*x^2+G*x^3)/(B+D*x+F*x^2+H*x^3)"); break;
    case 1232: strcpy(str, "f(x)=(A+C*(x-t)+E*(x-t)^2+G*(x-t)^3)/(B+D*(x-t)+F*(x-t)^2)"); break;
    /* Exponential functions */
    case 301: strcpy(str, "f(x)=A*exp(B*x)"); break;
    case 302: strcpy(str, "f(x)=A*exp(B*x)+C*exp(D*x)"); break;
    case 303: strcpy(str, "f(x)=A*exp(B*x)+C*exp(D*x)+E*exp(F*x)"); break;
    case 304: strcpy(str, "f(x)=A*exp(B*x)+C*exp(D*x)+E*exp(F*x)+G*exp(H*x)"); break;
    case 305: strcpy(str, "f(x)=A*exp(B*x)+C*exp(D*x)+E*exp(F*x)+G*exp(H*x)+I*exp(J*x)"); break;
    /* Feng function */
    case 1313: strcpy(str, "f(x)=(A*(x-t)-C-E)*exp(B*(x-t))+C*exp(D*(x-t))+E*exp(F*(x-t))"); break;
    case 1314: strcpy(str, "f(x)=(A*(x-t)-C-E-G)*exp(B*(x-t))+C*exp(D*(x-t))+E*exp(F*(x-t))+G*exp(H*(x-t))"); break;
    /* Lundqvist function */
    case 321: strcpy(str, "f(x)=A*exp(B*x)*(1-exp(C*x))"); break;
    /* Gamma variate function */
    case 1401: strcpy(str, "f(x)=A*((x-D)^B)*exp(-(x-D)/C) , when x>=D, else f(x)=0"); break;
    /* Gamma variate function with background */
    case 1402: strcpy(str, "f(x)=A*((x-D)^B)*exp(-(x-D)/C) + E , when x>=D, else f(x)=E"); break;
    /* Hill type functions */
    case 841: strcpy(str, "f(x)=(A*x^B)/(x^B+C)"); break;
    case 842: strcpy(str, "f(x)=1-((A*x^B)/(x^B+C))"); break;
    /* PET profile functions */
    case 2111: strcpy(str, "P(x)=(C/2)*(erf((x-d+R)/(sqrt(2)*FWHM/2355))-erf((x-d-R)/(sqrt(2)*FWHM/2355)))+bkg");
      break;
    /* Compartmental model functions */
    /* Graham's plasma curve smoothing function */
    case 9501: strcpy(str, "Cp(t)<=>Ci(t)<=>Ct(t)"); break;
    /* Extended Graham's plasma curve smoothing function */
    case 9502: strcpy(str, "Ce(t)<=>Cp(t)<=>Ci(t)<=>Ct(t)"); break;
    /* Extended Graham's plasma curve smoothing function with metabolite */
    case 9503: strcpy(str, "Cpa(t)<=>Cia(t)<=>Cta(t)->Ctm(t)<=>Cim(t)<=>Cpm(t)"); break;
    /* Huang's plasma metabolite model */
    case 9601: strcpy(str, "C4(t)<=>C3(t)<-C0(t)->C1(t)<=>C2(t)"); break;
    /* Extended Carson's plasma metabolite model */
    case 9602: strcpy(str, "Cpa(t)<=>Cta(t)->Ctm(t)<=>Cpm(t)"); break;
    /* New plasma metabolite model */
    case 9603: strcpy(str, "Cpa(t)->Ct1(t)<=>Cpm(t)<=>Ct2(t)"); break;
    default:  return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copies the name of the function to the specified string
    which must have space for >=128 characters.
*/
int functionnameFIT(
  /** The number of function */
  int type,
  /** Name of the function */
  char *str
) {
  strcpy(str, "");
  switch(type) {
    case 100: strcpy(str, "f(x)=A"); break;
    case 101: strcpy(str, "line"); break;
    case 102: strcpy(str, "2nd order polynomial"); break;
    case 103: strcpy(str, "3rd order polynomial"); break;
    case 104: strcpy(str, "4th order polynomial"); break;
    case 105: strcpy(str, "5th order polynomial"); break;
    case 106: strcpy(str, "6th order polynomial"); break;
    case 107: strcpy(str, "7th order polynomial"); break;
    case 108: strcpy(str, "8th order polynomial"); break;
    case 109: strcpy(str, "9th order polynomial"); break;
    case 211: strcpy(str, "1/1 order rational function"); break;
    case 221: strcpy(str, "2/1 order rational function"); break;
    case 222: strcpy(str, "2/2 order rational function"); break;
    case 232: strcpy(str, "3/2 order rational function"); break;
    case 233: strcpy(str, "3/3 order rational function"); break;
    case 1232: strcpy(str, "3/2 order rational function with delay"); break;
    case 301: strcpy(str, "exponential function"); break;
    case 302: strcpy(str, "sum of 2 exponential functions"); break;
    case 303: strcpy(str, "sum of 3 exponential functions"); break;
    case 304: strcpy(str, "sum of 4 exponential functions"); break;
    case 305: strcpy(str, "sum of 5 exponential functions"); break;
    case 1313: strcpy(str, "Feng model 2 function"); break;
    case 1314: strcpy(str, "Extended Feng model 2 function"); break;
    case 321: strcpy(str, "Lundqvist function"); break;
    case 1401: strcpy(str, "Gamma variate function"); break;
    case 1402: strcpy(str, "Gamma variate with background"); break;
    case 841: strcpy(str, "Hill function"); break;
    case 842: strcpy(str, "Hill function (1-f(x))"); break;
    case 2111: strcpy(str, "Image profile function"); break;
    case 9501: strcpy(str, "Graham's input function"); break;
    case 9502: strcpy(str, "Extended Graham's input function"); break;
    case 9503: strcpy(str, "Graham's input function with metabolite"); break;
    case 9601: strcpy(str, "Huang's plasma metabolite model"); break;
    case 9602: strcpy(str, "Extended Carson's plasma metabolite model"); break;
    case 9603: strcpy(str, "New plasma metabolite model"); break;
    default:  return(1);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate y=f(x).
\return Returns 0, if ok.
 */
int evalFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Time where to evaluate the function */
  double x,
  /** The value of the function is returned here */
  double *y
) {
  double a, b, f, sqrx, cubx, xt;
  int i, n;

  if(MATHFUNC_TEST) {
    printf("evalFIT(r, %g, %g): type=%d ;", x, *y, r->type);
    for(i=0; i<r->parNr; i++) printf(" %g", r->p[i]);
    printf("\n");
  }
  *y=NA;
  switch(r->type) {
    case 100:
    case 101:
    case 102:
    case 103:
    case 104:
    case 105:
    case 106:
    case 107:
    case 108:
    case 109:
      n=r->type-99; f=0.0; for(i=n-1; i>0; i--) {f+=r->p[i]; f*=x;} f+=r->p[0];
      *y=f;
      break;
    case 211:
      a=r->p[0]+r->p[2]*x; b=r->p[1]+r->p[3]*x; if(b==0.0) break;
      f=a/b; *y=f;
      break;
    case 221:
      sqrx=x*x;
      a=r->p[0]+r->p[2]*x+r->p[4]*sqrx; b=r->p[1]+r->p[3]*x; if(b==0.0) break;
      f=a/b; *y=f;
      break;
    case 222:
      sqrx=x*x;
      a=r->p[0]+r->p[2]*x+r->p[4]*sqrx;
      b=r->p[1]+r->p[3]*x+r->p[5]*sqrx; if(b==0.0) break;
      f=a/b; *y=f;
      break;
    case 232:
      sqrx=x*x; cubx=sqrx*x;
      a=r->p[0]+r->p[2]*x+r->p[4]*sqrx+r->p[6]*cubx;
      b=r->p[1]+r->p[3]*x+r->p[5]*sqrx; if(b==0.0) break;
      f=a/b; *y=f;
      break;
    case 233:
      sqrx=x*x; cubx=sqrx*x;
      a=r->p[0]+r->p[2]*x+r->p[4]*sqrx+r->p[6]*cubx;
      b=r->p[1]+r->p[3]*x+r->p[5]*sqrx+r->p[7]*cubx; if(b==0.0) break;
      f=a/b; *y=f;
      break;
    case 301:
      f=r->p[0]*exp(r->p[1]*x); *y=f;
      break;
    case 302:
      f=0;
      a=r->p[0]*exp(r->p[1]*x); f+=a;
      a=r->p[2]*exp(r->p[3]*x); f+=a;
      *y=f;
      break;
    case 303:
      f=0;
      a=r->p[0]*exp(r->p[1]*x); f+=a;
      a=r->p[2]*exp(r->p[3]*x); f+=a;
      a=r->p[4]*exp(r->p[5]*x); f+=a;
      *y=f;
      break;
    case 304:
      f=0;
      a=r->p[0]*exp(r->p[1]*x); f+=a;
      a=r->p[2]*exp(r->p[3]*x); f+=a;
      a=r->p[4]*exp(r->p[5]*x); f+=a;
      a=r->p[6]*exp(r->p[7]*x); f+=a;
      *y=f;
      break;
    case 305:
      f=0;
      a=r->p[0]*exp(r->p[1]*x); f+=a;
      a=r->p[2]*exp(r->p[3]*x); f+=a;
      a=r->p[4]*exp(r->p[5]*x); f+=a;
      a=r->p[6]*exp(r->p[7]*x); f+=a;
      a=r->p[8]*exp(r->p[9]*x); f+=a;
      *y=f;
      break;
    case 321:
      f=r->p[0]*exp(r->p[1]*x)*(1.0-exp(r->p[2]*x));
      *y=f;
      break;
    case 841:
      f=r->p[0]*pow(x, r->p[1]) / (pow(x, r->p[1]) + r->p[2]);
      *y=f;
      break;
    case 842:
      f=1.0-r->p[0]*pow(x, r->p[1]) / (pow(x, r->p[1]) + r->p[2]);
      *y=f;
      break;
    case 2111:
      xt=x-r->p[3]; a=sqrt(2.0)*(r->p[2]/2.355);
      f=r->p[4]+(r->p[0]/2.0)*(erf((xt+r->p[1])/a)-erf((xt-r->p[1])/a));
      *y=f;
      break;
    case 1232:
      xt=x-r->p[7]; if(xt<=0.0) {
        f=0.0;
      } else {
        sqrx=xt*xt; cubx=sqrx*xt;
        a=r->p[0]+r->p[2]*xt+r->p[4]*sqrx+r->p[6]*cubx;
        b=r->p[1]+r->p[3]*xt+r->p[5]*sqrx; if(b==0.0) break;
        f=a/b;
      }
      *y=f;
      break;
    case 1313:
      xt=x-r->p[6]; if(xt<0.0) {
        f=0.0;
      } else {
        f=0.0;
        a=(r->p[0]*(xt)-r->p[2]-r->p[4])*exp(r->p[1]*(xt)); f+=a;
        a=r->p[2]*exp(r->p[3]*(xt)); f+=a;
        a=r->p[4]*exp(r->p[5]*(xt)); f+=a;
      }
      *y=f;
      break;
    case 1314:
      xt=x-r->p[8]; if(xt<0.0) {
        f=0.0;
      } else {
        f=0.0;
        a=(r->p[0]*(xt)-r->p[2]-r->p[4]-r->p[6])*exp(r->p[1]*(xt)); f+=a;
        a=r->p[2]*exp(r->p[3]*(xt)); f+=a;
        a=r->p[4]*exp(r->p[5]*(xt)); f+=a;
        a=r->p[6]*exp(r->p[7]*(xt)); f+=a;
      }
      *y=f;
      break;
    case 1401:
      xt=x-r->p[3]; if(xt<=0.0 || r->p[2]==0.0) {
        f=0.0;
      } else {
        f=r->p[0]*pow(xt, r->p[1])*exp(-xt/r->p[2]);
      }
      *y=f;
      break;
    case 1402:
      xt=x-r->p[3]; if(xt<=0.0 || r->p[2]==0.0) {
        f=r->p[4];
      } else {
        f=r->p[0]*pow(xt, r->p[1])*exp(-xt/r->p[2]) + r->p[4];
      }
      *y=f;
      break;
    case 9501: *y=NA; break; /* cannot be applied to one point only */
    case 9502: *y=NA; break; /* cannot be applied to one point only */
    case 9503: *y=NA; break; /* cannot be applied to one point only */
    default:
      *y=NA;
  }
  if(*y==NA) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluates an array y[i]=f(x[i]).
\return Returns 0, if ok.
 */
int evaltacFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Times where to evaluate the function */
  double *x,
  /** Array for the function values */
  double *y,
  /** Nr of (x,y) data */
  int dataNr
) {
  int i;

  if(dataNr<1) return 1;
  for(i=0; i<dataNr; i++) if(evalFIT(r, x[i], y+i)) return 2;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluates yi=Integral of f(x) between 0 and x.
\return Returns 0, if ok.
 */
int ievalFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Time where to evaluate integral of the function */
  double x,
  /** The integral value of the function is returned here */
  double *yi
) {
  double a, f, xt, t;

  *yi=NA;
  switch(r->type) {
    case 301:
      if(fabs(r->p[1])>1.0e-12) f=(r->p[0]/r->p[1])*(exp(r->p[1]*x)-1.0);
      else f=r->p[0]*x;
      *yi=f;
      break;
    case 302:
      f=0;
      if(fabs(r->p[1])>1.0e-12) a=(r->p[0]/r->p[1])*(exp(r->p[1]*x)-1.0);
      else a=r->p[0]*x;
      f+=a;
      if(fabs(r->p[3])>1.0e-12) a=(r->p[2]/r->p[3])*(exp(r->p[3]*x)-1.0);
      else a=r->p[2]*x;
      f+=a;
      *yi=f;
      break;
    case 303:
      f=0;
      if(fabs(r->p[1])>1.0e-12) a=(r->p[0]/r->p[1])*(exp(r->p[1]*x)-1.0);
      else a=r->p[0]*x;
      f+=a;
      if(fabs(r->p[3])>1.0e-12) a=(r->p[2]/r->p[3])*(exp(r->p[3]*x)-1.0);
      else a=r->p[2]*x;
      f+=a;
      if(fabs(r->p[5])>1.0e-12) a=(r->p[4]/r->p[5])*(exp(r->p[5]*x)-1.0);
      else a=r->p[4]*x;
      f+=a;
      *yi=f;
      break;
    case 304:
      f=0;
      if(fabs(r->p[1])>1.0e-12) a=(r->p[0]/r->p[1])*(exp(r->p[1]*x)-1.0);
      else a=r->p[0]*x;
      f+=a;
      if(fabs(r->p[3])>1.0e-12) a=(r->p[2]/r->p[3])*(exp(r->p[3]*x)-1.0);
      else a=r->p[2]*x;
      f+=a;
      if(fabs(r->p[5])>1.0e-12) a=(r->p[4]/r->p[5])*(exp(r->p[5]*x)-1.0);
      else a=r->p[4]*x;
      f+=a;
      if(fabs(r->p[7])>1.0e-12) a=(r->p[6]/r->p[7])*(exp(r->p[7]*x)-1.0);
      else a=r->p[6]*x;
      f+=a;
      *yi=f;
      break;
    case 305:
      f=0;
      if(fabs(r->p[1])>1.0e-12) a=(r->p[0]/r->p[1])*(exp(r->p[1]*x)-1.0);
      else a=r->p[0]*x;
      f+=a;
      if(fabs(r->p[3])>1.0e-12) a=(r->p[2]/r->p[3])*(exp(r->p[3]*x)-1.0);
      else a=r->p[2]*x;
      f+=a;
      if(fabs(r->p[5])>1.0e-12) a=(r->p[4]/r->p[5])*(exp(r->p[5]*x)-1.0);
      else a=r->p[4]*x;
      f+=a;
      if(fabs(r->p[7])>1.0e-12) a=(r->p[6]/r->p[7])*(exp(r->p[7]*x)-1.0);
      else a=r->p[6]*x;
      f+=a;
      if(fabs(r->p[9])>1.0e-12) a=(r->p[8]/r->p[9])*(exp(r->p[8]*x)-1.0);
      else a=r->p[8]*x;
      f+=a;
      *yi=f;
      break;
    case 321:
      f=(r->p[0]/r->p[1])*exp(r->p[1]*x)
        - r->p[0]*exp((r->p[1]+r->p[2])*x)/(r->p[1]+r->p[2]);
      *yi=f;
      break;
    case 1313:
      t=r->p[6]; xt=x-t; f=0.0;
      if(xt>0.0) {
        if(r->p[1]!=0.0) {
          a= (r->p[0] + r->p[1]*(r->p[0]*t + r->p[2] + r->p[4]))
            /(r->p[1]*r->p[1]*exp(r->p[1]*t));
          f+= + exp(r->p[1]*x)*((r->p[0]*x)/(r->p[1]*exp(r->p[1]*t)) - a)
              - exp(r->p[1]*t)*((r->p[0]*t)/(r->p[1]*exp(r->p[1]*t)) - a);
        }
        if(r->p[3]!=0.0) f-= (r->p[2]/r->p[3])*(1.0 - exp(r->p[3]*xt));
        if(r->p[5]!=0.0) f-= (r->p[4]/r->p[5])*(1.0 - exp(r->p[5]*xt));
      }
      *yi=f;
      break;
    case 1314:
      t=r->p[8]; xt=x-t; f=0.0;
      if(xt>0.0) {
        if(r->p[1]!=0.0) {
          a= (r->p[0] + r->p[1]*(r->p[0]*t + r->p[2] + r->p[4] + r->p[6]))
            /(r->p[1]*r->p[1]*exp(r->p[1]*t));
          f+= + exp(r->p[1]*x)*((r->p[0]*x)/(r->p[1]*exp(r->p[1]*t)) - a)
              - exp(r->p[1]*t)*((r->p[0]*t)/(r->p[1]*exp(r->p[1]*t)) - a);
        }
        if(r->p[3]!=0.0) f-= (r->p[2]/r->p[3])*(1.0 - exp(r->p[3]*xt));
        if(r->p[5]!=0.0) f-= (r->p[4]/r->p[5])*(1.0 - exp(r->p[5]*xt));
        if(r->p[7]!=0.0) f-= (r->p[6]/r->p[7])*(1.0 - exp(r->p[7]*xt));
      }
      *yi=f;
      break;
    case 1401: *yi=NA; break; /* not yet applied */
    case 1402: *yi=NA; break; /* not yet applied */
    case 2111: *yi=NA; break; /* no application */
    case 9501: *yi=NA; break; /* cannot be applied to one point only */
    case 9502: *yi=NA; break; /* cannot be applied to one point only */
    case 9503: *yi=NA; break; /* cannot be applied to one point only */
    default:
      *yi=NA;
  }
  if(*yi==NA) return 3;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate an array yi[i]=Integral of f(x[i]) between 0 and x.
\return Returns 0, if ok.
*/
int ievaltacFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Times where to evaluate the function integrals */
  double *x,
  /** Array for the function integral values */
  double *yi,
  /** Nr of (x,yi) data */
  int dataNr
) {
  int i;

  if(dataNr<1) return 1;
  for(i=0; i<dataNr; i++) if(ievalFIT(r, x[i], yi+i)) return 2;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluates yd=Df(x).
\return Returns 0, if ok.
 */
int devalFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Time where to evaluate the derivative of the function */
  double x,
  /** The derivative of the function is returned here */
  double *yd
) {
  double f, xt;


  *yd=NA;
  switch(r->type) {
    case 301:
      break;
    case 302:
      break;
    case 303:
      break;
    case 304:
      break;
    case 305:
      break;
    case 321:
      f=r->p[0]*r->p[1]*exp(r->p[1]*x)*(1.0-exp(r->p[2]*x)) - r->p[0]*r->p[2]*exp((r->p[1]+r->p[2])*x);
      *yd=f;
      break;
    case 1313:
    case 1314:
      break;
    case 1401:
    case 1402:
      xt=x-r->p[3]; if(xt<=0.0 || r->p[2]==0.0) {
        f=0.0;
      } else {
        f=r->p[0]*pow(xt, r->p[1]-1.0)*exp(-xt/r->p[2])*(r->p[1]-(xt/r->p[2]));
      }
      *yd=f;
      break;
    case 2111: *yd=NA; break; /* no application */
    case 9501: *yd=NA; break; /* cannot be applied to one point only */
    case 9502: *yd=NA; break; /* cannot be applied to one point only */
    case 9503: *yd=NA; break; /* cannot be applied to one point only */
    default:
      *yd=NA;
  }
  if(*yd==NA) return(3);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluates an array yd[i]=Df(x[i]).
\return Returns 0, if ok.
 */
int devaltacFIT(
  /** Fit parameters of a single region */
  FitVOI *r,
  /** Times where to evaluate the function derivatives */
  double *x,
  /** Array for the function derivatives */
  double *yd,
  /** Nr of (x,yd) data */
  int dataNr
) {
  int i;

  if(dataNr<1) return(1);
  for(i=0; i<dataNr; i++) if(devalFIT(r, x[i], yd+i)) return(2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

