/******************************************************************************
  Copyright (c) 2002-2004 by Turku PET Centre

  result.c
  
  Procedures for
  - reading and writing RES (result) files


  Version:
  2002-05-26 Vesa Oikonen
  2002-07-23 VO
      Small changes in printing fields.
  2002-07-27 VO
      resWriteHTML() added.
      resWrite() applies resWriteHTML() if filename extension is *.htm(l)
  2002-07-30 VO
      Switches included in result data structure in result.h.
      Name of study (studynr) included in data structure. Filled in resRead().
      Included function resFName2study().
      Included functions resMedian() and resMean().
  2002-08-14 VO
      Floats -> Doubles.
  2002-10-10 VO
      resSortByName() and resCopyMHeader() added.
  2002-10-17 VO
      One digit less written for negative numbers.
  2002-11-13 VO
      Included function resDelete().
  2003-01-18 VO
      Added CL's and SD's for each result parameter.
  2003-02-18 VO
      Floating point numbers are written with uppercase E in resWriteHTML().
  2003-04-16 VO
      resRead can now read Plasmafile and refroi fields that contain spaces.
  2003-10-13 VO
      resMean() works now also if variance=0.
  2003-12-02 VO
      Added fields for tissue density, lumped constant, and plasma concentration.
  2004-01-26 VO
      Added field for beta.
  2004-07-12 VO
      Added field for Vb.
      Changes in comment style.
  2004-08-17 VO
      Added sw2.
  2004-08-23 VO
      resFName2study() replaced by new library function studynr_from_fname.
  2004-10-13 VO
      tm_isdst=-1 (unknown Daylight saving time).

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "include/result.h"
/*****************************************************************************/
/* Local functions */
int resQSortComp(const void *f1, const void *f2);
int resQSortName(const void *voi1, const void *voi2);
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for results. All data are cleared. */
void resEmpty(RES *res)
{
  if(res->_voidataNr>0) {
    free((char*)(res->voi));
    res->_voidataNr=0;
  }
  res->voiNr=0;
  res->parNr=0;
  res->program[0]=(char)0; res->refroi[0]=(char)0; res->datarange[0]=(char)0;
  res->datafile[0]=res->reffile[0]=res->plasmafile[0]=res->bloodfile[0]=(char)0;
  res->density=res->lc=res->concentration=res->beta=0.0;
  res->Vb=-1.0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate RES structure. This should be called once before first use. */
void resInit(RES *res)
{
  memset(res, 0, sizeof(RES));
  res->_voidataNr=0; res->voiNr=0; res->parNr=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for result data. Old data is destroyed. */
int resSetmem(RES *res, int voiNr)
{
  int ri, pi;

  /* Check that there is something to do */
  if(voiNr<1) return(1);

  /* Clear previous data, but only if necessary */
  if(res->_voidataNr>0 || res->voiNr>0) resEmpty(res);

  /* Allocate memory for regional curves */
  res->voi=(ResVOI*)calloc(voiNr, sizeof(ResVOI));
  if(res->voi==NULL) return(2);
  res->_voidataNr=voiNr;

  /* Set SDs and CLs to NA */
  for(ri=0; ri<res->_voidataNr; ri++) for(pi=0; pi<MAX_RESPARAMS; pi++)
    res->voi[ri].sd[pi]=res->voi[ri].cl1[pi]=res->voi[ri].cl2[pi]=NA;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print to stdout the contents of RES data structure. */
void resPrint(RES *res)
{
  resWrite(res, "stdout");
}
/*****************************************************************************/

/*****************************************************************************/
/** Read RES file contents to the specified data structure,
    emptying its old contents.
\return In case of an error, >0 is returned, and a description is written in reserrmsg.
 */
int resRead(char *filename, RES *res)
{
  FILE *fp;
  char *cptr, line[1024], *lptr, tmp[1024];
  int i, n;
  int yy, mm, dd, h, m, s;
  struct tm *st;
  time_t timet;
  long bookmark;


  if(RESULT_TEST) printf("resRead(%s, *res);\n", filename);
  /* Empty data */
  resEmpty(res);
  
  /* Open file */
  fp=fopen(filename, "r");
  if(fp==NULL) {strcpy(reserrmsg, "cannot open file"); return(1);}

  /*
   * Read data, each result set separately, saving only the first one
   */
  strcpy(reserrmsg, "wrong format");
  
  /* Read program name */
  while(fgets(line, 1024, fp)!=NULL) {
    /* Ignore empty and comment lines */
    if(strlen(line)<4 || line[0]=='#') continue; else break;
  }
  /* Check for string (c) or (C) */
  if(strstr(line, "(c)") || strstr(line, "(C)")) {
    strcpy(res->program, line);
  } else {fclose(fp); return(2);}
  
  /* Read calculation date and time */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Date:", 5)) {fclose(fp); return(4);}
  lptr=&line[5]; cptr=strtok(lptr, " \t\n\r");
  if(cptr!=NULL) {
    timet=time(NULL); st=localtime(&timet);
    sscanf(cptr, "%d-%d-%d", &yy, &mm, &dd);
    st->tm_mday=dd; st->tm_mon=mm-1; st->tm_year=yy-1900;
    cptr=strtok(NULL, " \t\n\r");
    if(cptr!=NULL) sscanf(cptr, "%d:%d:%d", &h, &m, &s);
    st->tm_hour=h; st->tm_min=m; st->tm_sec=s; st->tm_isdst=-1;
    res->time=mktime(st);
  }
  
  /* Read studynr, datafiles, ref region, data range, etc */
  do {
    while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
    n=0;
    if(strncmp(line, "Study", 5)==0) {
      lptr=&line[6]; cptr=strtok(lptr, " \t\n\r"); n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strncpy(res->studynr, cptr, MAX_STUDYNR_LEN);
    } else if(strncmp(line, "Data file", 9)==0) {
      lptr=&line[10]; cptr=strtok(lptr, " \t\n\r"); n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->datafile, cptr);
    } else if(strncmp(line, "ROI file", 8)==0) {
      lptr=&line[9]; cptr=strtok(lptr, " \t\n\r"); n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->datafile, cptr); 
    } else if(strncmp(line, "Plasma file", 11)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->plasmafile, cptr);
      cptr=res->plasmafile+strlen(res->plasmafile)-1;
      while(isspace((int)*cptr)) {*cptr=(char)0; cptr--;}
    } else if(strncmp(line, "Blood file", 10)==0) {
      lptr=&line[11]; cptr=strtok(lptr, " \t\n\r"); n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->bloodfile, cptr);
    } else if(strncmp(line, "Reference file", 14)==0) {
      lptr=&line[15]; cptr=strtok(lptr, " \t\n\r"); n=1;
      if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->reffile, cptr);
    } else if(strncmp(line, "Reference region", 16)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL && strlen(cptr)<64) strcpy(res->refroi, cptr);
      cptr=res->refroi+strlen(res->refroi)-1;
      while(isspace((int)*cptr)) {*cptr=(char)0; cptr--;}
    } else if(strncmp(line, "Fit time", 8)==0 || strncmp(line, "Data range", 10)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL && strlen(cptr)<128) strcpy(res->datarange, cptr);
      cptr=res->datarange+strlen(res->datarange)-1;
      while(isspace((int)*cptr)) {*cptr=(char)0; cptr--;}
    } else if(strncmp(line, "Tissue density", 14)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL) res->density=atof(cptr);
    } else if(strncmp(line, "Lumped constant", 15)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL) res->lc=atof(cptr);
    } else if(strncmp(line, "Concentration", 13)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL) res->concentration=atof(cptr);
    } else if(strncmp(line, "Beta", 4)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL) res->beta=atof(cptr);
    } else if(strncmp(line, "Vb", 2)==0) {
      cptr=strchr(line, ':')+1; while(isspace((int)*cptr)) cptr++; n=1;
      if(cptr!=NULL) res->Vb=atof(cptr);
    }
  } while(n);  

  /* Check now if data was weighted or not from the line read above */
  if(strncmp(line, "Data was weighted", 17)==0) res->isweight=1;
  else if(strncmp(line, "Data was not weighted", 21)==0) res->isweight=0;
  else {fclose(fp); return(6);}

  /* Read the result parameter title line */
  while(fgets(line, 1024, fp)!=NULL) if(strlen(line)>2 && line[0]!='#') break;
  if(strncmp(line, "Region  ", 8)) {fclose(fp); return(10);}
  cptr=&line[20]; while(isspace((int)*cptr)) cptr++;
  if(cptr!=NULL && strlen(cptr)<1024) strcpy(res->titleline, cptr);
  else {fclose(fp); return(11);}
  cptr=res->titleline+strlen(res->titleline)-1;
  while(isspace((int)*cptr)) {*cptr=(char)0; cptr--;}

  /* Read the nr of result lines */
  n=0; bookmark=ftell(fp); if(bookmark<0) {fclose(fp); return(21);}
  while(fgets(line, 1024, fp)!=NULL) {
    if(strlen(line)<20 && line[0]!='#') break;
    n++;
  }
  if(fseek(fp, bookmark, SEEK_SET)) {fclose(fp); return(22);}
  if(RESULT_TEST>1) printf("nr of result lines is %d\n", n);
  
  /* Allocate memory for regional results */
  if(resSetmem(res, n)) {
    strcpy(reserrmsg, "cannot allocate memory");
    fclose(fp); return(25);
  }
  
  /* Read regional results */
  res->voiNr=0;
  while(fgets(line, 1024, fp)!=NULL) {
    if(line[0]=='#') continue;
    if(strlen(line)<20) break;
    strcpy(tmp, line);
    /* Read region names */
    lptr=line;
    cptr=strtok(lptr, " \t\n\r"); if(cptr==NULL) {fclose(fp); return(31);}
    if(strlen(cptr)<7) strcpy(res->voi[res->voiNr].voiname, cptr);
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return(31);}
    if(strlen(cptr)<7) strcpy(res->voi[res->voiNr].hemisphere, cptr);
    if(strcmp(res->voi[res->voiNr].hemisphere, ".")==0) res->voi[res->voiNr].hemisphere[0]=(char)0;
    cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) {fclose(fp); return(31);}
    if(strlen(cptr)<7) strcpy(res->voi[res->voiNr].place, cptr);
    if(strcmp(res->voi[res->voiNr].place, ".")==0) res->voi[res->voiNr].place[0]=(char)0;
    /* Read results */
    i=0; lptr=tmp+20; cptr=strtok(lptr, " \t\n\r");
    if(cptr==NULL) {fclose(fp); return(32);}
    while(cptr!=NULL && i<MAX_RESPARAMS) {
      if(strlen(cptr)==1 && *cptr=='.') res->voi[res->voiNr].parameter[i++]=NA;
      else res->voi[res->voiNr].parameter[i++]=atof(cptr);
      cptr=strtok(NULL, " \t\n\r");
    }
    if(res->voiNr==0) {res->parNr=i;} else if(i<res->parNr) res->parNr=i;
    /* If 'region' name implies that this was confidence limit or sd, then */
    /* move the values into correct places, and do not increase the voiNr */
    if(res->voiNr==0) {res->voiNr++; continue;}
    if(strcmp(res->voi[res->voiNr].voiname, "CL")==0) {
      if(strcmp(res->voi[res->voiNr].hemisphere, "95%")==0) {
        if(strcmp(res->voi[res->voiNr].place, "Lower")==0)
          for(i=0; i<res->parNr; i++)
            res->voi[res->voiNr-1].cl1[i]=res->voi[res->voiNr].parameter[i];
        else if(strcmp(res->voi[res->voiNr].place, "Upper")==0)
          for(i=0; i<res->parNr; i++)
            res->voi[res->voiNr-1].cl2[i]=res->voi[res->voiNr].parameter[i];
        continue;
      }
    } else if(strcmp(res->voi[res->voiNr].voiname, "SD")==0) {
      for(i=0; i<res->parNr; i++)
        res->voi[res->voiNr-1].sd[i]=res->voi[res->voiNr].parameter[i];
      continue;
    }
    res->voiNr++;
  }
  if(res->parNr==0) {fclose(fp); return(33);}
  if(RESULT_TEST) printf("nr of results: %d ; nr of parameters: %d\n", res->voiNr, res->parNr);

  /* Seek for other results in the same file */
  while(fgets(line, 1024, fp)!=NULL) {
    /* Ignore empty and comment lines */
    if(strlen(line)<4 || line[0]=='#') continue; else break;
  }
  /* Check again for string (c) or (C) */
  if(strstr(line, "(c)") || strstr(line, "(C)")) {
    fprintf(stderr, "Warning: %s contains more then one set of results; only the 1st one is used.\n", filename);
  }

  /* Close file */
  fclose(fp);
  strcpy(reserrmsg, "");

  /* Fill studynr if it was not found in file */
  if(!res->studynr[0]) studynr_from_fname(filename, res->studynr);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write calculation results into specied file.
  If file exists, a backup file (%) is written also.
  If "stdout" is given as filename, output is directed to stdout.
  If filename extension is *.htm(l), file is saved in HTML format.
\return In case of an error, >0 is returned, and a description is written in reserrmsg.
*/
int resWrite(RES *res, char *filename)
{
  int i, j, n;
  char tmp[1024], is_stdout=0, *cptr;
  struct tm *st;
  FILE *fp;
  int partype[MAX_RESPARAMS]; /* 0=int, 1=double, 2=exp */
  double x, pint, max, *p;
  

  if(RESULT_TEST) printf("resWrite(*res, %s)\n", filename);
  /* Check that there is some data to write */
  if(res==NULL) {strcpy(reserrmsg, "error in result data"); return(1);}
  if(res->voiNr<1) {strcpy(reserrmsg, "no result data"); return(1);}

  /* Write results in HTML format, if necessary */
  cptr=strrchr(filename, '.');
  if(cptr!=NULL && (!strncmp(cptr, ".htm", 4) || !strncmp(cptr, ".HTM", 4)))
    return(resWriteHTML(res, filename));

  /* Check if writing to stdout */
  if(!strcmp(filename, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(filename, 0) != -1) {
    strcpy(tmp, filename); strcat(tmp, "%"); rename(filename, tmp);}
  strcpy(reserrmsg, "cannot write file");

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp = fopen(filename, "w")) == NULL) {
    strcpy(reserrmsg, "cannot open file"); return(2);}

  /* Program name */
  n=fprintf(fp, "%s\n", res->program);
  if(n==0) {
    strcpy(reserrmsg, "disk full");
    if(!is_stdout) fclose(fp); return(3);
  }

  /* Write calculation date and time */
  st=localtime(&res->time); strftime(tmp, 256, "%Y-%m-%d %T", st);
  fprintf(fp, "Date:         %s\n", tmp);

  /* Write the studynr */
  if(res->studynr[0]) fprintf(fp, "Study:        %s\n", res->studynr);

  /* Write the names of the original datafiles */
  if(res->datafile[0]) fprintf(fp, "Data file:    %s\n", res->datafile);
  if(res->plasmafile[0]) fprintf(fp, "Plasma file:  %s\n", res->plasmafile);
  if(res->bloodfile[0]) fprintf(fp, "Blood file:   %s\n", res->bloodfile);
  if(res->reffile[0]) fprintf(fp, "Reference file:  %s\n", res->reffile);
  if(res->refroi[0]) fprintf(fp, "Reference region:  %s\n", res->refroi);
  
  /* Write data range */
  if(res->datarange[0]) fprintf(fp, "Data range:   %s\n", res->datarange);

  /* Write constants */
  if(res->density>0.0) fprintf(fp, "Tissue density:  %g\n", res->density);
  if(res->lc>0.0) fprintf(fp, "Lumped constant: %g\n", res->lc);
  if(res->concentration>0.0) fprintf(fp, "Concentration:   %g\n", res->concentration);
  if(res->beta>0.0) fprintf(fp, "Beta:         %g\n", res->beta);
  if(res->Vb>=0.0) fprintf(fp, "Vb:           %g %%\n", res->Vb);

  /* Weighting */
  if(res->isweight) fprintf(fp, "Data was weighted.\n");
  else fprintf(fp, "Data was not weighted.\n");

  /* Collect info on column types */
  for(j=0; j<res->parNr; j++) {
    /* should column be printed as integers, floats or exponentials? */
    partype[j]=0; max=0.0;
    for(i=0; i<res->voiNr; i++) {
      x=res->voi[i].parameter[j]; if(modf(x, &pint)!=0.0) partype[j]=1;
      x=fabs(x); if(x>max) max=x;
    }
    if(partype[j]==1 && (max>=10.0 || max<0.1)) partype[j]=2;
  }

  /* Title line */
  strcpy(tmp, res->titleline); cptr=strtok(tmp, " \t\n\r");
  fprintf(fp, "\n%-20.20s ", "Region");
  j=0; while(cptr!=NULL && j<res->parNr) {
    fprintf(fp, " %-10.10s", cptr); cptr=strtok(NULL, " \t\n\r"); j++;}
  fprintf(fp, "\n");
  
  /* Write regional results */
  for(i=0; i<res->voiNr; i++) {
    if(res->voi[i].voiname[0]) strcpy(tmp, res->voi[i].voiname); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    if(res->voi[i].hemisphere[0]) strcpy(tmp, res->voi[i].hemisphere); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    if(res->voi[i].place[0]) strcpy(tmp, res->voi[i].place); else strcpy(tmp, ".");
    fprintf(fp, "%-6.6s ", tmp);
    p=res->voi[i].parameter;
    for(j=0; j<res->parNr; j++) switch(partype[j]) {
      case 0: fprintf(fp, " %10.0f", p[j]); break;
      case 1:
        if(p[j]>=0) fprintf(fp, " %10.4f", p[j]);
        else fprintf(fp, " %10.3f", p[j]);
        break;
      default:
        if(p[j]>=0) fprintf(fp, " %.4e", p[j]);
        else fprintf(fp, " %.3e", p[j]);
      break;
    }
    fprintf(fp, "\n");
    /* Write SD's, if they exist */
    p=res->voi[i].sd; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "%6.6s %6.6s %6.6s ", "SD", ".", ".");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, " %10.0f", p[j]); break;
          case 1:
            if(p[j]>=0) fprintf(fp, " %10.4f", p[j]);
            else fprintf(fp, " %10.3f", p[j]);
            break;
          default:
            if(p[j]>=0) fprintf(fp, " %.4e", p[j]);
            else fprintf(fp, " %.3e", p[j]);
            break;
        } else {
          fprintf(fp, " %10.10s", ".");
        }
      }
      fprintf(fp, "\n");
    }
    /* Write lower confidence limits, if they exist */
    p=res->voi[i].cl1; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "%6.6s %6.6s %6.6s ", "CL", "95%", "Lower");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, " %10.0f", p[j]); break;
          case 1:
            if(p[j]>=0) fprintf(fp, " %10.4f", p[j]);
            else fprintf(fp, " %10.3f", p[j]);
            break;
          default:
            if(p[j]>=0) fprintf(fp, " %.4e", p[j]);
            else fprintf(fp, " %.3e", p[j]);
            break;
        } else {
          fprintf(fp, " %10.10s", ".");
        }
      }
      fprintf(fp, "\n");
    }
    /* Write upper confidence limits, if they exist */
    p=res->voi[i].cl2; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "%6.6s %6.6s %6.6s ", "CL", "95%", "Upper");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, " %10.0f", p[j]); break;
          case 1:
            if(p[j]>=0) fprintf(fp, " %10.4f", p[j]);
            else fprintf(fp, " %10.3f", p[j]);
            break;
          default:
            if(p[j]>=0) fprintf(fp, " %.4e", p[j]);
            else fprintf(fp, " %.3e", p[j]);
            break;
        } else {
          fprintf(fp, " %10.10s", ".");
        }
      }
      fprintf(fp, "\n");
    }
  } /* next region */
  
  /* Close file */
  if(!is_stdout) fclose(fp);
  strcpy(reserrmsg, "");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write calculation results into specied file.
    If file exists, a backup file (%) is written also.
    If "stdout" is given as filename, output is directed to stdout
\return In case of an error, >0 is returned, and a description is written in reserrmsg.
 */
int resWriteHTML(RES *res, char *fname)
{
  int i, j, n, is_stdout=0;
  char tmp[1024], *cptr;
  FILE *fp;
  struct tm *st;
  int partype[MAX_RESPARAMS]; /* 0=int, 1=double, 2=exp */
  double x, pint, max, *p;


  if(RESULT_TEST) printf("resWriteHTML(*res, %s)\n", fname);
  /* Check that there is some data to write */
  if(res==NULL) {strcpy(reserrmsg, "error in result data"); return(1);}
  if(res->voiNr<1) {strcpy(reserrmsg, "no result data"); return(1);}
  /* Check if writing to stdout */
  if(!strcmp(fname, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(fname, 0) != -1) {
    strcpy(tmp, fname); strcat(tmp, "%"); rename(fname, tmp);}
  strcpy(reserrmsg, "cannot write file");

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp=fopen(fname, "w"))==NULL) {
    strcpy(reserrmsg, "cannot open file"); return(2);}

  /* Write HTML header */
  n=fprintf(fp, "<HTML><HEAD><TITLE>PET results</TITLE></HEAD>\n");
  if(n==0) {
    strcpy(reserrmsg, "disk full");
    if(!is_stdout) fclose(fp); return(3);
  }

  /* Start writing the body of the HTML file */
  fprintf(fp, "<BODY>\n");

  /* Write the title lines to the first table */
  fprintf(fp, "<table border=\"0\" width=\"100%%\">\n");
  /* Program name */
  fprintf(fp, "<tr><th align=left>Program:</th><td>%s</td></tr>\n", res->program);
  /* Write calculation date and time */
  st=localtime(&res->time); strftime(tmp, 256, "%Y-%m-%d %T", st);
  fprintf(fp, "<tr><th align=left>Date:</th><td>%s</td></tr>\n", tmp);
  /* Write the studynr */
  if(res->studynr[0])
    fprintf(fp, "<tr><th align=left>Study:</th><td>%s</td></tr>\n", res->studynr);
  /* Write the names of the original datafiles */
  if(res->datafile[0])
    fprintf(fp, "<tr><th align=left>Data file:</th><td>%s</td></tr>\n", res->datafile);
  if(res->plasmafile[0])
    fprintf(fp, "<tr><th align=left>Plasma file:</th><td>%s</td></tr>\n", res->plasmafile);
  if(res->bloodfile[0])
    fprintf(fp, "<tr><th align=left>Blood file:</th><td>%s</td></tr>\n", res->bloodfile);
  if(res->reffile[0])
    fprintf(fp, "<tr><th align=left>Reference file:</th><td>%s</td></tr>\n", res->reffile);
  if(res->refroi[0])
    fprintf(fp, "<tr><th align=left>Reference region:</th><td>%s</td></tr>\n", res->refroi);
  /* Write data range */
  if(res->datarange[0])
    fprintf(fp, "<tr><th align=left>Data range:</th><td>%s</td></tr>\n", res->datarange);
  /* Write constants */
  if(res->density>0.0)
    fprintf(fp, "<tr><th align=left>Tissue density:</th><td>%g</td></tr>\n", res->density);
  if(res->lc>0.0)
    fprintf(fp, "<tr><th align=left>Lumped constant:</th><td>%g</td></tr>\n", res->lc);
  if(res->concentration>0.0)
    fprintf(fp, "<tr><th align=left>Concentration:</th><td>%g</td></tr>\n", res->concentration);
  if(res->beta>0.0)
    fprintf(fp, "<tr><th align=left>Beta:</th><td>%g</td></tr>\n", res->beta);
  if(res->Vb>=0.0)
    fprintf(fp, "<tr><th align=left>Vb:</th><td>%g %%</td></tr>\n", res->Vb);
  /* Weighting */
  if(res->isweight)
    fprintf(fp, "<tr><th align=left>Weighting:</th><td>yes</td></tr>\n");
  else
    fprintf(fp, "<tr><th align=left>Weighting:</th><td>none</td></tr>\n");
  /* End the title table */
  fprintf(fp, "</table>\n");

  /* Collect info on column types */
  for(j=0; j<res->parNr; j++) {
    /* should column be printed as integers, floats or exponentials? */
    partype[j]=0; max=0.0;
    for(i=0; i<res->voiNr; i++) {
      x=res->voi[i].parameter[j]; if(modf(x, &pint)!=0.0) partype[j]=1;
      x=fabs(x); if(x>max) max=x;
    }
    if(partype[j]==1 && (max>=10.0 || max<0.1)) partype[j]=2;
  }

  /* Write the result data to the 2nd table */
  fprintf(fp, "<table border=\"1\" width=\"100%%\">\n");
  /* Write the result title line */
  fprintf(fp, "<tr align=left><th>Region</th><th>Hemisphere</th><th>Plane</th>\n");
  strcpy(tmp, res->titleline); cptr=strtok(tmp, " \t\n\r");
  i=0; while(cptr!=NULL && i<res->parNr) {
    fprintf(fp, "<th>%s</th>", cptr); cptr=strtok(NULL, " \t\n\r"); i++;}
  fprintf(fp, "</tr>\n");
  /* Write regional results */
  for(i=0; i<res->voiNr; i++) {
    fprintf(fp, "<tr>");
    fprintf(fp, "<th>%s</th>", res->voi[i].voiname);
    fprintf(fp, "<th>%s</th>", res->voi[i].hemisphere);
    fprintf(fp, "<th>%s</th>", res->voi[i].place);
    p=res->voi[i].parameter;
    for(j=0; j<res->parNr; j++) switch(partype[j]) {
      case 0: fprintf(fp, "<th>%.0f</th>", p[j]); break;
      case 1: fprintf(fp, "<th>%.4f</th>", p[j]); break;
      default: fprintf(fp, "<th>%.4E</th>", p[j]); break;
    }
    fprintf(fp, "</tr>\n");
    /* Write SD's, if they exist */
    p=res->voi[i].sd; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "<tr align=right>");
      fprintf(fp, "<th><font color=\"Blue\">%s</font></th><th></th><th></th>", "SD");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, "<th><font color=\"Blue\">%.0f</font></th>", p[j]); break;
          case 1: fprintf(fp, "<th><font color=\"Blue\">%.4f</font></th>", p[j]); break;
          default: fprintf(fp, "<th><font color=\"Blue\">%.4E</font></th>", p[j]); break;
        } else fprintf(fp, "<th></th>");
      }
      fprintf(fp, "</tr>\n");
    }
    /* Write lower confidence limits, if they exist */
    p=res->voi[i].cl1; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "<tr align=right>");
      fprintf(fp, "<th><font color=\"Green\">%s</font></th><th><font color=\"Green\">%s</font></th><th><font color=\"Green\">%s</font></th>", "CL", "95%", "Lower");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, "<th><font color=\"Green\">%.0f</font></th>", p[j]); break;
          case 1: fprintf(fp, "<th><font color=\"Green\">%.4f</font></th>", p[j]); break;
          default: fprintf(fp, "<th><font color=\"Green\">%.4E</font></th>", p[j]); break;
        } else fprintf(fp, "<th></th>");
      }
      fprintf(fp, "</tr>\n");
    }
    /* Write upper confidence limits, if they exist */
    p=res->voi[i].cl2; for(j=n=0; j<res->parNr; j++) if(p[j]!=NA) n++;
    if(n>0) {
      fprintf(fp, "<tr align=right>");
      fprintf(fp, "<th><font color=\"Green\">%s</font></th><th><font color=\"Green\">%s</font></th><th><font color=\"Green\">%s</font></th>", "CL", "95%", "Upper");
      for(j=0; j<res->parNr; j++) {
        if(p[j]!=NA) switch(partype[j]) {
          case 0: fprintf(fp, "<th><font color=\"Green\">%.0f</font></th>", p[j]); break;
          case 1: fprintf(fp, "<th><font color=\"Green\">%.4f</font></th>", p[j]); break;
          default: fprintf(fp, "<th><font color=\"Green\">%.4E</font></th>", p[j]); break;
        } else fprintf(fp, "<th></th>");
      }
      fprintf(fp, "</tr>\n");
    }
  }
  /* End the data table */
  fprintf(fp, "</table>\n");

  /* Stop writing the body of the HTML file, and end the file */
  n=fprintf(fp, "</BODY></HTML>\n");
  if(n==0) {
    strcpy(reserrmsg, "disk full");
    if(!is_stdout) fclose(fp); return(3);
  }

  /* Close file */
  if(!is_stdout) fclose(fp);
  strcpy(reserrmsg, "");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** See studynr_from_fname() */
int resFName2study(char *fname, char *studyNumber)
{
  return(studynr_from_fname(fname, studyNumber));
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the median and the lowest and highest value in the
    specified double array data of length nr.
    Note that array is sorted in this function.
    NULL pointer may be specified to function in place of an unwanted
    return parameter.
\return Returns 0 if succesfull.
 */
int resMedian(double *data, int nr, double *median, double *min, double *max)
{
  /* Check the arguments */
  if(data==NULL) return(1); if(nr<1) return(2);
  /* Sort data in increasing order */
  qsort(data, nr, sizeof(double), resQSortComp);
  /* Get minimum and maximum */
  if(min!=NULL) *min=data[0];
  if(max!=NULL) *max=data[nr-1];
  /* Calculate median */
  if(median!=NULL) {
    if(nr%2) *median=data[(nr-1)/2];
    else *median=0.5*(data[(nr/2)-1]+data[nr/2]);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the mean and sd in the specified double array data of length nr.
    NULL pointer may be specified to function in place of an unwanted
    return parameter.
\return Returns 0 if succesfull.
 */
int resMean(double *data, int nr, double *mean, double *sd)
{
  int i;
  double sum, ssum, v;

  /* Check the arguments */
  if(data==NULL) return(1); if(nr<1) return(2);
  /* Calculate avg and sd */
  for(i=0, sum=ssum=0.0; i<nr; i++) {
    sum+=data[i]; ssum+=data[i]*data[i];
  }
  if(mean!=NULL) *mean=sum/(double)nr;
  if(sd!=NULL) {
    if(nr>1) v=(ssum-sum*sum/(double)nr)/(double)(nr-1); else v=0.0;
    if(v>1.0E-12) *sd=sqrt(v); else *sd=0.0;
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int resQSortComp(const void *f1, const void *f2)
{
  if(*(double*)f1<*(double*)f2) return(-1);
  else if(*(double*)f1>*(double*)f2) return(1);
  else return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort RES regions by region name. */
void resSortByName(RES *res)
{
  if(res==NULL || res->voiNr<=1) return;
  qsort(res->voi, res->voiNr, sizeof(ResVOI), resQSortName);
  return;
}
/*****************************************************************************/

/*****************************************************************************/
int resQSortName(const void *voi1, const void *voi2)
{
  int res;

  res=strcmp( ((const ResVOI*)voi1)->voiname, ((const ResVOI*)voi2)->voiname );
  if(res!=0) return(res);
  res=strcmp( ((const ResVOI*)voi1)->hemisphere, ((const ResVOI*)voi2)->hemisphere );
  if(res!=0) return(res);
  res=strcmp( ((const ResVOI*)voi1)->place, ((const ResVOI*)voi2)->place );
  return(res);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy result main header information to another result structure. */
int resCopyMHeader(RES *res1, RES *res2)
{
  if(res1==NULL || res2==NULL) return(1);
  strcpy(res2->program, res1->program);
  res2->time=res1->time;
  res2->parNr=res1->parNr;
  strcpy(res2->studynr, res1->studynr);
  strcpy(res2->datafile, res1->datafile);
  strcpy(res2->reffile, res1->reffile);
  strcpy(res2->plasmafile, res1->plasmafile);
  strcpy(res2->bloodfile, res1->bloodfile);
  strcpy(res2->refroi, res1->refroi);
  strcpy(res2->datarange, res1->datarange);
  res2->density=res1->density;
  res2->lc=res1->lc;
  res2->concentration=res1->concentration;
  res2->beta=res1->beta;
  res2->Vb=res1->Vb;
  res2->isweight=res1->isweight;
  strcpy(res2->titleline, res1->titleline);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Delete specified region (0..voiNr-1) from the structure.
\return Returns 0 if ok.
 */
int resDelete(RES *res, int voi)
{
  int i;

  /* Check that region exists */
  if(voi>res->voiNr-1 || voi<0) return(1);
  /* If it is the last one, then just decrease the voiNr */
  if(voi==res->voiNr) {res->voiNr--; return(0);}
  /* Then we have to move the following regions in its place */
  for(i=voi+1; i<res->voiNr; i++)
    memcpy(&res->voi[i-1], &res->voi[i], sizeof(ResVOI));
  res->voiNr--;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Select VOIs (sets sw=1), whose names are matching specified string.
    If no string is specified, then all VOIs are selected.
\return Returns the number of matches, or <0, if an error occurred.
 */
int resSelect(RES *data, char *name)
{
  int i, j, n;
  char *p, n1[128], n2[128], n3[128], tmp[128], sname[1024];

  /* Select all, if no string was specified */
  if(name==NULL || strlen(name)==0) {
    for(i=0; i<data->voiNr; i++) data->voi[i].sw=1; return(data->voiNr);}
  /* Make a copy of 'name' and use it */
  strcpy(sname, name);
  /* Check if string contains several substrings (hemisphere and place) */
  n1[0]=n2[0]=n3[0]=(char)0;
  p=strtok(sname, " ,;\n\t|"); if(p!=NULL) strcpy(n1, p); else return(-1);
  p=strtok(NULL, " ,;\n\t|"); if(p!=NULL) {
    strcpy(n2, p); p=strtok(NULL, " ,;\n\t|"); if(p!=NULL) strcpy(n3, p);}
  /* Convert strings to lowercase */
  for(i=0; i<strlen(n1); i++) n1[i]=tolower(n1[i]);
  for(i=0; i<strlen(n2); i++) n2[i]=tolower(n2[i]);
  for(i=0; i<strlen(n3); i++) n3[i]=tolower(n3[i]);
  /* Search through the data */
  for(i=0, n=0; i<data->voiNr; i++) {
    data->voi[i].sw=0;
    sprintf(tmp, "%s%s%s", data->voi[i].voiname, data->voi[i].hemisphere,
                           data->voi[i].place);
    for(j=0; j<strlen(tmp); j++) tmp[j]=tolower(tmp[j]);
    if(strstr(tmp, n1)==NULL) continue;
    if(n2[0] && strstr(tmp, n2)==NULL) continue;
    if(n3[0] && strstr(tmp, n3)==NULL) continue;
    data->voi[i].sw=1; n++;
  }
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/

