/******************************************************************************
  Library file:  roi.c
  Description:   Procedures for reading, writing and processing 2D ROIs
                 of CTI ECAT format.

  Copyright (c) Turku PET Centre 1996-2004

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  1996-10-08 Vesa Oikonen
    First version.
  2000-11-24 VO
    nint() -> (int)rint().
  2001-10-03 VO
    Added some parenthesis.
  2002-02-18 VO
    i386 compatible version.
    Function names changed and procedures reorganized in functions.
    Possibility to apply ROIs to sinogram data was removed.
    Included functions for initiating and emptying ROI data.
    Removed the option for scaling ROIs to image with other matrix size.
  2002-11-22 Anne Seppnen
    New function to append rois to file.
  2003-01-23 AS
    Function roiAppend changed to write only one roi or all rois 
    from roi_list depending the 3rd parameter.
  2003-03-07 AS
    Function roiDel deletes ROI(s). 
    If ind=0 all ROIs are deleted, else del. ROI, which index is ind.
    Memory handling: _allocNr was added to ROI_list struct in roi.h file. 
  2003-07-02 CL
    Created the new API.
    ROIs are now stored in a linked list rather than a array.
    Rewrote RoiOnOff
    Rewrote Roi -> trace roi conversion functions
  2003-11-30 VO
    For now, calls temp_roundf() instead of roundf().
  2004-11-18 VO
    When reading trace ROIs with roiRead(), start and end points are connected
    if they are not so in the ROI file.
  2004-11-25 CL
    improved roi_onoff <-- kesken
    userdata is guaranteed to be NULL when ROI is loaded
    added roi module specific version information
    Merged following changes from 2003-10-21 version:
    Fixed ellipse and circle ROI bugs (radius was used as diameter)
    delete_roi and delete_roi_n now return a pointer to the next ROI in list.
  2004-12-22 VO
    Changed some comments etc, not affecting actual code.
    Note that improved roi_onoff() is applied only with "new" roi functions,
    old roi functions work as before.
    

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
/*****************************************************************************/
#include "include/ecat63.h"
#include "include/petc99.h"
#include "include/roi.h"
/*****************************************************************************/

#define SUPPORT_LEGACY

/** The ROI library error message */
char roierrmsg[128];

/*****************************************************************************/
/*** Return roi module version info ***/
const char *libpet_roi_version(void) {
    return "2004-11-25";
}

/** Allocated memory is freed, all data is cleared. */
void roi_empty(ROI_list *rl)
{
  int r;
  RoiList *next;
#ifdef SUPPORT_LEGACY 
  if(rl->roi) {
    for(r=0; r<rl->nr; r++) {
      if(rl->roi[r].point_nr>0) {
        free(rl->roi[r].x);
        free(rl->roi[r].y);
        rl->roi[r].point_nr=0;
      }
    }
  }
  if(rl->_allocNr>0) free(rl->roi);
  rl->_allocNr=0;
#endif
  rl->nr=0;
  while(rl->rois) {
    next=rl->rois->next;
    free(rl->rois);
    rl->rois=next;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Before any use, this should be called, but not later.
 *  Note. Rename to roi_init() when the old functions are removed
 */
void roiInit(ROI_list *rl)
{
  rl->roi=(ROI*)NULL;
  rl->nr=0;
#ifdef SUPPORT_LEGACY
  rl->_allocNr=0;
  rl->rois=NULL;
#endif
}

/*****************************************************************************/
/** Deletes the index:th ROI */
void roi_delete_n(ROI_list *rl, int ind) {
  RoiList *rois=rl->rois;
  int pos=0;
  while(pos<ind) {
    rois=rois->next;
    if(!rois) return;
    pos++;
  }
  if(rois->prev) rois->prev->next=rois->next; else rl->rois=rois->next;
  if(rois->next) rois->next->prev=rois->prev;
  if(rois->roi->x) free(rois->roi->x);
  if(rois->roi->y) free(rois->roi->y);
  free(rois->roi);
  free(rois);
  rl->nr--;
}

/*****************************************************************************/
/** Deletes ROI from roilist */
void roi_delete(ROI_list *rl, ROI *roi) {
  RoiList *rois=rl->rois;
  while(rois->roi!=roi) {
    rois=rois->next;
    if(!rois) return;
  }
  if(rois->prev) rois->prev->next=rois->next; else rl->rois=rois->next;
  if(rois->next) rois->next->prev=rois->prev;
  if(rois->roi->x) free(rois->roi->x);
  if(rois->roi->y) free(rois->roi->y);
  free(rois->roi);
  free(rois);
  rl->nr--;
}

/* To be removed */
#ifdef SUPPORT_LEGACY
/*****************************************************************************/
/** If ind=0, all ROIs in list are deleted keeping memory allocated,
 *  else only one ROI is deleted (ind=1, first ROI) keeping memory allocated.
 */
void roiDel(ROI_list *rl, int ind)
{
  int  j,r;
  
  if(ind==0){
    for(r=0; r<rl->nr; r++) {
      if(rl->roi[r].point_nr>0) {
        free(rl->roi[r].x);
        free(rl->roi[r].y);
        rl->roi[r].point_nr=0;
      }
    }
    rl->nr=0;
  }
  else {
     if(ind>rl->nr) return;
     /* free x and y for given ROI */
     if(rl->roi[ind-1].point_nr>0) {
        free(rl->roi[ind-1].x);
        free(rl->roi[ind-1].y);
        rl->roi[ind-1].point_nr=0;
     }
     /* copy memory */
     for(j=ind; j<rl->nr; j++)
	 memcpy(&rl->roi[j-1], &rl->roi[j], sizeof(ROI));
     rl->nr--;
  }
                     
}
#endif
/*****************************************************************************/
/** Adds ROI file contents (all ROIs) to specified data structure.
\return Returns 0, if OK.
 */
int roi_read(const char *fname, ROI_list *rl) {
  RoiList *newroientry;
  RoiList *roilist=rl->rois;
  ROI *newroi;
  int    i, j, n, ch;
  FILE  *fp;

  if(roilist) while(roilist->next) roilist=roilist->next;

  /* Open ROI file */
  if((fp=fopen(fname, "r")) == NULL) {
    strcpy(roierrmsg, "cannot open file");
    return 1;
  }

  n=0;
  while(1) {
    /* Allocate space for the new ROI and initialize it*/
    newroi=malloc(sizeof(ROI));
    memset(newroi,0,sizeof(ROI));
    newroi->userdata=NULL;

    /* Read first '*' */
    ch=fgetc(fp); if(ch==EOF || ch!='*') break;

    /* Read image filename */
    if(fscanf(fp, "%s ", newroi->imgfile)==EOF) {
      strcpy(roierrmsg, "cannot read image filename");
      fclose(fp);
      return 3;
    }

    /* Read zooms */
    if(fscanf(fp, "%f %f ", &newroi->zoom, &newroi->recon_zoom)==EOF) {
      strcpy(roierrmsg, "cannot read zoom values");
      fclose(fp);
      return 4;
    }
    if(newroi->recon_zoom==0.0) newroi->recon_zoom=1.0;

    /* Read matrix -> t */
    if(fscanf(fp, "%d %d %d %d %d %d %d %d ", &newroi->matnum, &newroi->type, &newroi->status,
         &newroi->pos_x, &newroi->pos_y, &newroi->w, &newroi->h, &newroi->t)==EOF) {
      strcpy(roierrmsg, "cannot read ROI definitions");
      fclose(fp);
      return 5;
    }

    /* Read ROI nr and name */
    if(fscanf(fp, "%d ", &newroi->roi)==EOF) {
      strcpy(roierrmsg, "cannot read ROI number");
      fclose(fp);
      return 6;
    }
    for(i=j=0; j<3; i++) {
      newroi->roiname[i]=(char)getc(fp);
      if(newroi->roiname[i]=='/') j++; else j=0;
    }
    j=i-3; newroi->roiname[j]='\0';
    /* delete spaces from ROI name */
    j=strlen(newroi->roiname);
    for(i=j-1; i>=0; i--)
      if(isspace((int)newroi->roiname[i])) newroi->roiname[i]=(char)0; else break;
    j=i; for(i=0; i<j; i++)
      if(!isspace((int)newroi->roiname[i])) {if(i>0) strcpy(newroi->roiname, newroi->roiname+i); break;}

    /* Read the number of points */
    fseek(fp,2,SEEK_CUR); /* skip 0 and space */
    if(fscanf(fp, "%d ", &newroi->point_nr)==EOF) {
      strcpy(roierrmsg, "cannot read nr of ROI border points");
      fclose(fp);
      return 8;
    }

    /* If TRACE ROI, read points */
    if(newroi->type==ROI_TRACE) {
      newroi->x=malloc(newroi->point_nr*sizeof(int));
      newroi->y=malloc(newroi->point_nr*sizeof(int));
      for(i=0; i<newroi->point_nr; i++)
        if(fscanf(fp, "%d %d", &newroi->x[i], &newroi->y[i]) == EOF) {
          strcpy(roierrmsg, "cannot read ROI border definition");
          free(newroi->x);
	  free(newroi->y);
	  free(newroi);
          fclose(fp);
	  return 9;
        }
      do {ch=getc(fp);} while(ch!=EOF && ch!='\n');
    }
    /* Append ROI to list */
    newroientry=malloc(sizeof(RoiList));
    newroientry->roi=newroi;
    newroientry->prev=roilist;
    newroientry->next=NULL;
    if(roilist) {
      roilist->next=newroientry;
      roilist=newroientry;
    } else {
      roilist=newroientry;
      rl->rois=roilist;
    }
    rl->nr++;
    n++;
  }
  fclose(fp);
  if(n==0) {
    strcpy(roierrmsg, "error in ROI file");
    return 2;
  }
  return 0;
}

#ifdef SUPPORT_LEGACY
/*****************************************************************************/
/** Adds ROI file contents (all ROIs) to specified data structure.
\return Returns 0, if OK.
 */
int roiRead(const char *fname, ROI_list *rl)
{
  int    i, j, n, ch;
  FILE  *fp;
  char   imgfile[FILENAME_MAX], roiname[FILENAME_MAX];
  float  zoom, recon_zoom;
  int    matrix, type, status, pos_x, pos_y, w, h, t, roi, npts, x, y;

  /* Open ROI file */
  if((fp=fopen(fname, "r")) == NULL) {
    strcpy(roierrmsg, "cannot open file");
    return(1);
  }

  n=0;
  while(1) {
    /* Read first '*' */
    ch=fgetc(fp); if(ch==EOF || ch!='*') break;

    /* Read image filename */
    if(fscanf(fp, "%s ", imgfile)==EOF) {
      strcpy(roierrmsg, "cannot read image filename");
      fclose(fp); return(3);
    }

    /* Read zooms */
    if(fscanf(fp, "%f %f ", &zoom, &recon_zoom)==EOF) {
      strcpy(roierrmsg, "cannot read zoom values");
      fclose(fp); return(4);}
    if(recon_zoom==0.0) recon_zoom=1.0;

    /* Read matrix -> t */
    if(fscanf(fp, "%d %d %d %d %d %d %d %d ", &matrix, &type, &status,
         &pos_x, &pos_y, &w, &h, &t)==EOF) {
      strcpy(roierrmsg, "cannot read ROI definitions");
      fclose(fp); return(5);
    }

    /* Read ROI nr and name */
    if(fscanf(fp, "%d ", &roi)==EOF) {
      strcpy(roierrmsg, "cannot read ROI number");
      fclose(fp); return(6);
    }
    for(i=j=0; j<3; i++) {
      roiname[i]=(char)getc(fp);
      if(roiname[i]=='/') j++; else j=0;
    }
    j=i-3; roiname[j]=(char)0;
    /* delete spaces from ROI name */
    j=strlen(roiname);
    for(i=j-1; i>=0; i--)
      if(isspace((int)roiname[i])) roiname[i]=(char)0; else break;
    j=i; for(i=0; i<j; i++)
      if(!isspace((int)roiname[i])) {if(i>0) strcpy(roiname, roiname+i); break;}

    /* Read the number of points */
    (void)getc(fp); (void)getc(fp); /* skip 0 and space */
    if(fscanf(fp, "%d ", &npts)==EOF) {
      strcpy(roierrmsg, "cannot read nr of ROI border points");
      fclose(fp); return(8);
    }

    /* Allocate memory for ROI */
    if(rl->_allocNr==0){ rl->roi=malloc(sizeof(ROI)); rl->_allocNr++;}
    else if(rl->_allocNr<=rl->nr){
       rl->roi=(ROI*)realloc((char*)rl->roi, (rl->nr+1)*sizeof(ROI));
       rl->_allocNr++;
    }
    if(rl->roi==NULL) {
      strcpy(roierrmsg, "out of memory");
      fclose(fp); rl->nr=0; return(11);
    } 
    

    /* Copy data */ 
    strcpy(rl->roi[rl->nr].imgfile, imgfile);
    rl->roi[rl->nr].zoom = zoom;
    rl->roi[rl->nr].recon_zoom = recon_zoom;
    rl->roi[rl->nr].matnum = matrix;
    rl->roi[rl->nr].type = type;
    rl->roi[rl->nr].status = status;
    rl->roi[rl->nr].pos_x = pos_x;
    rl->roi[rl->nr].pos_y = pos_y;
    rl->roi[rl->nr].w = w;
    rl->roi[rl->nr].h = h;
    rl->roi[rl->nr].t = t;
    rl->roi[rl->nr].roi = roi;
    strcpy(rl->roi[rl->nr].roiname, roiname);
    rl->roi[rl->nr].point_nr = npts; 
    rl->roi[rl->nr].userdata=NULL;
    rl->nr++;
    n++;

    /* If TRACE ROI, read points */
    if(type==ROI_TRACE) {
      if((rl->roi[rl->nr-1].x=malloc((npts+1)*sizeof(int)))==NULL) {
        strcpy(roierrmsg, "cannot read ROI border definition");
        fclose(fp); rl->nr--; return(12);
      }
      if((rl->roi[rl->nr-1].y=malloc((npts+1)*sizeof(int)))==NULL) {
        strcpy(roierrmsg, "cannot read ROI border definition");
        fclose(fp); rl->nr--; free((char*)rl->roi[rl->nr-1].x); return(13);
      }
      for(i=0; i<npts; i++)
        if(fscanf(fp, "%d %d", &x, &y) == EOF) {
          strcpy(roierrmsg, "cannot read ROI border definition");
          free((char*)rl->roi[rl->nr-1].x); free((char*)rl->roi[rl->nr-1].y);
          fclose(fp); rl->nr--; return(9);
        } else {
          rl->roi[rl->nr-1].x[i]=x; rl->roi[rl->nr-1].y[i]=y;
        }
      do {ch=getc(fp);} while(ch!=EOF && ch!='\n');
      /* make sure that ROI start and end are connected (extra memory was allocated previously) */
      if(rl->roi[rl->nr-1].x[npts-1]!=0 || rl->roi[rl->nr-1].y[npts-1]!=0) {
        rl->roi[rl->nr-1].x[npts]=0;
        rl->roi[rl->nr-1].y[npts]=0;
        npts++; rl->roi[rl->nr-1].point_nr++;
      }
    }
  }
  fclose(fp);
  if(n==0) {
    strcpy(roierrmsg, "error in ROI file");
    return(2);
  }
  /* Make points for others than trace ROIs */
  for(i=rl->nr-n; i<rl->nr; i++) {
    switch(rl->roi[i].type) {
      case ROI_RECTANGULAR:	j=roiComputeRect(&rl->roi[i]); break;
      case ROI_CIRCULAR:	j=roiComputeCirc(&rl->roi[i]); break;
      case ROI_ELLIPSE:		j=roiComputeElli(&rl->roi[i]); break;
      case ROI_TRACE:		j=0; break;
      default:			j=-1;
    }
    if(j) {
      strcpy(roierrmsg, "cannot read ROI border definition");
      rl->nr-=n; return(21);
    }
  }
  return(0);
}
/*****************************************************************************/
#endif

/*****************************************************************************/
/*
 *  int roi_compute_rect(ROI *roi);
 */ 
int roi_compute_rect(ROI *roi)
{
  roi->point_nr=4;
  if((roi->x=malloc(roi->point_nr*sizeof(int)))==NULL)
    return(-1);
  if((roi->y=malloc(roi->point_nr*sizeof(int)))==NULL) {
    free(roi->x); return(-1);}
  roi->x[0]=0; roi->y[0]=0;
  roi->x[1]=roi->w; roi->y[1]=0;
  roi->x[2]=roi->w; roi->y[2]=roi->h;
  roi->x[3]=0; roi->y[3]=roi->h;
  return(0);
}
#ifdef SUPPORT_LEGACY
int roiComputeRect(ROI *roi)
{
  roi->point_nr=5;
  if((roi->x=malloc(roi->point_nr*sizeof(int)))==NULL)
    return(-1);
  if((roi->y=malloc(roi->point_nr*sizeof(int)))==NULL) {
    free((char*)roi->x); return(-1);}
  roi->x[0]=0; roi->y[0]=0;
  roi->x[1]=roi->w; roi->y[1]=0;
  roi->x[2]=roi->w; roi->y[2]=roi->h;
  roi->x[3]=0; roi->y[3]=roi->h;
  roi->x[4]=0; roi->y[4]=0;
  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/*
 *  int roi_compute_circle(ROI *roi);
 */
int roi_compute_circle(ROI *roi) {
  int points,r;
  r=roi->w/2;
  roi->pos_x+=r;
  roi->pos_y+=roi->w;
  roi->point_nr=36; /* TODO: Some clever algorithm to decide how many points we need */
  roi->x=malloc(sizeof(int)*roi->point_nr);
  roi->y=malloc(sizeof(int)*roi->point_nr);
  for(points=0;points<roi->point_nr;points++) {
    float rad=(M_PI/180.0)*(360.0*(points/(float)roi->point_nr));
    roi->x[points]=sin(rad)*r;
    roi->y[points]=cos(rad)*r-r;
  }
  return 0;
}
#ifdef SUPPORT_LEGACY
int roiComputeCirc(ROI *roi)
{
  int i, d, x, y, n, r, s;

  r=roi->w;
  if((roi->x=malloc(sizeof(int)))==NULL) return(-1);
  if((roi->y=malloc(sizeof(int)))==NULL) {free((char*)roi->x); return(-1);}
  roi->x[0]=0; roi->y[0]=r;
  if(r==0) n=1; else {
    for(x=1, y=r, d=3-2*r, n=1; x<=y; x++) {
      if(d<0) d=d+4*x+6; else {d=d+4*(x-y)+10; y-=1;}
      if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
        return(-1);
      if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
        free((char*)roi->x); free((char*)roi->y); return(-1);}
      roi->x[n]=x; roi->y[n]=y;
      n++;
    }
  }
  for(i=n-1, s=i; i>0; i--) {
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return(-1);
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return(-1);}
    roi->x[n]=roi->y[--s]; roi->y[n++]=roi->x[s];
  }
  for(i=n-1, s=i; i>0; i--) {
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return(-1);
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return(-1);}
    roi->x[n]=roi->x[--s]; roi->y[n++]=-roi->y[s];
  }
  for(i=n-1, s=i; i>0; i--) {
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return(-1);
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return(-1);}
    roi->x[n]=-roi->x[--s]; roi->y[n++]=roi->y[s];
  }
  roi->point_nr=n;

  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/*
 *  int roi_compute_ellipse(ROI *roi)
 */ 
int roi_compute_ellipse(ROI *roi) {
  int points;
  roi->pos_x+=roi->w/2;
  roi->pos_y+=roi->h;
  roi->point_nr=40; /* TODO: Some clever algorithm to decide how many points we need */
  roi->x=malloc(sizeof(int)*roi->point_nr);
  roi->y=malloc(sizeof(int)*roi->point_nr);
  for(points=0;points<roi->point_nr;points++) {
    float rad=(M_PI/180.0)*(360.0*(points/(float)roi->point_nr));
    roi->x[points]=sin(rad)*(roi->w/2);
    roi->y[points]=cos(rad)*(roi->h/2)-roi->h/2;
  }
  return 0;

}
#ifdef SUPPORT_LEGACY
int roiComputeElli(ROI *roi)
{
  int i, ix, iy, ox, oy, n, s, w, h;

  w=roi->w; h=roi->h;
  if((roi->x=malloc(sizeof(int)))==NULL) return(-1);
  if((roi->y=malloc(sizeof(int)))==NULL) {free((char*)roi->x); return(-1);}
  roi->x[0]=0; roi->y[0]=h;
  n=1; oy=h;
  for(ix=1; ix<w; ix++) {
    iy=(jsqrt(h+h*h*(w-ix)*(w+ix)))/w; if(oy-iy>1) break;
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return -1;
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return -1;}
    roi->x[n]=ix; roi->y[n++]=iy;
    oy=iy;
  }
  ox=ix;
  for(iy=oy-1; iy>=0; iy--) {
    ix=(jsqrt(w+w*w*(h-iy)*(h+iy)))/h;
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return -1;
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return -1;}
    roi->x[n]=ix; roi->y[n++]=iy;
    ox=ix;
  }
  for(i=n-1, s=i; i>0; i--) {
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return -1;
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return -1;}
    roi->x[n]=roi->x[--s]; roi->y[n++]=-roi->y[s];
  }
  for(i=n-1, s=i; i>0; i--) {
    if((roi->x=(int*)realloc((char*)roi->x, (n+1)*sizeof(int)))==NULL)
      return -1;
    if((roi->y=(int*)realloc((char*)roi->y, (n+1)*sizeof(int)))==NULL) {
      free((char*)roi->x); free((char*)roi->y); return -1;}
    roi->x[n]=-roi->x[--s]; roi->y[n++]=roi->y[s];
  }
  roi->point_nr=n;

  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** Prints the contents of (individual) roi data structure. */
void roi_print(ROI *roi)
{
  int i;

  printf("imgfile: %s\n", roi->imgfile);
  printf("zoom: %f  recon_zoom: %f\n", roi->zoom, roi->recon_zoom);
  printf("matnum: %d ; plane: %d frame: %d\n",
    roi->matnum, roi_mplane(roi->matnum), roi_mframe(roi->matnum) );
  printf("type: %d status: %d\n", roi->type, roi->status);
  printf("pos_x=%d pos_y=%d w=%d h=%d t=%d\n",
    roi->pos_x, roi->pos_y, roi->w, roi->h, roi->t);
  printf("roi=%d _%s_\n", roi->roi, roi->roiname);
  printf("points=%d\n", roi->point_nr);
  if(roi->point_nr>0) {
    for(i=0; i<roi->point_nr; i++) printf("(%d,%d) ", roi->x[i], roi->y[i]);
    printf("\n");
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Fills the gaps between ROI points; join them with new points.
 *  Allocates memory for new point data in fx and fy.
\return Returns the number of new points or =< 0, if error.
 */
int roiFillGaps(int *x, int *y, int nr, int **ffx, int **ffy)
{
  int i, j, x1, x2=0, y1, y2=0, dx, dy, nx, ny, *fx, *fy, n;
  float k;

  /* Check data */
  if(nr<1) return(0);

  /* Allocate memory for new data */
  if((fx=malloc(nr*sizeof(int)))==NULL) return -1;
  if((fy=malloc(nr*sizeof(int)))==NULL) {free((char*)fx); return -1;}

  /* Process from first to last (last->first is done already) */
  for(i=1, n=j=0; i<nr; i++) {
    x1=x[i-1]; x2=x[i]; y1=y[i-1]; y2=y[i]; dx=x2-x1; dy=y2-y1;
    fx[j]=x1; fy[j++]=y1;
    if(abs(dx)<2 && abs(dy)<2) continue;
    if(abs(dx)>=abs(dy)) {
      k=(float)dy/(float)dx;
      if(x2>x1)
        for(nx=x1+1; nx<x2; nx++) {
          ny=y1+(int)temp_roundf(k*(nx-x1));
          fx[j]=nx; fy[j++]=ny; n++;
          if((fx=(int*)realloc((char*)fx, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fy); return -1;}
          if((fy=(int*)realloc((char*)fy, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fx); return -1;}
        }
      else
        for(nx=x1-1; nx>x2; nx--) {
          ny=y1+(int)temp_roundf(k*(nx-x1));
          fx[j]=nx; fy[j++]=ny; n++;
          if((fx=(int*)realloc((char*)fx, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fy); return -1;}
          if((fy=(int*)realloc((char*)fy, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fx); return -1;}
        }
    } else {
      k=(float)dx/(float)dy;
      if(y2>y1)
        for(ny=y1+1; ny<y2; ny++) {
          nx=x1+(int)temp_roundf(k*(ny-y1));
          fx[j]=nx; fy[j++]=ny; n++;
          if((fx=(int*)realloc((char*)fx, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fy); return -1;}
          if((fy=(int*)realloc((char*)fy, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fx); return -1;}
        }
      else
        for(ny=y1-1; ny>y2; ny--) {
          nx=x1+(int)temp_roundf(k*(ny-y1));
          fx[j]=nx; fy[j++]=ny; n++;
          if((fx=(int*)realloc((char*)fx, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fy); return -1;}
          if((fy=(int*)realloc((char*)fy, (nr+n)*sizeof(int)))==NULL) {
            free((char*)fx); return -1;}
        }
    }
  }
  fx[j]=x2; fy[j++]=y2; n=j;
  *ffx=fx; *ffy=fy;
  return(n);
}
/*****************************************************************************/

#ifdef SUPPORT_LEGACY
/*****************************************************************************/
/** The matrix is filled with 0's (outside of ROI), 1's (on the ROI border),
 *  and 2's (inside the ROI border).
 *  If ROI extends outside image borders, those points are ignored.
 *  Matrix coordinates are up-to-bottom and left-to-right.
\return Function returns 0, if everything is OK.
 */
int roiOnOff(
  /** Definitions for one region-of-interest */
  ROI *roi,
  /** Image matrix x dimension */
  int dimx,
  /** Image matrix y dimension */
  int dimy,
  /** Image reconstruction zoom */
  float zoom,
  /** Allocated memory for output matrix[dimx][dimy] */
  char **m
) {
  int i, j, n, *x, *y, pos_x, pos_y, *fx, *fy;
  float f;


  /* Check the parameters */
  if(roi==NULL || dimx<2 || dimy<2 || zoom<1.0e-6 || m==NULL) {
    strcpy(roierrmsg, "invalid arguments for roiOnOff()");
    return(1);
  }

  /* Make scaled ROI points */
  strcpy(roierrmsg, "out of memory");
  if((x=malloc(roi->point_nr*sizeof(int)))==NULL) return(2);
  if((y=malloc(roi->point_nr*sizeof(int)))==NULL) {
    free((char*)x); return(2);}
  f=roi->zoom; if(f<=0.0) f=1.0;
  pos_x=(int)temp_roundf(((float)roi->pos_x)/f);
  pos_y=(int)temp_roundf(((float)roi->pos_y)/f);
  f=roi->zoom*(roi->recon_zoom/zoom); if(f<=0.0) f=1.0;
  for(i=0; i<roi->point_nr; i++) {
    x[i]=pos_x+(int)temp_roundf((float)roi->x[i]/f);
    y[i]=pos_y+(int)temp_roundf((float)roi->y[i]/f);
  }

  /* Fill the gaps between points */
  n=roiFillGaps(x, y, roi->point_nr, &fx, &fy);
  if(n<1) {
    strcpy(roierrmsg, "cannot fill the gaps in ROI border");
    free((char*)x); free((char*)y); return(3);
  }

  /* Set matrix */
  for(i=0; i<dimy; i++) for(j=0; j<dimx; j++) m[j][i]=(char)0;
  for(i=0; i<n; i++) if(fx[i]>=0 && fx[i]<dimx && fy[i]>=0 && fy[i]<dimy)
    m[fx[i]][fy[i]]=(char)1;

  /* Fill the inside of ROI */
  for(i=0; i<dimy; i++) if(m[0][i]==0) m[0][i]=2; else break;
  for(i=dimy-1; i>=0; i--) if(m[0][i]==0) m[0][i]=2; else break;
  for(i=0; i<dimy; i++) if(m[dimx-1][i]==0) m[dimx-1][i]=2; else break;
  for(i=dimy-1; i>=0; i--) if(m[dimx-1][i]==0) m[dimx-1][i]=2; else break;
  for(i=0; i<dimx; i++) if(m[i][0]==0) m[i][0]=2; else break;
  for(i=dimx-1; i>=0; i--) if(m[i][0]==0) m[i][0]=2; else break;
  for(i=0; i<dimx; i++) if(m[i][dimy-1]==0) m[i][dimy-1]=2; else break;
  for(i=dimx-1; i>=0; i--) if(m[i][dimy-1]==0) m[i][dimy-1]=2; else break;
  for(i=0; i<dimy; i++) for(j=0; j<dimx; j++) {
    if(m[j][i]!=2) continue;
    if(i>0 && m[j][i-1]==0) {m[j][i-1]=2; j=-1; i-=2; continue;}
    if(j>0 && m[j-1][i]==0) {m[j-1][i]=2; j-=2; continue;}
    if(j+1<dimx) {if(m[j+1][i]!=1) m[j+1][i]=2;}
    if(i+1<dimy) {if(m[j][i+1]!=1) m[j][i+1]=2;}
    m[j][i]=3;
  }
  for(i=0; i<dimy; i++) for(j=0; j<dimx; j++)
    if(m[j][i]==0) m[j][i]=2; else if(m[j][i]>1) m[j][i]=0;
    //if(m[j][i]<2) m[j][i]=1; else m[j][i]=0;

  /* Free memory */
  free((char*)x); free((char*)y);
  free((char*)fx); free((char*)fy);

  return(0);
}
#endif

/*** Fill the matrix with a Trace ROI ***/
/* A structure for polygon edges. Used in roi_onoff() */
typedef struct {
    /** Minimium and maximium Y coordinates */
    float minY,maxY;
    /** X coordinate of minY */
    float x;
    /** slope */
    float m;
} PolygonEdge;

static void fill_traceroi(char **m,int dimx,int dimy,float f,int pos_x,int pos_y,int *roix,int *roiy,int point_nr) {
    PolygonEdge *edges;
    int edge,p,y;
    int *px;

    edges=malloc(sizeof(PolygonEdge)*point_nr);
    edge=0;
    for(p=0;p<point_nr;p++) {
        float maxX;
        int n;
        if(p==point_nr-1) n=0; else n=p+1;
        if(roiy[p]>roiy[n]) {
            edges[edge].minY=roiy[n]*f+pos_y;
            edges[edge].x=roix[n]*f+pos_x;
            edges[edge].maxY=roiy[p]*f+pos_y;
            maxX=roix[p]*f+pos_x;
        } else if(roiy[p]<roiy[n]) {
            edges[edge].minY=roiy[p]*f+pos_y;
            edges[edge].x=roix[p]*f+pos_x;
            edges[edge].maxY=roiy[n]*f+pos_y;
            maxX=roix[n]*f+pos_x;
        } else continue;
        edges[edge].m=(maxX-edges[edge].x)/(edges[edge].maxY-edges[edge].minY);
        edge++;
    }
    px=malloc(sizeof(int)*point_nr);
    for(y=0;y<dimy;y++) {
        int pc,x,odd;
        pc=0;
        for(p=0;p<edge;p++) {
            if(temp_roundf(edges[p].minY)<=y && y<temp_roundf(edges[p].maxY)){
                px[pc]=temp_roundf(edges[p].x+edges[p].m*(y-temp_roundf(edges[p].minY)));
                if(px[pc]<0) px[pc]=0;
                pc++;
            }
        }
        odd=0;
        for(x=0;x<dimx;x++) {
            m[x][y]=odd;
            for(p=0;p<pc;p++) {
                if(x==px[p]) {
                    if(pc>1) odd=!odd;
                    m[x][y]=1;
                }
            }
        }
    }
    free(px);
    free(edges);
}

/** The matrix is filled with 0's (outside of ROI) and 1's (inside
 *  the ROI border).
 *  If ROI extends outside image borders, those points are ignored.
 *  Matrix coordinates are up-to-bottom and left-to-right.
\return Function returns 0, if everything is OK.
 */

int roi_onoff(
  /** Definitions for one region-of-interest */
  ROI *roi,
  /** Image matrix x dimension */
  int dimx,
  /** Image matrix y dimension */
  int dimy,
  /** Image reconstruction zoom */
  float zoom,
  /** Allocated memory for output matrix[dimx][dimy] */
  char **m
) {
  int x,y,r;
  float f,pos_x,pos_y;

  /* Check the parameters */
  if(roi==NULL || dimx<2 || dimy<2 || zoom<1.0e-6 || m==NULL) {
    strcpy(roierrmsg, "invalid arguments for roiOnOff()");
    return 1;
  }
  /* Prepare */
  f=zoom/roi->zoom;
  pos_x=roi->pos_x*f;
  pos_y=roi->pos_y*f;
  for(y=0;y<dimy;y++)
    memset(m[y],0,dimx);

  /* Fill the matrix */
  switch(roi->type) {
    case ROI_TRACE:		    /* Trace ROI */
        fill_traceroi(m,dimx,dimy,f,pos_x,pos_y,roi->x,roi->y,roi->point_nr);
      break;
    case ROI_RECTANGULAR:	/* Rectangle ROI */
      for(x=pos_x;x<pos_x+roi->w*f;x++)
        for(y=pos_y;y<pos_y+roi->h*f;y++)
          m[x][y]=1;
      break;
    case ROI_CIRCULAR:{	    /* Circle ROI */
      int x1,x2;
      int y1,y2;
      r=roi->w*f;
      for(y=0;y<r;y++) {
        y1=pos_y-y;
	    y2=pos_y+y;
        x1=temp_roundf(sqrt(r*r-y*y));
        x2=pos_x+x1;
        x1=pos_x-x1;
        if(x1<0) x1=0;
        if(x2>dimx) x2=dimx;
        while(x1<x2) {
          if(y1>=0 && y1<dimy) m[x1][y1]=1;
          if(y2>=0 && y2<dimy) m[x1][y2]=1;
          x1++;
        }
      }
    } break;
    case ROI_ELLIPSE: {		/* Ellipse ROI */
      int x1,x2,a;
      int y1,y2,b;
      a=roi->w*f;
      b=roi->h*f;
      for(y=0;y<b;y++) {
        y1=pos_y-y;
        y2=pos_y+y;
        x1=sqrt(1-(y*y)/(float)(b*b))*a;
        x2=pos_x+x1;
        x1=pos_x-x1;
        if(x1<0) x1=0;
        if(x2>dimx) x2=dimx;
        while(x1<x2) {
          if(y1>=0 && y1<dimy) m[x1][y1]=1;
          if(y2>=0 && y2<dimy) m[x1][y2]=1;
          x1++;
        }
      }
    } break;
    default:
      strcpy(roierrmsg,"invalid roi type");
      return 2;
  }
  return 0;
}
/*****************************************************************************/
/*
 *  int roi_save(char *fname, ROI_list *roilist)
 */
int roi_save(char *fname, ROI_list *rl) {
  RoiList *rois;
  FILE *fp;

  /* Check arguments */
  if(!fname[0] || rl==NULL || rl->nr<1) {
    strcpy(roierrmsg, "invalid arguments for roiSave()");
    return 1;
  }

  /* Create file */
  if((fp=fopen(fname, "w"))==NULL) {
    strcpy(roierrmsg, "cannot open file");
    return 2;
  }
  /* Loop and save the ROIs */
  rois=rl->rois;
  while(rois) {
    if(roi_append(fp,rois->roi)) return 3;
    rois=rois->next;
  }
  return 0;
}


#ifdef SUPPORT_LEGACY
/*****************************************************************************/
/*
 *  int roiSave(char *fname, ROI_list *roilist)
 */
int roiSave(char *fname, ROI_list *rl)
{
  int i, j;
  FILE *fp;

  /* Check arguments */
  if(!fname[0] || rl==NULL || rl->nr<1) {
    strcpy(roierrmsg, "invalid arguments for roiSave()");
    return(1);
  }

  /* Create file */
  if((fp=fopen(fname, "w"))==NULL) {
    strcpy(roierrmsg, "cannot open file");
    return(2);
  }

  for(i=0; i<rl->nr; i++) {
    /* Write first line */
    if(fprintf(fp, "*%s %f %f %d %d %d %d %d %d %d %d %d %s///%d %d\n",
      rl->roi[i].imgfile, rl->roi[i].zoom, rl->roi[i].recon_zoom,
      rl->roi[i].matnum, rl->roi[i].type, rl->roi[i].status,
      rl->roi[i].pos_x, rl->roi[i].pos_y, rl->roi[i].w, rl->roi[i].h,
      rl->roi[i].t, rl->roi[i].roi, rl->roi[i].roiname,
      0, rl->roi[i].point_nr ) < 10 ) {
        strcpy(roierrmsg, "cannot write data");
        fclose(fp); return(3);
    }
    /* Write second line, if trace ROI */
    if(rl->roi[i].type!=ROI_TRACE) continue;
    for(j=0; j<rl->roi[i].point_nr; j++)
      fprintf(fp, "%d %d ", rl->roi[i].x[j], rl->roi[i].y[j]);
    fprintf(fp, "\n");
  }

  /* Close file */
  fclose(fp);

  return(0);
}
#endif

/*****************************************************************************/
/*
 *  int roi_append(FILE *fp, ROI *roi)
 */
int roi_append(FILE *fp,ROI *roi) {
  /* Write first line */
  if(fprintf(fp, "*%s %f %f %d %d %d %d %d %d %d %d %d %s///%d %d\n",
    roi->imgfile, roi->zoom, roi->recon_zoom,
    roi->matnum, roi->type, roi->status,
    roi->pos_x, roi->pos_y, roi->w, roi->h,
    roi->t, roi->roi, roi->roiname,
    0, roi->point_nr ) < 10 ) {
      strcpy(roierrmsg, "cannot write data");
      fclose(fp); return 1;
   }
   /* Write second line, if trace ROI */
   if(roi->type==ROI_TRACE) {
     int j;
     for(j=0; j<roi->point_nr; j++)
       fprintf(fp, "%d %d ", roi->x[j], roi->y[j]);
     fprintf(fp, "\n");
   }
   return 0;
}

/*****************************************************************************/
/*
 *  int roi_append_n(FILE *fp, ROI_list *rl,int ind)
 */
int roi_append_n(FILE *fp,ROI_list *rl,int ind) {
  RoiList *rois=rl->rois;
  int pos=0;
  while(pos<ind) {
    rois=rois->next;
    pos++;
  }
  return roi_append(fp,rois->roi);
}


#ifdef SUPPORT_LEGACY
/*****************************************************************************/
/*
 *  int roiAppend(char *fname, ROI_list *roilist)
 */
int roiAppend(char *fname, ROI_list *rl, int ind)
{
  int i, j;
  FILE *fp;

  /* Check arguments */
  if(!fname[0] || rl==NULL || rl->nr<1) {
    strcpy(roierrmsg, "invalid arguments for roiAppend()");
    return(1);
  }

  /* Create file */
  if((fp=fopen(fname, "a"))==NULL) {
    strcpy(roierrmsg, "cannot open file");
    return(2);
  }

  /* Check ind, which roi to append to file */
  if(ind==0){
        for(i=0; i<rl->nr; i++) {
           /* Write first line */
           if(fprintf(fp, "*%s %f %f %d %d %d %d %d %d %d %d %d %s///%d %d\n",
            rl->roi[i].imgfile, rl->roi[i].zoom, rl->roi[i].recon_zoom,
            rl->roi[i].matnum, rl->roi[i].type, rl->roi[i].status,
            rl->roi[i].pos_x, rl->roi[i].pos_y, rl->roi[i].w, rl->roi[i].h,
            rl->roi[i].t, rl->roi[i].roi, rl->roi[i].roiname,
            0, rl->roi[i].point_nr ) < 10 ) {
              strcpy(roierrmsg, "cannot write data");
              fclose(fp); return(3);
          }
          /* Write second line, if trace ROI */
          if(rl->roi[i].type!=ROI_TRACE) continue;
          for(j=0; j<rl->roi[i].point_nr; j++)
            fprintf(fp, "%d %d ", rl->roi[i].x[j], rl->roi[i].y[j]);
          fprintf(fp, "\n");
        }

    }
  else if(ind>0){
         i=ind-1;
         /* Write first line */
         if(fprintf(fp, "*%s %f %f %d %d %d %d %d %d %d %d %d %s///%d %d\n",
           rl->roi[i].imgfile, rl->roi[i].zoom, rl->roi[i].recon_zoom,
           rl->roi[i].matnum, rl->roi[i].type, rl->roi[i].status,
           rl->roi[i].pos_x, rl->roi[i].pos_y, rl->roi[i].w, rl->roi[i].h,
           rl->roi[i].t, rl->roi[i].roi, rl->roi[i].roiname,
           0, rl->roi[i].point_nr ) < 10 ) {
             strcpy(roierrmsg, "cannot write data");
             fclose(fp); return(3);
         }
         /* Write second line, if trace ROI */
         //if(rl->roi[i].type!=ROI_TRACE) continue;
         for(j=0; j<rl->roi[i].point_nr; j++)
            fprintf(fp, "%d %d ", rl->roi[i].x[j], rl->roi[i].y[j]);
            fprintf(fp, "\n");
           
    }
  


    /* Close file */
    fclose(fp);
    return(0);
}
/*****************************************************************************/
#endif

/*****************************************************************************/
int jsqrt(int n)
{
  int x, x2, q, r, t;

  if(n<2) return n;
  t=x=n; while(t>>=2) x>>=1; x++;
  while(1) {
    q=n/x; r=n%x; 
    if(x<=q) {x2=x+2; if(q<x2 || (q==x2 && r==0)) break;}
    x=(x+q)>>1;
  }
  return x;
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the plane number of specified matrix */
int roi_mplane(int matnum)
{
  Matval matval;

  mat_numdoc(matnum, &matval);
  return matval.plane;
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the frame number of specified matrix */
int roi_mframe(int matnum)
{
  Matval matval;

  mat_numdoc(matnum, &matval);
  return matval.frame;
}
/*****************************************************************************/

/*****************************************************************************/

