/******************************************************************************

  Copyright (c) 2001-2004 by Turku PET Centre

  simulate.c
  
  Procedures for simulating PET time-activity curves.

  Version:
  2001-03-31 Vesa Oikonen
  2001-06-03 VO
    Included c2lSIM().
  2001-07-02 VO
    Changes in equation presentation, corresponding to cm_equations.doc
    Included functions rtcmSIM(), srtmSIM(), and trtmSIM().
  2001-07-08 VO
    The function trtmSIM() is simplified, as in rtcm_equations.doc
  2002-03-13 VO
    Included function c2vlSIM().
  2003-03-04 VO
    Included function simHuangmet().
  2003-07-07 VO
    Functions c3sSIM(), c3pSIM(), c3vsSIM(), c3vpSIM(), c2lSIM(),
    c2vlSIM(), rtcmSIM(), srtmSIM() and trtmSIM():
    the first sample time need not to be zero for precise result, and
    sequential samples can have equal sample times.
    Function simHuangmet(): small simplification in the code.
  2004-09-17 VO
    Doxygen style comments.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "include/integr.h"
#include "include/dft.h"
#include "include/simulate.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 1-3 tissue compartment model (in series) and
    plasma TAC, at plasma TAC times.

    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta, ctb and/or ctc can be given; if compartmental TACs are not
    required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int c3sSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial activities */
  double *ca,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double k4,
  /** Rate constant of the model */
  double k5,
  /** Rate constant of the model */
  double k6,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL */
  double *ctc
) {
  int i, model, parNr;
  double b, c, d, w, z, dt2;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Check actual parameter number */
  if(k1<=0.0) return 3;
  if(k3<=0.0) {k3=0.0; model=1; if(k2<=0.0) {k2=0.0; parNr=1;} else parNr=2;}
  else if(k5<=0.0) {k5=0.0; model=2; if(k4<=0.0) {k4=0.0; parNr=3;} else parNr=4;}
  else {model=3; if(k6<=0.0) {k6=0.0; parNr=5;} else parNr=6;}
  if(SIMULATE_TEST) printf("simulate(): model=%d parNr=%d\n", model, parNr);

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      d=ct3i_last+dt2*ct3_last;
      w=k4 + k5 - (k5*k6*dt2)/(1.0+k6*dt2);
      z=1.0+w*dt2;
      /* 1st tissue compartment and its integral */
      ct1 = (
          + k1*z*cai + (k3*k4*dt2 - (k2+k3)*z)*b
          + k4*c + k4*k6*dt2*d/(1.0+k6*dt2)
        ) / ( z*(1.0 + dt2*(k2+k3)) - k3*k4*dt2*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - w*c + k6*d/(1.0+k6*dt2)) / z;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct2i - k6*d) / (1.0 + k6*dt2);
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* copy values to argument arrays; set very small values to zero */
    ct[i]=ct1+ct2+ct3; if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    if(cta!=NULL) {cta[i]=ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    if(ctc!=NULL) {ctc[i]=ct3; if(fabs(ctc[i])<1.0e-12) ctc[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 1-3 tissue compartment model (2nd and 3rd
    compartments in parallel) and plasma TAC, at plasma TAC times.

    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta, ctb and/or ctc can be given; if compartmental TACs are not
    required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int c3pSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial activities */
  double *ca,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double k4,
  /** Rate constant of the model */
  double k5,
  /** Rate constant of the model */
  double k6,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL */
  double *ctc
) {
  int i;
  double dt2, r, s, u, v, w;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Check parameters */
  if(k1<=0.0) return 3;

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      r=1.0+k4*dt2;
      s=1.0+k6*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      w=ct3i_last+dt2*ct3_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - (k2 + (k3/r) + (k5/s))*u + (k4/r)*v + (k6/s)*w )
            / ( 1.0 + dt2*(k2 + (k3/r) + (k5/s)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct1i - k6*w) / s;
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* copy values to argument arrays; set very small values to zero */
    ct[i]=ct1+ct2+ct3; if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    if(cta!=NULL) {cta[i]=ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    if(ctc!=NULL) {ctc[i]=ct3; if(fabs(ctc[i])<1.0e-12) ctc[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 1-3 tissue compartment model (in series) and
    plasma TAC, at plasma TAC times, considering also arterial and venous
    vasculature.

    Memory for cpet must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta, ctb, ctc, ctab and/or ctvb can be given; if compartmental
    TACs are not required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.",

\return Function returns 0 when succesful, else a value >= 1.
*/
int c3vsSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial plasma activities */
  double *ca,
  /** Array of arterial blood activities */
  double *cb,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double k4,
  /** Rate constant of the model */
  double k5,
  /** Rate constant of the model */
  double k6,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  double f,
  /** Vascular volume fraction */
  double vb,
  /** Arterial fraction of vascular volume */
  double fa,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *cpet,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL */
  double *ctc,
  /** Pointer for arterial TAC in tissue, or NULL */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL */
  double *ctvb
) {
  int i;
  double b, c, d, w, z, dt2, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;

  /* Check parameters */
  if(k1<=0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      d=ct3i_last+dt2*ct3_last;
      w=k4 + k5 - (k5*k6*dt2)/(1.0+k6*dt2);
      z=1.0+w*dt2;
      /* 1st tissue compartment and its integral */
      ct1 = (
          + k1*z*cai + (k3*k4*dt2 - (k2+k3)*z)*b
          + k4*c + k4*k6*dt2*d/(1.0+k6*dt2)
        ) / ( z*(1.0 + dt2*(k2+k3)) - k3*k4*dt2*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - w*c + k6*d/(1.0+k6*dt2)) / z;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct2i - k6*d) / (1.0 + k6*dt2);
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1; cvb = cb[i] - dct/f;} else cvb=cb[i];
    /* copy values to argument arrays; set very small values to zero */
    cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2+ct3);
    if(fabs(cpet[i])<1.0e-12) cpet[i]=0.0;
    if(cta!=NULL) {cta[i]=(1.0-vb)*ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=(1.0-vb)*ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    if(ctc!=NULL) {ctc[i]=(1.0-vb)*ct3; if(fabs(ctc[i])<1.0e-12) ctc[i]=0.0;}
    if(ctab!=NULL) {ctab[i]=va*cb[i]; if(fabs(ctab[i])<1.0e-12) ctab[i]=0.0;}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb; if(fabs(ctvb[i])<1.0e-12) ctvb[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 1-3 tissue compartment model (2nd and 3rd
    compartments in parallel) and plasma TAC, at plasma TAC times,
    considering also arterial and venous vasculature.

    Memory for cpet must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta, ctb, ctc, ctab and/or ctvb can be given; if compartmental
    TACs are not required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.",

\return Function returns 0 when succesful, else a value >= 1.
*/
int c3vpSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial plasma activities */
  double *ca,
  /** Array of arterial blood activities */
  double *cb,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double k4,
  /** Rate constant of the model */
  double k5,
  /** Rate constant of the model */
  double k6,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  double f,
  /** Vascular volume fraction */
  double vb,
  /** Arterial fraction of vascular volume */
  double fa,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *cpet,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb,
  /** Pointer for 3rd compartment TAC to be simulated, or NULL */
  double *ctc,
  /** Pointer for arterial TAC in tissue, or NULL */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL */
  double *ctvb
) {
  int i;
  double dt2, r, s, u, v, w, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last, ct3, ct3_last;
  double ct1i, ct1i_last, ct2i, ct2i_last, ct3i, ct3i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;

  /* Check parameters */
  if(k1<=0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct3_last=ct1i_last=ct2i_last=ct3i_last=0.0;
  ct1=ct2=ct3=ct1i=ct2i=ct3i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      r=1.0+k4*dt2;
      s=1.0+k6*dt2;
      u=ct1i_last+dt2*ct1_last;
      v=ct2i_last+dt2*ct2_last;
      w=ct3i_last+dt2*ct3_last;
      /* 1st tissue compartment and its integral */
      ct1 = ( k1*cai - (k2 + (k3/r) + (k5/s))*u + (k4/r)*v + (k6/s)*w )
            / ( 1.0 + dt2*(k2 + (k3/r) + (k5/s)) );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - k4*v) / r;
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
      /* 3rd tissue compartment and its integral */
      ct3 = (k5*ct1i - k6*w) / s;
      ct3i = ct3i_last + dt2*(ct3_last+ct3);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1; cvb = cb[i] - dct/f;} else cvb=cb[i];
    /* copy values to argument arrays; set very small values to zero */
    cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2+ct3);
    if(fabs(cpet[i])<1.0e-12) cpet[i]=0.0;
    if(cta!=NULL) {cta[i]=(1.0-vb)*ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=(1.0-vb)*ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    if(ctc!=NULL) {ctc[i]=(1.0-vb)*ct3; if(fabs(ctc[i])<1.0e-12) ctc[i]=0.0;}
    if(ctab!=NULL) {ctab[i]=va*cb[i]; if(fabs(ctab[i])<1.0e-12) ctab[i]=0.0;}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb; if(fabs(ctvb[i])<1.0e-12) ctvb[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
    ct3_last=ct3; ct3i_last=ct3i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Interpolates TACs to automatically determined sample times with
    smaller intervals in the beginning.
    Only data in y arrays are interpolated; data in y2 and y3 are not used.
\return Function returns 0 when succesful, else a value >= 1.
*/
int autointerpolateDFT(
  /** Data to be interpolated is read from this array */
  DFT *dft,
  /** Interpolated data is written in this array */
  DFT *dft2,
  /** The length of interpolated/extrapolated data */
  double endtime
) {
  int i, newnr, maxNr=10000;
  double t, dt;


  /* Check the arguments */
  if(dft->frameNr<1 || dft->voiNr<1) return 1;
  if(endtime<1.0 || endtime>1.0e12) return 2;

  /* Calculate the number of interpolated data points */
  t=0.0; dt=0.02; newnr=1;
  if(SIMULATE_TEST) printf("%05d: %12.5f  %10.5f\n", newnr, t, dt);
  while(t+dt<endtime && newnr<maxNr-1) {
    t+=dt; dt*=1.05; newnr++;
    if(SIMULATE_TEST) printf("%05d: %12.5f  %10.5f\n", newnr, t, dt);
  }
  dt=endtime-t; t=endtime; newnr++;
  if(SIMULATE_TEST) printf("%05d: %12.5f  %10.5f\n", newnr, t, dt);

  /* Allocate memory for interpolated data */
  emptyDFT(dft2); if(setmemDFT(dft2, newnr, dft->voiNr)) return 3;
  /* Copy header info */
  (void)copymainhdrDFT(dft, dft2); dft2->voiNr=dft->voiNr;
  for(i=0; i<dft->voiNr; i++) if(copyvoihdrDFT(dft, i, dft2, i)) return 4;
  /* Set times */
  dft2->timetype=0;
  t=0.0; dt=0.02; i=0; dft2->x[i++]=t;
  while(t+dt<endtime && newnr<maxNr-1) {t+=dt; dt*=1.05; dft2->x[i++]=t;}
  dt=endtime-t; t=endtime; dft2->x[i++]=t; dft2->frameNr=i;

  /* Interpolate */
  for(i=0; i<dft->voiNr; i++)
    if(interpolate(dft->x, dft->voi[i].y, dft->frameNr,
         dft2->x, dft2->voi[i].y, NULL, NULL, dft2->frameNr)) {
      emptyDFT(dft2); return 5;
    }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 2 tissue compartment model (in series) and
    plasma TAC, at plasma TAC times.
    In contrary to the common model, kLoss represents a direct loss rate from
    the 2nd tissue compartment to venous plasma.

    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta and ctb can be given; if compartmental TACs are not
    required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int c2lSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial activities */
  double *ca,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double kLoss,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb
) {
  int i;
  double b, c, dt2;
  double cai, ca_last, t_last;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Check actual parameter number */
  if(k1<=0.0) return 3;

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct1i_last=ct2i_last=ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = (k1*cai - (k2+k3)*b) / (1.0 + (k2+k3)*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - kLoss*c) / (1.0 + kLoss*dt2);
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* copy values to argument arrays; set very small values to zero */
    ct[i]=ct1+ct2; if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    if(cta!=NULL) {cta[i]=ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using 2 tissue compartment model and plasma TAC,
    at plasma TAC times, considering also arterial and venous vasculature.
    The efflux from 2nd tissue compartment (at rate kL) goes directly to blood.

    Memory for cpet must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cta, ctb, ctab and/or ctvb can be given; if compartmental
    TACs are not required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

    If blood flow is set to 0, function assumes that f>>k1, and Cvb=Cab.",

\return Function returns 0 when succesful, else a value >= 1.
*/
int c2vlSIM(
  /** Array of time values */
  double *t,
  /** Array of arterial plasma activities */
  double *ca,
  /** Array of arterial blood activities */
  double *cb,
  /** Number of values in TACs */
  int nr,
  /** Rate constant of the model */
  double k1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double kL,
  /** Blood flow; if 0, function assumes that f>>k1, and Cvb=Cab. */
  double f,
  /** Vascular volume fraction */
  double vb,
  /** Arterial fraction of vascular volume */
  double fa,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *cpet,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb,
  /** Pointer for arterial TAC in tissue, or NULL */
  double *ctab,
  /** Pointer for venous TAC in tissue, or NULL */
  double *ctvb
) {
  int i;
  double dt2, b, c, va, vv;
  double cai, ca_last, t_last, dct, cvb;
  double ct1, ct1_last, ct2, ct2_last;
  double ct1i, ct1i_last, ct2i, ct2i_last;


  /* Check for data */
  if(nr<2) return 1;
  if(cpet==NULL) return 2;

  /* Check parameters */
  if(k1<=0.0) return 3;
  if(vb<0.0 || vb>=1.0) return 4;
  if(fa<=0.0 || fa>1.0) return 5;
  va=fa*vb; vv=(1.0-fa)*vb;

  /* Calculate curves */
  t_last=0.0; cai=ca_last=0.0;
  ct1_last=ct2_last=ct1i_last=ct2i_last=ct1=ct2=ct1i=ct2i=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* arterial integral */
      cai+=(ca[i]+ca_last)*dt2;
      /* Calculate partial results */
      b=ct1i_last+dt2*ct1_last;
      c=ct2i_last+dt2*ct2_last;
      /* 1st tissue compartment and its integral */
      ct1 = (k1*cai - (k2+k3)*b) / (1.0 + (k2+k3)*dt2 );
      ct1i = ct1i_last + dt2*(ct1_last+ct1);
      /* 2nd tissue compartment and its integral */
      ct2 = (k3*ct1i - kL*c) / (1.0 + kL*dt2);
      ct2i = ct2i_last + dt2*(ct2_last+ct2);
    }
    /* Venous curve */
    if(f>0.) {dct = k1*ca[i] - k2*ct1 - kL*ct2; cvb = cb[i] - dct/f;}
    else cvb=cb[i];
    /* copy values to argument arrays; set very small values to zero */
    cpet[i]= va*cb[i] + vv*cvb + (1.0-vb)*(ct1+ct2);
    if(fabs(cpet[i])<1.0e-12) cpet[i]=0.0;
    if(cta!=NULL) {cta[i]=(1.0-vb)*ct1; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=(1.0-vb)*ct2; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    if(ctab!=NULL) {ctab[i]=va*cb[i]; if(fabs(ctab[i])<1.0e-12) ctab[i]=0.0;}
    if(ctvb!=NULL) {ctvb[i]=vv*cvb; if(fabs(ctvb[i])<1.0e-12) ctvb[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; ca_last=ca[i];
    ct1_last=ct1; ct1i_last=ct1i;
    ct2_last=ct2; ct2i_last=ct2i;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using reference tissue compartment model (original) and
    reference region TAC, at reference region TAC times.

    Memory for ct must be allocated in the calling program.
    To retrieve the separate tissue compartment TACs, pointer to allocated
    memory for cf and/or cb can be given; if compartmental TACs are not
    required, NULL pointer can be given instead.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int rtcmSIM(
  /** Array of time values */
  double *t,
  /** Reference region activities */
  double *cr,
  /** Number of values in TACs */
  int nr,
  /** Ratio K1/K1' */
  double R1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Rate constant of the model */
  double k4,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct,
  /** Pointer for 1st compartment TAC to be simulated, or NULL */
  double *cta,
  /** Pointer for 2nd compartment TAC to be simulated, or NULL */
  double *ctb
) {
  int i;
  double f, b, w, dt2;
  double cri, cr_last, t_last;
  double cf, cf_last, cb, cb_last;
  double cfi, cfi_last, cbi, cbi_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Calculate curves */
  t_last=0.0; cri=cr_last=0.0;
  cf_last=cb_last=cfi_last=cbi_last=cf=cb=cfi=cbi=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* reference integral */
      cri+=(cr[i]+cr_last)*dt2;
      /* partial results */
      f=cfi_last+dt2*cf_last;
      b=cbi_last+dt2*cb_last;
      w=k2 + k3 + k2*k4*dt2;
      /* 1st tissue compartment and its integral */
      cf = ( (1.0 + k4*dt2)*(R1*cr[i] + k2*cri) + k4*b - w*f ) /
           ( 1.0 + dt2*(w+k4) );
      cfi = cfi_last + dt2*(cf_last+cf);
      /* 2nd tissue compartment and its integral */
      cb = (k3*cfi - k4*b) / (1.0 + k4*dt2);
      cbi = cbi_last + dt2*(cb_last+cb);
    }
    /* copy values to argument arrays; set very small values to zero */
    ct[i]=cf+cb; if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    if(cta!=NULL) {cta[i]=cf; if(fabs(cta[i])<1.0e-12) cta[i]=0.0;}
    if(ctb!=NULL) {ctb[i]=cb; if(fabs(ctb[i])<1.0e-12) ctb[i]=0.0;}
    /* prepare to the next loop */
    t_last=t[i]; cr_last=cr[i];
    cf_last=cf; cfi_last=cfi;
    cb_last=cb; cbi_last=cbi;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using reference tissue compartment model (simplified)
    and reference region TAC, at reference region TAC times.

    Memory for ct must be allocated in the calling program.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int srtmSIM(
  /** Array of time values */
  double *t,
  /** Reference region activities */
  double *cr,
  /** Number of values in TACs */
  int nr,
  /** Ratio K1/K1' */
  double R1,
  /** Rate constant of the model */
  double k2,
  /** Binding potential */
  double BP,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct
) {
  int i;
  double dt2;
  double cri, cr_last, t_last;
  double ct_last, cti, cti_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Calculate curves */
  t_last=0.0; cri=cr_last=0.0;
  cti=ct_last=cti_last=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* reference integral */
      cri+=(cr[i]+cr_last)*dt2;
      /* Tissue compartment and its integral */
      ct[i] = ( R1*cr[i] + k2*cri - (k2/(1.0+BP))*(cti_last+dt2*ct_last) ) /
              ( 1.0 + dt2*(k2/(1.0+BP)) );
      cti = cti_last + dt2*(ct_last+ct[i]);
    }
    /* set very small values to zero */
    if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    /* prepare to the next loop */
    t_last=t[i]; cr_last=cr[i];
    ct_last=ct[i]; cti_last=cti;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulates tissue TAC using reference tissue compartment model (transport
    limited in ref region) and reference region TAC, at reference region TAC
    times.

    Memory for ct must be allocated in the calling program.

    The units of rate constants must be related to the time unit; 1/min and min,
    or 1/sec and sec.

\return Function returns 0 when succesful, else a value >= 1.
*/
int trtmSIM(
  /** Array of time values */
  double *t,
  /** Reference region activities */
  double *cr,
  /** Number of values in TACs */
  int nr,
  /** Ratio K1/K1' */
  double R1,
  /** Rate constant of the model */
  double k2,
  /** Rate constant of the model */
  double k3,
  /** Pointer for TAC array to be simulated; must be allocated */
  double *ct
) {
  int i;
  double dt2;
  double cri, cr_last, t_last;
  double ct_last, cti, cti_last;


  /* Check for data */
  if(nr<2) return 1;
  if(ct==NULL) return 2;

  /* Calculate curves */
  t_last=0.0; cri=cr_last=0.0;
  cti=ct_last=cti_last=0.0;
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last);
    /* calculate values */
    if(dt2<0.0) {
      return 5;
    } else if(dt2>0.0) {
      /* reference integral */
      cri+=(cr[i]+cr_last)*dt2;
      /* Tissue compartment and its integral */
      ct[i] = ( R1*cr[i] + R1*k3*cri - (k2+k3)*(cti_last+dt2*ct_last) ) /
              ( 1.0 + dt2*(k2+k3) );
      cti = cti_last + dt2*(ct_last+ct[i]);
    }
    /* set very small values to zero */
    if(fabs(ct[i])<1.0e-12) ct[i]=0.0;
    /* prepare to the next loop */
    t_last=t[i]; cr_last=cr[i];
    ct_last=ct[i]; cti_last=cti;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Simulation of TACs of parent tracer, and 1-2 of its metabolites in plasma
    using Huang's compartmental model.

    The units of model parameters must be related to the sample time unit;
    1/min and min, or 1/sec and sec.

    Pointers to memory for output TACs must be specified, or NULL if TAC is not
    needed.

\return Returns 0, if ok.
*/
int simHuangmet(
  /** Input: Sample times (preferably with short intervals)  */
  double *t,
  /** Input: Measured total plasma TAC */
  double *ctot,
  /** Input: Nr of samples */
  int nr,
  /** Input: Model parameters */
  double k01,
  /** Input: Model parameters */
  double k12,
  /** Input: Model parameters */
  double k21,
  /** Input: Model parameters */
  double k03,
  /** Input: Model parameters */
  double k34,
  /** Input: Model parameters */
  double k43,
  /** Output: unchanged (parent) tracer TAC */
  double *c0,
  /** Output: TAC of the 1st metabolite */
  double *c1,
  /** Output: TAC of the 2nd metabolite */
  double *c3
) {
  int i;
  double dt2, t_last, ictot, ctot_last;
  double c0_, c1_, c2_, c3_, c4_;
  double ic0_, ic1_, ic2_, ic3_, ic4_;
  double c0_last, c1_last, c2_last, c3_last, c4_last;
  double ic0_last, ic1_last, ic2_last, ic3_last, ic4_last;
  double a, b, am1, am2, am3, am4;


  /* Check input */
  if(t==NULL || ctot==NULL || nr<2) return(1);
  if(k01<0 || k12<0 || k21<0 || k03<0 || k34<0 || k43<0) return(2);
  if(t[0]<0.0) return(3);

  /* Compute the TACs */
  t_last=0.0; ictot=ctot_last=0.0;
  c0_=c0_last=ic0_=ic0_last=0.0;
  c1_=c1_last=ic1_=ic1_last=0.0;
  c2_=c2_last=ic2_=ic2_last=0.0;
  c3_=c3_last=ic3_=ic3_last=0.0;
  c4_=c4_last=ic4_=ic4_last=0.0;
  if(SIMULATE_TEST)
    printf("%6.6s %4.4s %10.10s %10.10s %10.10s %10.10s %10.10s %10.10s\n",
      "t", "dt/2", "ictot", "C0", "C1", "C2", "C3", "C4");
  for(i=0; i<nr; i++) {
    /* delta time / 2 */
    dt2=0.5*(t[i]-t_last); if(dt2<0.0) return(5);
    if(dt2>0.0) {
      /* Compute temp constants */
      a=k01+k12-(k12*k21*dt2/(1.0+dt2*k21));
      b=k03+k34-(k34*k43*dt2/(1.0+dt2*k43));
      am1=ic1_last+dt2*c1_last;
      am2=ic2_last+dt2*c2_last;
      am3=ic3_last+dt2*c3_last;
      am4=ic4_last+dt2*c4_last;
    
      /* Compute the "input" i.e. ctot integral */
      ictot+=(ctot[i]+ctot_last)*dt2;
      /* Compute C1(t) and its integral */
      c1_= ( k01*(1.0-k03*dt2/(1.0+dt2*b))*ictot
            -(a-k01*k03*dt2/(1.0+dt2*b))*am1
            +(k21/(1.0+dt2*k21))*am2
            -(k01/(1.0+dt2*b))*am3
            -(k01*k43*dt2/((1.0+dt2*b)*(1.0+dt2*k43)))*am4
           ) / ( 1.0+dt2*(a-k01*k03*dt2/(1.0+dt2*b)) );
      ic1_= ic1_last + dt2*(c1_+c1_last);
      /* Compute C2(t) and its integral */
      c2_= (k12*ic1_-k21*am2)/(1.0+dt2*k21);
      ic2_= ic2_last + dt2*(c2_+c2_last);

      /* Compute C3(t) and its integral */
      c3_= ( k03*(1.0-k01*dt2/(1.0+dt2*a))*ictot
            -(b-k01*k03*dt2/(1.0+dt2*a))*am3
            +(k43/(1.0+dt2*k43))*am4
            -(k03/(1.0+dt2*a))*am1
            -(k03*k21*dt2/((1.0+dt2*a)*(1.0+dt2*k21)))*am2
           ) / ( 1.0+dt2*(b-k01*k03*dt2/(1.0+dt2*a)) );
      ic3_= ic3_last + dt2*(c3_+c3_last);
      /* Compute C4(t) and its integral */
      c4_= (k34*ic3_-k43*am4)/(1.0+dt2*k43);
      ic4_= ic4_last + dt2*(c4_+c4_last);

      /* Compute the C0(t) */
      c0_=ctot[i]-c1_-c3_; /*if(c0_<0.0) return(-1);*/
    }
    if(SIMULATE_TEST)
      printf("%6.2f %4.2f %10.2e %10.2e %10.2e %10.2e %10.2e %10.2e\n",
        t[i], dt2, ictot, c0_, c1_, c2_, c3_, c4_ );
    /* Set output data */
    if(c0) c0[i]=c0_; if(c1) c1[i]=c1_; if(c3) c3[i]=c3_;
    /* Prepare for the next sample */
    c0_last=c0_; c1_last=c1_; c2_last=c2_; c3_last=c3_; c4_last=c4_;
    ic0_last=ic0_; ic1_last=ic1_; ic2_last=ic2_; ic3_last=ic3_; ic4_last=ic4_;
    ctot_last=ctot[i]; t_last=t[i];
  } /* next sample */

  /* Check that all of the last activities are >=0 */
  /* Due to the noise in total plasma TAC, this cannot be applied */
  /*if(c0_<0.0 || c1_<0.0 || c2_<0.0 || c3_<0.0 || c4_<0.0) return(-1);*/

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

