/*****************************************************************************
  Copyright (c) 2002,2003 by Turku PET Centre

  tgo

  Topografical minimization algorithm, that searches the global minimum of 
  a function using clusterization. Calls a local minimization 
  algorithm. Based on an algorithm by Aimo Torn and Sami Viitanen.
  See the article 
  Topografical Global optimization in: C.A. Floudas and 
  P.M. Pardalos (eds.) Recent advances in Global Optimization, 
  Princeton University Press, 1992
  or webpage www.abo.fi/~atorn/ProbAlg/Page53.html

  Version:
  1.0  2002-10-22 Kaisa Sederholm
  1.1  2002-11-12 KS
       Paramater samNr included so that the user can specify number of points
       to sample if he/she wants, else it is fixed to be 100
       2002-11-14 VO
       samplNr is set before used.
       2003-03-25 VO
       Corrected a bug in comparison with the closest neighbours.
       Changes in test prints.
       Tiny changes to make this routine a bit faster.
       Should work even if no actual topografic minimum is found.
       More point data moved into struct.
       2003-03-29 VO
       Added several powell routines to make sure that the bottom of global
       minimum is reached.
       2004-07-12 VO
       Added Doxygen-style comments.
       2004-09-20 VO
       Doxygen style comments are corrected.


*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "include/powell.h"
#include "include/tgo.h"
/*****************************************************************************/
#ifndef RAND_MAX
#define RAND_MAX 32767
#endif
/*****************************************************************************/
#ifndef TGO_SAMPLNR  /* Max 4000, MUST BE EVEN NUMBER */
#define TGO_SAMPLNR 100
#endif
/*****************************************************************************/
/** Topografical minimization algorithm, that searches the global minimum of 
    a function using clusterization. Calls a local minimization 
    algorithm. Based on an algorithm by Aimo Torn and Sami Viitanen.
\return Returns 0, if ok.
*/
int tgo(
  /** Lower limits for the parameters */
  double *lowlim,
  /** Upper limits for the parameters */
  double *uplim,
  /** The object function */
  double (*objf)(double*),
  /** Dimension = nr of parameters */
  int dim,
  /** Nr of neighbours to investigate */
  int neighNr,
  /** Function value at global minimum */
  double *fmin,
  /** Global minimum = parameter estimates */
  double *gmin,
  /** Nr of points to sample in one iteration */
  int samNr
) {
  int i, j, k, l, IDmin, itNr, samplNr, tgoNr, topoNr, ret;
  double *delta, temp, min, *tempp, deltaf, tol;
  TGO_POINT *sampled_points;


  if(TGO_TEST>0)
    printf("in tgo(lowlim, uplim, objf(), %d, %d, fmin, gmin, %d)\n",
      dim, neighNr, samNr);

  /* Check input */
  if(samNr<=0) samplNr=TGO_SAMPLNR; else samplNr=samNr;
  if(lowlim==NULL || uplim==NULL || objf==NULL || dim<=0 || neighNr<=0)
    return(1);

  /* Allocate memory */
  sampled_points=(TGO_POINT*)calloc(samplNr, sizeof(TGO_POINT));
  delta=(double*)malloc(dim*sizeof(double));
  tempp=(double*)malloc(samplNr*sizeof(double));
  if(sampled_points==NULL || delta==NULL || tempp==NULL) return(2);
  for(i=0; i<samplNr; i++) sampled_points[i].topomin=0;

  /* Set seed for random number generator */
  srand(15345); //srand(time(NULL));

  /* Make sure "neighNr" isn't specified too big  */
  if(neighNr>samplNr-1) neighNr=samplNr-1;

	
  /*
   *  Iteration nr of TGO
   */

  tgoNr=dim;

  for(l=0; l<tgoNr; l++) {

    if(TGO_TEST>1) printf("Loop # %d: \n",l+1);
    /*
     *  Sample N points in the feasible region and 
     *  compute the object function values. 
     */
    randomParameters(sampled_points, dim, samplNr, lowlim, uplim);
    for(i=0; i<samplNr; i++) {
      if(sampled_points[i].topomin==0)
        sampled_points[i].fvalue=objf(sampled_points[i].par);
    }
    if(TGO_TEST>5) {
      printf("Sampled points:\n");
      for(j=0; j<samplNr; j++) {
        printf("%d", j+1);
        for(i=0; i<dim; i++) printf(" %e ", sampled_points[j].par[i]);
        printf("=>%e\n", sampled_points[j].fvalue);
      }
    }
    if(TGO_TEST>3) {
      for(k=0; k<dim; k++) {
        temp=0.0; deltaf=-1000.0; tol=9.9e20;
        for(j=0; j<samplNr; j++) {
          temp+=sampled_points[j].par[k];
          if(sampled_points[j].par[k]<tol) tol=sampled_points[j].par[k];
          if(sampled_points[j].par[k]>deltaf) deltaf=sampled_points[j].par[k];
        }
        temp/=(double)samplNr;
        printf("Parameter %d: avg=%e min=%e max=%e\n", k+1, temp, tol, deltaf);
      }
      temp=0.0; deltaf=-1000.0; tol=9.9e20;
      for(j=0, k=0; j<samplNr; j++) {
        if(sampled_points[j].fvalue==BAD_FIT) {k++; continue;}
        temp+=sampled_points[j].fvalue;
        if(sampled_points[j].fvalue<tol) tol=sampled_points[j].fvalue;
        if(sampled_points[j].fvalue>deltaf) deltaf=sampled_points[j].fvalue;
      }
      temp/=(double)(samplNr-k);
      printf("Function value: avg=%e min=%e max=%e bad fits %d/%d\n",
        temp, tol, deltaf, k, samplNr);
    }

    /*
     *  For each point i find out if it is a "topografic minimum" 
     *  = better than k neighbour points
     */
    /* Save the distances to point i in a vector */
    /* Find the closest neighbour from {x1,..,xn}/{xi} k times, */
    /* extracting it after comparing */
    for(i=0, topoNr=0; i<samplNr; i++) {
      sampled_points[i].topomin=0; /*printf(" %d ", i);*/

      /* Compute the distances */
      for(j=0; j<samplNr; j++) {
        if(i==j)
          tempp[j]=1e+99;
        else {
          for(k=0, tempp[j]=0.0; k<dim; k++) {
            temp=sampled_points[i].par[k]-sampled_points[j].par[k];
            tempp[j] += temp*temp;
          }
          /* Distance is computed as square root */
          /* but for this purpose it is not needed, and sqrt() is slow */
          /*tempp=sqrt(tempp);*/
        }
      }

      /* Find the closest neighbours */
      for(j=0; j<neighNr; j++) {
        min=tempp[0]; IDmin=0;
        for(k=1; k<samplNr; k++) {
          if(tempp[k]<min) {min=tempp[k]; IDmin=k;}
        }
        /*printf("n %d ",IDmin);*/
        tempp[IDmin]=1e+99;
        /* If point i is worse than one of the closest neighbours, then go to */
        /* the next point i+1, else add it to the topografic minima (TM) */	  
        if(sampled_points[IDmin].fvalue<sampled_points[i].fvalue) break;
      }
      if(j==neighNr) {sampled_points[i].topomin=1; topoNr++;}
      /* printf("tm %d \n", sampled_points[i].topomin);*/
    } /* next sample */
    if(TGO_TEST>1) printf("  %d topografical minima\n", topoNr);
    /* Check that if no topografical minimum was found, then set the smallest */
    /* minumum as 'topografical' minimum */
    if(topoNr==0) {
      min=sampled_points[0].fvalue; IDmin=0;
      for(k=1; k<samplNr; k++)
        if(sampled_points[k].fvalue<min) {min=sampled_points[k].fvalue; IDmin=k;}
      sampled_points[IDmin].topomin=1;
      if(TGO_TEST>2)
        printf("  ; therefore minimum was set at %e\n", sampled_points[IDmin].fvalue);
    }
    if(TGO_TEST>2) {
      for(i=0, min=1e+99, IDmin=0; i<samplNr; i++) if(sampled_points[i].topomin==1) {
        if(sampled_points[i].fvalue<min) {min=sampled_points[i].fvalue; IDmin=i;}
      }
      printf("  best topografical min:");
      for(k=0; k<dim; k++) printf(" %e", sampled_points[IDmin].par[k]);
      printf(" => %e\n", sampled_points[IDmin].fvalue);
    }
  } /* end of tgo iterations */

  free(tempp);

  
  /*
   *  Use the points in TM as starting points for local optimization
   *  with Powell
   */
  if(TGO_TEST>2) printf("Topografig minima:\n");
  POWELL_TEST=0;
  for(j=0; j<samplNr; j++) if(sampled_points[j].topomin==1) {
    itNr=60; tol=0.00001;
    if(TGO_TEST>2) printf("%d.    ", j);
    for(k=0; k<dim; k++) {
      if(TGO_TEST>2) printf("%e ", sampled_points[j].par[k]);
      delta[k]=0.02*(uplim[k]-lowlim[k]);
    }
    if(TGO_TEST>2) printf("=> %e ", sampled_points[j].fvalue);
    ret=powell(sampled_points[j].par, delta, dim, tol, &itNr,
               &sampled_points[j].fvalue, objf);
    if(ret > 1) {
      free(sampled_points); free(delta);
      if(TGO_TEST) printf("powell error %d\n", ret);
      return(5);
    }
    if(TGO_TEST>2) printf("=> %e (itNr=%d) ", sampled_points[j].fvalue, itNr);
    /* Rerun of Powell */
    itNr=100; tol=0.000001;
    for(k=0; k<dim; k++) delta[k]=0.001*(uplim[k]-lowlim[k]);
    ret=powell(sampled_points[j].par, delta, dim, tol, &itNr,
               &sampled_points[j].fvalue, objf);
    if(ret > 1) {
      free(sampled_points); free(delta);
      if(TGO_TEST) printf("powell error %d\n", ret);
      return(6);
    }
    if(TGO_TEST>2) printf("=> %e (itNr=%d)\n", sampled_points[j].fvalue, itNr);
  }
    

  /*
   *  Find the best result of Powell and run Powell again from
   *  this point with better accuracy
   */

  if(TGO_TEST>2) printf("The minima given by Powell:\n");
  for(j=0, min=1e+99, IDmin=0; j<samplNr; j++) if(sampled_points[j].topomin==1) {
    if(TGO_TEST>2) {
      for(i=0; i<dim;i++) printf("%e ", sampled_points[j].par[i]); 
      printf("=> %e\n", sampled_points[j].fvalue);
    }
    if(sampled_points[j].fvalue<min) {min=sampled_points[j].fvalue; IDmin=j;}
  }
  if(TGO_TEST>1) {
    printf("Best topographical minimum:");
    for(i=0; i<dim;i++) printf("%e ", sampled_points[IDmin].par[i]);
    printf("-> %e \n", sampled_points[IDmin].fvalue); 
  }
  /* Rerun of Powell */
  deltaf=0.0002; tol=0.000001;
  do {
    itNr=100;
    for(k=0; k<dim; k++) delta[k]=deltaf*(uplim[k]-lowlim[k]);
    ret=powell(sampled_points[IDmin].par, delta, dim, tol, &itNr,
               &sampled_points[IDmin].fvalue, objf);
    if(ret > 1) {
      free(sampled_points); free(delta);
      if(TGO_TEST) printf("powell error %d\n", ret);
      return(7);
    }
    if(TGO_TEST>1) printf("  powell once more with %d iteration(s) -> WSS=%e\n",
      itNr, sampled_points[IDmin].fvalue);
    deltaf*=0.5; tol*=0.5;
  } while(itNr>1 && deltaf>0.0000000001);

  /* Save best point to gmin */
  for(j=0; j<dim; j++) gmin[j]=sampled_points[IDmin].par[j];
  *fmin=sampled_points[IDmin].fvalue;


  free(sampled_points);
  free(delta);
  if(TGO_TEST>0) printf("out of tgo\n");
  return(0);

} /* end tgo */

/*****************************************************************************/
void randomParameters(TGO_POINT *p, int parNr, int sNr,
		      double *low, double *up)
{
  int i, j;
  double dif;

  for(j=0; j<parNr; j++) {
    dif=up[j]-low[j];
    if(dif<=0.0) {
      for(i=0; i<sNr; i++) if(p[i].topomin==0) p[i].par[j]=low[j];
    } else {
      for(i=0; i<sNr; i++) if(p[i].topomin==0) {
        p[i].par[j]=((double)rand()/(double)RAND_MAX) * dif + low[j];
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/

