/******************************************************************************
  Copyright (c) 2000-2006 by Turku PET Centre

  File:        dft.c
  Description: Contains functions for reading and writing DFT files and
               processing TAC data.
  Known bugs:  Line length cannot be larger than max int value, but this is not
               checked. This may limit the curve number in some systems.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2000-07-07 Vesa Oikonen
    First created.
  2000-07-26 VO
    _type is set to 0 (not 2) in case of "plain" data without titles.
    For plain datafiles only frame mid times are saved.
  2000-08-07 VO
    Checks for overlapping and very small gaps in frametimesDFT().
  2000-09-09 VO
    Added place for weights in the data structure.
    Weights are saved as a VOI in DFT files.
    Included function copyvoiDFT().
  2000-09-18 VO
    Weights are not saved with "plain" data.
  2000-09-21 VO
    Included function addnullframeDFT().
  2000-11-30 VO
    Fixed a bug in addnullframeDFT().
  2000-12-13 VO
    Minor changes in source code, not affecting the compiled code.
  2001-01-12 VO
    Included function sortDFT().
    addnullframeDFT(): x2[0]=x1[1]
  2001-01-24 VO
    Extracting study number works as intended with MSDOS pathnames.
  2001-01-30 VO
    selectDFT() does not change its argument 'name'.
  2001-01-31 VO
    Additional data is initialized in addmemDFT().
  2001-03-25 VO
    Included functions copymainhdrDFT() and copyvoihdrDFT().
  2001-03-31 VO
    Fixed a bug in copymainhdrDFT().
  2001-04-10 VO
    Minor change in selectDFT().
  2001-04-26 VO
    overflowDFT() returns now different error numbers.
  2001-10-30 VO
    Recognizes specifically DFT fit format (but does not read).
  2002-02-21 VO
    addDFT() allows different units if _type is 0.
  2002-07-30 VO
    memset() added to initDFT().
  2002-10-10 VO
    sortDFT() rewritten.
  2002-12-01 VO
    Change in frametimesDFT(); now works with just one frame.
  2003-02-12 VO
    Change in readDFT(); ignores data line after comment sign '#'.
  2003-06-19 VO
    Included function na_fillDFT().
  2003-08-13 VO
    Included "timetypes" um and mm for measuring distances.
    Strings "Distance" and "Distances" and units "um", "mm" and s are
    identified in addition to the previous "Time" and "Times" and
    "sec" and "min".
  2003-08-14 VO
    Unknown timetype is read/written as string "()".
  2003-12-05 KS & VO
    Included functions kBqMinDFT() and kBqMaxDFT().
  2004-01-21 VO
    Returns error as it should when trying to read empty file.
  2004-01-26 VO
    Included function dftdup().
    dftadd() applies copyvoihdrDFT().
    Function infos are changed.
  2004-01-27 VO
    Included function dftMovevoi().
  2004-05-23 VO
    Changed some comments into Doxygen style.
    Included function dftSortByFrame().
  2004-05-25 VO
    Included function dftWriteHTML().
  2004-08-17 VO
    Included function dftDelete().
  2004-08-23 VO
    Function names with *DFT end changed to start with dft*.
    New library function studynr_from_fname() is applied.
  2004-08-26 VO
    Unit check in dftAdd() is changed to be case-insensitive.
  2004-09-05 VO
    Reads and copies and prints voi.name but does not yet write it.
    Included function dftSelectRegions().
  2004-09-20 VO
    Doxygen style comments corrected.
  2004-12-04 VO
    dftRead() tries to read also old *.roi.kbq / *.roi.nci files.
  2004-12-09 VO
    Included function dftSortByPlane().
  2005-06-17 VO
    Added one digit in printing floats in exponential format because
    of MinGW compiler.
  2005-10-30 VO
    New fields in DFT are initiated and copied when necessary.
    Included function dft_fill_hdr_from_IFT().
    dftRead():
      Tries to read header info in comments.
    dftWrite():
      Writes newline always before '#' in comments.
      Writes HTML format if filename extension is *.HTM(L).
    dftWriteHTML():
      Big changes in HTML codes, but table looks as before.
  2005-11-30 VO
    dftRead():
      Tries to read IDWC and IF file, based on filename extension.
  2005-12-05 Kaisa Sederholm
    dftWrite():
      Added warning if the function is writing the file with a name
      that allready exists.
  2006-02-01 VO
    "allready" -> "already".
  2006-05-15 VO
    Corrected bugs in dftFrametimes() which changed time unit
    when sample number was 1.
  2006-06-03 VO
    Some strcmp() -> strcasecmp().
    dftRead(): atof() replaced by decimal point insensitive atof_dpi().


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "decpoint.h"
/*****************************************************************************/
#include "include/dft.h"
#include "include/ncifile.h"
#include "include/idwc.h"
#include "include/if.h"
/* local function definitions */
int dftQSortName(const void *voi1, const void *voi2);
int dftQSortPlane(const void *voi1, const void *voi2);
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for DFT. All data is cleared. */
void dftEmpty(DFT *data)
{
  if(data->_voidataNr>0) free((char*)(data->voi));
  if(data->_dataSize>0) free((char*)(data->_data));
  data->_dataSize=data->_voidataNr=0;
  data->frameNr=data->voiNr=0;
  data->studynr[0]=data->comments[0]=data->unit[0]=(char)0;
  data->radiopharmaceutical[0]=data->isotope[0]=data->decayCorrected=(char)0;
  data->scanStartTime[0]=data->injectionTime[0]=(char)0;
  data->timeunit=data->timetype=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate DFT structure. This should be called once before use. */
void dftInit(DFT *data)
{
  memset(data, 0, sizeof(DFT));
  data->_voidataNr=data->_dataSize=data->_dataSize=0;
  data->frameNr=data->voiNr=0;
  data->studynr[0]=data->comments[0]=data->unit[0]=(char)0;
  data->radiopharmaceutical[0]=data->isotope[0]=data->decayCorrected=(char)0;
  data->scanStartTime[0]=data->injectionTime[0]=(char)0;
  data->timeunit=data->timetype=data->isweight=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read DFT file contents into specified data structure, emptying its old
    contents.
\return Returns 0 when successful, in case of error sets dfterrmsg.
 */
int dftRead(char *filename, DFT *data)
{
  FILE *fp;
  char *cptr, *line, *lptr;
  int i, j, c, type=0, voiNr=0, frameNr=0, longest=0;
  double f;
  IFT ift;


  /* Empty data */
  dftEmpty(data);

  /* Check from filename if this is an IDWC file */
  if(fncasematch("*.idwc", filename)==1 || fncasematch("*.idw", filename)==1) {
    if(idwcRead(filename, data)==0) return(0); else return(1);
  }
  /* Check from filename if this is an IF file */
  if(fncasematch("*.if", filename)==1) {
    if(ifRead(filename, data)==0) return(0); else return(1);
  }

  /* Try to read information from an interfile-type header */
  iftInit(&ift);
  if(iftRead(&ift, filename, 1)!=0) iftEmpty(&ift);

  /* Open file */
  fp=fopen(filename, "r");
  if(fp==NULL) {
    strcpy(dfterrmsg, "cannot open file"); iftEmpty(&ift); return 1;
  }

  /* Decide file type */
  type=dftType(fp);
  if(type==0) { /* unknown format */
    fclose(fp); iftEmpty(&ift);
    strcpy(dfterrmsg, "unknown file format"); return(1);
  } else if(type==3) {
    strcpy(dfterrmsg, "cannot read fit file"); fclose(fp); iftEmpty(&ift); return 1;
  } else if(type==4) {
    /* Try to read the old format */
    fclose(fp); i=roikbqRead(filename, data); if(i) {iftEmpty(&ift); return 1;}
    return(0);
  }

  /* Get the length of the longest line */
  i=0; while((c=fgetc(fp))!=EOF) {
    if(c==10 || c==13) {if(i>longest) longest=i; i=0;} else i++;}
  if(i>longest) longest=i;
  rewind(fp); longest+=2;
  /* and allocate memory for string of that length */
  line=(char*)malloc((longest+1)*sizeof(char));
  if(line==NULL) {strcpy(dfterrmsg, "out of memory"); iftEmpty(&ift); fclose(fp); return 2;}

  /* Get the frame number */
  i=0; while(fgets(line, longest, fp)!=NULL && *line) {
    if(line[0]=='#') continue;
    cptr=strchr(line, '#'); if(cptr!=NULL) *cptr=(char)0;
    for(j=0; j<strlen(line); j++)
      if(isalnum((int)line[j]) || line[j]=='.') {i++; break;}
  }
  rewind(fp); frameNr=i; if(type==1) frameNr-=4; /* title lines */
  if(frameNr<1) {strcpy(dfterrmsg, "contains no data"); iftEmpty(&ift); fclose(fp); return 1;}

  /* Get the number of curves */
  /* find first line that is not an empty or a comment line */
  /* In plain data file that is first frame, which starts with time */
  /* In normal DFT file that is 1st title line, which starts with 'DFT' */
  while(fgets(line, longest, fp)!=NULL) {
    lptr=line; if(line[0]=='#') continue;
    cptr=strtok(lptr, " \t\n\r"); if(cptr==NULL) continue;
    i=0; while((cptr=strtok(NULL, " \t\n\r"))!=NULL) i++;
    break;
  }
  rewind(fp); voiNr=i;
  if(voiNr<1) {strcpy(dfterrmsg, "contains no curves"); iftEmpty(&ift); fclose(fp); return 1;}

  /* Allocate memory for data */
  if(dftSetmem(data, frameNr, voiNr)) {
    strcpy(dfterrmsg, "out of memory"); iftEmpty(&ift); fclose(fp); return 2;}

  /* For plain data files, set defaults for title data */
  if(type==2) {
    for(i=0; i<voiNr; i++) {
      sprintf(data->voi[i].voiname, "%06d", i+1);
      strcpy(data->voi[i].name, data->voi[i].voiname);
    }
    /* Default: times in minutes, and frame mid time */
    data->timeunit=0; data->timetype=0;
  }

  /*
   *  Try to read information from an interfile-type header
   */
  if(ift.keyNr>0) dft_fill_hdr_from_IFT(data, &ift);
  iftEmpty(&ift);

  /* Read title lines, if they exist */
  i=0; if(type==1) do {
    if(fgets(line, longest, fp)==NULL) {
      strcpy(dfterrmsg, "wrong format"); iftEmpty(&ift); fclose(fp); return 3;}
    lptr=line;
    /* Check for comment line */
    if(line[0]=='#') { /* Read comment only if there is space left */
      if(strlen(data->comments)+strlen(line)<_DFT_COMMENT_LEN)
        strcat(data->comments, line);
      continue;
    }
    cptr=strchr(line, '#'); if(cptr!=NULL) *cptr=(char)0;
    /* Read first token, and check for empty lines as well */
    cptr=strtok(lptr, " \t\n\r"); if(cptr==NULL) continue; else i++;
    if(i==1) { /* VOI names */
      for(j=0; j<voiNr; j++) {
        if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
          strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
        if(strlen(cptr)>1 || *cptr!='.') strncpy(data->voi[j].voiname, cptr, 6);
        else sprintf(data->voi[j].voiname, "%06d", j);
        strcpy(data->voi[j].name, data->voi[j].voiname);
      }
    } else if(i==2) { /* 2nd VOI names (hemispheres) */
      strncpy(data->studynr, cptr, MAX_STUDYNR_LEN);
      for(j=0; j<voiNr; j++) {
        if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
          strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
        if(strlen(cptr)>1||*cptr!='.') {
          strncpy(data->voi[j].hemisphere,cptr,6);
          strcat(data->voi[j].name, " ");
          strcat(data->voi[j].name, data->voi[j].hemisphere);
        } else {
          strcat(data->voi[j].name, " .");
        }
      }
    } else if(i==3) { /* unit and VOI place (planes) */
      strncpy(data->unit, cptr, 12);
      for(j=0; j<voiNr; j++) {
        if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
          strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
        if(strlen(cptr)>1 || *cptr!='.') {
          strncpy(data->voi[j].place, cptr, 6);
          strcat(data->voi[j].name, " ");
          strcat(data->voi[j].name, data->voi[j].place);
        } else {
          strcat(data->voi[j].name, " .");
        }
      }
    } else if(i==4) { /* time type, time unit, and VOI sizes */
      /* time type */
      if(strcasecmp(cptr, "Time")==0) data->timetype=0;
      else if(strcasecmp(cptr, "Times")==0) data->timetype=3;
      else if(strcasecmp(cptr, "Start")==0) data->timetype=1;
      else if(strcasecmp(cptr, "End")==0) data->timetype=2;
      else if(strcasecmp(cptr, "Distance")==0) data->timetype=0;
      else if(strcasecmp(cptr, "Distances")==3) data->timetype=0;
      else {strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
      /* time unit */
      if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
        strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
      if(strcasecmp(cptr, "(min)")==0) data->timeunit=0;
      else if(strcasecmp(cptr, "(sec)")==0) data->timeunit=1;
      else if(strcasecmp(cptr, "(s)")==0) data->timeunit=1;
      else if(strcasecmp(cptr, "(um)")==0) data->timeunit=11;
      else if(strcasecmp(cptr, "(mm)")==0) data->timeunit=12;
      else if(strcmp(cptr, "()")==0) data->timeunit=-1;
      else {strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
      /* volumes */
      for(j=0; j<voiNr; j++) {
        if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
          strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
        if(strlen(cptr)==1 && *cptr=='.') data->voi[j].size=0.0;
        else data->voi[j].size=atof_dpi(cptr);
      }
    }
  } while(i<4);


  /* Read data lines */
  i=0; while(fgets(line, longest, fp)!=NULL) {
    lptr=line;
    /* Check for comment line */
    if(line[0]=='#') { /* Read comment only if there is space left */
      if(strlen(data->comments)+strlen(line)<_DFT_COMMENT_LEN)
        strcat(data->comments, line);
      continue;
    }
    cptr=strchr(line, '#'); if(cptr!=NULL) *cptr=(char)0;
    /* Read first token, and check for empty lines as well */
    cptr=strtok(lptr, " \t\n\r"); if(cptr==NULL) continue;
    if(i<frameNr) {
      /* Read time(s) */
      f=atof_dpi(cptr);
      if(data->timetype==3) {
        data->x1[i]=f; cptr=strtok(NULL, " \t\n\r"); data->x2[i]=atof_dpi(cptr);
      } else data->x[i]=f;
      /* curve data */
      for(j=0; j<voiNr; j++) {
        if((cptr=strtok(NULL, " \t\n\r"))==NULL) {
          strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}
        if(strlen(cptr)==1 && *cptr=='.') data->voi[j].y[i]=NA;
        else data->voi[j].y[i]=atof_dpi(cptr);
      }
    }
    i++;
  }
  if(i!=frameNr) {strcpy(dfterrmsg, "wrong format"); fclose(fp); return 3;}

  /* Close file */
  fclose(fp); free((char*)line);

  /* Set voiNr and frameNr, and type */
  data->voiNr=voiNr; data->frameNr=frameNr;
  if(type==2) type=0; data->_type=type;

  /* Set study number from filename, if necessary */
  if(strlen(data->studynr)==0) studynr_from_fname(filename, data->studynr);

  /* If weight 'voi' was read, move it to its own place */
  for(i=0; i<data->voiNr; i++) if(!strcasecmp(data->voi[i].voiname, "weight")) {
    data->isweight=1;
    for(j=0; j<data->frameNr; j++) data->w[j]=data->voi[i].y[j];
    /* Move the following VOIs one step backwards */
    for(c=i+1; c<data->voiNr; c++) if(dftCopyvoi(data, c, c-1)) {
      strcpy(dfterrmsg, "cannot read weight"); return 4;}
    data->voiNr--;
    break;
  }
  /* If no weights were found, set uniform weight */
  if(!data->isweight) for(i=0; i<data->frameNr; i++) data->w[i]=1.0;

  /* Calculate frame mid times, or frame start and end times */
  dftFrametimes(data);

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine the type of DFT file.
\return 0=unknown; 1=normal DFT; 2=plain data; 3=fit file ; 4=nci file
 */
int dftType(FILE *fp)
{
  char tmp[256];
  int c;

  /* Find first line that is not empty or comment line */
  rewind(fp);
  while(fgets(tmp, 4, fp) != NULL) {if(strlen(tmp) && tmp[0]!='#') break;}
  rewind(fp);

  /* Check for identification strings */
  if(strncasecmp(tmp, "DFT", 3)==0) return 1;
  else if(strncasecmp(tmp, "FIT1", 3)==0) return 3;
  else if(strncasecmp(tmp, "cpt", 3)==0) return 4;

  /* Binary data? */
  while((c=fgetc(fp))!=EOF) if(!isalnum(c) && !isspace(c) && !isgraph(c)) {
    rewind(fp); return 0;}
  rewind(fp); return 2;
}
/*****************************************************************************/

/*****************************************************************************/
/** Prints to stdout the contents of DFT data structure.
    Mainly for testing purposes.
 */
void dftPrint(DFT *data)
{
  int voi, frame;

  printf("Number of curves: %d     Number of data points: %d\n",
    data->voiNr, data->frameNr);
  printf("Study: '%s'  Unit: '%s'\n", data->studynr, data->unit);
  printf("Time unit and type: %d %d\n", data->timeunit, data->timetype);
  if(strlen(data->radiopharmaceutical))
    printf("Radiopharmaceutical: %s\n", data->radiopharmaceutical);
  if(strlen(data->isotope)) printf("Isotope: %s\n", data->isotope);
  if(strlen(data->scanStartTime))
    printf("Scan start time: %s\n", data->scanStartTime);
  if(strlen(data->injectionTime))
    printf("Injection time: %s\n", data->injectionTime);
  if(data->decayCorrected==1) printf("Corrected for physical decay: yes\n");
  else if(data->decayCorrected==2) printf("Corrected for physical decay: no\n");
  printf("_datasize = %d\n", data->_dataSize);
  for(voi=0; voi<data->voiNr; voi++) {
    if(strlen(data->voi[voi].name)>0)
      printf("\nROI name: '%s' Size: %g\n",
        data->voi[voi].name, data->voi[voi].size);
    else
      printf("\nROI name: '%s' '%s' '%s'  Size: %g\n",
        data->voi[voi].voiname, data->voi[voi].hemisphere, data->voi[voi].place,
        data->voi[voi].size);
    for(frame=0; frame<data->frameNr; frame++) {
      printf("%03d:  %11.3e %11.3e %11.3e    %11.3e %11.3e %11.3e\n",
        frame+1, data->x[frame], data->x1[frame], data->x2[frame],
        data->voi[voi].y[frame], data->voi[voi].y2[frame],
        data->voi[voi].y3[frame]);
    }
  }
  printf("Comments:\n");
  if(strlen(data->comments)>0) printf("%s\n", data->comments);
  printf("Weights:\n");
  if(data->isweight)
    for(frame=0; frame<data->frameNr; frame++)
      printf(" %03d  %11.3e %11.3e  %11.3e\n", frame+1,
        data->x1[frame], data->x2[frame], data->w[frame]);
  else
    printf(" contains no weights.\n");
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for DFT data and sets data pointers. Old data is destroyed.
 */
int dftSetmem(DFT *data, int frameNr, int voiNr)
{
  int i, n;
  double *d;


  /* Clear previous data */
  dftEmpty(data);

  /* Allocate memory for curves */
  data->voi=(Voi*)calloc(voiNr, sizeof(Voi));
  if(data->voi==NULL) return 1;
  data->_voidataNr=voiNr;

  /* Allocate memory for frames */
  /* For 3 axes, weights, and for curves (3 for each VOI) */
  /* And one extra 'frame' for overflow testing */
  n=(frameNr+1)*(3+1+3*voiNr);
  data->_data=(double*)calloc(n, sizeof(double));
  if(data->_data==NULL) return 1;
  data->_dataSize=n;

  /* Set pointers for curve data */
  d=data->_data; data->x = d;
  d += frameNr + 1; data->x1 = d;
  d += frameNr + 1; data->x2 = d;
  d += frameNr + 1; data->w = d;
  for(i=0; i<voiNr; i++) {
    d += frameNr + 1; data->voi[i].y = d;
    d += frameNr + 1; data->voi[i].y2 = d;
    d += frameNr + 1; data->voi[i].y3 = d;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for specified nr of additional data VOIs.
    Old data is left unchanged.
 */
int dftAddmem(DFT *data, int voiNr)
{
  int i, j, n;
  double *d;


  /* Allocate memory for curves */
  data->voi=(Voi*)realloc(data->voi, (voiNr+data->_voidataNr)*sizeof(Voi));
  if(data->voi==NULL) return 1;
  data->_voidataNr+=voiNr;

  /* Allocate memory for frames */
  /* For 3 axes, weights, and for curves (3 for each VOI) */
  /* And one extra 'frame' for overflow testing */
  n=(data->frameNr+1)*(3*voiNr);
  data->_data=(double*)realloc(data->_data, (n+data->_dataSize)*sizeof(double));
  if(data->_data==NULL) return 1;
  data->_dataSize+=n;

  /* Set pointers for curve data */
  d=data->_data; data->x = d; data->x[data->frameNr]=0.0;
  d += data->frameNr + 1; data->x1 = d; data->x1[data->frameNr]=0.0;
  d += data->frameNr + 1; data->x2 = d; data->x2[data->frameNr]=0.0;
  d += data->frameNr + 1; data->w = d; data->w[data->frameNr]=0.0;
  for(i=0; i<data->_voidataNr; i++) {
    d += data->frameNr + 1; data->voi[i].y = d;
    d += data->frameNr + 1; data->voi[i].y2 = d;
    d += data->frameNr + 1; data->voi[i].y3 = d;
  }

  /* Initiate values */
  for(i=data->voiNr; i<data->_voidataNr; i++) {
    strcpy(data->voi[i].name, "");
    strcpy(data->voi[i].voiname, "");
    strcpy(data->voi[i].hemisphere, "");
    strcpy(data->voi[i].place, "");
    data->voi[i].size=1.0;
    data->voi[i].sw=data->voi[i].sw2=data->voi[i].sw3=0;
    for(j=0; j<data->frameNr+1; j++)
      data->voi[i].y[j]=data->voi[i].y2[j]=data->voi[i].y3[j]=0.0;
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add the specified voi [0,voiNr-1] from data2 to data1.
    Allocates memory for additional data VOI, if necessary.
\return Returns 0 if ok.
 */
int dftAdd(DFT *data1, DFT *data2, int voi)
{
  int i, n;

  /* Check that voi exists */
  if(data2->voiNr<=voi || voi<0) {
    strcpy(dfterrmsg, "there is no region to combine"); return 8;}

  /* Check that frame number etc is the same */
  if(data1->frameNr!=data2->frameNr ||
     (data1->_type>0 && data2->_type>0 &&
      (data1->timeunit!=data2->timeunit || strcasecmp(data1->unit, data2->unit))
   )) {
    strcpy(dfterrmsg, "data does not match"); return 8;}

  /* Allocate more memory if necessary */
  if(data1->_voidataNr==data1->voiNr)
    if(dftAddmem(data1, 1)) {
      strcpy(dfterrmsg, "cannot allocate memory"); return 8;}

  /* Copy data */
  n=data1->voiNr;
  (void)dftCopyvoihdr(data2, voi, data1, n);
  for(i=0; i<data1->frameNr; i++) {
    data1->voi[n].y[i]=data2->voi[voi].y[i];
    data1->voi[n].y2[i]=data2->voi[voi].y2[i];
    data1->voi[n].y3[i]=data2->voi[voi].y3[i];
  }
  data1->voiNr+=1;

  /* If data2 contains weights and data1 does not, then copy those */
  if(data2->isweight && !data1->isweight)
    for(i=0; i<data1->frameNr; i++) data1->w[i]=data2->w[i];

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Select VOIs (sets sw=1), whose names are matching specified string.
    If no string is specified, then all VOIs are selected.
\return Returns the number of matches, or <0, if an error occurred.
 */
int dftSelect(DFT *data, char *name)
{
  int i, j, n;
  char *p, n1[128], n2[128], n3[128], tmp[128], sname[1024];

  /* Select all, if no string was specified */
  if(name==NULL || strlen(name)==0) {
    for(i=0; i<data->voiNr; i++) data->voi[i].sw=1; return data->voiNr;}
  /* Make a copy of 'name' and use it */
  strcpy(sname, name);
  /* Check if string contains several substrings (hemisphere and place) */
  n1[0]=n2[0]=n3[0]=(char)0;
  p=strtok(sname, " ,;\n\t|"); if(p!=NULL) strcpy(n1, p); else return -1;
  p=strtok(NULL, " ,;\n\t|"); if(p!=NULL) {
    strcpy(n2, p); p=strtok(NULL, " ,;\n\t|"); if(p!=NULL) strcpy(n3, p);}
  /* Convert strings to lowercase */
  for(i=0; i<strlen(n1); i++) n1[i]=tolower(n1[i]);
  for(i=0; i<strlen(n2); i++) n2[i]=tolower(n2[i]);
  for(i=0; i<strlen(n3); i++) n3[i]=tolower(n3[i]);
  /* Search through the data */
  for(i=0, n=0; i<data->voiNr; i++) {
    data->voi[i].sw=0;
    sprintf(tmp, "%s%s%s", data->voi[i].voiname, data->voi[i].hemisphere,
                           data->voi[i].place);
    for(j=0; j<strlen(tmp); j++) tmp[j]=tolower(tmp[j]);
    if(strstr(tmp, n1)==NULL) continue;
    if(n2[0] && strstr(tmp, n2)==NULL) continue;
    if(n3[0] && strstr(tmp, n3)==NULL) continue;
    data->voi[i].sw=1; n++;
  }
  return n;
}
/*****************************************************************************/

/*****************************************************************************/
/** Select the VOIs that have matching region name or number.
    Sets sw=1 or sw=0. This will replace dftSelect().
\return Returns the number of selected VOIs, or <0 in case of an error.
 */
int dftSelectRegions(
  /** Pointer to DFT data where VOIs are selected */
  DFT *dft,
  /** Name or VOI number which is searched */
  char *region_name,
  /** 1=Non-matching VOIs are deselected, 0=Old selections are preserved */
  int reset
) {
  int ri, match_nr=0;

  /* Check the input */
  if(dft==NULL || dft->voiNr<1 || strlen(region_name)<1) return(-1);
  /* Reset all selections if required */
  if(reset!=0) for(ri=0; ri<dft->voiNr; ri++) dft->voi[ri].sw=0;
  /* Check each VOI */
  for(ri=0; ri<dft->voiNr; ri++) {
    if(rnameMatch(dft->voi[ri].name, ri+1, region_name)!=0) {
      dft->voi[ri].sw=1; match_nr++;
    }
  }
  return(match_nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Change time unit from min to sec, without checking original unit. */
void dftMin2sec(DFT *data)
{
  int i;

  for(i=0; i<data->frameNr; i++) {
    data->x[i]*=60.; data->x1[i]*=60.; data->x2[i]*=60.;}
  data->timeunit=1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Change time unit from sec to min, without checking original unit. */
void dftSec2min(DFT *data)
{
  int i;

  for(i=0; i<data->frameNr; i++) {
    data->x[i]/=60.; data->x1[i]/=60.; data->x2[i]/=60.;}
  data->timeunit=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate frame mid or start and end times. Timetype is not changed. */
void dftFrametimes(DFT *data)
{
  int i, j, sw=0;
  double f, fs;

  if(data->timeunit<0) return;

  /* If times are in min, change to sec to allow rounding to ints */
  if(data->timeunit==0) {sw=1; dftMin2sec(data);}

  /* Decide what to do, and get it done */
  if(data->timetype==0) { /* frame start and end times from mid times */
    if(data->timeunit>10) return; /* Don't touch distances */
    /* Easy, if only one frame */
    if(data->frameNr==1) {
      if(data->x[0]<=0.0) {data->x1[0]=data->x[0]; data->x2[0]=0.0;}
      else {data->x1[0]=0.0; data->x2[0]=2.0*data->x[0];}
      if(sw) dftSec2min(data);
      return;
    }
    /* Fill start and end times with -999 */
    for(i=0; i<data->frameNr; i++) data->x1[i]=data->x2[i]=-999.;
    /* Search for sequences of nearly same frame lengths */
    for(i=1; i<data->frameNr-1; i++) {
      f=data->x[i]-data->x[i-1]; fs=data->x[i+1]-data->x[i];
      if(fabs(f-fs)>=2.0) continue; else f=(f+fs)/2.0;
      data->x1[i-1]=data->x[i-1]-f/2.0; data->x2[i-1]=data->x[i-1]+f/2.0;
      data->x1[i]=data->x[i]-f/2.0; data->x2[i]=data->x[i]+f/2.0;
      data->x1[i+1]=data->x[i+1]-f/2.0; data->x2[i+1]=data->x[i+1]+f/2.0;
      /* Check for negatives */
      for(j=i-1; j<i+2; j++) {
        if(data->x1[j]<0.0) data->x1[j]=0.0;
        if(data->x2[j]<0.0) data->x2[j]=0.0;
      }
    }
    /* If out-of-sequence frames were left out, fill those to the nearest one */
    i=0;  /* first frame */
    if(data->x1[i]<0) {
      if(data->x1[i+1]>0) data->x2[i]=data->x1[i+1];
      else data->x2[i]=(data->x[i+1]+data->x[i])/2.0;
      data->x1[i]=2.0*data->x[i]-data->x2[i];
    }
    i=data->frameNr-1;  /* last frame */
    if(data->x1[i]<0) {
      if(data->x2[i-1]>0) data->x1[i]=data->x2[i-1];
      else data->x1[i]=(data->x[i-1]+data->x[i])/2.0;
      data->x2[i]=2.0*data->x[i]-data->x1[i];
    }
    /* other frames */
    for(i=1; i<data->frameNr-1; i++) if(data->x1[i]<0.0) {
      /* which frame is nearest? */
      if(data->x[i]-data->x[i-1] < data->x[i+1]-data->x[i]) { /* last one */
        if(data->x2[i-1]>0) data->x1[i]=data->x2[i-1];
        else data->x1[i]=(data->x[i-1]+data->x[i])/2.0;
        data->x2[i]=2.*data->x[i]-data->x1[i];
      } else { /* next one */
        if(data->x1[i+1]>0) data->x2[i]=data->x1[i+1];
        else data->x2[i]=(data->x[i+1]+data->x[i])/2.0;
        data->x1[i]=2.0*data->x[i]-data->x2[i];
      }
    }
    /* Check for negatives */
    for(i=0; i<data->frameNr; i++) {
      if(data->x1[i]<0.0) data->x1[i]=0.0;
      if(data->x2[i]<0.0) data->x2[i]=data->x1[i];
    }
    /* Check for overlapping and very small gaps */
    for(i=1; i<data->frameNr; i++) {
      f=data->x1[i]-data->x2[i-1];
      if(f<1.0) data->x1[i]=data->x2[i-1]=(data->x1[i]+data->x2[i-1])/2.0;
    }
  } else if(data->timetype==3) { /* mid times from frame start and end times */
    for(i=0; i<data->frameNr; i++) data->x[i]=0.5*(data->x1[i]+data->x2[i]);
  } else if(data->timetype==1) { /* frame start times -> end and mid times */
    if(data->timeunit>10) return; /* Don't touch distances */
    for(i=0; i<data->frameNr-1; i++) data->x2[i]=data->x1[i+1];
    data->x2[data->frameNr-1]=data->x1[data->frameNr-1]+
      (data->x2[data->frameNr-2]-data->x1[data->frameNr-2]);
    for(i=0; i<data->frameNr; i++) data->x[i]=0.5*(data->x1[i]+data->x2[i]);
  } else if(data->timetype==2) { /* frame end times -> start and mid times */
    if(data->timeunit>10) return; /* Don't touch distances */
    data->x1[0]=0.0;
    for(i=1; i<data->frameNr; i++) data->x1[i]=data->x2[i-1];
    for(i=0; i<data->frameNr; i++) data->x[i]=0.5*(data->x1[i]+data->x2[i]);
  }

  /* If times were in min, change them back */
  if(sw) dftSec2min(data);

  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check for overflows in data structure. Returns 0, if ok. */
int dftOverflow(DFT *data)
{
  int i;

  if(data->frameNr<1 || data->voiNr<1) return 0;
  if(data->x[data->frameNr]!=0.0) return 1;
  if(data->x1[data->frameNr]!=0.0) return 2;
  if(data->x2[data->frameNr]!=0.0) return 3;
  for(i=0; i<data->voiNr; i++) {
    if(data->voi[i].y[data->frameNr]!=0.0) return 4;
    if(data->voi[i].y2[data->frameNr]!=0.0) return 5;
    if(data->voi[i].y3[data->frameNr]!=0.0) return 6;
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DFT data into specied file.
    If file exists, a backup file (%) is written also.
    The file format specified in data is applied.
 */
int dftWrite(DFT *data, char *filename)
{
  int i, j, n, sw;
  char *cptr, tmp[1024], is_stdout=0;
  FILE *fp;


  /* Check that there is some data to write */
  if(data->voiNr<1 || data->frameNr<1) {
    strcpy(dfterrmsg, "no data"); return 1;}

  /* If extension is *.HTM(L), write in HTML format */
  cptr=strrchr(filename, '.'); if(cptr!=NULL) cptr++;
  if(cptr!=NULL) {
    if(strcasecmp(cptr, "HTM")==0 || strcasecmp(cptr, "HTML")==0)
      return(dftWriteHTML(data, filename, 1));
  }

  /* Check if writing to stdout */
  if(!strcasecmp(filename, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(filename, 0) != -1) {
   strcpy(tmp, filename); strcat(tmp, "%"); rename(filename, tmp);
    fprintf(stdout, "Warning: a file called %s already existed and was renamed as %s%%.\n", filename, filename);
}

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp = fopen(filename, "w")) == NULL) {
    strcpy(dfterrmsg, "cannot open file"); fclose(fp); return 2;}

  /* Write title lines */
  if(data->_type==1) {
    /* 1st line with filetype identification string and region names */
    n=fprintf(fp, "%-20.20s", DFT_VER);
    if(n==0) {strcpy(dfterrmsg, "disk full"); fclose(fp); return 3;}
    for(i=0; i<data->voiNr; i++) fprintf(fp," %-11.11s",data->voi[i].voiname);
    if(data->isweight)  fprintf(fp," %-11.11s", "weight");
    fprintf(fp, "\n");
    /* 2nd line with study identification and 2nd name (hemisphere) */
    if(strlen(data->studynr)) strcpy(tmp, data->studynr); else strcpy(tmp, ".");
    fprintf(fp, "%-20.20s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(strlen(data->voi[i].hemisphere)) strcpy(tmp, data->voi[i].hemisphere);
      else strcpy(tmp, ".");
      fprintf(fp, " %-11.11s", tmp);
    }
    if(data->isweight) fprintf(fp," %-11.11s", ".");
    fprintf(fp, "\n");
    /* 3rd line with unit and plane names */
    if(strlen(data->unit)) strcpy(tmp, data->unit); else strcpy(tmp, ".");
    fprintf(fp, "%-20.20s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(strlen(data->voi[i].place)) strcpy(tmp, data->voi[i].place);
      else strcpy(tmp, ".");
      fprintf(fp, " %-11.11s", tmp);
    }
    if(data->isweight) fprintf(fp," %-11.11s", ".");
    fprintf(fp, "\n");
    /* 4th line with time type & unit and region volumes */
    switch(data->timetype) {
      case 0:
        if(data->timeunit<11) strcpy(tmp, "Time ");
        else strcpy(tmp, "Distance ");
        break;
      case 1: strcpy(tmp, "Start "); break;
      case 2: strcpy(tmp, "End "); break;
      case 3:
        if(data->timeunit<11) strcpy(tmp, "Times ");
        else strcpy(tmp, "Distances ");
        break;
    }
    if(data->timeunit==0) strcat(tmp, "(min)");
    else if(data->timeunit==1) strcat(tmp, "(sec)");
    else if(data->timeunit==11) strcat(tmp, "(um)");
    else if(data->timeunit==12) strcat(tmp, "(mm)");
    else strcat(tmp, "()");
    fprintf(fp, "%-20.20s", tmp);
    for(i=0; i<data->voiNr; i++) {
      if(data->voi[i].size>=0.0) sprintf(tmp, "%11.3e", data->voi[i].size);
      else strcpy(tmp, ".");
      fprintf(fp, " %-.11s", tmp);
    }
    if(data->isweight) fprintf(fp," %-.11s", ".");
    fprintf(fp, "\n");
  }

  /* Make sure that only frame mid times are saved without titles */
  if(data->_type==0 && data->timetype>0) sw=0; else sw=data->timetype;

  /* Write data */
  for(j=0; j<data->frameNr; j++) {
    /* Time(s) */
    switch(sw) {
      case 0: fprintf(fp, "%17.5f   ", data->x[j]); break;
      case 1: fprintf(fp, "%17.5f   ", data->x1[j]); break;
      case 2: fprintf(fp, "%17.5f   ", data->x2[j]); break;
      case 3: fprintf(fp, "%10.4f %9.4f", data->x1[j], data->x2[j]); break;
    }
    /* y values */
    for(i=0; i<data->voiNr; i++) {
      if(data->voi[i].y[j]!=NA) sprintf(tmp, "%11.3e", data->voi[i].y[j]);
      else strcpy(tmp, ".");
      fprintf(fp, " %-.11s", tmp);
    }
    if(data->isweight && data->_type==1) {
      sprintf(tmp, "%11.3e", data->w[j]); fprintf(fp," %-.11s", tmp);}
    n=fprintf(fp, "\n");
    if(n==0) {strcpy(dfterrmsg, "disk full"); fclose(fp); return 3;}
  }

  /* Write comments */
  n=strlen(data->comments);
  if(n) for(i=0; i<n; i++) {
    if(i>0 && data->comments[i]=='#' &&
       (data->comments[i-1]!='\n' && data->comments[i-1]!='\r'))
      fputc('\n', fp);
    fputc(data->comments[i], fp);
  }

  /* Close file */
  if(!is_stdout) fclose(fp);

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy VOI data inside DFT data structure from one place to another. */
int dftCopyvoi(DFT *data, int from, int to)
{
  int i;

  /* Check that required data exists */
  if(to>=data->_voidataNr || from>=data->_voidataNr) return 1;
  if(from==to) return 0;

  /* Copy VOI info */
  strcpy(data->voi[to].name, data->voi[from].name);
  strcpy(data->voi[to].voiname, data->voi[from].voiname);
  strcpy(data->voi[to].hemisphere, data->voi[from].hemisphere);
  strcpy(data->voi[to].place, data->voi[from].place);
  data->voi[to].size=data->voi[from].size;
  data->voi[to].sw=data->voi[from].sw;
  data->voi[to].sw2=data->voi[from].sw2;
  data->voi[to].sw3=data->voi[from].sw3;
  /* Copy VOI curves */
  for(i=0; i<data->frameNr; i++) {
    data->voi[to].y[i]=data->voi[from].y[i];
    data->voi[to].y2[i]=data->voi[from].y2[i];
    data->voi[to].y3[i]=data->voi[from].y3[i];
  }

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Move VOI in DFT structure from one position to another. */
int dftMovevoi(DFT *dft, int from, int to)
{
  int ri;
  size_t voisize;
  Voi voi;

  if(from<0 || to<0) return(1);
  if(from+1>dft->_voidataNr || to+1>dft->_voidataNr) return(2);
  if(from==to) return(0);
  voisize=sizeof(Voi);
  memcpy(&voi, dft->voi+from, voisize);
  if(from>to) for(ri=from; ri>to; ri--)
    memcpy(dft->voi+ri, dft->voi+(ri-1), voisize);
  else for(ri=from; ri<to; ri++)
    memcpy(dft->voi+ri, dft->voi+(ri+1), voisize);
  memcpy(dft->voi+ri, &voi, voisize);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Delete specified TAC (0..voiNr-1) from the DFT structure.
\return Returns 0 if ok.
 */
int dftDelete(DFT *dft, int voi)
{
  int i, ret;

  /* Check that region exists */
  if(voi>dft->voiNr-1 || voi<0) return(1);
  /* If it is the last one, then just decrease the voiNr */
  if(voi==dft->voiNr) {dft->voiNr--; return(0);}
  /* Otherwise we have to move the following regions in its place */
  for(i=voi+1; i<dft->voiNr; i++) {
    ret=dftMovevoi(dft, i, i-1); if(ret) return(10+ret);
  }
  dft->voiNr--;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy main header info from dft1 to dft2. */
int dftCopymainhdr(DFT *dft1, DFT *dft2)
{
  strcpy(dft2->studynr, dft1->studynr);
  strcpy(dft2->unit, dft1->unit);
  dft2->timeunit=dft1->timeunit; dft2->timetype=dft1->timetype;
  strcpy(dft2->comments, dft1->comments);
  strcpy(dft2->radiopharmaceutical, dft1->radiopharmaceutical);
  strcpy(dft2->isotope, dft1->isotope);
  strcpy(dft2->scanStartTime, dft1->scanStartTime);
  strcpy(dft2->injectionTime, dft1->injectionTime);
  dft2->decayCorrected=dft1->decayCorrected;
  dft2->_type=dft1->_type;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy voi header info from dft1.voi[from] to dft2.voi[to]. */
int dftCopyvoihdr(DFT *dft1, int from, DFT *dft2, int to)
{
  /* Check that required data exists */
  if(to>=dft2->_voidataNr || from>=dft1->_voidataNr) return 1;

  /* Copy VOI info */
  strcpy(dft2->voi[to].name, dft1->voi[from].name);
  strcpy(dft2->voi[to].voiname, dft1->voi[from].voiname);
  strcpy(dft2->voi[to].hemisphere, dft1->voi[from].hemisphere);
  strcpy(dft2->voi[to].place, dft1->voi[from].place);
  dft2->voi[to].size=dft1->voi[from].size;
  dft2->voi[to].sw=dft1->voi[from].sw;
  dft2->voi[to].sw2=dft1->voi[from].sw2;
  dft2->voi[to].sw3=dft1->voi[from].sw3;

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Makes a duplicate of DFT structure pointed to by dft1 into dft2.
    Any existing content of dft2 will be deleted. Dft2 must be initiated.
\return Returns 0 if ok.
 */
int dftdup(DFT *dft1, DFT *dft2)
{
  int ri, fi, ret;

  /* Empty the new data */
  dftEmpty(dft2);
  /* Is that it? Is there any contents in dft1? */
  if(dft1->voiNr==0 && dft1->frameNr==0) {
    ret=dftCopymainhdr(dft1, dft2); return(ret);
  }
  /* Allocate memory for dft2 */
  ret=dftSetmem(dft2, dft1->frameNr, dft1->voiNr); if(ret) return(ret);
  dft2->voiNr=dft1->voiNr; dft2->frameNr=dft1->frameNr;
  /* Copy the contents */
  ret=dftCopymainhdr(dft1, dft2); if(ret) return(ret);
  for(ri=0; ri<dft1->voiNr; ri++) {
    ret=dftCopyvoihdr(dft1, ri, dft2, ri); if(ret) return(ret);
    for(fi=0; fi<dft1->frameNr; fi++) {
      dft2->voi[ri].y[fi]=dft1->voi[ri].y[fi];
      dft2->voi[ri].y2[fi]=dft1->voi[ri].y2[fi];
      dft2->voi[ri].y3[fi]=dft1->voi[ri].y3[fi];
    }
  }
  for(fi=0; fi<dft1->frameNr; fi++) {
    dft2->x[fi]=dft1->x[fi];
    dft2->x1[fi]=dft1->x1[fi]; dft2->x2[fi]=dft1->x2[fi];
    dft2->w[fi]=dft1->w[fi];
  }
  dft2->isweight=dft1->isweight;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Include a frame with time 0, unless one already exists. */
int dftAddnullframe(DFT *data)
{
  int i, j, n;
  DFT temp;


  /* Check whether nullframe exists */
  if(data->x[0]==0.0) return 0;

  /* Allocate memory for temp data */
  dftInit(&temp);
  if(dftSetmem(&temp, data->frameNr, data->voiNr)) return 1;
  temp.frameNr=data->frameNr; temp.voiNr=data->voiNr;

  /* Copy data to temp */
  strcpy(temp.unit, data->unit);
  temp.timeunit=data->timeunit; temp.timetype=data->timetype;
  temp.isweight=data->isweight; temp._type=data->_type;
  strcpy(temp.comments, data->comments);
  for(j=0; j<data->frameNr; j++) {
    temp.x[j]=data->x[j]; temp.x1[j]=data->x1[j]; temp.x2[j]=data->x2[j];
    temp.w[j]=data->w[j];
    for(i=0; i<data->voiNr; i++) {
      temp.voi[i].y[j]=data->voi[i].y[j];
      temp.voi[i].y2[j]=data->voi[i].y2[j];
      temp.voi[i].y3[j]=data->voi[i].y3[j];
    }
  }
  for(i=0; i<data->voiNr; i++) {
    strcpy(temp.voi[i].name, data->voi[i].name);
    strcpy(temp.voi[i].voiname, data->voi[i].voiname);
    strcpy(temp.voi[i].hemisphere, data->voi[i].hemisphere);
    strcpy(temp.voi[i].place, data->voi[i].place);
    temp.voi[i].size=data->voi[i].size;
    temp.voi[i].sw=data->voi[i].sw;
    temp.voi[i].sw2=data->voi[i].sw2;
    temp.voi[i].sw3=data->voi[i].sw3;
  }

  /* Reallocate memory for data */
  dftEmpty(data);
  if(dftSetmem(data, temp.frameNr+1, temp.voiNr)) {dftEmpty(&temp); return 2;}

  /* Set nullframe */
  data->x[0]=data->x1[0]=data->x2[0]=0.0; data->w[0]=0.0;
  for(i=0; i<temp.voiNr; i++)
    data->voi[i].y[0]=data->voi[i].y2[0]=data->voi[i].y3[0]=0.0;

  /* Copy data back from temp */
  data->voiNr=temp.voiNr;
  strcpy(data->unit, temp.unit);
  data->timeunit=temp.timeunit; data->timetype=temp.timetype;
  data->isweight=temp.isweight; data->_type=temp._type;
  strcpy(data->comments, temp.comments);
  for(j=0, n=1; j<temp.frameNr; j++) {
    if(temp.x[j]<0.0) continue;
    if(n==1) data->x2[0]=temp.x1[j];
    data->x[n]=temp.x[j]; data->x1[n]=temp.x1[j]; data->x2[n]=temp.x2[j];
    data->w[n]=temp.w[j];
    for(i=0; i<temp.voiNr; i++) {
      data->voi[i].y[n]=temp.voi[i].y[j];
      data->voi[i].y2[n]=temp.voi[i].y2[j];
      data->voi[i].y3[n]=temp.voi[i].y3[j];
    }
    n++;
  }
  data->frameNr=n;
  for(i=0; i<temp.voiNr; i++) {
    strcpy(data->voi[i].name, temp.voi[i].name);
    strcpy(data->voi[i].voiname, temp.voi[i].voiname);
    strcpy(data->voi[i].hemisphere, temp.voi[i].hemisphere);
    strcpy(data->voi[i].place, temp.voi[i].place);
    data->voi[i].size=temp.voi[i].size;
    data->voi[i].sw=temp.voi[i].sw;
    data->voi[i].sw2=temp.voi[i].sw2;
    data->voi[i].sw3=temp.voi[i].sw3;
  }

  dftEmpty(&temp);

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort DFT regions in alphabetical order by their name. */
int dftSort(DFT *data)
{
  if(data==NULL) return(1);
  if(data->voiNr<=1) return(0);
  qsort(data->voi, data->voiNr, sizeof(Voi), dftQSortName);
  return(0);
}
int dftQSortName(const void *voi1, const void *voi2)
{
  int res;

  res=strcasecmp( ((Voi*)voi1)->name, ((Voi*)voi2)->name );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->voiname, ((Voi*)voi2)->voiname );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->hemisphere, ((Voi*)voi2)->hemisphere );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->place, ((Voi*)voi2)->place );
  return(res);
}
/*****************************************************************************/

/*****************************************************************************/
/** Sort DFT regions in alphabetical order by their plane. */
int dftSortPlane(DFT *data)
{
  if(data==NULL) return(1);
  if(data->voiNr<=1) return(0);
  qsort(data->voi, data->voiNr, sizeof(Voi), dftQSortPlane);
  return(0);
}
int dftQSortPlane(const void *voi1, const void *voi2)
{
  int res;

  res=strcasecmp( ((Voi*)voi1)->place, ((Voi*)voi2)->place );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->name, ((Voi*)voi2)->name );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->voiname, ((Voi*)voi2)->voiname );
  if(res!=0) return(res);
  res=strcasecmp( ((Voi*)voi1)->hemisphere, ((Voi*)voi2)->hemisphere );
  return(res);
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace NA's in basic DFT data with interpolated values.
    If extrapolation is necessary, then the values (0,0) and
    (Infinity,last measured) are assumed.
\return Returns 0, if NA's could be filled with sensible values.
 */
int dftNAfill(DFT *dft)
{
  int ri, fi, fj;
  double x1, x2, y1, y2, x, y;

  if(dft->voiNr<1 || dft->frameNr<1) return(1);
  for(ri=0; ri<dft->voiNr; ri++) for(fi=0; fi<dft->frameNr; fi++) {
    if(dft->x[fi]==NA) return(2);
    if(dft->voi[ri].y[fi]==NA) {
      /* NA's before zero time are always replaced with 0 */
      if(dft->x[fi]<0.0) {dft->voi[ri].y[fi]=0.0; continue;}
      x=dft->x[fi];
      /* Get the previous data that is not NA */
      for(x1=y1=NA, fj=fi-1; fj>=0; fj--) if(dft->voi[ri].y[fj]!=NA) {
        x1=dft->x[fj]; y1=dft->voi[ri].y[fj]; break;
      }
      if(x1==NA || y1==NA) x1=y1=0.0;
      /* Get the following data that is not NA */
      for(x2=y2=NA, fj=fi+1; fj<dft->frameNr; fj++) if(dft->voi[ri].y[fj]!=NA) {
        x2=dft->x[fj]; y2=dft->voi[ri].y[fj]; break;
      }
      if(x2==NA || y2==NA) for(fj=fi-1; fj>=0; fj--) if(dft->voi[ri].y[fj]!=NA) {
        x2=dft->x[fj]; y2=dft->voi[ri].y[fj]; break;
      }
      if(x2==NA || y2==NA) return(2);
      /* Calculate new value */
      if(x2==x1) y=0.5*(y1+y2); else y=y2-(x2-x)*(y2-y1)/(x2-x1);
      dft->voi[ri].y[fi]=y;
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the lowest activity value in DFT */
double dft_kBqMin(DFT *data) {

  int i, j;
  double min=1e+99;

  for(i=0; i<data->voiNr ;i++) {
    for(j=0; j<data->frameNr; j++) {
      if(data->voi[i].y[j]!=NA && data->voi[i].y[j]<min)
	min=data->voi[i].y[j];
    }
  }
  return (min);
}
/*****************************************************************************/

/*****************************************************************************/
/** Returns the highest activity value in DFT */
double dft_kBqMax(DFT *data) {

  int i, j;
  double max=-1e+99;

  for(i=0; i<data->voiNr ;i++){
    for(j=0; j<data->frameNr; j++){
      if(data->voi[i].y[j]!=NA && data->voi[i].y[j]>max)
	max=data->voi[i].y[j];
    }
  }
  return(max);
}
/*****************************************************************************/

/*****************************************************************************/
/** Sorts TAC frames by increasing sample time.
\return Returns 0 if ok.
 */
int dftSortByFrame(DFT *dft)
{
  int ri, fi, fj;
  double d;

  if(dft==NULL || dft->voiNr<1 || dft->frameNr<1) return(1);
  for(fi=0; fi<dft->frameNr-1; fi++) for(fj=fi+1; fj<dft->frameNr; fj++) {
    if(dft->x[fj]>=dft->x[fi]) continue;
    d=dft->x[fi];  dft->x[fi]=dft->x[fj];   dft->x[fj]=d;
    d=dft->x1[fi]; dft->x1[fi]=dft->x1[fj]; dft->x1[fj]=d;
    d=dft->x2[fi]; dft->x2[fi]=dft->x2[fj]; dft->x2[fj]=d;
    d=dft->w[fi];  dft->w[fi]=dft->w[fj];   dft->w[fj]=d;
    for(ri=0; ri<dft->voiNr; ri++) {
      d=dft->voi[ri].y[fi]; dft->voi[ri].y[fi]=dft->voi[ri].y[fj]; dft->voi[ri].y[fj]=d;
      d=dft->voi[ri].y2[fi]; dft->voi[ri].y2[fi]=dft->voi[ri].y2[fj]; dft->voi[ri].y2[fj]=d;
      d=dft->voi[ri].y3[fi]; dft->voi[ri].y3[fi]=dft->voi[ri].y3[fj]; dft->voi[ri].y3[fj]=d;
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write DFT contents in HTML table format
    If file exists, a backup file (%) is written also.
    If "stdout" is given as filename, output is directed to stdout
    In case of an error, description is written in dfterrmsg.
\return Returns 0 if ok.
*/
int dftWriteHTML(
  /** Input DFT */
  DFT *dft,
  /** HTML filename */
  char *fname,
  /** Table orientation: 1=original, 2=transposed */
  int orientation
) {
  int ri, fi, is_stdout=0, ret;
  char tmp[FILENAME_MAX];
  FILE *fp;


  /* Check input */
  strcpy(dfterrmsg, "invalid input to dftWriteHTML()");
  if(dft==NULL || dft->frameNr<1 || dft->voiNr<1) return(1);
  if(fname==NULL || strlen(fname)<1) return(1);
  strcpy(dfterrmsg, "");
  /* Check if writing to stdout */
  if(!strcasecmp(fname, "stdout")) is_stdout=1;

  /* Check if file exists; backup, if necessary */
  if(!is_stdout && access(fname, 0) != -1) {
    strcpy(tmp, fname); strcat(tmp, "%"); rename(fname, tmp);}
  strcpy(dfterrmsg, "cannot write file");

  /* Open output file */
  if(is_stdout) fp=(FILE*)stdout;
  else if((fp=fopen(fname, "w"))==NULL) {
    strcpy(dfterrmsg, "cannot open file"); return(2);}

  /* Write HTML header */
  ret=fprintf(fp, "<html><head><title>PET data</title></head>\n");
  if(ret==0) {
    strcpy(dfterrmsg, "disk full");
    if(!is_stdout) fclose(fp); return(3);
  }

  /* Start writing the body of the HTML file */
  fprintf(fp, "<body>\n");

  /* Write the title lines, if there are titles, into 1st table */
  if(dft->_type>0 && orientation!=2) {
    fprintf(fp, "<table border=\"1\" width=\"95%%\">\n");
    fprintf(fp, "<thead align=\"left\">\n");
    /* Empty cell and Region names */
    fprintf(fp, "<tr><td></td>\n");
    for(ri=0; ri<dft->voiNr; ri++)
      fprintf(fp, "<th>%s</th>\n", dft->voi[ri].voiname);
    fprintf(fp, "</tr>\n");
    /* Study number and Hemispheres */
    fprintf(fp, "<tr><th>%s</th>\n", dft->studynr);
    for(ri=0; ri<dft->voiNr; ri++)
      fprintf(fp, "<th>%s</th>\n", dft->voi[ri].hemisphere);
    fprintf(fp, "</tr>\n");
    /* Unit and Places */
    fprintf(fp, "<tr><th>%s</th>\n", dft->unit);
    for(ri=0; ri<dft->voiNr; ri++)
      fprintf(fp, "<th>%s</th>\n", dft->voi[ri].place);
    fprintf(fp, "</tr>\n");
    /* Time unit and Volumes */
    switch(dft->timeunit) {
      case 0: strcpy(tmp, "min"); break;
      case 1: strcpy(tmp, "sec"); break;
      case 11: strcpy(tmp, "um"); break;
      case 12: strcpy(tmp, "mm"); break;
      default: strcpy(tmp, "min"); break;
    }
    fprintf(fp, "<tr><th>%s</th>\n", tmp);
    for(ri=0; ri<dft->voiNr; ri++)
      fprintf(fp, "<th>%g</th>\n", dft->voi[ri].size);
    fprintf(fp, "</tr>\n");
    /* End the title table */
    fprintf(fp, "</thead>\n");
  }
  if(dft->_type>0 && orientation==2) { /* Titles for transposed table */
    fprintf(fp, "<table border=\"1\" width=\"95%%\">\n");
    fprintf(fp, "<thead align=\"left\">\n");
    /* Study number and Unit */
    fprintf(fp, "<tr><th>%s</th>\n", dft->studynr);
    fprintf(fp, "<th>%s</th></tr>\n", dft->unit);
    /* End the title table */
    fprintf(fp, "</theads>\n");
  }

  /* Write the DFT data into body of the table */
  fprintf(fp, "<tbody align=\"left\">\n");
  /* If transposed, write titles */
  if(orientation==2) {
    fprintf(fp, "<tr><th>Region</th><th>Hemisphere</th><th>Plane</th>\n");
    fprintf(fp, "<th>Volume</th></tr>\n");
  }
  if(orientation!=2) { /* Not transposed */
    for(fi=0; fi<dft->frameNr; fi++) {
      /* Time */
      fprintf(fp, "<tr><th>%g</th>\n", dft->x[fi]);
      /* Values */
      for(ri=0; ri<dft->voiNr; ri++)
        fprintf(fp, "<th>%g</th>", dft->voi[ri].y[fi]);
      fprintf(fp, "</tr>\n");
    }
  } else { /* transposed */
    for(ri=0; ri<dft->voiNr; ri++) {
      /* Region names and volume */
      fprintf(fp, "<tr><th>%s</th><th>%s</th><th>%s</th>\n",
        dft->voi[ri].voiname, dft->voi[ri].hemisphere, dft->voi[ri].place);
      fprintf(fp, "<td>%g</td>\n", dft->voi[ri].size);
      /* Values */
      for(fi=0; fi<dft->frameNr; fi++) {
        fprintf(fp, "<td>%g</td>", dft->voi[ri].y[fi]);
      }
      fprintf(fp, "</tr>\n");
    }
  }
  /* End the table body and table */
  fprintf(fp, "</tbody></table>\n");

  /* Stop writing the body of the HTML file, and end the file */
  ret=fprintf(fp, "</body></html>\n\n");
  if(ret==0) {
    strcpy(dfterrmsg, "disk full");
    if(!is_stdout) fclose(fp); return(3);
  }

  /* Close file */
  if(!is_stdout) fclose(fp);
  strcpy(dfterrmsg, "");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read certain keys from IFT and set DFT fields accordingly.
\return Returns the nr of identified keys.
 */
int dft_fill_hdr_from_IFT(
  DFT *dft,
  IFT *ift
) {
  int ki, ri, ok_nr=0;
  char keystr[256], *cptr;


  /* Check for study number */
  strcpy(keystr, "studynr"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "study number"); ki=iftGet(ift, keystr);}
  if(ki==-1) {strcpy(keystr, "study_number"); ki=iftGet(ift, keystr);}
  if(ki>=0) {
    strncpy(dft->studynr, ift->item[ki].value, MAX_STUDYNR_LEN);
    dft->studynr[MAX_STUDYNR_LEN]=(char)0;
    ok_nr++;
  }

  /* Check for time unit */
  strcpy(keystr, "timeunit"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "time unit"); ki=iftGet(ift, keystr);}
  if(ki==-1) {strcpy(keystr, "time_unit"); ki=iftGet(ift, keystr);}
  if(ki==-1) {strcpy(keystr, "Time units"); ki=iftGet(ift, keystr);}
  if(ki>=0) {
    if(strcasecmp(ift->item[ki].value, "min")==0) {
      dft->timeunit=0; ok_nr++;
    } else if(strcasecmp(ift->item[ki].value, "sec")==0) {
      dft->timeunit=1; ok_nr++;
    } else if(strcasecmp(ift->item[ki].value, "um")==0) {
      dft->timeunit=11; ok_nr++;
    } else if(strcasecmp(ift->item[ki].value, "mm")==0) {
      dft->timeunit=12; ok_nr++;
    }
  }

  /* Check for sample unit */
  strcpy(keystr, "unit"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "Activity units"); ki=iftGet(ift, keystr);}
  if(ki>=0) {
    strncpy(dft->unit, ift->item[ki].value, 12); dft->unit[12]=(char)0;
    ok_nr++;
  }

  /* Check for region names */
  ri=0;
  do {
    strcpy(keystr, "voiname"); ki=iftGetNth(ift, keystr, ri+1);
    if(ki>=0) {
      strncpy(dft->voi[ri].name, ift->item[ki].value, MAX_REGIONNAME_LEN);
      dft->voi[ri].name[MAX_REGIONNAME_LEN]=(char)0;
      rnameSplit(ift->item[ki].value, dft->voi[ri].voiname,
        dft->voi[ri].hemisphere, dft->voi[ri].place, 6);
    }
    ri++; ok_nr++;
  } while(ki>=0 && ri<dft->_voidataNr);

  /* Check for region volumes */
  strcpy(keystr, "sizes"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "volumes"); ki=iftGet(ift, keystr);}
  if(ki>=0 && strlen(ift->item[ki].value)) {
    ri=0; cptr=strtok(ift->item[ki].value, " ;\t");
    while(cptr!=NULL && ri<dft->_voidataNr) {
      if(strcmp(cptr, ".")==0) {ri++; continue;}
      dft->voi[ri++].size=atof_dpi(cptr);
      cptr=strtok(NULL, " ;\t");
    }
    ok_nr++;
  }

  /* Check for the name of radiopharmaceutical */
  strcpy(keystr, "radiopharmaceutical"); ki=iftGet(ift, keystr);
  if(ki>=0 && strlen(ift->item[ki].value)) {
    strncpy(dft->radiopharmaceutical, ift->item[ki].value, 31);
    dft->radiopharmaceutical[31]=(char)0;
    ok_nr++;
  }

  /* Check for isotope name */
  strcpy(keystr, "isotope"); ki=iftGet(ift, keystr);
  if(ki>=0 && strlen(ift->item[ki].value)) {
    strncpy(dft->isotope, ift->item[ki].value, 6); dft->isotope[6]=(char)0;
    ok_nr++;
  }

  /* Check if there is anything about correction for physical decay */
  strcpy(keystr, "decay_correction"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "decay correction"); ki=iftGet(ift, keystr);}
  if(ki>=0 && strlen(ift->item[ki].value)) {
    if(strncasecmp(ift->item[ki].value, "Yes", 1)==0) {
      dft->decayCorrected=1; ok_nr++;
    } else if(strncasecmp(ift->item[ki].value, "No", 1)==0) {
      dft->decayCorrected=2; ok_nr++;
    }
  }

  /* Check for injection time */
  strcpy(keystr, "injection time"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "injection_time"); ki=iftGet(ift, keystr);}
  if(ki>=0 && strlen(ift->item[ki].value)) {
    /* check if time is in correct format; otherwise try to correct it */
    if(ift->item[ki].value[4]=='-' && ift->item[ki].value[7]=='-' &&
       ift->item[ki].value[13]==':' && ift->item[ki].value[16]==':') {
      strncpy(dft->injectionTime, ift->item[ki].value, 19);
      dft->injectionTime[19]=(char)0; ok_nr++;
    } else if(ift->item[ki].value[2]=='.' && ift->item[ki].value[5]=='.' &&
              ift->item[ki].value[13]==':' && ift->item[ki].value[16]==':') {
      sprintf(dft->injectionTime, "%4.4s-%2.2s-%2.2s %8.8s",
        ift->item[ki].value+6, ift->item[ki].value+3, ift->item[ki].value,
        ift->item[ki].value+11);
      ok_nr++;
    }
  }

  /* Check for scan start time */
  strcpy(keystr, "scan start time"); ki=iftGet(ift, keystr);
  if(ki==-1) {strcpy(keystr, "scan_start_time"); ki=iftGet(ift, keystr);}
  if(ki==-1) {strcpy(keystr, "scan_start"); ki=iftGet(ift, keystr);}
  if(ki==-1) {strcpy(keystr, "scan start"); ki=iftGet(ift, keystr);}
  if(ki>=0 && strlen(ift->item[ki].value)) {
    /* check if time is in correct format; otherwise try to correct it */
    if(ift->item[ki].value[4]=='-' && ift->item[ki].value[7]=='-' &&
       ift->item[ki].value[13]==':' && ift->item[ki].value[16]==':') {
      strncpy(dft->scanStartTime, ift->item[ki].value, 19);
      dft->scanStartTime[19]=(char)0; ok_nr++;
    } else if(ift->item[ki].value[2]=='.' && ift->item[ki].value[5]=='.' &&
              ift->item[ki].value[13]==':' && ift->item[ki].value[16]==':') {
      sprintf(dft->scanStartTime, "%4.4s-%2.2s-%2.2s %8.8s",
        ift->item[ki].value+6, ift->item[ki].value+3, ift->item[ki].value,
        ift->item[ki].value+11);
      ok_nr++;
    }
  }

  return(ok_nr);
}
/*****************************************************************************/

/*****************************************************************************/

