/* Does this need to be in libtpcmigio ??? */
/******************************************************************************
  Copyright (c) 2003,2004 by Turku PET Centre

  imgproc.c
  
  Additional procedures for processing 4D PET image data.
  
  Version:
  2003-07-01 Vesa Oikonen
  2003-07-07 VO
      Included functions imgMax() and imgSetScanner().
  2003-07-27 VO
      Included functions imgArithm(), imgArithmFrame() and imgArithmConst().
  2003-08-04 VO
      imgSetScanner() supports HR+.
  2003-09-24 VO
      Changed parameters for GE Advance in imgSetScanner()
      as requested by Roman Krais:
      -axialFOV 150 -> 153 (18 rings * 8.5 mm Ring spacing)
      -sampleDistance 1.95730 -> 1.970177 (from GE email).
  2003-09-30 VO
      Included function imgAbsMax().
  2003-10-04 VO
      Included function imgMinMax().
  2003-10-13 VO
      Included function imgFrameIntegral();
  2003-12-03 VO
      Including HRRT in imgSetScanner(), but only pixel size for now.
      Included image flip functions.
  2003-12-14 VO
      imgSetScanner() sets scanner in IMG.
  2003-12-18 VO
      Included function imgFrameMinMax().
  2004-03-18 VO
      HRRT parameters set in imgSetScanner() (from PN).
  2004-06-23 VO
      Also HRRT pixel size set based on image dimension and zoom.
  2004-07-18 VO
      imgAverageTAC replaced (in practise) by new imgAverageTemplateTAC().
      Included function imgThresholdTemplate().
  2004-07-20 VO
      Included function imgThresholdByTemplate().
      Included function imgRawCountsPerTime().
      More doxygen-style comments.
  2004-09-20 VO
      Doxygen style comments are corrected.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "img.h"
#include "imgproc.h"
/*****************************************************************************/

/*****************************************************************************/
/** Simple arithmetics between matching IMG planes and frames.
    Spefify the operation as one of characters +, -, /, :, *, ., x.
    Results that are higher than ulimit are set to ulimit.
\return Returns 0, if ok.
 */
int imgArithm(
  /** The first IMG data  */
  IMG *img1,
  /** The second IMG data */
  IMG *img2,
  /** Operation, one of the characters +, -, /, :, *, ., x  */
  char operation,
  /** Results that are higher than ulimit are set to ulimit */
  float ulimit
) {
  int pi, yi, xi, fi;

  /* Check the arguments */
  if(img1->status!=IMG_STATUS_OCCUPIED || img2->status!=IMG_STATUS_OCCUPIED)
    return(1);
  if(img1->dimx!=img2->dimx || img1->dimy!=img2->dimy) return(1);
  /* Check the plane numbers */
  if(img1->dimz!=img2->dimz) return(2);
  for(pi=0; pi<img1->dimz; pi++)
    if(img1->planeNumber[pi]!=img2->planeNumber[pi]) return(2);
  /* Check the frame numbers */
  if(img1->dimt!=img2->dimt) return(3);

  /* Operate */
  switch(operation) {
    case '+':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]+=img2->m[pi][yi][xi][fi];
      break;
    case '-':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]-=img2->m[pi][yi][xi][fi];
      break;
    case '/':
    case ':':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++) {
          if(fabs(img2->m[pi][yi][xi][fi])>1.0E-5)
            img1->m[pi][yi][xi][fi]/=img2->m[pi][yi][xi][fi];
          else img1->m[pi][yi][xi][fi]=0.0;
        }
      break;
    case '*':
    case 'x':
    case '.':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]*=img2->m[pi][yi][xi][fi];
      break;
    default: return(1);
  }

  /* Check for limits */
  if(ulimit>0.0) {
    for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
      for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
        if(img1->m[pi][yi][xi][fi]>ulimit) img1->m[pi][yi][xi][fi]=ulimit;
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Simple arithmetics between IMG and specified constant.
    Spefify the operation as one of characters +, -, /, :, *, ., x.
    Results that are higher than ulimit are set to ulimit.
\return Returns 0, if ok.
 */
int imgArithmConst(
  /** IMG data which is modified */
  IMG *img,
  /** Constant value which is used to modify the data */
  float operand,
  /** Operation, one of the characters +, -, /, :, *, ., x */
  char operation,
  /** Results that are higher than ulimit are set to ulimit */
  float ulimit
) {
  int pi, yi, xi, fi;

  /* Check the arguments */
  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  switch(operation) {
    case '+':
      for(pi=0; pi<img->dimz; pi++) for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++) for(fi=0; fi<img->dimt; fi++)
          img->m[pi][yi][xi][fi]+=operand;
      break;
    case '-':
      for(pi=0; pi<img->dimz; pi++) for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++) for(fi=0; fi<img->dimt; fi++)
          img->m[pi][yi][xi][fi]-=operand;
      break;
    case '/':
    case ':':
      if(fabs(operand)<1.0e-100) return(2);
      for(pi=0; pi<img->dimz; pi++) for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++) for(fi=0; fi<img->dimt; fi++)
          img->m[pi][yi][xi][fi]/=operand;
      break;
    case '*':
    case 'x':
    case '.':
      for(pi=0; pi<img->dimz; pi++) for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++) for(fi=0; fi<img->dimt; fi++)
          img->m[pi][yi][xi][fi]*=operand;
      break;
    default: return(1);
  }

  /* Check for limits */
  if(ulimit>0.0) {
    for(pi=0; pi<img->dimz; pi++) for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++) for(fi=0; fi<img->dimt; fi++)
        if(img->m[pi][yi][xi][fi]>ulimit) img->m[pi][yi][xi][fi]=ulimit;
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Simple arithmetics between matching IMG planes and the first frame of img2.
    Spefify the operation as one of characters +, -, /, :, *, ., x.
    Results that are higher than ulimit are set to ulimit.
\return Returns 0, if ok.
 */
int imgArithmFrame(
  /** IMG data which is modified */
  IMG *img1,
  /** Operand IMG data, only the first frame is used */
  IMG *img2,
  /** Operation, one of the characters +, -, /, :, *, ., x */
  char operation,
  /** Results that are higher than ulimit are set to ulimit */
  float ulimit
) {
  int pi, yi, xi, fi;

  /* Check the arguments */
  if(img1->status!=IMG_STATUS_OCCUPIED || img2->status!=IMG_STATUS_OCCUPIED)
    return(1);
  if(img1->dimx!=img2->dimx || img1->dimy!=img2->dimy) return(1);
  /* Check the plane numbers */
  if(img1->dimz!=img2->dimz) return(2);
  for(pi=0; pi<img1->dimz; pi++)
    if(img1->planeNumber[pi]!=img2->planeNumber[pi]) return(2);

  /* Operate */
  switch(operation) {
    case '+':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]+=img2->m[pi][yi][xi][0];
      break;
    case '-':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]-=img2->m[pi][yi][xi][0];
      break;
    case '/':
    case ':':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) {
          if(fabs(img2->m[pi][yi][xi][0])>1.0E-8)
            for(fi=0; fi<img1->dimt; fi++)
              img1->m[pi][yi][xi][fi]/=img2->m[pi][yi][xi][0];
          else for(fi=0; fi<img1->dimt; fi++) img1->m[pi][yi][xi][fi]=0.0;
        }
      break;
    case '*':
    case 'x':
    case '.':
      for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
        for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
          img1->m[pi][yi][xi][fi]*=img2->m[pi][yi][xi][0];
      break;
    default: return(1);
  }

  /* Check for limits */
  if(ulimit>0.0) {
    for(pi=0; pi<img1->dimz; pi++) for(yi=0; yi<img1->dimy; yi++)
      for(xi=0; xi<img1->dimx; xi++) for(fi=0; fi<img1->dimt; fi++)
        if(img1->m[pi][yi][xi][fi]>ulimit) img1->m[pi][yi][xi][fi]=ulimit;
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Creates a template image based on lower and upper threshold values.
    This function allocates memory for the template.
    If pixel value in original image is >=minValue and <=maxValue, the
    corresponding template pixel is set to 1, otherwise to 0.
    Only the first frame of images is applied.
\return Returns 0 if ok.
 */
int imgThresholdTemplate(
  /** Original image */
  IMG *img,
  /** Lower threshold */
  float minValue,
  /** Upper threshold */
  float maxValue,
  /** Template image */
  IMG *timg
) {
  int fi=0, zi, xi, yi, ret;

  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  /* Allocate memory for the template data */
  ret=imgAllocate(timg, img->dimz, img->dimy, img->dimx, 1);
  if(ret) return(ret);

  /* set template header information */
  imgCopyhdr(img, timg);
  timg->start[0]=img->start[0]; timg->end[0]=img->end[img->dimt-1];
  timg->mid[0]=(timg->start[0]+timg->end[0])/2.0;

  /* Make template */
  fi=0;
  for(zi=0; zi<img->dimz; zi++)
    for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++) {
      if(img->m[zi][yi][xi][fi]<minValue || img->m[zi][yi][xi][fi]>maxValue) 
        timg->m[zi][yi][xi][fi]=0.0; else timg->m[zi][yi][xi][fi]=1.0;
    }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Threshold IMG by a template.
    Sets pixel values in img to thrValue, if corresponding pixel value in
    template is == 0. Only first plane of template is used.
\return Returns 0, if ok.
 */
int imgThresholdByTemplate(
  /** Image to threshold */
  IMG *img,
  /** Threshold template (1 frame) where 0=cut off */
  IMG *template,
  /** Value which is written in cut off pixels */
  float thrValue
) {
  int fi, zi, xi, yi;

  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(template->status!=IMG_STATUS_OCCUPIED) return(2);
  for(zi=0; zi<img->dimz; zi++)
    for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++)
      if(template->m[zi][yi][xi][0]==0.0) {  
        for(fi=0; fi<img->dimt; fi++) img->m[zi][yi][xi][fi]=thrValue;
      }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates an average time-activity curve of all pixels or bins in the
    specified IMG data.
\return Returns 0, if ok.
 */
int imgAverageTAC(IMG *img, float *tac)
{
  return(imgAverageTemplateTAC(img, NULL, tac));
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates an average time-activity curve of pixels or bins in the
    specified IMG data.
    Template image specifies the pixels that are included in average
    (0=excluded, otherwise included).
    If all pixels are to be averaged, then NULL can be given instead of
    template.
\return Returns 0, if ok.
 */
int imgAverageTemplateTAC(
  /** (Dynamic) IMG data from which cluster TAC is computed */
  IMG *img,
  /** Template: 0=excluded, otherwise included */
  IMG *timg,
  /** Allocated float array for the TAC */
  float *tac
) {
  int zi, yi, xi, fi, pxlNr;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(tac==NULL) return(2);
  if(timg!=NULL && timg->status<IMG_STATUS_OCCUPIED) return(3);
  pxlNr=img->dimz*img->dimy*img->dimx; if(pxlNr<1) return(4);
  for(fi=0; fi<img->dimt; fi++) {
    tac[fi]=0.0; if(timg!=NULL) pxlNr=0;
    for(zi=0; zi<img->dimz; zi++) {
      for(yi=0; yi<img->dimy; yi++) for(xi=0; xi<img->dimx; xi++)
        if(timg==NULL)
          tac[fi]+=img->m[zi][yi][xi][fi];
        else if(timg->m[zi][yi][xi][0]!=0.0) {
          tac[fi]+=img->m[zi][yi][xi][fi]; pxlNr++;}
    }
    if(pxlNr>0) tac[fi]/=(float)pxlNr; else return(5);
    printf("pxlNr=%d/%d\n", pxlNr, img->dimz*img->dimy*img->dimx);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates the Area-Under-Curve of an average time-activity curve of all
    pixels or bins in the specified IMG data.
\return Returns 0, if ok.
 */
int imgAverageAUC(
  /** (Dynamic) IMG data */
  IMG *img,
  /** Allocate float array for the AUC-TAC */
  float *avgauc
) {
  int pi, yi, xi, fi, pxlNr;
  float fv, fl;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(avgauc==NULL) return(2);
  *avgauc=0.0;
  pxlNr=img->dimz*img->dimy*img->dimx; if(pxlNr<1) return(3);
  for(fi=0; fi<img->dimt; fi++) {
    fv=0.0; fl=img->end[fi]-img->start[fi]; if(fl<=0.0) return(4);
    for(pi=0; pi<img->dimz; pi++)
      for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++)
          fv+=img->m[pi][yi][xi][fi];
    fv/=(float)pxlNr;
    *avgauc+=fv*fl;
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace IMG data values by their log10 values.
\return Returns 0, if ok.
 */
int imgLog10(IMG *img)
{
  int pi, yi, xi, fi;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(img->dimt<1 || img->dimz<1 || img->dimy<1 || img->dimx<1) return(2);
  for(pi=0; pi<img->dimz; pi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++)
        for(fi=0; fi<img->dimt; fi++) {
          if(img->m[pi][yi][xi][fi]<=0.0) img->m[pi][yi][xi][fi]=0.0;
          else img->m[pi][yi][xi][fi]=log10(img->m[pi][yi][xi][fi]);
        }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searche the max pixel value in the IMG data.
\return Returns 0, if ok.
 */
int imgMax(IMG *img, float *maxvalue)
{
  int pi, yi, xi, fi;
  float f;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(maxvalue==NULL) return(2); else *maxvalue=0.0;
  if(img->dimt<1 || img->dimz<1 || img->dimy<1 || img->dimx<1) return(3);
  f=img->m[0][0][0][0];
  for(pi=0; pi<img->dimz; pi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++)
        for(fi=0; fi<img->dimt; fi++) {
          if(img->m[pi][yi][xi][fi]>f) f=img->m[pi][yi][xi][fi];
        }
  *maxvalue=f;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches the max absolute pixel value in the IMG data.
    Sets maxvalue to the absolute max value with sign.
\return Returns 0, if ok.
 */
int imgAbsMax(IMG *img, float *maxvalue)
{
  int pi, yi, xi, fi;
  float f;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(maxvalue==NULL) return(2); else *maxvalue=0.0;
  if(img->dimt<1 || img->dimz<1 || img->dimy<1 || img->dimx<1) return(3);
  f=img->m[0][0][0][0];
  for(pi=0; pi<img->dimz; pi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++)
        for(fi=0; fi<img->dimt; fi++) {
          if(fabs(img->m[pi][yi][xi][fi])>fabs(f)) f=img->m[pi][yi][xi][fi];
        }
  *maxvalue=f;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches the min and max pixel value in the IMG data.
\return Returns 0, if ok.
 */
int imgMinMax(IMG *img, float *minvalue, float *maxvalue)
{
  int pi, yi, xi, fi;
  float mi, ma;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(minvalue==NULL || maxvalue==NULL) return(2);
  *minvalue=*maxvalue=0.0;
  if(img->dimt<1 || img->dimz<1 || img->dimy<1 || img->dimx<1) return(3);
  mi=ma=img->m[0][0][0][0];
  for(pi=0; pi<img->dimz; pi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++)
        for(fi=0; fi<img->dimt; fi++) {
          if(img->m[pi][yi][xi][fi]>ma) ma=img->m[pi][yi][xi][fi];
          else if(img->m[pi][yi][xi][fi]<mi) mi=img->m[pi][yi][xi][fi];
        }
  *minvalue=mi; *maxvalue=ma;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Searches the min and max pixel value in one frame (1..dimt) of the IMG data.
\return Returns 0, if ok.
 */
int imgFrameMinMax(IMG *img, int frame, float *minvalue, float *maxvalue)
{
  int pi, yi, xi, fi;
  float mi, ma;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  if(minvalue==NULL || maxvalue==NULL) return(2);
  *minvalue=*maxvalue=0.0; fi=frame-1;
  if(img->dimt<frame || img->dimz<1 || img->dimy<1 || img->dimx<1) return(3);
  if(frame<1) return(4);
  mi=ma=img->m[0][0][0][fi];
  for(pi=0; pi<img->dimz; pi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++) {
        if(img->m[pi][yi][xi][fi]>ma) ma=img->m[pi][yi][xi][fi];
        else if(img->m[pi][yi][xi][fi]<mi) mi=img->m[pi][yi][xi][fi];
      }
  *minvalue=mi; *maxvalue=ma;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Integration from first frame (0..last) to last frame (first..dimt) to iimg,
    which is allocated here.
    Frames do not have to be continuous in time. Time unit in integral is sec.
    Raw data (sinogram) must be divided by frame durations before calling this.
\return Returns 0 if ok, otherwise >0.
*/
int imgFrameIntegral(
  /** (Dynamic) IMG data */
  IMG *img,
  /** First frame to include in AUC; 0..dimt-1 */
  int first,
  /** Last frame to include in AUC; 0..dimt-1 */
  int last,
  /** Pointer to initiated and empty AUC IMG data */
  IMG *iimg
) {
  int zi, yi, xi, fi, ret;
  float fstart, fend, dur, x, y, k;

  /* Check the arguments */
  if(img==NULL || iimg==NULL || first<0 || first>last) return(1);
  if(img->status!=IMG_STATUS_OCCUPIED) return(2);
  fstart=img->start[first]; fend=img->end[last];

  /* Allocate memory for the integral */
  imgEmpty(iimg);
  ret=imgAllocate(iimg, img->dimz, img->dimy, img->dimx, 1);
  if(ret) {imgEmpty(iimg); return(3);}

  /* Copy header information */
  imgCopyhdr(img, iimg);
  if(img->type==IMG_TYPE_IMAGE && img->unit==3) iimg->unit=4;
  iimg->start[0]=fstart; iimg->end[0]=fend;
  iimg->mid[0]=0.5*(iimg->start[0]+iimg->end[0]);
  if(img->type==IMG_TYPE_RAW) iimg->decayCorrected=0;
  else iimg->decayCorrected=1;
  iimg->decayCorrFactor[0]=0.0;

  /* Integrate the first frame */
  fi=first;
  dur=img->end[fi]-img->start[fi]; if(dur<0.0) {imgEmpty(iimg); return(4);}
  for(zi=0; zi<img->dimz; zi++)
    for(yi=0; yi<img->dimy; yi++)
      for(xi=0; xi<img->dimx; xi++)
        iimg->m[zi][yi][xi][0]=dur*img->m[zi][yi][xi][fi];
  /* Add integrals of the following frames */
  for(fi=first+1; fi<=last; fi++) {
    dur=img->end[fi]-img->start[fi]; if(dur<0.0) {imgEmpty(iimg); return(4);}
    /* Add the integral of this frame to the previous integral */
    for(zi=0; zi<img->dimz; zi++)
      for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++)
          iimg->m[zi][yi][xi][0]+=dur*img->m[zi][yi][xi][fi];
    /* Check whether frames are contiguous */
    dur=img->start[fi]-img->end[fi-1]; if(dur<=1.0E-10) continue;
    /* When not, calculate the integral between frames */
    x=0.5*(img->start[fi]+img->end[fi-1]);
    for(zi=0; zi<img->dimz; zi++)
      for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++) {
          k=(img->m[zi][yi][xi][fi]-img->m[zi][yi][xi][fi-1])
            /(img->mid[fi]-img->mid[fi-1]);
          y=img->m[zi][yi][xi][fi-1]+k*(x-img->mid[fi-1]);
          iimg->m[zi][yi][xi][0]+=dur*y;
        }    
  } /* next frame */

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Divide or multiply raw data (sinogram) counts by frame duration.
    If IMG is not raw data, division is quietly not done.
\return Returns 0 if ok.
*/
int imgRawCountsPerTime(
  /** IMG containing raw data */
  IMG *img,
  /** 1=division, 0=multiply  */
  int operation
) {
  int fi, zi, xi, yi;
  float f;

  /* Check the input data */
  if(operation!=0 && operation!=1) return(1);
  if(img==NULL) return(2);
  if(img->status!=IMG_STATUS_OCCUPIED) return(3);
  if(img->type!=IMG_TYPE_RAW) return(0);
  for(fi=0; fi<img->dimt; fi++) {
    f=img->end[fi]-img->start[fi]; if(f<=1.0E-12) return(4);
  }
  for(fi=0; fi<img->dimt; fi++) {
    f=img->end[fi]-img->start[fi]; if(operation==1) f=1.0/f;
    for(zi=0; zi<img->dimz; zi++)
      for(yi=0; yi<img->dimy; yi++)
        for(xi=0; xi<img->dimx; xi++)
          img->m[zi][yi][xi][fi]*=f;
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data horizontally (left-right) */
void imgFlipHorizontal(IMG *img)
{
  int zi, yi, from, to;
  float *col_ptr;

  for(zi=0; zi<img->dimz; zi++) {
    for(yi=0; yi<img->dimy; yi++) {
      for(from=0, to=img->dimx-1; from<to; from++, to--) {
        col_ptr=img->m[zi][yi][from];
        img->m[zi][yi][from]=img->m[zi][yi][to];
        img->m[zi][yi][to]=col_ptr;
      }
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data vertically (up-down) */
void imgFlipVertical(IMG *img)
{
  int zi, from, to;
  float **row_ptr;

  for(zi=0; zi<img->dimz; zi++) {
    for(from=0, to=img->dimy-1; from<to; from++, to--) {
      row_ptr=img->m[zi][from];
      img->m[zi][from]=img->m[zi][to];
      img->m[zi][to]=row_ptr;
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Flip IMG data planes (head-toes). To work properly, the plane numbers
    must be contiguous.
 */
void imgFlipPlanes(IMG *img)
{
  int from, to;
  float ***plane_ptr;

  for(from=0, to=img->dimz-1; from<to; from++, to--) {
    plane_ptr=img->m[from];
    img->m[from]=img->m[to];
    img->m[to]=plane_ptr;
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Sets scanner specific parameters in IMG data.
    If possible, set image zoom before calling this.
\return Returns 0, if ok.
 */
int imgSetScanner(
  /** IMG data which is filled with scanner specific information */
  IMG *img,
  /** SCANNER_ECAT931, SCANNER_ADVANCE, SCANNER_HRPLUS, SCANNER_HRRT */
  int scanner_type
) {
  int rayNr;
  
  if(img->status<IMG_STATUS_OCCUPIED) return(1);
  img->scanner=scanner_type;
  /* Set zoom to 1.0, if it is not set */
  if(img->zoom<=0.0) img->zoom=1.0;
  /* Then the others */
  if(scanner_type==SCANNER_ECAT931) {
    rayNr=192;
    img->axialFOV=108.;
    img->transaxialFOV=600.826;
    img->sampleDistance=3.12932;
    img->sizez=6.75;
  } else if(scanner_type==SCANNER_ADVANCE) {
    rayNr=281;
    img->axialFOV=153.;
    img->transaxialFOV=550.;
    img->sampleDistance=1.970177;
    img->sizez=4.25;
  } else if(scanner_type==SCANNER_HRPLUS) {
    rayNr=288;
    img->axialFOV=155.2;
    img->transaxialFOV=583.;
    img->sampleDistance=2.25; /* bin size */
    img->sizez=2.425;
  } else if(scanner_type==SCANNER_HRRT) {
    rayNr=256;
    img->axialFOV=252.28;
    img->transaxialFOV=312.;
    img->sampleDistance=1.08; /* bin size */
    img->sizez=img->sizex=img->sizey=1.218750;
  } else
    return(2);
  /* If this is image, then set also pixel sizes */
  if(img->type==IMG_TYPE_IMAGE) {
    if(scanner_type!=SCANNER_HRRT) {
      img->sizex=img->sizey=
        img->sampleDistance*(float)rayNr / ( (float)img->dimx*img->zoom );
    } else {
      img->sizex=img->sizey=
        img->transaxialFOV / ( (float)img->dimx*img->zoom );
    }
  }
  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

