/*
  Copyright (c)  2002-2004 by Turku PET Centre

  img.h - definitions for img.c

  Version:
  2002-01-05 Vesa Oikonen
  2002-03-28 VO
    sampleDistance included in IMG structure.
  2002-08-23 VO
    _dataType included in IMG structure.
  2002-12-01 VO
    imgDecayCorrection() can be used also to remove decay correction.
  2002-12-03 VO
    imgCopyhdr() included.
  2002-12-23 VO
    patientName included in IMG structure.
    imgIsotope() included.
    decayCorrFactor[] included in IMG structure.
  2003-09-04 VO
    _fileFormat and its definitions included in IMG structure.
  2003-10-04 VO
    One more _fileFormat definition added (Analyze little endian byte order).
  2003-12-14 VO
    Memory for all pixels is allocated in one chunk;
    pointers are changed accordingly.
    (Patient) orientation included in IMG structure.
    Scanner (type) included in IMG structure.
    Scanner type definitions moved here from imgproc.h.
  2003-12-18 VO
    Added struct IMG_PIXEL and IMG_RANGE.
    Defined function imgExtractRange().
  2004-05-23 VO
    Added comments.
  2004-07-26 VO
    More comments.
  2004-08-23 VO
    MAX_STUDYNR_LEN applied.


*/
#ifndef _IMG_H
#define _IMG_H
/*===========================================================================*/
#include "studynr.h"
/*===========================================================================*/
/** Definitions for img status */
#define IMG_STATUS_UNINITIALIZED 0
#define IMG_STATUS_INITIALIZED   1
#define IMG_STATUS_OCCUPIED      2
#define IMG_STATUS_ERROR         3
/** Definitions for image type */
#define IMG_TYPE_IMAGE 1
#define IMG_TYPE_RAW   2
/** Definitions for file format */
#define IMG_UNKNOWN   0
#define IMG_E63       1
#define IMG_E7        2
#define IMG_E7_2D     3
#define IMG_ANA       11
#define IMG_ANA_L     12 /* little endian variant */
#define IMG_INTERFILE 21
/** Definition for scanner type */
#define SCANNER_UNKNOWN 0
#define SCANNER_ECAT931 12
#define SCANNER_ADVANCE 12096
#define SCANNER_HRPLUS 3
#define SCANNER_HRRT 4
/*===========================================================================*/
int IMG_TEST;
/*===========================================================================*/
typedef struct {
  /** [1..dimx] */
  int x;
  /** [1..dimy] */
  int y;
  /** [1..dimz] */
  int z;
  /** [1..dimt] */
  int f;
} IMG_PIXEL;
typedef struct {
  /** [1..dimx] */
  int x1, x2;
  /** [1..dimy] */
  int y1, y2;
  /** [1..dimz] */
  int z1, z2;
  /** [1..dimt] */
  int f1, f2;
} IMG_RANGE;
/*===========================================================================*/
  
/*===========================================================================*/
/**
   4D image data structure
 */
typedef struct {

  /*
   *  State of image
   */
  /** Image status */
  char status;
  /** Pointer to _imgStatusMessage, describing current status */
  const char *statmsg;

  /*
   *  Information on the study
   */
  /** for units see imgUnit() in img.c */
  char unit;
  /** study identification code, i.e. (consequental) study number */
  char studyNr[MAX_STUDYNR_LEN+1];
  /** and patient name */
  char patientName[32];
  /** Name of radiopharmaceutical */
  char radiopharmaceutical[32];
  /** Half-life of isotope (sec) */
  float isotopeHalflife;
  /** Decay correction: 0=not corrected, 1=corrected */
  char decayCorrected;
  /** Scan start time and date */
  time_t scanStart;
  /** Patient orientation (see ECAT 7.2 format) */
  int orientation;

  /*
   *  Information on the image
   */
  /** IMG_TYPE_IMAGE, IMG_TYPE_RAW */
  char type;
  /** Reconstruction zoom factor */
  float zoom;
  /** Scanner axial FOV (mm) */
  float axialFOV;
  /** Scanner transaxial FOV (mm) */
  float transaxialFOV;
  /** Scanner sample distance (mm) */
  float sampleDistance;
  /** Pixel size (mm) */
  float sizex;
  /** Pixel size (mm) */
  float sizey;
  /** Pixel size (mm) */
  float sizez;
  /** Saved data type; default 0 is always ok */
  int _dataType;
  /** File format: IMG_UNKNOWN, IMG_E63, IMG_E7, IMG_E7_2D, ...
      default 0 is always ok */
  int _fileFormat;
  /** Scanner type */
  int scanner;

  /*
   *  Image data
   */
  /* Dimensions */
  /** Dimension of Time (t) */
  unsigned short int dimt;
  /** Dimension of Column (c/x) */
  unsigned short int dimx;
  /** Dimension of Row (r/y) */
  unsigned short int dimy;
  /** Dimension of Plane (p/z) */
  unsigned short int dimz;
  /** 'Hidden' pointers for actual data */
  float *_pxl;
  float **_col;
  float ***_row;
  float ****_pln;
  float *_header;
  /* Pointers for data to be used */
  /** Pointer to image data in matrix format m[plane][row][col][frame] */
  float ****m;
  /** Pointer to image data in matrix format plane[plane][row][col][frame] */
  float ****plane;
  /** Pointer to image data in matrix format row[row][col][frame] */
  float ***row;
  /** Pointer to image data in matrix format column[col][frame] */
  float **column;
  /** Pointer to image data in matrix format pixel[frame] */
  float *pixel;
  /** Plane numbers (numbers need not be consequential) */
  int *planeNumber;

  /*
   *  Frame times
   */
  /** Frame start time (sec) */
  float *start;
  /** Frame end time (sec) */
  float *end;
  /** Frame mid time (sec) */
  float *mid;

  /*
   *  Frame weights
   */
  /** Weights: 0=not weighted, 1=weighted, 2=also SD known */
  char isWeight;
  /** Frame weight factor */
  float *weight;
  /** Frame S.D. for weighting */
  float *sd;
  
  /*
   *  Decay correction factors for each frame
   */
  /** Decay correction factor for each frame; included in pixel values */
  float *decayCorrFactor;
                              

} IMG;
/*===========================================================================*/
  
/*===========================================================================*/
/*
 *  Function definitions
 */
/* Initialization and memory handling of img data */
extern void imgInit(IMG *image);
extern void imgEmpty(IMG *image);
extern int imgAllocate(IMG *image, int planes, int rows, int columns, int frames);
/* Retrieving image information */
extern char *imgUnit(int dunit);
extern char *imgIsotope(IMG *img);
extern void imgInfo(IMG *image);
extern int imgDecayCorrection(IMG *img, int mode);
extern int imgCopyhdr(IMG *image1, IMG *image2);
extern int imgExtractRange(IMG *img1, IMG_RANGE r, IMG *img2);
/*===========================================================================*/

/*===========================================================================*/
#endif /* _IMG_H */

