/******************************************************************************
  Copyright (c) 2003,2004 by Turku PET Centre

  vol.c
  
  Procedures for
  - storing and processing 3D PET image volume data (no time information).
  Depends on IMG functions in library.
  
  Version:
  2003-12-18 Vesa Oikonen
    Added 3D structures VOL and SVOL and related functions.
  2004-01-29 VO
    Added functions vol2img() and svol2img.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include <petc99.h>
#include <imgproc.h>
#include "include/img.h"
#include "include/vol.h"
/*****************************************************************************/
/*
 *  Status (error) messages from volume processing
 */
char *_volStatusMessage[] = {
  /*  0 */ "ok",
  /*  1 */ "fault in calling routine",
  /*  2 */ "out of memory"
};
/*****************************************************************************/

/*****************************************************************************/
/** Initiate volume before any use of VOL data; this should be called once.
 */
void volInit(VOL *vol)
{
  if(VOL_TEST) printf("volInit()\n");
  memset(vol, 0, sizeof(VOL));
  vol->status=IMG_STATUS_INITIALIZED;
  vol->statmsg=_volStatusMessage[0];
  vol->orientation=0;
  vol->dimx=vol->dimy=vol->dimz=0;
  vol->sizex=vol->sizey=vol->sizez=0;
  vol->v=(float***)NULL;
  vol->voxel=(float*)NULL;
  vol->column=(float*)NULL;
  vol->row=(float**)NULL;
  vol->plane=(float***)NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Initiate short int volume before any use of SVOL data;
    this should be called once.
 */
void svolInit(SVOL *svol)
{
  if(VOL_TEST) printf("svolInit()\n");
  memset(svol, 0, sizeof(SVOL));
  svol->status=IMG_STATUS_INITIALIZED;
  svol->statmsg=_volStatusMessage[0];
  svol->orientation=0;
  svol->dimx=svol->dimy=svol->dimz=0;
  svol->sizex=svol->sizey=svol->sizez=0;
  svol->scale_factor=1.0;
  svol->v=(short int***)NULL;
  svol->voxel=(short int*)NULL;
  svol->column=(short int*)NULL;
  svol->row=(short int**)NULL;
  svol->plane=(short int***)NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for volume.
 */
void volEmpty(VOL *vol)
{
  if(VOL_TEST) printf("volEmpty()\n");
  if(vol->status<IMG_STATUS_OCCUPIED) return;
  /* Free up memory */
  if(vol->_vxl!=NULL) free(vol->_vxl);
  if(vol->_col!=NULL) free(vol->_col);
  if(vol->_row!=NULL) free(vol->_row);
  if(vol->_pln!=NULL) free(vol->_pln);
  /* Set variables */
  vol->statmsg=_volStatusMessage[0];
  vol->orientation=0;
  vol->dimx=vol->dimy=vol->dimz=0;
  vol->sizex=vol->sizey=vol->sizez=0;
  vol->v=(float***)NULL;
  vol->voxel=(float*)NULL;
  vol->column=(float*)NULL;
  vol->row=(float**)NULL;
  vol->plane=(float***)NULL;
  /* Set status */
  vol->status=IMG_STATUS_INITIALIZED;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for short int volume.
 */
void svolEmpty(SVOL *svol)
{
  if(VOL_TEST) printf("svolEmpty()\n");
  if(svol->status<IMG_STATUS_OCCUPIED) return;
  /* Free up memory */
  if(svol->_vxl!=NULL) free(svol->_vxl);
  if(svol->_col!=NULL) free(svol->_col);
  if(svol->_row!=NULL) free(svol->_row);
  if(svol->_pln!=NULL) free(svol->_pln);
  /* Set variables */
  svol->statmsg=_volStatusMessage[0];
  svol->orientation=0;
  svol->dimx=svol->dimy=svol->dimz=0;
  svol->sizex=svol->sizey=svol->sizez=0;
  svol->scale_factor=1.0;
  svol->v=(short int***)NULL;
  svol->voxel=(short int*)NULL;
  svol->column=(short int*)NULL;
  svol->row=(short int**)NULL;
  svol->plane=(short int***)NULL;
  /* Set status */
  svol->status=IMG_STATUS_INITIALIZED;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for 3D image volume. Returns 0 if ok.
 */
int volAllocate(VOL *vol, int planes, int rows, int columns)
{
  unsigned short int zi, ri;
  int vxlNr, vi;
  float **rptr, *cptr;

  if(VOL_TEST) printf("voiAllocate(*vol, %d, %d, %d)\n", planes, rows, columns);
  /* Check arguments */
  if(vol==NULL) return(1); else vol->statmsg=_volStatusMessage[1];
  if(vol->status==IMG_STATUS_UNINITIALIZED) return(1);
  if(planes<1 || rows<1 || columns<1) return(2);
  vxlNr=planes*rows*columns;

  /* Check if correct volume size is already allocated */
  if(vol->status>=IMG_STATUS_OCCUPIED) {
    if(planes==vol->dimz && rows==vol->dimy && columns==vol->dimx) {
      for(vi=0; vi<vxlNr; vi++) vol->_vxl[vi]=0;
      return(0); /* that's it */
    } else {
      volEmpty(vol);
    }
  }
  /* Allocate memory for volume data */
  vol->_pln=(float***)malloc(planes*sizeof(float**));
  if(vol->_pln==NULL) {
    return(5);}
  vol->_row=(float**)malloc(planes*rows*sizeof(float*));
  if(vol->_row==NULL) {
    free(vol->_pln); return(6);}
  vol->_col=vol->_vxl=(float*)calloc(planes*rows*columns, sizeof(float));
  if(vol->_vxl==NULL) {
    free(vol->_pln); free(vol->_row); return(8);
  }
  /* Set data pointers */
  rptr=vol->_row; cptr=vol->_col;
  for(zi=0; zi<planes; zi++) {
    vol->_pln[zi]=rptr;
    for(ri=0; ri<rows; ri++) {
      *rptr++=cptr; cptr+=columns;
    }
  }
  vol->v=vol->_pln;
  vol->plane=vol->_pln;
  vol->column=vol->_col;
  vol->row=vol->_row;
  vol->voxel=vol->_vxl;
  /* Ok */
  vol->dimz=planes; vol->dimy=rows; vol->dimx=columns;
  vol->statmsg=_volStatusMessage[0];
  vol->status=IMG_STATUS_OCCUPIED;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for 3D short int volume. Returns 0 if ok.
 */
int svolAllocate(SVOL *svol, int planes, int rows, int columns)
{
  unsigned short int zi, ri;
  int vxlNr, vi;
  short int **rptr, *cptr;

  if(VOL_TEST) printf("svoiAllocate(*svol, %d, %d, %d)\n", planes, rows, columns);
  /* Check arguments */
  if(svol==NULL) return(1); else svol->statmsg=_volStatusMessage[1];
  if(svol->status==IMG_STATUS_UNINITIALIZED) return(1);
  if(planes<1 || rows<1 || columns<1) return(2);
  vxlNr=planes*rows*columns;

  /* Check if correct volume size is already allocated */
  if(svol->status>=IMG_STATUS_OCCUPIED) {
    if(planes==svol->dimz && rows==svol->dimy && columns==svol->dimx) {
      for(vi=0; vi<vxlNr; vi++) svol->_vxl[vi]=0;
      return(0); /* that's it */
    } else {
      svolEmpty(svol);
    }
  }
  /* Allocate memory for volume data */
  svol->_pln=(short int***)malloc(planes*sizeof(short int**));
  if(svol->_pln==NULL) {
    return(5);}
  svol->_row=(short int**)malloc(planes*rows*sizeof(short int*));
  if(svol->_row==NULL) {
    free(svol->_pln); return(6);}
  svol->_col=svol->_vxl=(short int*)calloc(planes*rows*columns, sizeof(short int));
  if(svol->_vxl==NULL) {
    free(svol->_pln); free(svol->_row); return(8);
  }
  /* Set data pointers */
  rptr=svol->_row; cptr=svol->_col;
  for(zi=0; zi<planes; zi++) {
    svol->_pln[zi]=rptr;
    for(ri=0; ri<rows; ri++) {
      *rptr++=cptr; cptr+=columns;
    }
  }
  svol->v=svol->_pln;
  svol->plane=svol->_pln;
  svol->column=svol->_col;
  svol->row=svol->_row;
  svol->voxel=svol->_vxl;
  /* Ok */
  svol->dimz=planes; svol->dimy=rows; svol->dimx=columns;
  svol->statmsg=_volStatusMessage[0];
  svol->status=IMG_STATUS_OCCUPIED;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy one time frame (1..dimt) from 4D image to 3D volume.
 *  Vol can be but need not to be allocated.
 *  Returns 0 if ok.
 */
int img2vol(IMG *img, VOL *vol, int frame)
{
  int ret;
  unsigned short int zi, yi, xi, fi;

  if(VOL_TEST) printf("img2vol(img, %d, vol)\n", frame);
  /* Check input */
  if(vol==NULL) return(1);
  vol->statmsg=_volStatusMessage[1];
  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(frame<1 || img->dimt<frame) return(2);
  if(vol->status==IMG_STATUS_UNINITIALIZED) return(1);

  /* Allocate memory (if needed) for volume */
  ret=volAllocate(vol, img->dimz, img->dimy, img->dimx);
  if(ret) return(ret);

  /* Copy data */
  fi=frame-1;
  vol->orientation=img->orientation;
  vol->sizex=img->sizex; vol->sizey=img->sizey; vol->sizez=img->sizez;
  for(zi=0; zi<vol->dimz; zi++)
    for(yi=0; yi<vol->dimy; yi++)
      for(xi=0; xi<vol->dimx; xi++)
        vol->v[zi][yi][xi]=img->m[zi][yi][xi][fi];

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy one time frame (1..dimt) from 4D image to 3D short int volume.
 *  Svol can be but need not to be allocated.
 *  Returns 0 if ok.
 */
int img2svol(IMG *img, SVOL *svol, int frame)
{
  int ret;
  unsigned short int zi, yi, xi, fi;
  float fmin, fmax, g;

  if(VOL_TEST) printf("img2svol(img, %d, svol)\n", frame);
  /* Check input */
  if(svol==NULL) return(1);
  svol->statmsg=_volStatusMessage[1];
  if(img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(frame<1 || img->dimt<frame) return(2);
  if(svol->status==IMG_STATUS_UNINITIALIZED) return(1);

  /* Allocate memory (if needed) for volume */
  ret=svolAllocate(svol, img->dimz, img->dimy, img->dimx);
  if(ret) return(ret);

  /* Copy data */
  fi=frame-1;
  svol->orientation=img->orientation;
  svol->sizex=img->sizex; svol->sizey=img->sizey; svol->sizez=img->sizez;
  ret=imgFrameMinMax(img, frame, &fmin, &fmax); if(ret) return(10+ret);
  if(fabs(fmin)>fabs(fmax)) g=fabs(fmin); else g=fabs(fmax);
  if(g!=0) g=32766./g; else g=1.0;
  for(zi=0; zi<svol->dimz; zi++)
    for(yi=0; yi<svol->dimy; yi++)
      for(xi=0; xi<svol->dimx; xi++)
        svol->v[zi][yi][xi]=(short int)temp_roundf(g*img->m[zi][yi][xi][fi]);
  svol->scale_factor=1.0/g;

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy 3D volume as one time frame (1..dimt) into 4D image.
 *  Img must be allocated.
 *  Returns 0 if ok.
 */
int vol2img(VOL *vol, IMG *img, int frame)
{
  unsigned short int zi, yi, xi, fi;

  if(VOL_TEST) printf("vol2img(vol, img, %d)\n", frame);
  /* Check input */
  if(vol==NULL || vol->status!=IMG_STATUS_OCCUPIED) return(1);
  vol->statmsg=_volStatusMessage[1];
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(frame<1 || img->dimt<frame) return(2);
  if(img->dimx!=vol->dimx || img->dimy!=vol->dimy) return(3);
  if(img->dimz!=vol->dimz) return(4);

  /* Copy data */
  fi=frame-1;
  for(zi=0; zi<vol->dimz; zi++)
    for(yi=0; yi<vol->dimy; yi++)
      for(xi=0; xi<vol->dimx; xi++)
        img->m[zi][yi][xi][fi]=vol->v[zi][yi][xi];

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy 3D short int volume as one time frame (1..dimt) into 4D image.
 *  Img must be allocated.
 *  Returns 0 if ok.
 */
int svol2img(SVOL *svol, IMG *img, int frame)
{
  unsigned short int zi, yi, xi, fi;

  if(VOL_TEST) printf("svol2img(svol, img, %d)\n", frame);
  /* Check input */
  if(svol==NULL || svol->status!=IMG_STATUS_OCCUPIED) return(1);
  svol->statmsg=_volStatusMessage[1];
  if(img==NULL || img->status!=IMG_STATUS_OCCUPIED) return(1);
  if(frame<1 || img->dimt<frame) return(2);
  if(img->dimx!=svol->dimx || img->dimy!=svol->dimy) return(3);
  if(img->dimz!=svol->dimz) return(4);

  /* Copy data */
  fi=frame-1;
  for(zi=0; zi<svol->dimz; zi++)
    for(yi=0; yi<svol->dimy; yi++)
      for(xi=0; xi<svol->dimx; xi++)
        img->m[zi][yi][xi][fi]=(svol->scale_factor)*(float)svol->v[zi][yi][xi];

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Prints volume information to specified file pointer, e.g. stdout
 */
void volInfo(VOL *vol, FILE *fp)
{
  if(VOL_TEST) printf("volInfo()\n");
  if(vol->status<=IMG_STATUS_UNINITIALIZED) {
    fprintf(fp, "Volume data is not initialized.\n"); return;}
  if(vol->status==IMG_STATUS_INITIALIZED) {
    fprintf(fp, "Volume data is initialized but empty.\n"); return;}
  if(vol->status==IMG_STATUS_ERROR) fprintf(stdout, "Volume data has errors.\n");
  fprintf(fp, "Volume status: %s\n", vol->statmsg);
  fprintf(fp, "Patient orientation: %d\n", vol->orientation);
  fprintf(fp, "Voxel sizes (x, y, z): %g %g %g mm\n",
    vol->sizex, vol->sizey, vol->sizez);
  fprintf(fp, "Dimensions (x, y, z): %d %d %d\n",
    vol->dimx, vol->dimy, vol->dimz);
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Prints short int volume information to specified file pointer, e.g. stdout
 */
void svolInfo(SVOL *svol, FILE *fp)
{
  if(VOL_TEST) printf("svolInfo()\n");
  if(svol->status<=IMG_STATUS_UNINITIALIZED) {
    fprintf(fp, "Volume data is not initialized.\n"); return;}
  if(svol->status==IMG_STATUS_INITIALIZED) {
    fprintf(fp, "Volume data is initialized but empty.\n"); return;}
  if(svol->status==IMG_STATUS_ERROR) fprintf(stdout, "Volume data has errors.\n");
  fprintf(fp, "Volume status: %s\n", svol->statmsg);
  fprintf(fp, "Patient orientation: %d\n", svol->orientation);
  fprintf(fp, "Voxel sizes (x, y, z): %g %g %g mm\n",
    svol->sizex, svol->sizey, svol->sizez);
  fprintf(fp, "Dimensions (x, y, z): %d %d %d\n",
    svol->dimx, svol->dimy, svol->dimz);
  fprintf(fp, "Scale factor: %g\n", svol->scale_factor);
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Prints matrix values inside specified range to file pointer.
 */
void volContents(VOL *vol, VOL_RANGE r, FILE *fp)
{
  int zi, yi, xi;

  if(vol->status!=IMG_STATUS_OCCUPIED) return;
  if(r.z1<1 || r.y1<1 || r.x1<1) return;
  if(r.z2<r.z1 || r.y2<r.y1 || r.x2<r.x1) return;
  if(r.z2>vol->dimz || r.y2>vol->dimy || r.x2>vol->dimx) return;

  for(zi=r.z1-1; zi<r.z2; zi++) {
    fprintf(fp, "pl=%03d ", zi+1);
    for(xi=r.x1-1; xi<r.x2; xi++) fprintf(fp, " x=%05d", xi+1);
    fprintf(fp, "\n");
    for(yi=r.y1-1; yi<r.y2; yi++) {
      fprintf(fp, "y=%05d", yi+1);
      for(xi=r.x1-1; xi<r.x2; xi++)
        fprintf(fp, " %7.3f", vol->v[zi][yi][xi]);
      fprintf(fp, "\n");
    }
  }
}
/*****************************************************************************/

/*****************************************************************************/
/** Finds max voxel inside specified volume range.
    Returns 0 if ok.
 */
int volMax(VOL *vol, VOL_RANGE r, VOL_PIXEL *p, float *maxv)
{
  int zi, yi, xi;

  if(vol->status!=IMG_STATUS_OCCUPIED) return(1);
  if(r.z1<1 || r.y1<1 || r.x1<1) return(2);
  if(r.z2<r.z1 || r.y2<r.y1 || r.x2<r.x1) return(3);
  if(r.z2>vol->dimz || r.y2>vol->dimy || r.x2>vol->dimx) return(4);

  zi=r.z1-1; yi=r.y1-1; xi=r.x1-1; *maxv=vol->v[zi][yi][xi];
  if(p!=NULL) {p->z=zi+1; p->y=yi+1; p->x=xi+1;}
  for(zi=r.z1-1; zi<r.z2; zi++) {
    for(yi=r.y1-1; yi<r.y2; yi++) {
      for(xi=r.x1-1; xi<r.x2; xi++) if(*maxv<vol->v[zi][yi][xi]) {
        *maxv=vol->v[zi][yi][xi];
        if(p!=NULL) {p->z=zi+1; p->y=yi+1; p->x=xi+1;}
      }
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates average voxel value inside specified volume range.
    Returns 0 if ok.
 */
int volAvg(VOL *vol, VOL_RANGE r, float *avg)
{
  int zi, yi, xi, n=0;

  if(vol->status!=IMG_STATUS_OCCUPIED) return(1);
  if(r.z1<1 || r.y1<1 || r.x1<1) return(2);
  if(r.z2<r.z1 || r.y2<r.y1 || r.x2<r.x1) return(3);
  if(r.z2>vol->dimz || r.y2>vol->dimy || r.x2>vol->dimx) return(4);

  *avg=0.0;
  for(zi=r.z1-1; zi<r.z2; zi++) {
    for(yi=r.y1-1; yi<r.y2; yi++) {
      for(xi=r.x1-1; xi<r.x2; xi++) {
        *avg+=vol->v[zi][yi][xi]; n++;
      }
    }
  }
  if(n>0) *avg/=(float)n;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
