/******************************************************************************

  File:        weight.c
  Description: Functions for setting weight factors based on SIF.

  Copyright (c) 2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2005-01-15 Vesa Oikonen
    First created. Functions from sif.c.
  2005-01-16 VO
    Weights are scaled so that average is 1.0.
  2005-04-26 CL
    Merged libsif to libtpcimgio


******************************************************************************/

/*****************************************************************************/
#include "sif.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate weights for frames in SIF data, normalized between 0-1.
    Weights are calculated from formula
      weight=(frame duration)^2 / (trues in a frame).
    Before calling this routine, trues must be calculated as total counts -
    randoms.
    Counts in SIF are not corrected for physical decay. Therefore, isotope
    halflife must be known, if weights are to be calculated for decay
    corrected TACs. Isotope halflife must be set to 0, if weights are used
    for TACs that are not corrected for decay.

    Mazoyer BM, Huesman RH, Budinger TF, Knittel BL. Dynamic PET data analysis.
    J Comput Assist Tomogr 1986; 10:645-653.
 */
void sifWeight(
  /** Pointer to SIF data */
  SIF *data,
  /** Halflife (sec) of isotope label;
      If halflife is 0, the weights are calculated for non-decay corrected data.
      If halflife (sec) is >0, the weights are calculated using decay corrected
      trues, but trues data is not changed. */
  double halflife
) {
  int i;
  double f, d;

  if(SIF_TEST) printf("sifWeight(*sif, %g)\n", halflife);
  /* Calculate weights */
  for(i=0; i<data->frameNr; i++) {
    if(data->trues[i]<1.0) data->trues[i]=1.0;
    f=data->x2[i]-data->x1[i]; if(f<=0.0) f=1.0;
    if(halflife<=1.0E-8) 
      d=1.0;
    else
      d=exp( ((data->x1[i]+data->x2[i])/2.0)*0.693147/halflife );
    data->weights[i]=(f*f)/(d*data->trues[i]);
    /*printf("%3d %g %g\n", i, data->trues[i], data->weights[i]);*/
  }

#if(0)
  /* Scale weights between [0,1] */
  f=data->weights[0];
  for(i=1; i<data->frameNr; i++) if(data->weights[i]>f) f=data->weights[i];
  for(i=0; i<data->frameNr; i++) data->weights[i]/=f;
#else
  /* Scale weights so that average weight is 1.0 */
  for(i=0, f=0.0; i<data->frameNr; i++) f+=data->weights[i];
  f/=(double)data->frameNr;
  for(i=0; i<data->frameNr; i++) data->weights[i]/=f;
#endif

  return;
}
/*****************************************************************************/

/*****************************************************************************/

