/******************************************************************************

  File:        iftsrch.c
  Description: Search functions for IFT contents.

  Copyright (c) 2004,2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2005-01-05 Vesa Oikonen
    First created. Functions from ift.c.


******************************************************************************/

/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
/*****************************************************************************/
#include "include/ift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Find the key in the IFT and return the index [0..keyNr-1].
    Key is case insensitive.
\return Returns -1 if key was not found, or other negative value in case of an error
 */
int iftGet(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Pointer to the key string; contents are replaced by
      the correct key string */
  char *key
) {
  int li;

  if(IFT_TEST) printf("iftGet(*ift, \"%s\", value)\n", key);
  if(ift==NULL) {iftSetStatus(ift, 1); return(-10);}
  if(key==NULL || strlen(key)<1) {iftSetStatus(ift, 1); return(-11);}
  
  /* Search the list */
  for(li=0; li<ift->keyNr; li++) {
    if(strcasecmp(ift->item[li].key, key)==0) {
      strcpy(key, ift->item[li].key);
      iftSetStatus(ift, 0); return(li);
    }
  }
  iftSetStatus(ift, 6);
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the Nth key with similar name in the IFT and return the index
    [0..keyNr-1]. Key is case insensitive.
\return Returns -1 if key was not found, or other negative value in case of an error
 */
int iftGetNth(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Pointer to the key string; contents are replaced by
      the correct key string */
  char *key,
  /** Nth (1..) insidence of key is searched. */
  int n
) {
  int li, found_nr=0;

  if(IFT_TEST) printf("iftGetNth(*ift, \"%s\", %d)\n", key, n);
  if(ift==NULL) {iftSetStatus(ift, 1); return(-10);}
  if(key==NULL || strlen(key)<1) {iftSetStatus(ift, 1); return(-11);}
  if(n<1) {iftSetStatus(ift, 1); return(-11);}
  
  /* Search the list */
  for(li=0; li<ift->keyNr; li++) {
    if(strcasecmp(ift->item[li].key, key)==0) {
      strcpy(key, ift->item[li].key); found_nr++;
      if(n==found_nr) {iftSetStatus(ift, 0); return(li);}
    }
  }
  iftSetStatus(ift, 6);
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the Nth item of IFT where the specified string is found in the key.
    Comparison is case sensitive.
\return Returns -1 if key was not found, or other negative value in
 case of an error, and the index [0..keyNr-1] if matching key is found.
 */
int iftFindNthKey(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Pointer to the case-sensitive (partial) key string */
  char *str,
  /** Nth (1..keyNr-1) insidence of value is searched. */
  int n
) {
  int li, found_nr=0;

  if(IFT_TEST) printf("iftFindNthKey(*ift, \"%s\", %d)\n", str, n);
  if(ift==NULL) {iftSetStatus(ift, 1); return(-10);}
  if(str==NULL || strlen(str)<1) {iftSetStatus(ift, 1); return(-11);}
  if(n<1) {iftSetStatus(ift, 1); return(-11);}
  
  /* Search the list */
  for(li=0; li<ift->keyNr; li++) {
    if(strstr(ift->item[li].key, str)!=NULL) {
      found_nr++;
      if(n==found_nr) {iftSetStatus(ift, 0); return(li);}
    }
  }
  iftSetStatus(ift, 9);
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the Nth item of IFT where the specified string is found in the value.
    Comparison is case sensitive.
\return Returns -1 if key was not found, or other negative value in
 case of an error, and the index [0..keyNr-1] if matching value is found.
 */
int iftFindNthValue(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Pointer to the case-sensitive (partial) value string */
  char *str,
  /** Nth (1..keyNr-1) insidence of value is searched. */
  int n
) {
  int li, found_nr=0;

  if(IFT_TEST) printf("iftFindNthValue(*ift, \"%s\", %d)\n", str, n);
  if(ift==NULL) {iftSetStatus(ift, 1); return(-10);}
  if(str==NULL || strlen(str)<1) {iftSetStatus(ift, 1); return(-11);}
  if(n<1) {iftSetStatus(ift, 1); return(-11);}
  
  /* Search the list */
  for(li=0; li<ift->keyNr; li++) {
    if(strstr(ift->item[li].value, str)!=NULL) {
      found_nr++;
      if(n==found_nr) {iftSetStatus(ift, 0); return(li);}
    }
  }
  iftSetStatus(ift, 8);
  return(-1);
}
/*****************************************************************************/

/*****************************************************************************/

