/******************************************************************************
File: studynr.c

Description:
Functions for processing PET study number.

Copyright (c) 2004,2006 Turku PET Centre

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
For more information on open source licenses see http://opensource.org.

Modification history:
2004-08-23 Vesa Oikonen
  First introduced.
2004-09-05 VO
  Included functions fnmatch(), fncasematch(), rnameSplit() and rnameMatch().
2004-09-20 VO
  Doxygen style comments corrected.
2006-03-16 VO
  Corrected comment.

******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
/*****************************************************************************/
#include "include/studynr.h"
/*****************************************************************************/

/*****************************************************************************/
/** Extract study number (max 8 chars) from filename (may include path).
    Fname is not modified.
    Memory (>=MAX_STUDYNR_LEN+1 chars) for studynr must be allocated when
    calling this.
\return Returns 0 if successful.
 */
int studynr_from_fname(char *fname, char *studynr)
{
  int i;
  char *cptr;

  if(fname==NULL || studynr==NULL) return(1);
  for(i=0; i<=MAX_STUDYNR_LEN; i++) studynr[i]=(char)0;
  /* Remove path */
  cptr=strrchr(fname, '/'); if(cptr==NULL) cptr=strrchr(fname, '\\');
  if(cptr==NULL) cptr=fname; else cptr++;
  i=strlen(cptr); if(i>MAX_STUDYNR_LEN) i=MAX_STUDYNR_LEN;
  strncpy(studynr, cptr, i);
  /* Remove extensions */
  cptr=strchr(studynr+1, '.'); if(cptr!=NULL) *cptr=(char)0;
  /* Remove non-digit characters from the end until length is >=6 */
  for(i=strlen(studynr)-1; i>5; i--)
    if(isdigit((int)studynr[i])) break; else studynr[i]=(char)0;
  /* Check the length of study number */
  if(strlen(studynr)<1) return(5);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if string fname matches string key, which may contain wildcards
    ? and *.
\return Returns 1 if strings do match and 0 if not.
 */
int fnmatch(char *key, char *fname)
{
  char *key_ptr=NULL, *fname_ptr=NULL;

  while((*key)&&(*fname)) {
    if((*key=='?')||(*key==*fname)) {
      key++; fname++;
    } else if(*key=='*') {
      if(*(key+1)==*fname) {key_ptr=key++; fname_ptr=fname+1;}
      else {fname++; if(*(key+1)=='?') {key_ptr=key++; fname_ptr=fname;}}
    } else if((key_ptr!=NULL) && (*fname_ptr)) {
      return(fnmatch(key_ptr, fname_ptr));
    } else {
      return(0);
    }
  }
  if((*fname)&&(key_ptr!=NULL)) {return(fnmatch(key_ptr, fname_ptr));}
  else {if(*key=='*') key++; return(*key==*fname);}
}
/*****************************************************************************/

/*****************************************************************************/
/** Case-independent check whether string fname matches string key,
    which may contain wildcards ? and *.
\return Returns 1 if strings do match and 0 if not.
 */
int fncasematch(char *key, char *fname)
{
  char *key_ptr=NULL, *fname_ptr=NULL;

  while((*key)&&(*fname)) {
    if((*key=='?')||(toupper((int)*key)==toupper((int)*fname))) {
      key++; fname++;
    } else if(*key=='*') {
      if(toupper((int)*(key+1))==toupper((int)*fname)) {key_ptr=key++; fname_ptr=fname+1;}
      else {fname++; if(*(key+1)=='?') {key_ptr=key++; fname_ptr=fname;}}
    } else if((key_ptr!=NULL) && (*fname_ptr)) {
      return(fnmatch(key_ptr, fname_ptr));
    } else {
      return(0);
    }
  }
  if((*fname)&&(key_ptr!=NULL)) {return(fnmatch(key_ptr, fname_ptr));}
  else {if(*key=='*') key++; return(toupper((int)*key)==toupper((int)*fname));}
}
/*****************************************************************************/

/*****************************************************************************/
/** Split region name into 1-3 subparts of given max length.
\return Returns the number of subparts.
 */
int rnameSplit(
  /** Region name to split (string is not edited) */
  char *rname,
  /** Pointer to 1st subname (anatomical region) */
  char *name1,
  /** Pointer to 2nd subname (usually hemisphere) */
  char *name2,
  /** Pointer to 3rd subname (usually image plane) */
  char *name3,
  /** Max lenght of subnames, excluding terminal null */
  int max_name_len
) {
  char temp[MAX_REGIONNAME_LEN+1], *cptr;
  int nr=0;

  if(rname==NULL || name1==NULL || name2==NULL || name3==NULL) return(0);
  if(max_name_len<1) return(0);
  name1[0]=name2[0]=name3[0]=(char)0;
  strncpy(temp, rname, MAX_REGIONNAME_LEN); temp[MAX_REGIONNAME_LEN]=(char)0;
  cptr=strtok(temp, " \t\n\r"); if(cptr==NULL) return(nr);
  strncpy(name1, cptr, max_name_len); name1[max_name_len]=(char)0; nr++;
  cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) return(nr);
  strncpy(name2, cptr, max_name_len); name2[max_name_len]=(char)0; nr++;
  cptr=strtok(NULL, " \t\n\r"); if(cptr==NULL) return(nr);
  strncpy(name3, cptr, max_name_len); name3[max_name_len]=(char)0; nr++;
  return(nr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Test whether region name or number matches with a test string.
    Test string can contain wildcards.
    If test string contains only one subname, it is tested against whole rname.
    If it contains 2-3 subnames, those are tested against the corresponding
    tokens in rname. Subname '.' stands for empty name.
    Number is tested only if test string contains one token of all digits.
\return Returns 1, in case of match or 0 if not matched.
 */
int rnameMatch(
  /** Region name which is tested */
  char *rname,
  /** Region number (1..) */
  int rnr,
  /** Test string */
  char *test_str
) {
  char region[3][MAX_REGIONNAME_LEN+1];
  char test[3][MAX_REGIONNAME_LEN+1];
  int ni, m, rtoknr=0, ttoknr=0;

  /* Check the input */
  if(rname==NULL || strlen(rname)<1) return(0);
  if(test_str==NULL || strlen(test_str)<1) return(0);
  /* Split region names into 1-3 subnames */
  rtoknr=rnameSplit(rname, region[0], region[1], region[2], MAX_REGIONNAME_LEN);
  ttoknr=rnameSplit(test_str, test[0], test[1], test[2], MAX_REGIONNAME_LEN);
  if(rtoknr==0 || ttoknr==0) return(0);
  /* If more than one subname to test for, then test against corresponding
     subnames */
  if(ttoknr>1) {
    for(ni=0; ni<ttoknr; ni++) {
      if( strcmp(test[ni], ".") && fncasematch(test[ni], region[ni])==0 ) {
        return(0);}
    }
    return(1);
  }
  /* If just one subname to test for */
  /* If it contains only digits, then we assume that it is region number */
  for(ni=m=0; ni<strlen(test[0]); ni++)
    if(isdigit((int)test[0][ni])==0) {m++; break;}
  if(m==0 && (ni=atoi(test[0]))>0) { /* it indeed is an acceptable  number */
    if(rnr==ni) return(1); else return(0);
  }
  /* It is name; chack if it is found anywhere in the region name */
  for(ni=0; ni<rtoknr; ni++) if(fncasematch(test[0], region[ni])) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

