/*****************************************************************************

  bootstrap.c    (c) 2003,2004 by Turku PET Centre


  Procedure for counting the confidence intervals and standard deviations
  for estimates of parameters of compartmental PET models.

  This is based on the article "Estimation of component and parametric
  distributions in spectral analysis" by Turkheimer, Sokoloff, Bertoldo, 
  Lucignani, Reivich, Jaggi and Schmidt in J Cereb Blood Flow Metabol. 1998;
  18:1211-1222.

  Standard deviation calculation based on the book 
  "An introduction to bootstrap" by Efron & Tibshirani, 1993,
  Chapman & Hall, New York.


  Version:
  1.0 2003-02-13 Kaisa Sederholm
  1.1 2003-02-17 KS
    Added the option to calculate the standard deviation.
  1.2 2003-02-19 VO  
    Added to the PET library; made the necessary changes.
    Sorting is done using qsort().
  1.3 2004-01-22 VO
    rint() is temporarily replaced by temp_roundf() (in petc99.c).
      2004-09-17 VO
    Doxygen style comments.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <time.h>
/*****************************************************************************/
#include "petc99.h"
#include "include/powell.h"
#include "include/bootstrap.h"
/*****************************************************************************/
#ifndef RAND_MAX
#define RAND_MAX 32767
#endif
/*****************************************************************************/
#ifndef ITER_NR  /* Nr of estimates to be created */
#define ITER_NR 200 
#endif
/*****************************************************************************/
/** local function definitions */
int bootstrapQSort(const void *par1, const void *par2);
/*****************************************************************************/
int bs_parNr, bs_frameNr;
double *bs_parameter;
double *bs_uplim;
double *bs_lowlim;
double *bs_weight;
double (*bs_func)(double*);
/*****************************************************************************/

/*****************************************************************************/
/** Bootstrap method.

  Original weights are assumed to be inversely proportional to variance.
  Square root is used, because bootstrap assumes them to be proportinal 
  to standard deviation. 
  If only standard deviation is wanted then cLim1 and cLim2 can be set to 
  be NULL, and if only the confidence limits are wanted then SD can be set
  to be NULL.

\return Return values:
  0, if ok.
  1 - 3 if some of the given parameters is not qualified.  
  4, if Powell fails, and 5, if out of memory.

*/
int bootstrap(
  /** vector to put the lower limits to               */
  double *cLim1,
  /** vector to put the upper limits to               */
  double *cLim2,
  /** vector to put the standard deviations to        */
  double *SD,
  /** Parameter estimates                             */
  double *parameter,
  /** Lower limits for the parameters                 */
  double *lowlim,
  /** Upper limits for the parameters                 */
  double *uplim,
  /** Nr of time points in tissue data                */
  int frameNr,
  /** measured tissue TAC values (will be changed)    */
  double *tac,
  /** fitted tissue TAC values (preserved)            */
  double *fitTac,
  /** Nr of parameters                                */
  int parNr,
  /** weights                                         */
  double *weight,
  /** The object function                             */
  double (*objf)(double*)
) {

  int ret, i, j, powellItNr, lowindex, upindex;
  double fret=0.0, *parMean, *chainptr, *chain, **matrix, *delta;
  double help, *error, *wError, *biasEst, *unbiasPar, *estInOrder;



  /* Checking the given parameters */
  strcpy(bserrmsg, "");
  if(frameNr<1 || parNr<1 ) {
    strcpy(bserrmsg, "either framenumber or parameternumber is negative");
    return(1);
  }
  if(parameter==NULL || objf==NULL || tac==NULL || fitTac==NULL){
    strcpy(bserrmsg, "some of the given parameters are not pointing anywhere");
    return(2);
  }
  for(i=0; i<parNr; i++) {
    if(lowlim[i]>uplim[i]) {
      strcpy(bserrmsg, "given limit values are not qualified");
      return(3);
    }
  }
 
  /* Allocating memory */
  bs_parameter=(double*)malloc(parNr*sizeof(double));
  bs_weight=(double*)malloc(frameNr*sizeof(double));
  delta=(double*)malloc(parNr*sizeof(double));
  parMean=(double*)malloc(parNr*sizeof(double));
  chain=(double*)malloc((ITER_NR*parNr)*sizeof(double));
  matrix=(double**)malloc(parNr*sizeof(double*));
  error=(double*)malloc(frameNr*sizeof(double));
  wError=(double*)malloc(frameNr*sizeof(double));
  biasEst=(double*)malloc(parNr*sizeof(double));
  unbiasPar=(double*)malloc(parNr*sizeof(double));
  if(bs_parameter==NULL || bs_weight==NULL || delta==NULL || parMean==NULL ||
     chain==NULL || matrix==NULL || error==NULL || wError==NULL ||
     biasEst==NULL || unbiasPar==NULL) {
    strcpy(bserrmsg, "out of memory"); return(5);
  }
  for(i=0, chainptr=chain; i<parNr; i++) {
    matrix[i]=(chainptr + i*ITER_NR);
  }

  bs_parNr=parNr;
  bs_frameNr=frameNr;
  bs_func=objf;
  bs_lowlim=lowlim;
  bs_uplim=uplim;
  for(i=0; i<bs_parNr; i++) {
    bs_parameter[i]=parameter[i];
  }
  for(i=0; i<bs_frameNr; i++) { /* copy and check the weights */
    if(weight[i]<=0) bs_weight[i]=1;
    else bs_weight[i]=weight[i];
  }
  
  if(BS_TEST) {printf("in bootstrap()\n");}

  /* calculate the errors and weighted errors*/ 
  for(i=0; i<bs_frameNr; i++) {
    error[i]=tac[i]-fitTac[i];    
    wError[i]=error[i]/sqrt(bs_weight[i]);    
  }
  if(BS_TEST>1) {
    printf("weighted errors:\n");
    for(i=0; i<bs_frameNr; i++) printf("%g ", wError[i]);
    printf("\n");
  }

  srand(6348);
  
  /*
   *  bootstrap iterations
   */
  if(BS_TEST>1) printf("Bootstrap matrix:");

  for(i=0; i<ITER_NR; i++) {
      
    /* sample a new error distribution (to the pointer error) */
    for(j=0; j<bs_frameNr; j++) {
      error[j]=
        wError[(int)((bs_frameNr)*(double)rand()/((double)(RAND_MAX)+1))];
      tac[j]=fitTac[j]+bs_weight[j]*error[j];
    }

    /* Powell local search */
    for(j=0; j<bs_parNr; j++) {
      delta[j]=(bs_uplim[j]-bs_lowlim[j])/100.;
      bs_parameter[j]=parameter[j];
    }      
    powellItNr=400;
    ret=powell(bs_parameter, delta, bs_parNr, 0.00001, &powellItNr,
               &fret, bs_func);
    if(ret > 1)	{
      sprintf(bserrmsg, "error %d in powell()", ret);
      free(bs_parameter);
      free(bs_weight);
      free(delta);
      free(parMean);
      free(matrix);
      free(chain);
      free(error);
      free(wError);
      free(biasEst);
      free(unbiasPar);
      return(4);
    }

    for(j=0; j<bs_parNr; j++) {
      matrix[j][i]=bs_parameter[j];
    }
    if(BS_TEST>1) {
      for(j=0; j<bs_parNr; j++) printf("%g ", bs_parameter[j]);
      printf("\n");
    }

  } /* end of bootstrap iterations */ 

  /* Computing the mean of each parameter and estimates for bias */
  for(i=0; i<bs_parNr; i++) {
    for(j=0, help=0; j<ITER_NR;j++) help+=matrix[i][j];
    parMean[i]=help/(double)ITER_NR;
  }
  for(i=0; i<bs_parNr; i++) {
    biasEst[i]=parMean[i]-parameter[i];
    /*unbiasPar[i]=parameter[i]-biasEst[i];*/
  }

  /* Computing the standard deviation for each parameter estimate */
  if(SD!=NULL) {
    if(BS_TEST) printf("Standard deviations:\n");
    for(i=0; i<bs_parNr; i++) {
      for(j=0, help=0; j<ITER_NR; j++) {
        help+=(matrix[i][j]-parMean[i])*(matrix[i][j]-parMean[i])/(ITER_NR-1);
      }
      SD[i]=sqrt(help); if(BS_TEST) printf("%g\n", SD[i]);
    }
  }

  if(cLim1!=NULL && cLim2!=NULL) {
    if(BS_TEST) printf("Confidence intervals:\n");
    /* Computing the 95% confidence intervals for each parameter estimate */
    for(i=0; i<bs_parNr; i++) {

      /* Arrange estimate samples in ascending order */
      estInOrder=matrix[i];
      qsort(estInOrder, ITER_NR, sizeof(double), bootstrapQSort); 

      /* Get the indexes within which 95% of samples lie in */
      lowindex=(int)temp_roundf(0.025*ITER_NR);
      upindex=(int)temp_roundf(0.975*ITER_NR)-1;
      if(estInOrder[lowindex]-biasEst[i]<1e-99) cLim1[i]=0.0;
      else cLim1[i]=estInOrder[lowindex]-biasEst[i];
      if(estInOrder[upindex]-biasEst[i]<1e-99) cLim2[i]=0.0;
      else cLim2[i]=estInOrder[upindex]-biasEst[i];
      if(BS_TEST) {
        printf("%g - %g\n", cLim1[i], cLim2[i]);
      }
    }
  }

  free(bs_parameter);
  free(bs_weight);
  free(delta);
  free(parMean);
  free(matrix);
  free(chain);
  free(error);
  free(wError);
  free(biasEst);
  free(unbiasPar);

  if(BS_TEST) {printf("end of bootstrap()\n");}
  return(0);

} /* end bootstrap() */
/*****************************************************************************/
 
/*****************************************************************************/
int bootstrapQSort(const void *par1, const void *par2)
{
  if( *((double*)par1) < *((double*)par2)) return(-1);
  else if( *((double*)par1) > *((double*)par2)) return(1);
  else return(0);
}
/*****************************************************************************/

/*****************************************************************************/

