/******************************************************************************

  Copyright (c) 1999-2004 by Turku PET Centre

  integr.c
  
  Procedures for
    linear interpolation and integration of PET and blood/plasma TACs
  to be incorporated in C programs.

  Version:
  1999-12-01 Jani Ruotsalainen
  2000-07-06 Vesa Oikonen
    The correct contents written for interpolate2pet() and petintegrate().
    Any output (y values or integrals) can be left out by specifying NULL
    as an argument.
  2000-07-26 VO
    Calculation of 2nd integral corrected in interpolate2pet().
  2000-09-20 VO
    Original data in interpolate() may contain only one value.
  2001-01-14 VO
    petintegrate(): frame length can be 0.
    included <stdlib.h>
  2001-02-21 VO
    Included function interpolate4pet(). It replaces interpolate2pet(),
    which will be removed later when all programs using it have been changed.
  2001-03-17 VO
    Included function qinterpolate(); it is similar to interpolate but faster.
  2001-03-22 VO
    Corrected a bug in interpolate4pet().
  2001-03-31 VO
    interpolate4pet() interface changed.
    qinterpolate() corrected to work with negative and equal times.
    Previous interpolate() removed, qinterpolate() renamed as interpolate().
    interpolate4pet() corrected to work with negative and equal times.
  2001-07-23 VO
    Data is tested to contain at least one time point, not two as before.
  2001-09-28 VO
    Corrected to avoid warnings during compiling with -Wall option.
  2002-02-12 VO
    Malloc is avoided in petintegrate(), if space for newyi is provided.
  2004-01-14 VO
    Information on functions (comments) are changed.
    Change in petintegrate() and fpetintegrate(): frames can be partially
    overlapping.
  2004-01-20 VO
    Included functions petintegral(), petintegrate2fe() and their float
    versions. Function petintegrate2fe() will replace petintegrate().
    Included float version of interpolate() function.
    Included float version of integrate() function.
    Included float version of petintegrate() function.
    Included float version of interpolate4pet() function.
    Function interpolate2pet() is commented out.
  2004-09-17 VO
    Doxygen style comments.


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "include/integr.h"
/*****************************************************************************/

/*****************************************************************************/
/** Linear interpolation and integration.

  It is assumed, that both original and new interpolated data represent
  the actual values at specified time points (not from framed data).
  The integration is calculated dot-by-dot.

  If NULL is spefied for newy[], newyi[] and/or newyii[], their values are
  not calculated.

  Original data and new x values must be sorted by ascending x.
  Subsequent x values can have equal values, enabling the use of step functions.
  Negative x (time) values can be processed.
  If necessary, the data is extrapolated assuming that:
    1) y[inf]=y[nr-1] and
    2) if x[0]>0 and y[0]>0, then an imaginary line is drawn from
       (0,0) to (x[0],y[0]).

*/
int interpolate(
  /** Input data x (time) values */
  double *x,
  /** Input data y values */
  double *y,
  /** Number of values in input data */
  int nr,
  /** Output data x values */
  double *newx,
  /** Interpolated (extrapolated) y values; NULL can be given */
  double *newy,
  /** Integrals; NULL can be given */
  double *newyi,
  /** 2nd integrals; NULL can be given */
  double *newyii,
  /** Nr of values in output data */
  int newnr
) {
  int i, j;
  double ty, tyi, tyii, ly, li, lii, lil;


  if(INTEGR_TEST) printf("in interpolate()\n");
  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;

  /* Set output values before the input data has started */
  /* If newx is also negative, set values to zero */
  for(j=0; j<newnr; j++) {
    if(newx[j]>=x[0] || newx[j]>=0.0) break;
    ty=tyi=tyii=0.0;
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
  }
  /* Now newx[j]>=x[0] or newx[j]>=0 */
  /* If newx is still smaller than x[0], but positive, then interpolate */
  /* between (0,0) and (x[0], y[0]) */
  if(newx[j]>=0.0) for(;j<newnr; j++) {
    if(newx[j]>x[0]) break;
    if(x[0]==0.0) {ty=tyi=tyii=0.0; if(newx[j]==x[0]) ty=y[0];}
    else {ty=newx[j]*y[0]/x[0]; tyi=0.5*ty*newx[j]; tyii=0.5*tyi*newx[j];}
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
  }
  if(j==newnr) return 0;
  /* Now newx[j]>=x[0] */
  if(newx[j]<x[0]) return 3;
  /* Calculate input data values at x[0] */
  ly=y[0]; if(x[0]<=0.0) {li=lii=lil=0.0;}
  else {lil=li=0.5*x[0]*ly; lii=0.5*x[0]*li;}
  /* Calculate input data values at x[i] */
  for(i=1; i<nr && j<newnr; i++) {
    /* calculate output values between x[i-1] and x[i] */
    for(;j<newnr;j++) {
      if(newx[j]>x[i]) break;
      ty=((y[i]-y[i-1])/(x[i]-x[i-1]))*(newx[j]-x[i-1]) + y[i-1];
      if(newy!=NULL) newy[j]=ty;
      tyi=li + 0.5*(ty+y[i-1])*(newx[j]-x[i-1]); if(newyi!=NULL) newyi[j]=tyi;
      if(newyii!=NULL) newyii[j]=tyii=lii + 0.5*(tyi+li)*(newx[j]-x[i-1]);
    }
    /* calculate integrals of original data at x[i] */
    li+=0.5*(y[i]+y[i-1])*(x[i]-x[i-1]);
    lii+=0.5*(li+lil)*(x[i]-x[i-1]);
    lil=li;
  }
  /* new values after the last x */
  for(;j<newnr; j++) {
    ty=y[nr-1]; if(newy!=NULL) newy[j]=ty;
    tyi=li + ty*(newx[j]-x[i-1]); if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii=lii + 0.5*(tyi+li)*(newx[j]-x[i-1]);
  }
  
  if(INTEGR_TEST) printf("out interpolate()\n");
  return 0;
}

/** float version of interpolate() */
int finterpolate(
  /** Input data x (time) values */
  float *x,
  /** Input data y values */
  float *y,
  /** Number of values in input data */
  int nr,
  /** Output data x values */
  float *newx,
  /** Interpolated (extrapolated) y values; NULL can be given */
  float *newy,
  /** Integrals; NULL can be given */
  float *newyi,
  /** 2nd integrals; NULL can be given */
  float *newyii,
  /** Nr of values in output data */
  int newnr
) {
  int i, j;
  float ty, tyi, tyii, ly, li, lii, lil;


  if(INTEGR_TEST) printf("in finterpolate()\n");
  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;

  /* Set output values before the input data has started */
  /* If newx is also negative, set values to zero */
  for(j=0; j<newnr; j++) {
    if(newx[j]>=x[0] || newx[j]>=0.0) break;
    ty=tyi=tyii=0.0;
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
  }
  /* Now newx[j]>=x[0] or newx[j]>=0 */
  /* If newx is still smaller than x[0], but positive, then interpolate */
  /* between (0,0) and (x[0], y[0]) */
  if(newx[j]>=0.0) for(;j<newnr; j++) {
    if(newx[j]>x[0]) break;
    if(x[0]==0.0) {ty=tyi=tyii=0.0; if(newx[j]==x[0]) ty=y[0];}
    else {ty=newx[j]*y[0]/x[0]; tyi=0.5*ty*newx[j]; tyii=0.5*tyi*newx[j];}
    if(newy!=NULL) newy[j]=ty;
    if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii;
  }
  if(j==newnr) return 0;
  /* Now newx[j]>=x[0] */
  if(newx[j]<x[0]) return 3;
  /* Calculate input data values at x[0] */
  ly=y[0]; if(x[0]<=0.0) {li=lii=lil=0.0;}
  else {lil=li=0.5*x[0]*ly; lii=0.5*x[0]*li;}
  /* Calculate input data values at x[i] */
  for(i=1; i<nr && j<newnr; i++) {
    /* calculate output values between x[i-1] and x[i] */
    for(;j<newnr;j++) {
      if(newx[j]>x[i]) break;
      ty=((y[i]-y[i-1])/(x[i]-x[i-1]))*(newx[j]-x[i-1]) + y[i-1];
      if(newy!=NULL) newy[j]=ty;
      tyi=li + 0.5*(ty+y[i-1])*(newx[j]-x[i-1]); if(newyi!=NULL) newyi[j]=tyi;
      if(newyii!=NULL) newyii[j]=tyii=lii + 0.5*(tyi+li)*(newx[j]-x[i-1]);
    }
    /* calculate integrals of original data at x[i] */
    li+=0.5*(y[i]+y[i-1])*(x[i]-x[i-1]);
    lii+=0.5*(li+lil)*(x[i]-x[i-1]);
    lil=li;
  }
  /* new values after the last x */
  for(;j<newnr; j++) {
    ty=y[nr-1]; if(newy!=NULL) newy[j]=ty;
    tyi=li + ty*(newx[j]-x[i-1]); if(newyi!=NULL) newyi[j]=tyi;
    if(newyii!=NULL) newyii[j]=tyii=lii + 0.5*(tyi+li)*(newx[j]-x[i-1]);
  }
  
  if(INTEGR_TEST) printf("out finterpolate()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
/** int interpolate2pet(
      double orig_x[], double orig_y[], int orig_nr,
      double new_x1[], double new_x2[], double value[], double integral[],
      double 2nd_integral[], int new_nr)

  Like interpolate(), except that interpolated values are not calculated
  as a value at specified time point, but as an average during the whole
  time frame newx1-newx2.
  Original data must be sorted by ascending x. Subsequent x values can have
  equal values, enabling the use of step functions.
  The newx2 must always be higher than newx1.
  It is assumed, that original data is not from framed data, but that
  the values represent the actual value at specified time point, which
  allows the integration to be calculated dot-by-dot.
  The integral values at frame end times are returned.
  If NULL is spefied for *newy, *newyi and/or *newyii, their values are
  not calculated.
*/
int interpolate2pet(double *x, double *y, int nr, double *newx1, double *newx2, 
                    double *newy, double *newyi, double *newyii, int newnr)
{
  int i, j, k, n;
  double *tx, *ty, *ti, ig1=0.0, ig2=0.0, dt;


  if(nr<1 || newnr<1) return 1;

  /* Allocate memory for temporary data */
  n=nr+2*newnr+1;
  tx=(double*)malloc(n*sizeof(double));
  ty=(double*)malloc(n*sizeof(double));
  ti=(double*)malloc(n*sizeof(double));
  if(tx==NULL || ty==NULL || ti==NULL) return 2;

  /* Fill the temp data */
  /* Make sure that we begin with point (0,0) */
  tx[0]=ty[0]=ti[0]=0; n=1;
  /* Copy original data */
  for(i=0; i<nr; i++) if(x>=0) {tx[n]=x[i]; ty[n]=y[i]; n++;}
  /* Allow extrapolation by setting the y at the last requested x to */
  /* the last y value of the original data */
  dt=newx2[0]; for(i=newnr-1; i>=0; i--) if(newx2[i]>dt) dt=newx2[i];
  if(dt>tx[n-1]) {tx[n]=dt; ty[n]=ty[n-1]; n++;}
  /* Insert new x values to temp data and interpolate values for those */
  for(i=0; i<newnr; i++) {
    /* Search the correct place for newx1 */
    for(j=0; j<n; j++) {
      if(newx1[i]==tx[j]) break; /* It already exists */
      if(newx1[i]<tx[j]) { /* Here is the place */
        /* Move temp data forward to make space for newx */
        for(k=n; k>j; k--) {ty[k]=ty[k-1]; tx[k]=tx[k-1];}
        /* Calculate a value for this newx */
        tx[j]=newx1[i]; 
        ty[j]=((ty[j+1]-ty[j-1])/(tx[j+1]-tx[j-1]))*(tx[j]-tx[j-1])+ty[j-1];
        n++; break;
      }
    }
    /* Search the correct place for newx2 */
    for(j=0; j<n; j++) {
      if(newx2[i]==tx[j]) break; /* It already exists */
      if(newx2[i]<tx[j]) { /* Here is the place */
        /* Move temp data forward to make space for newx */
        for(k=n; k>j; k--) {ty[k]=ty[k-1]; tx[k]=tx[k-1];}
        /* Calculate a value for this newx */
        tx[j]=newx2[i]; 
        ty[j]=((ty[j+1]-ty[j-1])/(tx[j+1]-tx[j-1]))*(tx[j]-tx[j-1])+ty[j-1];
        n++; break;
      }
    }
  }
  /* Calculate integrals for the temp data */
  for(i=1; i<n; i++) {
    dt=tx[i]-tx[i-1];
    if(dt<=0) ti[i]=ti[i-1]; else ti[i]=ti[i-1]+dt*(ty[i-1]+ty[i])/2.0;
  }

  /* Calculate averages if requested */
  if(newy!=NULL) for(i=0; i<newnr; i++) {
    /* Search the integral at frame start and end */
    for(j=0; j<n; j++) if(newx1[i]==tx[j]) {ig1=ti[j]; break;}
    for(k=j; k<n; k++) if(newx2[i]==tx[k]) {ig2=ti[k]; break;}
    /* Divide the difference of integrals with difference in times */
    dt=newx2[i]-newx1[i]; if(dt<=0) newy[i]=0; else newy[i]=(ig2-ig1)/dt;
  }

  /* Copy integrals if requested */
  if(newyi!=NULL) for(i=0; i<newnr; i++) {
    /* Search the integral at frame end */
    for(j=0; j<n; j++) if(newx2[i]==tx[j]) {newyi[i]=ti[j]; break;}
  }

  /* Calculate 2nd integrals if requested */
  if(newyii!=NULL) {
    /* Calculate 2nd integrals for temp data overwriting y values */
    ty[0]=0;
    for(i=1; i<n; i++) {
      dt=tx[i]-tx[i-1];
      if(dt<=0) ty[i]=ty[i-1]; else ty[i]=ty[i-1]+dt*(ti[i-1]+ti[i])/2.0;
    }
    /* Search the integral at frame end */
    for(i=0; i<newnr; i++) for(j=0; j<n; j++)
      if(newx2[i]==tx[j]) {newyii[i]=ty[j]; break;}
  }

  /* Free memory */
  free((char*)tx); free((char*)ty); free((char*)ti);

  return 0;
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** Linear integration from time 0 to x[0..nr-1].
    If x[0] is >0, then the beginning is interpolated from it to (0,0).
\return Returns 0 if OK, or 1, if error.
*/
int integrate(
  /** Original x values; duplicates are not allowed, all must be >=0.
      Data must be sorted by ascending x */
  double *x,
  /** Original y values */
  double *y,
  /** Nr of values */
  int nr,
  /** Array for integrals */
  double *yi
) {
  int j;

  yi[0]=0.5*y[0]*x[0];
  for(j=1; j<nr; j++) yi[j]=yi[j-1]+0.5*(y[j]+y[j-1])*(x[j]-x[j-1]);

  return 0;
}

/** float version of integrate() */
int fintegrate(
  /** Original x values; duplicates are not allowed, all must be >=0.
      Data must be sorted by ascending x */
  float *x,
  /** Original y values */
  float *y,
  /** Nr of values */
  int nr,
  /** Array for integrals */
  float *yi
) {
  int j;

  yi[0]=0.5*y[0]*x[0];
  for(j=1; j<nr; j++) yi[j]=yi[j-1]+0.5*(y[j]+y[j-1])*(x[j]-x[j-1]);

  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates integrals of PET data at frame end times.

    Data does not have to be continuous, but it must be increasing in time.
    For faster performance in repetitive calls, allocate memory for integral[]
    even if it is not needed.
\return Returns 0 if ok.
*/
int petintegrate(
  /** Array of frame start times */
  double *x1,
  /** Array of frame end times */
  double *x2,
  /** Array of y values (avg during each frame) */
  double *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  double *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  double *newyii
) {
  int i, allocated=0;
  double *ti, x, a;


  if(INTEGR_TEST) printf("in petintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(double*)malloc(nr*sizeof(double)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  if(x1[0]>0) {x=(x1[0]+x2[0])/2.0; a=(x1[0]*(y[0]/x)*x1[0])/2.0; ti[0]+=a;}

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out petintegrate()\n");
  return 0;
}

/* Float version of the same function */
int fpetintegrate(
  /** Array of frame start times */
  float *x1,
  /** Array of frame end times */
  float *x2,
  /** Array of y values (avg during each frame) */
  float *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  float *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  float *newyii
) {
  int i, allocated=0;
  float *ti, x, a;


  if(INTEGR_TEST) printf("in fpetintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(float*)malloc(nr*sizeof(float)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  if(x1[0]>0) {x=(x1[0]+x2[0])/2.0; a=(x1[0]*(y[0]/x)*x1[0])/2.0; ti[0]+=a;}

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out fpetintegrate()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Interpolate and integrate TAC to PET frames.

  It is assumed, that original data is not from framed data, but that
  the values represent the actual value at specified time point, which
  allows the integration to be calculated dot-by-dot.

  If NULL is specified for *newy, *newyi and/or *newyii, their values are
  not calculated.

  Original data and new x values must be sorted by ascending x.
  Subsequent x values can have equal values, enabling the use of step functions.
  The newx2 must always be higher or equal than newx1.
  Frames must not overlap. Negative x (time) values can be processed.
  If necessary, the data is extrapolated assuming that 1) y[inf]=y[nr-1] and
  2) if x[0]>0 and y[0]>0, then an imaginary line is drawn from (0,0) to
  (x[0],y[0]).
*/
int interpolate4pet(
  /** Times of original data */
  double *x,
  /** Values of original data */
  double *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times */
  double *newx1,
  /** PET frame end times */
  double *newx2, 
  /** Mean value during PET frame, or NULL */
  double *newy,
  /** Integral at frame mid time, or NULL */
  double *newyi,
  /** 2nd integral at frame mid time, or NULL */
  double *newyii,
  /** Number of PET frames */
  int newnr
) {
/*
  Variables in the function:
  on = the index of original data curve
  oy1= y value of original data at last index on-1 (at ox1)
  oy2= y value of original data at current index on (at ox2)
  oi1= integral of original data at last index on-1
  oi2= integral of original data at current index on
  oii1=2nd integral of original data at last index on-1
  oii2=2nd integral of original data at current index on
  nn = the index of new data curve
  ol = length between original data (ox2-ox1)
  fl = frame length
  ni1= integral of new data at frame start time
  ni2= integral of new data at frame end time
  nx = frame start/mid/end time, where ny is to be calculated
  ny = curve value to be interpolated at nx
  nyi= integral value at nx
*/
  int on, nn;
  double ox1, ox2, oy1, oy2, oi1, oi2, oii1, oii2;
  double ni1, ni2, fl, ol, nx, ny, nyi;


  if(INTEGR_TEST) printf("in interpolate4pet()\n");
  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* One output frame at a time */
  /* Set output values to 0 as long as input data has not started */
  for(nn=0; nn<newnr; nn++) if(newx2[nn]<=x[0] && newx2[nn]<=0.0) {
    if(newy!=NULL) newy[nn]=0.0;
    if(newyi!=NULL) newyi[nn]=0.0;
    if(newyii!=NULL) newyii[nn]=0.0;
  } else break;
  /* Set initial 'input' data */
  on=0; oi1=oii1=oi2=oii2=0.0;
  if(x[0]>0.0) ox1=oy1=ox2=oy2=0.0; else {ox1=ox2=x[0]; oy1=oy2=y[0];} 
  /* Now there must be some input data inside the current time frame */
  for(; nn<newnr; nn++) {
    //printf("nn=%d on=%d ; oi1=%g oii1=%g\n", nn, on, oi1, oii1);
    /* Calculate and check frame length */
    fl=newx2[nn]-newx1[nn]; if(fl<0.0) return 4;
    /* Check that output frames do not overlap */
    if(nn>0 && newx1[nn]<newx2[nn-1]) return 5;
    /* go through original data until frame start is reached */
    nx=newx1[nn];
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1]; /*printf("nn=%d on=%d nx=%g (%g,%g)-(%g,%g)\n", nn, on, nx, ox1, oy1, ox2, oy2);*/
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    ni1=oi1+0.5*(oy1+ny)*(nx-ox1);
    /* go through original data until frame mid is reached */
    nx=0.5*(newx1[nn]+newx2[nn]);
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1];
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    nyi=oi1+0.5*(oy1+ny)*(nx-ox1); if(newyi!=NULL) newyi[nn]=nyi;
    if(newyii!=NULL) newyii[nn]=oii1+0.5*(oi1+nyi)*(nx-ox1);
    /* go through original data until frame end is reached */
    nx=newx2[nn];
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1];
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    ni2=oi1+0.5*(oy1+ny)*(nx-ox1);
    if(newy!=NULL) {
      if(fl>0.) newy[nn]=(ni2-ni1)/fl;
      else newy[nn]=ny;
    }
  }

  if(INTEGR_TEST) printf("out interpolate4pet()\n");
  return 0;
}

/** float version of interpolate4pet() */
int finterpolate4pet(
  /** Times of original data */
  float *x,
  /** Values of original data */
  float *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times */
  float *newx1,
  /** PET frame end times */
  float *newx2, 
  /** Mean value during PET frame, or NULL */
  float *newy,
  /** Integral at frame mid time, or NULL */
  float *newyi,
  /** 2nd integral at frame mid time, or NULL */
  float *newyii,
  /** Number of PET frames */
  int newnr
) {
  int on, nn;
  float ox1, ox2, oy1, oy2, oi1, oi2, oii1, oii2;
  float ni1, ni2, fl, ol, nx, ny, nyi;


  if(INTEGR_TEST) printf("in finterpolate4pet()\n");
  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* One output frame at a time */
  /* Set output values to 0 as long as input data has not started */
  for(nn=0; nn<newnr; nn++) if(newx2[nn]<=x[0] && newx2[nn]<=0.0) {
    if(newy!=NULL) newy[nn]=0.0;
    if(newyi!=NULL) newyi[nn]=0.0;
    if(newyii!=NULL) newyii[nn]=0.0;
  } else break;
  /* Set initial 'input' data */
  on=0; oi1=oii1=oi2=oii2=0.0;
  if(x[0]>0.0) ox1=oy1=ox2=oy2=0.0; else {ox1=ox2=x[0]; oy1=oy2=y[0];} 
  /* Now there must be some input data inside the current time frame */
  for(; nn<newnr; nn++) {
    //printf("nn=%d on=%d ; oi1=%g oii1=%g\n", nn, on, oi1, oii1);
    /* Calculate and check frame length */
    fl=newx2[nn]-newx1[nn]; if(fl<0.0) return 4;
    /* Check that output frames do not overlap */
    if(nn>0 && newx1[nn]<newx2[nn-1]) return 5;
    /* go through original data until frame start is reached */
    nx=newx1[nn];
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1]; /*printf("nn=%d on=%d nx=%g (%g,%g)-(%g,%g)\n", nn, on, nx, ox1, oy1, ox2, oy2);*/
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    ni1=oi1+0.5*(oy1+ny)*(nx-ox1);
    /* go through original data until frame mid is reached */
    nx=0.5*(newx1[nn]+newx2[nn]);
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1];
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    nyi=oi1+0.5*(oy1+ny)*(nx-ox1); if(newyi!=NULL) newyi[nn]=nyi;
    if(newyii!=NULL) newyii[nn]=oii1+0.5*(oi1+nyi)*(nx-ox1);
    /* go through original data until frame end is reached */
    nx=newx2[nn];
    while(x[on]<=nx && on<nr) { /* integrate original data */
      ox2=x[on]; oy2=y[on];
      ol=ox2-ox1;
      if(ol>0.0) {
        oi2=oi1+0.5*(oy2+oy1)*(ox2-ox1); oii2=oii1+0.5*(oi2+oi1)*(ox2-ox1);
      } else {
        oi2=oi1; oii2=oii1;
      }
      ox1=ox2; oy1=oy2; oi1=oi2; oii1=oii2; on++;
    }
    ox1=x[on-1]; oy1=y[on-1];
    if(on<nr) {ox2=x[on]; oy2=y[on]; ny=oy1+(nx-ox1)*(oy2-oy1)/(ox2-ox1);}
    else {ox2=nx; oy2=y[on-1]; ny=oy1;}
    ni2=oi1+0.5*(oy1+ny)*(nx-ox1);
    if(newy!=NULL) {
      if(fl>0.) newy[nn]=(ni2-ni1)/fl;
      else newy[nn]=ny;
    }
  }

  if(INTEGR_TEST) printf("out finterpolate4pet()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame mid times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0, then an imaginary line is drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegral(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  double *ie,
  /** 2nd integrals at frame mid time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  double integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    box_integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+half_integral; box_integral+=frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out petintegral()\n");
  return(0);
}

/** float version of petintegral() */
int fpetintegral(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  float *ie,
  /** 2nd integrals at frame mid time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  float integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    box_integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+half_integral; box_integral+=frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out fpetintegral()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame end times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0, then an imaginary line is drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegrate2fe(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  double *e,
  /** integrals at frame end time */
  double *ie,
  /** 2nd integrals at frame end time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral;
  double value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out petintegrate2fe()\n");
  return(0);
}

/** float version of petinetgrate2fe() */
int fpetintegrate2fe(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  float *e,
  /** integrals at frame end time */
  float *ie,
  /** 2nd integrals at frame end time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral;
  float value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out fpetintegrate2fe()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/

