/**
   Copyright (c) 2004,2005 by Turku PET Centre
   
   File: libtpcmodel.c
   
   Purpose: methods for testing functions in the library libtpcmodel,
   and for printing out libtpcmodel information, such as Readme,
   History and build information. 

   NOTICE! This program should always have the same version with the
   library.
   
   Version:
   
   2005-02-22 1.0.0 Jarkko Johansson
   2005-04-27 1.1.0 JJ
   2005-06-09 1.2.0 Kaisa Sederholm
     Changes following from addition of robust estimation functions 
     to this library e.g. test_re() function added
   2005-12-27 1.2.1 Vesa Oikonen
     See tgo.c.
*/
/*****************************************************************************/
/** Library name. */
#define LIB_NAME "libtpcmodel"

/** Serial numbers for function names. */
#define ROBUSTEST 1

/** Serial numbers for function names. */
#define SWAP 1
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
/*****************************************************************************/
#include "img.h"
#include "libtpcmodelv.h"
#include "median.h"
#include "lms.h"
#include "lts.h"
#include "mestim.h"
/*****************************************************************************/
/*****************************************************************************/
#define MAXVAL 100000;
/*****************************************************************************/

// Test function declarations:
/**
   Function for testing library functions related to image arithmetics.
*/
int test_re();

// Functions for initializing image data:

/**
   Function for drawing a cros on the center of the image region,
   of width 2 pixels and value 1.
 */
void drawCross(IMG *img);

/**
   Function for drawing an unsymmetric cross for testing image transformation
   functions.
 */
void drawUnSymmetricCross(IMG *img);

/** 
    Function for printing the usage information of the libtpcmodel test program.
*/
void print_usage();

// Verbose mode switch:
int VERBOSE = 0;

int main (int argc, char *argv[ ])
{
    int c;
    int errflg=0, functionflag=0, exit_code=0;
    //    int  planes1, rows1, columns1, frames1, planes2, rows2, columns2, frames2, zoom=2;
    //    IMG img1, img2;
    //    char filename1[100], filename2[100];
    extern char *optarg;
    extern int optind, optopt;
    
    while ((c = getopt(argc, argv, "vVbBrRhHf:F:")) != -1) {
      switch (c) {

      case 'V':
      case 'v':
	// Set verbose flag on:
	VERBOSE = 1;
	break;

      case 'B':
      case 'b':
	// Print out the build information:
	libtpcmodel_print_build(stdout);
	return(0);

      case 'r':
	// Print out the Readme message:
	libtpcmodel_print_readme(stdout);
	return(0);
      case 'R':
	// Print out the Readme message with Doxygen tags:
	libtpcmodel_print_dreadme(stdout);
	return(0);

      case 'h':
	// Print out the History message:
	libtpcmodel_print_history(stdout);
	return(0);
      case 'H':
	// Print out the History message with Doxygen tags:
	libtpcmodel_print_dhistory(stdout);
	return(0);

      case 'f':
      case 'F':
	// Set function serial number to be tested:
	functionflag = atoi(optarg);
	break;

      case ':': // -f without argument
	fprintf(stderr,"Option -%c requires an argument\n", optopt);
	errflg++;
	break;
      case '?':
	fprintf(stderr,"Unrecognised option: -%c\n", optopt);
	errflg++;
      }
    }// End parsing command line options...

    if (errflg) {
      print_usage();
      return(2);
    }

    if(!functionflag){
      print_usage();
      return(0);
    }

    libtpcmodel_print_build(stdout);


    switch(functionflag){
    case ROBUSTEST:
      exit_code = test_re();
      break;
   default:
      break;
    }
    
    exit(exit_code);
}
 
/* Check robust functions */

int test_re(){
 
  double data[100], est, var;
  int dataNr, n;
  int error_code = 0;

  /* For a single data point 0, all functions should give 0 as result */ 

  data[0]=data[1]=0.;
  dataNr=2;

  est=dmedian(data, dataNr);
  if(est<0.0001 && est>-0.0001){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: dmedian() operation succeeded for one data point.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: dmedian() failed for one data point.");} 
    return (1);
  }
  
  est=mEstim(data, dataNr, 10, 0.5);
  if(est<0.0001 && est>-0.0001){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: mEstim() operation succeeded for one data point.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: mEstim() failed for one data point.");} 
    return (1);
  }

  est=least_median_of_squares(data, dataNr);
  if(est<0.0001 && est>-0.0001){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: least_median_of_squares() operation succeeded for one data point.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: least_median_of_squares() failed for one data point.");} 
    return (1);
  }


  error_code=least_trimmed_square(data, dataNr, &est, &var);
  if(error_code){
    if(est<0.0001 && est>-0.0001){  
      if(VERBOSE){printf("\n    Test SUCCESFULL: least_trimmed_square() succeeded with error code: %i\n",error_code);}
    }
    else if(VERBOSE){printf("\n    Test FAILED: least_trimmed_square() failed with error code: %i\n",error_code);}
    return(error_code);
  }
  else{
    if(est<0.0001 && est>-0.0001){  
      if(VERBOSE){printf("\n    Test SUCCESFULL: least_trimmed_square() operation succeeded for one data point.");} 
    }
    else{       
      if(VERBOSE){printf("\n    Test FAILED: least_trimmed_square() failed for one data point.");} 
      return (1);
    }
  }

  /* For a uniform distribution 1,2,3,4,5,6,7,8,9,10 all functions should give 5.5 as result */

  for(n=0; n<10; n++) data[n]=n+1;
  dataNr=10;

  est=dmedian(data, dataNr);
  if(est<5.50001 && est>5.49999){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: dmedian() operation succeeded for uniform distribution.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: dmedian() failed for uniform distribution.");} 
    return (2);
  }
  
  est=mEstim(data, dataNr, 10, 0.5);
  if(est<5.50001 && est>5.49999){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: mEstim() operation succeeded for uniform distribution.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: mEstim() failed for uniform distribution."); }
    return (2);
  }


  /* LMS and LTS methods aren't able to handle uniform distributions */


  /* For skewed distribution 2.1,3.1,3.3,3.5,3.6,3.7,4.5,5.2,6.0,7.4  
     median is 3.6, lms 3.8 */

  data[0]=2.1;
  data[1]=3.1;
  data[2]=3.3;
  data[3]=3.5;
  data[4]=3.6;
  data[5]=3.7;
  data[6]=4.5;
  data[7]=5.2;
  data[8]=6.0;
  data[9]=7.4; 

  /*median is the average of data points 3.6 and 3.7*/

  est=dmedian(data, dataNr);
  if(est<3.6501 && est>3.6499){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: dmedian() operation succeeded for skewed distribution.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: dmedian() failed for skewed distribution.");} 
    return (3);
  }

  /*LMS finds the "shortest half" 3.1-4.5 and the result is midpoint of 
    these values (3.8).*/

  est=least_median_of_squares(data, dataNr);
  if(est<3.8001 && est>3.7999){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: least_median_of_squares() operation succeeded for skewed distribution.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: least_median_of_squares() failed for skewed distribution.");} 
    return (3);
  }


   /* For skewed distribution 1, 1, 1, 1, 10
      hubers M-estimator is about 1.243, lts estimate is 1*/

  data[0]=1.;
  data[1]=1.;
  data[2]=1.;
  data[3]=1.;
  data[4]=10.;
  dataNr=5;


  /*Huber's M-estimator with one iteration and cutoffpoint 0.5 should
    give approximately 1.243 for this data according to algorithm described in
    Nonlinear_signal_prosessing_lecture_notes in 
    www.cs.tut.fi/~eeroh/nonlin.html*/ 

  est=mEstim(data, dataNr, 1, 0.5);

  if(est<1.244 && est>1.242){  
    if(VERBOSE){printf("\n    Test SUCCESFULL: mEstim() operation succeeded for skewed distribution.");} 
  }
  else{       
    if(VERBOSE){printf("\n    Test FAILED: mEstim() failed for skewed distribution.");} 
    return (3);
  }
  
  /*LTS should maybe have more sophisticated test data.
    The results could also be compared to reqular least square*/

   error_code=least_trimmed_square(data, dataNr, &est, &var);
  if(error_code){
    if(est<1.0001 && est>0.9999){  
      if(VERBOSE){printf("\n    Test SUCCESFULL: least_trimmed_square() succeeded with error code: %i\n",error_code);}
    }
    else if(VERBOSE){printf("\n    Test FAILED: least_trimmed_square() failed with error code: %i\n",error_code);}
    return(error_code);
  }
  else{
    if(est<1.0001 && est>0.9999){  
      if(VERBOSE){printf("\n    Test SUCCESFULL: least_trimmed_square() operation succeeded for skewed distribution.");} 
    }
    else{       
      if(VERBOSE){printf("\n    Test FAILED: least_trimmed_square() failed for skewed distribution.");} 
      return (3);
    }
  }
 
  return (error_code);
}



void drawCross(IMG *img){

  int plane, row, column, frame, r_limit, c_limit;

// Draw a symmetric cross on all planes and frames in the image:
  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++)
    for(row = 0; row < img->dimy; row++)
      for(column = 0; column < img->dimx; column++)
	for(frame = 0; frame < img->dimt; frame++){
	  if((row >= r_limit) && (row <= r_limit + 1))
	    img->m[plane][row][column][frame] = 1;
	  if((column >= c_limit) && (column <= c_limit + 1)\
	     && !((row >= r_limit) && (row <= r_limit + 1)))
	    img->m[plane][row][column][frame] = 1;
	}

}

void drawUnSymmetricCross(IMG *img){


  int plane, row, column, frame, r_limit, c_limit;

// Draw an unsymmetric cross on all planes and frames in the image:
  r_limit = (img->dimy - 2)/2;
  c_limit = (img->dimx - 2)/2;
  for(plane = 0; plane < img->dimz; plane++)
    for(row = 0; row < img->dimy; row++)
      for(column = 0; column < img->dimx; column++)
	for(frame = 0; frame < img->dimt; frame++){
	  if((row >= r_limit) && (row <= r_limit + 1)){
	    if(column < c_limit) // left 
	      img->m[plane][row][column][frame] = 1;
	    if(column > c_limit + 1) // right
	      img->m[plane][row][column][frame] = 3;
	    if((column >= c_limit) && (column <= c_limit + 1)) //center
	      img->m[plane][row][column][frame] = 5;
	  }
	  else{
	    if((column >= c_limit) && (column <= c_limit + 1)){	     
	      if(row < r_limit) // top
		img->m[plane][row][column][frame] = 4;
	      if(row > r_limit) // bottom
		img->m[plane][row][column][frame] = 2;

	    }
	    else{
	      img->m[plane][row][column][frame] = 0;
	    }
	  }
	}

}

void print_usage(){

  libtpcmodel_print_build(stdout);

  puts("\n"
       "   Methods for testing functions in the library "LIB_NAME"\n"
       "   and for printing out associated information, such as Readme,\n"
       "   History and build information. \n"
       "\n"
       "   Usage: "LIB_NAME" [-Options]\n"
       "\n"
       "   Options:\n"
       "   -h | H \n"
       "       Print out the "LIB_NAME" History message. Include Doxygen style tags\n"
       "       by using option '-H'.\n"
       "   -r | R \n"
       "       Print out the "LIB_NAME" Readme message. Include Doxygen style tags\n"
       "       by using option '-R'.\n"
       "   -b\n"
       "       Print out the "LIB_NAME" build information.\n"
       "   -f<function serial number>\n"
       "       Runs test procedures for the functions corresponding to given\n"
       "       'function serial number'. Serial numbers are defined in file\n"
       "       libtpcmodel.c.\n"
       "   -v\n"
       "       Run in verbose mode.\n"
       "\n"
       "   E.g.: "LIB_NAME" -h\n"
       "         "LIB_NAME" -v -f 1 \n"
       "\n"
	);

  fflush(stdout);
}
