/*****************************************************************************

  File:        lms.c
  Description: This file contains routines for calculating least median of
               squares estimate for single data.

  Copyright (c) 2004,2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  0.1 2004-08-18 Kaisa Sederholm
  0.2 2004-10-05 KS
      2005-01-05 VO
      Cahnges in comments not affecting compiled code.
      2005-04-26 CL
      Merged with libtpcimgp
      2005-06-10 KS
      Added comments
      (Removed from libtpcimgp to libtpcmodel)


*****************************************************************************/
#include "include/lms.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
/*****************************************************************************/
/* local function definitions */
int lmsQSort(const void *par1, const void *par2);
/*****************************************************************************/
/** Fit a constant (horisontal straight line) to the data by minimising
    the median of squared residuals.

    The algorithm is described in
    P.J. Rousseeuw: Least Median of Squares Regression, Journal of the 
    American Statistical Association, Vol. 79, No. 388 (1984), 871-880.
\return Returns the LMS estimate.
 */
double least_median_of_squares(
  /** Data array */
  double *data,
  /** Number of data values */
  int n
) {

  int i, odd=1, half=floor(n/2), smallnr;
  double small, *help;
  if(fmod((double)n, 2.0)<1e-99) odd=0;
  double *halfs_data;

  halfs_data=(double*)malloc((half+odd) * sizeof(double)); 

  help=data;
  /* data in ascending order*/
  qsort(help, n, sizeof(double), lmsQSort); 

     /*if n is even number */

       for(i=0; i<half+odd; i++){
         halfs_data[i]=data[half+i]-data[i];
       }

     i=smallnr=0;
     for(i=1, small=halfs_data[0]; i<half+odd; i++){
       if(halfs_data[i]<small){
         small=halfs_data[i];
         smallnr=i;
       }
       }

     return (data[half+smallnr]+data[smallnr])/2.0;

}

/** Compares two numbers
\return Returns -1 if value1<value2, 1 if value1>value2 and 0 otherwise
*/

int lmsQSort(
    /** value nr 1*/
    const void *par1, 
    /** value nr 2*/
    const void *par2)
{
  if( *((double*)par1) < *((double*)par2)) return(-1);
  else if( *((double*)par1) > *((double*)par2)) return(1);
  else return(0);
}
