/*****************************************************************************

  File:        median.c
  Description: Functions related to calculation of median value.

  Copyright (c) 2003-2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  2003-10-09 Vesa Oikonen
    First version.
  2004-09-17 VO
    Doxygen style comments.
  2005-01-05 VO
    Changes in comments not affecting compiled code.
  2005-04-26 CL
    Merged with libtpcimgp
  2005-06-10 Kaisa Sederholm
    Removed from libtpcimgp into libtpcmodel


******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
/*****************************************************************************/
#include "include/median.h"
/*****************************************************************************/

/*****************************************************************************/
/** Returns the kth smallest value in data[0..n]. Array is partially sorted.
    Algorithm is based on the book Wirth N. Algorithms + data structures =
    programs. Englewood Cliffs, Prentice-Hall, 1976.
\return Returns the kth smallest value in data[0..n].
*/
double d_kth_smallest(
  /** Pointer to data; array is partially sorted */
  double *data,
  /** Length of data array */
  int n,
  /** kth smallest value will be returned */
  int k
) {
  int i, j, l, m;
  double x, s;

  l=0; m=n-1;
  while(l<m) {
    x=data[k]; i=l; j=m;
    do {
      while(data[i]<x) i++;
      while(x<data[j]) j--;
      if(i<=j) {s=data[i]; data[i]=data[j]; data[j]=s; i++; j--;}
    } while(i<=j);
    if(j<k) l=i;
    if(k<i) m=j;
  }
  return(data[k]);
}
/****************************************************************************/

/****************************************************************************/
/** Returns the median in array data[0..n]. Array is partially sorted.
    Algorithm is based on the book Wirth N. Algorithms + data structures =
    programs. Englewood Cliffs, Prentice-Hall, 1976.
\return Returns the median in array data[0..n].
*/
double dmedian(
  /** Pointer to data; array is partially sorted */
  double *data,
  /** Length of data array */
  int n
) {
  int k;
  double d1, d2;

  if(n<1) return(0.0);
  if(n%2) {
    k=(n-1)/2; return(d_kth_smallest(data, n, k));
  } else {
    k=n/2; d1=d_kth_smallest(data, n, k-1); d2=d_kth_smallest(data, n, k);
    return(0.5*(d1+d2));
  }
}
/****************************************************************************/

/****************************************************************************/

