/*****************************************************************************

  File:        mestim.c
  Description: This file contains routines for calculating Hubers M-estimator
               for single data.

  Copyright (c) 2005 Turku PET Centre

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details:
  http://www.gnu.org/copyleft/lesser.html

  You should have received a copy of the GNU Lesser General Public License
  along with this library/program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 

  Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi

  Modification history:
  0.1 2005-03-02 Kaisa Sederholm
      2005-04-22 KS
      Corrected condition if(data[in]!=theta) to be
      if(data[in]<0.99999*theta || data[in]>1.00001*theta)
  0.2 2005-05-06 KS
      added parameter cutoff
  1.0 2005-06-02 KS
      2005-06-10 KS
      Added comments
      (Remoced from libtpcimgp into libtpcmodel)
   
*****************************************************************************/
#include "include/mestim.h"
#include "include/median.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
/*****************************************************************************/
/** Fit a constant (horisontal straight line) to the data with M-estimator.

    The algorithm is described in the lecture notes of Nonlinear signal 
    processing course of Tampere university of technology
    http://www.cs.tut.fi/~eeroh/nonlin.html

\return Returns Hubers M-estimator for single dataset.
 */
double mEstim(
  /** Data array */
  double *data,
  /** Number of data values */
  int nr,
  /**Number of iterations */
  int iterNr,
  /** cutoff point */
  double cutoff
) {

  int ii, in;
  double theta, sum1, sum2, help;

  theta=dmedian(data, nr);
  for(ii=0; ii<iterNr; ii++){
    sum1=sum2=0;
    for(in=0; in<nr; in++){
      if(data[in]<0.9999*theta || data[in]>1.0001*theta){
        help=huber(data[in]-theta, cutoff);
        sum1=sum1+data[in]*(huber(data[in]-theta, cutoff))/(data[in]-theta);
        sum2=sum2+(huber(data[in]-theta, cutoff))/(data[in]-theta);
      }else{
	sum1=sum1+cutoff*data[in];
        sum2=sum2+cutoff; 
      }
    }
    theta=sum1/sum2;
  }
  return theta;
 }

/**Hubers function
\return Returns x if |x|<b, and b otherwise.
*/
double huber(
  /** parameter x*/
  double x, 
  /** cutoff point*/
double b){

  double help;


  if(x<-b) help=-b;
  else help=x;
  if(help<b){return help;}
  else{ return b;}

}

