/*****************************************************************************

  powell.c      (c) 1995-2004 by Turku PET Centre


  Description:
  Powell function minimization routines.
  Based on Numerical recipes in C (Press et al.).

  Version:
  1995-10-08 Vesa Oikonen
  1995-12-08 VO
    A minor bug fixed.
  1999-08-01 VO
    Floats->Doubles
  2002-07-11 VO
    Functions and arguments are renamed.
    Function returning the WSS must be specified as an argument.
    Function returns the smallest WSS, or BAD_FIT in case of error.
    Test prints added.
  2002-07-08 Kaisa Sederholm
    Old versions of functions taken out
  2002-08-14 VO
    Included in petlib.
    The rest of the local functions renamed to prevent mix-ups.
  2003-03-29 VO
    Removed nonnecessary tests.
  2004-07-12 VO
    Added Doxygen-style comments.
  2004-09-20 VO
    Doxygen style comments are corrected.



******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "include/powell.h"
/*****************************************************************************/
/* Local variables */
int _powell_ncom;
double _powell_pcom[MAX_PARAMETERS], _powell_xicom[MAX_PARAMETERS];
double (*_powellFunc)(double*);
/*****************************************************************************/
/* Local functions */
void _powell_linmin(double *p, double *xi, int n, double *fret);
double _powell_brent(double ax, double bx, double cx, double tol, double *xmin);
double _powell_f1dim(double x);
void _powell_mnbrak(double *ax, double *bx, double *cx, double *fa, double *fb,
      double *fc);
/* Local "inline" functions */
double _powell_sqr(double x) {return (x*x);}
void _powell_shft(double *a, double *b, double *c, double *d) {*a=*b; *b=*c; *c=*d;}
double _powell_sign(double a, double b) {return((b>=0.0) ? fabs(a):-fabs(a));}
double _powell_fmax(double a, double b) {return((a>b) ? a:b);}
/*****************************************************************************/

/*****************************************************************************/
/** Powell function minimization routine.
\return Returns 0, if succesful, 1 if required tolerance was not reached, and >1 in case of an error.
 */
int powell(
  /** Initial guess and final set of parameters */
  double *p,
  /** Initial changes for parameters, ==0 if fixed */
  double *delta,
  /** Nr of parameters */
  int parNr,
  /** Fractional tolerance (for WSS); 0<ftol<1 */
  double ftol,
  /** Max nr of iterations, and nr of required iters */
  int *iterNr,
  /** Function return value (WSS) at minimum */
  double *fret,
  /** Function to minimize (must return the WSS) */
  double (*_fun)(double*)
) {
  int i, j, ibig, iterMax, fixed[MAX_PARAMETERS];
  double xi[MAX_PARAMETERS][MAX_PARAMETERS]; /* Matrix for directions */
  double pt[MAX_PARAMETERS], ptt[MAX_PARAMETERS], xit[MAX_PARAMETERS];
  double del, fp, fptt, t;


  if(POWELL_TEST>0) printf("in powell(,,%d,%g,%d,,)\n", parNr, ftol, *iterNr);
  *fret=BAD_FIT;
  if(p==NULL) return(11); if(delta==NULL) return(12);
  if(parNr<1) return(21); if(ftol<=0.0) return(22); if(ftol>=1.0) return(23);
  if((*iterNr)<1) return(24);

  /* SetUp */
  _powellFunc=_fun;
  iterMax=*iterNr; /* save the max nr of iterations */
  _powell_ncom=parNr;
  /* Function value at initial point */
  *fret=(*_powellFunc)(p);
  /* Save the initial point */
  for(j=0; j<parNr; j++) pt[j]=p[j];
  /* Check which parameters are fixed */
  for(i=0; i<parNr; i++) if(fabs(delta[i])<1.0e-20) fixed[i]=1; else fixed[i]=0;

  /* Initiate matrix for directions */
  for(i=0; i<parNr; i++)
    for(j=0; j<parNr; j++)
      if(i==j) xi[i][j]=delta[i]; else xi[i][j]=0.0;

  /* Iterate */
  for(*iterNr=1; ; (*iterNr)++) {
    if(POWELL_TEST>0) printf("  iteration %d\n", *iterNr);
    fp=*fret; ibig=0; del=0.0; /* largest function decrease */

    /* In each iteration, loop over all directions in the set */
    for(i=0; i<parNr; i++) {
      if(fixed[i]) continue; /* do nothing with fixed parameters */
      for(j=0; j<parNr; j++) if(fixed[j]) xit[j]=0.0; else xit[j]=xi[j][i];
      fptt=*fret;
      /* minimize along direction xit */
      _powell_linmin(p, xit, parNr, fret);
      if(fabs(fptt-(*fret))>del) {del=fabs(fptt-(*fret)); ibig=i;}
    }

    /* Check if done */
    if(2.0*fabs(fp-(*fret)) <= ftol*(fabs(fp)+fabs(*fret))) break;
    if((*iterNr)>=iterMax) return(1);

    /* Construct the extrapolated point and the average direction moved */
    for(j=0; j<parNr; j++) {
      ptt[j]=2.0*p[j]-pt[j]; xit[j]=p[j]-pt[j];
      pt[j]=p[j]; /* save the old starting point */
    }
    fptt=(*_powellFunc)(ptt);
    if(fptt<fp) {
      t=2.0*(fp-2.0*(*fret)+fptt)*_powell_sqr(fp-(*fret)-del)-del*_powell_sqr(fp-fptt);
      if(t<0.0) {
        _powell_linmin(p, xit, parNr, fret);
        for(j=0; j<parNr; j++) {
          xi[j][ibig]=xi[j][parNr-1]; xi[j][parNr-1]=xit[j];}
      }
    }
  } /* next iteration */
  if(POWELL_TEST>0) printf("out of powell() in good order.\n");

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
void _powell_linmin(double *p, double *xi, int n, double *fret)
{
  int i;
  double xx, xmin, fx, fb, fa, bx, ax;

  if(POWELL_TEST>0) printf("_powell_linmin()\n");
  _powell_ncom=n;
  for(i=0; i<n; i++) {_powell_pcom[i]=p[i]; _powell_xicom[i]=xi[i];}
  ax=0.0; xx=1.0;
  _powell_mnbrak(&ax, &xx, &bx, &fa, &fx, &fb);
  *fret=_powell_brent(ax, xx, bx, 2.0e-4, &xmin);
  for(i=0; i<n; i++) {xi[i]*=xmin; p[i]+=xi[i];}
}
/*****************************************************************************/
double _powell_brent(double ax, double bx, double cx, double tol, double *xmin)
{
  const int ITMAX = 100;
  const double CGOLD = 0.3819660;
  const double ZEPS = 1.0e-10;
  int iter;
  double a, b, d=0.0, etemp, fu, fv, fw, fx, p, q, r;
  double e=0.0, tol1, tol2, u, v, w, x, xm;

  if(POWELL_TEST>0) printf("_powell_brent()\n");
  a=(ax<cx ? ax:cx); b=(ax>cx ? ax:cx); x=w=v=bx; fw=fv=fx=_powell_f1dim(x);
  for(iter=0; iter<ITMAX; iter++) {
    if(POWELL_TEST>0) printf("     iter=%d\n", iter);
    xm=0.5*(a+b); tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
    if(fabs(x-xm)<=(tol2-0.5*(b-a))) {*xmin=x; return(fx);}
    if(fabs(e)>tol1) {
      r=(x-w)*(fx-fv); q=(x-v)*(fx-fw); p=(x-v)*q-(x-w)*r;
      q=2.0*(q-r); if(q>0.0) p=-p; q=fabs(q);
      etemp=e; e=d;
      if(fabs(p)>=fabs(0.5*q*etemp) || p<=q*(a-x) || p>=q*(b-x))
        d=CGOLD*(e=(x>=xm ? a-x:b-x));
      else {
        d=p/q; u=x+d; if(u-a<tol2 || b-u<tol2) d=_powell_sign(tol1, xm-x);}
    } else {d=CGOLD*(e=(x>=xm ? a-x:b-x));}
    u=(fabs(d)>=tol1 ? x+d:x+_powell_sign(tol1, d));
    fu=_powell_f1dim(u);
    if(fu<=fx) {
      if(u>=x) a=x; else b=x;
      _powell_shft(&v, &w, &x, &u); _powell_shft(&fv, &fw, &fx, &fu);
    } else {
      if(u<x) a=u; else b=u;
      if(fu<=fw || w==x) {v=w; w=u; fv=fw; fw=fu;}
      else if(fu<=fv || v==x || v==w) {v=u; fv=fu;}
    }
  }
  *xmin=x;
  return(fx);
}
/*****************************************************************************/
double _powell_f1dim(double x)
{
  int i;
  double f, xt[MAX_PARAMETERS];

  if(POWELL_TEST>0) printf("_powell_f1dim(%g)\n", x);
  for(i=0; i<_powell_ncom; i++) xt[i]=_powell_pcom[i]+x*_powell_xicom[i];
  f=(*_powellFunc)(xt);
  return(f);
}
/*****************************************************************************/
void _powell_mnbrak(double *ax, double *bx, double *cx, double *fa, double *fb,
  double *fc)
{
  const double GOLD = 1.618034;
  const double GLIMIT = 100.0;
  const double TINY = 1.0e-20;
  double ulim, u, r, q, fu, dum=0.0;

  if(POWELL_TEST>0) printf("_powell_mnbrak()\n");
  *fa=_powell_f1dim(*ax); *fb=_powell_f1dim(*bx);
  if(*fb>*fa) {_powell_shft(&dum, ax, bx, &dum); _powell_shft(&dum, fb, fa, &dum);}
  *cx=(*bx)+GOLD*(*bx-*ax); *fc=_powell_f1dim(*cx);
  while((*fb)>(*fc)) {
    r=(*bx-*ax)*(*fb-*fc); q=(*bx-*cx)*(*fb-*fa);
    u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/(2.0*_powell_sign(_powell_fmax(fabs(q-r),TINY),q-r));
    ulim=(*bx)+GLIMIT*(*cx-*bx);
    if(((*bx)-u)*(u-(*cx)) > 0.0) {
      fu=_powell_f1dim(u);
      if(fu < *fc) {*ax=(*bx); *bx=u; *fa=(*fb); *fb=fu; return;}
      else if(fu > *fb) {*cx=u; *fc=fu; return;}
      u=(*cx)+GOLD*(*cx-*bx);
      fu=_powell_f1dim(u);
    } else if((*cx-u)*(u-ulim) > 0.0) {
      fu=_powell_f1dim(u);
      if(fu < *fc) {
        q=*cx+GOLD*(*cx-*bx); r=_powell_f1dim(u);
        _powell_shft(bx, cx, &u, &q); _powell_shft(fb, fc, &fu, &r);
      }
    } else if((u-ulim)*(ulim-*cx) >= 0.0) {
      u=ulim; fu=_powell_f1dim(u);
    } else {
      u=(*cx)+GOLD*(*cx-*bx);
      fu=_powell_f1dim(u);
    }
    _powell_shft(ax, bx, cx, &u); _powell_shft(fa, fb, fc, &fu);
  }
}
/*****************************************************************************/

/*****************************************************************************/

