#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <stdio.h>

#include "main.h"
#include "colormap.h"
#include "colortables.h"
#include "message_dialog.h"

#define SWAPLONG(x) ((((x)&0x000000ff)<<24)|(((x)&0x0000ff00)<<8)|(((x)&0x00ff0000)>>8)|(((x)&0xff000000)>>24))

/* Exported globals */
XColor yait_palette[2][PALETTE_COLORS];
XColor extra_palette[EXTRA_COLORS];
int displaydepth;
int haveRWpalette;

/* Internally used globals */
static int use_truecolor=0;
static const char *extracolornames[EXTRA_COLORS]= {"Red","Green"};
static int swap_endian;
static unsigned long dynamic_palette[2][PALETTE_COLORS];

/* Internally used functions */
static void allocate_dynamic_colormap(Display *display,int screen, XColor *map,unsigned long *colors);
static void allocate_extra_colors(Display *display,int screen,XColor *map);


/* Allocate the colormap */
void allocate_colors(Display *display, int screen) {
  XVisualInfo visual_info;
  
  displaydepth=DefaultDepth(display,screen);

  if(displaydepth<8) {
    printf("Unsupported color depth %d !\n",displaydepth);
    exit(1);
  }
  if(XMatchVisualInfo(display,screen,displaydepth,TrueColor,&visual_info)) {
    printf("Read/write palette unavailable, using TrueColor mode.\n");
    use_truecolor=1;
    haveRWpalette=0;
  } else {
    if(!XMatchVisualInfo(display,screen,displaydepth,PseudoColor,&visual_info)) {
      if(!XMatchVisualInfo(display,screen,displaydepth,DirectColor,&visual_info)) {
        printf("Cannot get PseudoColor or DirectColor !\n");
        exit(1);
      }
    }
    haveRWpalette=1;
    printf("Read/write palette available.\nAllocating primary study colormap... ");
    allocate_dynamic_colormap(display,screen,yait_palette[0],dynamic_palette[0]);
    printf("Allocating secondary study colormap... ");
    allocate_dynamic_colormap(display,screen,yait_palette[1],dynamic_palette[1]);
  }
  /* Allocate some extra colors that are used for decorations, ROIs and such */
  allocate_extra_colors(display,screen,extra_palette);
  /* Load the colormap */
  colormap_setscaleN(display,studies[0].color,studies[0].c_lower,studies[0].c_upper,0);
  colormap_setscaleN(display,studies[1].color,studies[1].c_lower,studies[0].c_upper,1);
  /* Check byteorder */
  /* If the computer YaIT runs on has a different byteorder than the one  */
  /* where the XServer runs on, we need to swap the bytes when generating */
  /* XImages. (eg. Sun and PC machines have different byteorder.)	  */
  if((ntohs(0xF0A0)!=0xF0A0 && ImageByteOrder(display)==LSBFirst) ||
     (ntohs(0xF0A0)==0xF0A0 && ImageByteOrder(display)!=LSBFirst))
    swap_endian=0;
  else
    swap_endian=1;
  printf("My byteorder is %s endian, server byteorder is %s endian.\n",ntohs(0xF0A0)!=0xF0A0?"little":"big",ImageByteOrder(display)==LSBFirst?"little":"big");
}

static void allocate_dynamic_colormap(Display *display,int screen,XColor *map,unsigned long *colors) {
  unsigned long plane_masks[1];
  int ncolors=PALETTE_COLORS;
  Colormap cmap;

  cmap=DefaultColormap(display,screen);
  if(XAllocColorCells(display,cmap,False,plane_masks,0,colors,ncolors)==0) {
    fprintf(stderr,"Error occured while attempting to allocate %d colors.\n",ncolors);
    exit(1);
  }
  printf("allocated %d colors.\n",ncolors);
}

static void allocate_extra_colors(Display *display,int screen,XColor *map) {
  Colormap cmap;
  XColor exact;
  int r;
  cmap=DefaultColormap(display,screen);
  for(r=0;r<EXTRA_COLORS;r++) {
    if(!XAllocNamedColor(display,cmap,extracolornames[r],&map[r],&exact)) {
      fprintf(stderr,"allocate_extra_colors(): Cannot allocate color \"%s\"\n",extracolornames[r]);
      continue;
    }
  }
}

/* Set the colormap */
void colormap_setscale(Display *display, const float scale[][4],int scalelen,int min,int max,int cmapN) {
  XColor tmpcolors[PALETTE_COLORS],tmpcolors2[PALETTE_COLORS];
  Colormap cmap;
  int *scalepoints,curpoint;
  double *scales;
  int width;
  int c,j,v1,v2;
  cmap=DefaultColormap(display,DefaultScreen(display));
  scalepoints=(int*)malloc(sizeof(int)*scalelen);
  scales=(double*)malloc(sizeof(double)*(scalelen-1));
  for(c=0;c<scalelen;c++) {
    scalepoints[c]=scale[c][0]*PALETTE_COLORS;
  }
  for(c=0;c<(scalelen-1);c++) {
    scales[c]=scalepoints[c+1]-scalepoints[c];
  }
  curpoint=0;
  for(c=0;c<PALETTE_COLORS;c++) {
    double k;
    int t,n;
    t=curpoint;
    n=curpoint+1;
    k=((c-scalepoints[t])/scales[t]);
    tmpcolors[c].red=(((1.0-k)*scale[t][1])+(k*scale[n][1]))*0xffff;
    tmpcolors[c].green=(((1.0-k)*scale[t][2])+(k*scale[n][2]))*0xffff;
    tmpcolors[c].blue=(((1.0-k)*scale[t][3])+(k*scale[n][3]))*0xffff;
    if(curpoint<scalelen)
      while(c==scalepoints[curpoint+1]) curpoint++;
  }
  free(scalepoints);
  free(scales);
  /* Compress scale */
  v1=studies[cmapN].c_lower*PALETTE_COLORS/100.0;
  v2=studies[cmapN].c_upper*PALETTE_COLORS/100.0;
  width=v2-v1;
  for(c=0;c<width;c++) {
    j=c*PALETTE_COLORS/width;
    tmpcolors2[c]=tmpcolors[j];
  }
  /* Shift the scale to the right */
  for(c=0;c<v2-v1;c++) {
    yait_palette[cmapN][c+v1]=tmpcolors2[c];
  }
  for(c=0;c<v1;c++)
    yait_palette[cmapN][c]=tmpcolors[0];
  for(c=v2;c<PALETTE_COLORS;c++)
    yait_palette[cmapN][c]=tmpcolors[PALETTE_COLORS-1];
  /* Update colormap entries  */
  if(use_truecolor) {
    for(c=0;c<PALETTE_COLORS;c++) {
       if(!XAllocColor(display,cmap,&yait_palette[cmapN][c])) {
        fprintf(stderr,"Error: Cannot allocate color (%d,%d,%d) !\n",c,c,c);
	fatal_error_dialog("Cannot allocate color !");
	return;
      }
    }
  } else {
    for(c=0;c<PALETTE_COLORS;c++) {
      yait_palette[cmapN][c].pixel=dynamic_palette[cmapN][c];
      yait_palette[cmapN][c].flags=DoRed|DoGreen|DoBlue;
    }
    XStoreColors(display,cmap,yait_palette[cmapN],PALETTE_COLORS);
  }
} 

/* Set the colormap by number */
void colormap_setscaleN(Display *display, int scaleN,int min,int max,int cmapN) {
  switch(scaleN) {
     case GRAYSCALE:
     colormap_setscale(display,gray_scale,GRAYSCALE_LEN,min,max,cmapN);
     break;
   case INVERSESCALE:
     colormap_setscale(display,inverse_scale,INVERSESCALE_LEN,min,max,cmapN);
     break;
   case RAINBOWSCALE:
     colormap_setscale(display,rainbow_scale,RAINBOWSCALE_LEN,min,max,cmapN);
     break;
   case HEATSCALE:
     colormap_setscale(display,heat_scale,HEATSCALE_LEN,min,max,cmapN);
     break;
   case AIPSSCALE:
     colormap_setscale(display,aips_scale,AIPSSCALE_LEN,min,max,cmapN);
     break;
    default:
      fprintf(stderr,"colormap_setscaleN(%x,%d,%d,%d,%d): Color scale '%d' is unknown\n",(int)display,scaleN,min,max,cmapN,scaleN);
  }
}

void screen_format_ximage_update(int len,int depth, unsigned char *srcdata, unsigned char *targdata,int cmapN) {
  int j;
  switch(depth) {
    case 32:
    case 24:
      if(swap_endian) {
        for(j=0;j<len;j++) {
	  (*(unsigned long*)targdata)=SWAPLONG(yait_palette[cmapN][*srcdata].pixel);
	  targdata+=4;
  	  srcdata++;
        }
      } else {
        for(j=0;j<len;j++) {
	  (*(unsigned long*)targdata)=yait_palette[cmapN][*srcdata].pixel;
	  targdata+=4;
  	  srcdata++;
        }
      }
      break;
    case 16:
      if(swap_endian) {
        for(j=0;j<len;j++) {
	  *((unsigned short*)targdata)=yait_palette[cmapN][*srcdata].pixel<<8 | yait_palette[cmapN][*srcdata].pixel>>8 ;
	  targdata+=2;
	  srcdata++;
	}
      } else {
        for(j=0;j<len;j++) {
          *((unsigned short*)targdata)=yait_palette[cmapN][*srcdata].pixel;
	  targdata+=2;
	  srcdata++;
	}
      }
      break;
    case 8:
      for(j=0;j<len;j++) {
        *targdata=(unsigned char)yait_palette[cmapN][*srcdata].pixel;
	targdata++;
	srcdata++;
      }
      break;
    default:
      fprintf(stderr,"screen_format_ximage(): Unknown color depth %d !\n",depth);
      fatal_error_dialog("Unknown color depth !");
      break;
  }
}

XImage *screen_format_ximage(XImage *src,int cmapN) {
  unsigned char *tmp_data=NULL;
  XImage *newimage;
  Display *display;
  Visual *visual;
  int screen;
  int bitmap_pad;
  display=XtDisplay(toplevel);
  screen=XDefaultScreen(display);
  bitmap_pad=src->bitmap_pad;
  visual=XDefaultVisual(display,screen);
  //if(displaydepth>8)
    tmp_data=malloc(src->width*src->height*((displaydepth==24?32:displaydepth)/8));
  //else return src;
  screen_format_ximage_update(src->width*src->height,displaydepth,src->data,tmp_data,cmapN);
  newimage=XCreateImage(display, visual, displaydepth,ZPixmap,src->xoffset, tmp_data,src->width,src->height,bitmap_pad,src->bytes_per_line*(displaydepth==24?32:displaydepth)/8);
  if(newimage==NULL) {
    fprintf(stderr,"screen_format_ximage(): XCreateImage() returned NULL !\n");
    fatal_error_dialog("Couldn't create new image !");
  }
  return newimage;
}

