#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <img.h>

#include "main.h"
#include "images.h"
#include "zoom_images.h"
#include "colormap.h"
#include "export.h"
#include "message_dialog.h"
#include "petimage.h"

/* For BMPs, image data must be saved in little-endian format */
#ifdef _BIG_ENDIAN
#define SWAPS(x) ((((x)&0x00ff)<<8)|(((x)&0xff00)>>8))
#define SWAPL(x) ((((x)&0x000000ff)<<24)|(((x)&0x0000ff00)<<8)|(((x)&0x00ff0000)>>8)|(((x)&0xff000000)>>24))
#else
#define SWAPS(x) (x)
#define SWAPL(x) (x)
#endif

static XImage *matrix_to_ximage(ExportOptions options);
static XImage *zoom_to_ximage(ExportOptions options);
static void copy_area(char *src,char *dest,int src_x,int src_y,int src_w,int src_h,int dest_x,int dest_y,int dest_w,int colorshift);
static int ximage2bmp(XImage *image,char *bmpfilename);

/* Export menu callback */
void export_cb(Widget w, XtPointer client_data, XtPointer call_data) {
  switch((int)client_data) {
    case 0:
      printf("Export BMP...\n");
      if(zoom_matrix.zoom==0) {
        export_dialog(EXPORT_BMP);
      } else {
        ExportOptions *options=malloc(sizeof(ExportOptions));
        options->type=EXPORT_BMP;
        export_save(options);
      }
      break;
    default:
      fprintf(stderr,"export_cb(): Unknown menu item '%d' !\n",(int)client_data);
      break;
  }
}

/* Export */
void export_image(Widget widget, XtPointer client_data, XtPointer call_data) {
  XmFileSelectionBoxCallbackStruct *cbs =
     (XmFileSelectionBoxCallbackStruct *)call_data;
  ExportOptions *options=(ExportOptions*)client_data;
  char *filename;
  XImage *image;
  XtUnmanageChild(widget);
  if(options->imagesPerRow>2 && strlen(studies[1].filename) && options->imagesPerRow%2) options->imagesPerRow--;
  image=export_to_ximage(*options);
  if(!image) return;
  filename=XmStringUnparse((XmString)(cbs->value),NULL,XmCHARSET_TEXT,XmCHARSET_TEXT,NULL,0,XmOUTPUT_ALL);
  printf("export_image(): filename=\"%s\"\n",filename);
  switch(options->type) {
    case EXPORT_BMP:
      if(ximage2bmp(image,filename)) {
        message_dialog("Can't save BMP image !","Error");
      }
      break;
  }
  XDestroyImage(image);
  free(options);
  free(filename);
};

/* Create an XImage with the content to be exported */
XImage *export_to_ximage(ExportOptions options) {
  if(zoom_matrix.zoom)
    return zoom_to_ximage(options);
  return matrix_to_ximage(options);
}

/* Create an XImage with the matrix view */
static XImage *matrix_to_ximage(ExportOptions options) {
  Display *display;
  Visual *visual;
  XImage *image;
  ImageList *list;
  int width,height,pos;
  int imagecount=0,x,y;
  char *data;
  GC gc;
  /* Get image sizes and such */
  display=XtDisplay(toplevel);
  visual=DefaultVisual(display,XDefaultScreen(display));
  gc=DefaultGC(display,XDefaultScreen(display));
  list=get_image_list();
  while(list) {imagecount++; list=list->next;}
  width=(options.imagesPerRow<imagecount?options.imagesPerRow:imagecount)*(studies[0].imagepack.dimx+options.spacing)+options.spacing;
  height=((int)(0.9+(imagecount/(float)options.imagesPerRow)))*(studies[0].imagepack.dimy+options.spacing)+options.spacing;
  if(width<0 || width>10000 || height<0 || height>10000) {
    fprintf(stderr,"matrix_to_ximage(): Strange values !\n");
    fprintf(stderr,"options.imagesPerRow=%d , options.spacing=%d\n",options.imagesPerRow,options.spacing);
    fprintf(stderr,"width=%d, height=%d\n",width,height);
    error_dialog("Got strange values in matrix_to_ximage() !");
    return NULL;
  }
  printf("matrix_to_ximage(): width=%d, height=%d\n",width,height);
  /* Create data */
  data=malloc(width*height);
  memset(data,0,width*height);
  /* Paint image */
  list=get_image_list();
  pos=1;
  x=options.spacing;
  y=options.spacing;
  while(list) {
    copy_area(list->img->image->data,data,0,0,list->img->image->width,list->img->image->height,x,y,width,list->img->studyN*PALETTE_COLORS);
    if(pos==options.imagesPerRow) {
      x=options.spacing;
      y+=list->img->image->height+options.spacing;
      pos=1;
    } else {
      pos++;
      x+=list->img->image->width+options.spacing;
    }
    list=list->next;
  }
  /* Create XImage */
  image=XCreateImage(display,visual,8,ZPixmap,0,data,width,height,8,0);
  return image;
}

/* Zoomed image to exportable XImage */
/* Create an XImage with the matrix view */
static XImage *zoom_to_ximage(ExportOptions options) {
  Display *display;
  Visual *visual;
  XImage *image,*zoomed[2];
  int width,height;
  char *data;
  GC gc;
  display=XtDisplay(toplevel);
  visual=DefaultVisual(display,XDefaultScreen(display));
  gc=DefaultGC(display,XDefaultScreen(display));
  /* Get images */
  zoomed[0]=get_zoom_image()->image[0];
  zoomed[1]=get_zoom_image()->image[1];
  width=zoomed[0]->width;
  height=zoomed[0]->height;
  if(zoomed[1]) {
    width+=zoomed[1]->width;
  }
  /* Create data */
  data=malloc(width*height);
  memset(data,0,width*height);
  /* Paint image */
  copy_area(zoomed[0]->data,data,0,0,zoomed[0]->width,zoomed[0]->height,0,0,width,0);
  if(zoomed[1]) {
    copy_area(zoomed[1]->data,data,0,0,zoomed[1]->width,zoomed[1]->height,width,0,width,PALETTE_COLORS);
  }
  /* Create XImage */
  image=XCreateImage(display,visual,8,ZPixmap,0,data,width,height,8,0);
  return image;
}

/* Copy an image to another image */
static void copy_area(char *src,char *dest,int src_x,int src_y,int src_w,int src_h,int dest_x,int dest_y,int dest_w,int colorshift) {
  int x,y;
  src+=src_y*src_w+src_x;
  dest+=dest_y*dest_w+dest_x;
  for(y=0;y<src_h;y++) {
    memcpy(dest,src,src_w);
    if(colorshift)
      for(x=0;x<src_w;x++)
        dest[x]+=colorshift;
    dest+=dest_w;
    src+=src_w;
  }
}

static void write2(FILE *fp,short s) {
  s=SWAPS(s);
  fwrite(&s,2,1,fp);
}

static void write4(FILE *fp,long l) {
  l=SWAPL(l);
  fwrite(&l,4,1,fp);
}
/* Convert an XImage to BMP and write it out */
static int ximage2bmp(XImage *image,char *bmpfilename) {
  FILE *fp;
  long dataoffset,colorcount,len,datalen;
  char signature[2]="BM",padbytes[3];
  long map,c,y,pad;
  char *ptr;
  /* Calculate values */
  colorcount=PALETTE_COLORS*2;
  dataoffset=14+40+(4*colorcount);
  pad=((((image->width-1)/4)+1)*4)-image->width;
  datalen=(image->width+pad)*image->height;
  len=dataoffset+datalen;
  /* Open file */
  fp=fopen(bmpfilename,"w");
  if(!fp) {
    fprintf(stderr,"ximage2bmp(image,\"%s\"): Couldn't open file for writing !",bmpfilename);
    return 1;
  }
  /* Write out headers */
  /* Write main header */
  fwrite(signature,1,2,fp);		/* Signature */
  write4(fp,len);			/* File length */
  write4(fp,0);				/* Unused */
  write4(fp,dataoffset);		/* Offset to data */
  /* Write infoheader */
  write4(fp,40);			/* Header length */
  write4(fp,image->width);		/* Width */
  write4(fp,image->height);		/* Height */
  write2(fp,1);				/* Planes */
  write2(fp,8);				/* Bits per pixel */
  write4(fp,0);				/* No compression */
  write4(fp,0);				/* Compressed imagesize */
  write4(fp,100);			/* Horizontal resolution: pixels/meter */
  write4(fp,100);			/* Vertical resolution: pixels/meter */
  write4(fp,colorcount);		/* Colors used */
  write4(fp,0);				/* All colors are important */
  /* Write colortable */
  for(map=0;map<2;map++)
    for(c=0;c<PALETTE_COLORS;c++) {
      char ct[4];
      ct[2]=yait_palette[map][c].red>>8;
      ct[1]=yait_palette[map][c].green>>8;
      ct[0]=yait_palette[map][c].blue>>8;
      ct[3]=0;
      fwrite(ct,1,4,fp);
    }
  /* Write image data */
  ptr=image->data+((image->height-1)*image->width);
  memset(padbytes,0,3);
  for(y=0;y<image->height;y++) {
    fwrite(ptr,1,image->width,fp);
    fwrite(padbytes,1,pad,fp);
    ptr-=image->width;
  }
  fclose(fp);
  return 0;
}

