#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <X11/Xlib.h>
#include <Xm/Xm.h>
#include <Xm/TextF.h>
#include "petimage.h"
#include "colorslide.h"
#include "main.h"
#include "colormap.h"
#include "message_dialog.h"
#include "images.h"
#include "zoom_images.h"
#include "util.h"

typedef struct _TMP_ImageList {
  YaIT_Image *image;
  int frame,plane;
  struct _TMP_ImageList *prev,*next;
} TMP_ImageList;

static ImageList *image_list=NULL,*last_image=NULL;
static YaIT_Image *selected_image=NULL;
static int negative_values;

/* Internally used functions */
static void select_image(Widget widget, XtPointer client_data, XtPointer call_data);
static void add_image(IMG *imgpack,ScaleType scale,TMP_ImageList **list,int frame,int plane,float *min,float *max,int curWidget,int widgetsPerRow,int studyN);

/* Are there any negative pixel values */
void set_negative_values(int val) {
  negative_values = val;
}

/* Return the image list */
ImageList *get_image_list(void) {return image_list;}

/* Return the selected image */
YaIT_Image *get_selected_image(void) {return selected_image;}

/* Create a YaIT_Image structure */
YaIT_Image *create_YaIT_Image(XImage *image) {
  YaIT_Image *img;
  img=(YaIT_Image*)malloc(sizeof(YaIT_Image));
  memset(img,0,sizeof(YaIT_Image));
  img->image=image;
  return img;
}

/* Add a new YaIT_Image to image list */
void image_list_add(YaIT_Image *image) {
  ImageList *newentry;
  newentry=(ImageList*)malloc(sizeof(ImageList));
  newentry->img=image;
  newentry->prev=last_image;
  newentry->next=NULL;
  if(last_image) last_image->next=newentry;
  last_image=newentry;
  if(!image_list) image_list=newentry;
}

/* Refresh the image list with the new palette */
void refresh_image_list(int studyN) {
  ImageList *ptr=image_list;
  Display *d;
  int len;
  if(!image_list) return;
  d=XtDisplay(toplevel);
  len=ptr->img->image->width*ptr->img->image->height; /* We trust that all the images are of same size */
  while(ptr) {
    if(studyN<0 || ptr->img->studyN==studyN)
      screen_format_ximage_update(len,displaydepth,ptr->img->image->data,ptr->img->dimage->data,ptr->img->studyN);
    ptr=ptr->next;
  }
}

/* Clear images */
void clear_images(void) {
  ImageList *list=image_list,*next;
  while(list) {
    XtVaSetValues((Widget)list->img->widget,petimageNimage,NULL,NULL);
    XtDestroyWidget((Widget)list->img->widget);
    XDestroyImage(list->img->image);
    XDestroyImage(list->img->dimage);
    free(list->img);
    next=list->next;
    free(list);
    list=next;
  }
  image_list=NULL;
  last_image=NULL;
  selected_image=NULL;
}

/* Reload the image list */
void update_image_list(Study *study) {
  char error_msg[256], tmp[8];
  XmStringTable str_list;
  int i, j;
  INT_list iframes,iplanes;
  /*printf("update_image_list()\n");*/

  /* Test, is there frames, that user has asked for */
  if( study->framelist.i[ study->framelist.nr-1]>study->imagepack.dimt){
     sprintf(error_msg,"Error: wrong frame, there is no frame %d!\n",
	    study->framelist.i[ study->framelist.nr-1]);
     error_dialog(error_msg);
     return;
  }
  iframes=intMerge(&studies[0].framelist,&studies[1].framelist);
  str_list=(XmStringTable)XtMalloc(iframes.nr*sizeof (XmString *));
  for(i=0; i<iframes.nr; i++){
       sprintf(tmp, "%d", iframes.i[i]);
       str_list[i] = XmStringCreateLocalized(tmp);
  }
  XtVaSetValues (frame_spin, XmNnumValues, iframes.nr, 
                             XmNvalues, str_list, NULL);
  for (i = 0; i < iframes.nr; i++)
        XmStringFree (str_list[i]);
  XtFree ((XtPointer) str_list);
  
  /* Test, is there planes, that user has asked for */
  for(i=0; i< study->planelist.nr; i++){ 
    for(j=0; j<study->imagepack.dimz; j++){ 
      if( study->planelist.i[i]==study->imagepack.planeNumber[j]) break;
      else if(j==study->imagepack.dimz-1){  
         sprintf(error_msg, "Error: wrong plane, there is no plane %d!\n",
	    study->planelist.i[ study->planelist.nr-1]);
	 error_dialog(error_msg);
         return; 
      }  
    }
  }
  iplanes=intMerge(&studies[0].planelist,&studies[1].planelist);
  str_list=(XmStringTable)XtMalloc(iplanes.nr*sizeof (XmString *));
  for(i=0; i<iplanes.nr; i++){
        sprintf(tmp, "%d", iplanes.i[i]);
        str_list[i] = XmStringCreateLocalized (tmp);
	/*printf("spin plane[%d]=%d\n", i, study->planelist.i[i]);*/
  }
  XtVaSetValues (plane_spin, XmNnumValues, iplanes.nr, 
                             XmNvalues, str_list, NULL);

  for (i = 0; i < iplanes.nr; i++)
         XmStringFree (str_list[i]);
  XtFree ((XtPointer) str_list);
 
  panels.panel_maxnum=iplanes.nr*iframes.nr;

  /* Remove the old images (if any) */
  clear_images();
  /* Add the new ones */
  add_images(panels.scale,0);
}

void add_images(ScaleType scale,char replace) {
  int a,j,b,frame,plane=0;
  ImageList *imglist=image_list;
  int xsize,ysize;
  float min[2],max[2];
  int widgetsPerRow,curWidget=0;
  int studyN,studyCount=1;
  TMP_ImageList *imagelists[2]={NULL,NULL};

  if(strlen(studies[1].filename)>=4) studyCount++;

  xsize=studies[0].imagepack.dimx;
  ysize=studies[0].imagepack.dimy;
  if(imagesPerRow>0)
    widgetsPerRow=imagesPerRow;
  else
    widgetsPerRow=WIDTH/xsize;

  if(studyCount>1 && widgetsPerRow%2) widgetsPerRow++;
  if(scale==ScaleFixed) { /* For fixed scale, get the minimium and maximium values of all images */
    min[0]=10000.0; max[0]=-10000.0;
    min[1]=10000.0; max[1]=-10000.0;
    get_minmax(&studies[0],&min[0],&max[0]);
    if(studyCount>1)
      get_minmax(&studies[1],&min[1],&max[1]);
  }
  negative_values=0;
  /* Read the images */
  for(studyN=0;studyN<studyCount;studyN++) {
    for(b=0;b<studies[studyN].framelist.nr;b++) {
      /* Find frame and plane */
      frame=studies[studyN].framelist.i[b]-1;
      for(a = 0; a < studies[studyN].planelist.nr; a++){
        for(j=0; j<studies[studyN].imagepack.dimz; j++){
          if(studies[studyN].planelist.i[a]==studies[studyN].imagepack.planeNumber[j]){
            plane=studies[studyN].planelist.i[a]-1;
            break;
          }
        }
	/*printf("add_image(): frame=%d, plane=%d\n",frame,plane);*/
        add_image(&studies[studyN].imagepack,scale,&imagelists[studyN],frame,plane,&min[studyN],&max[studyN],curWidget,widgetsPerRow,studyN);
      }
    }
  }
  /* Rewind the lists */
  for(studyN=0;studyN<studyCount;studyN++)
    if(imagelists[studyN])
      while(imagelists[studyN]->prev)
        imagelists[studyN]=imagelists[studyN]->prev;
  /* Replace images / add widgets */
  while(imagelists[0] || imagelists[1]) {
    TMP_ImageList *next;
    char name[30];
    for(studyN=0;studyN<studyCount;studyN++) {
      Arg arg[10];
      int argN;
      if(!imagelists[studyN]) continue;
      next=imagelists[studyN]->next;
      if(replace) { /* Replace existing images */
        if(!imglist) {
          fprintf(stderr,"add_images(%d,%d): image_list doesn't have enough images !\n",scale,replace);
          fatal_error_dialog("Missing images !");
          return;
        }
	XtVaSetValues((Widget)imglist->img->widget,petimageNimage,imagelists[studyN]->image->dimage,NULL);
        XDestroyImage(imglist->img->image);
	XDestroyImage(imglist->img->dimage);
        imglist->img->image=imagelists[studyN]->image->image;
	imglist->img->dimage=imagelists[studyN]->image->dimage;
	imglist=imglist->next;
      } else { /* Create new images and widgets */
        image_list_add(imagelists[studyN]->image);
	frame=imagelists[studyN]->frame+1;
	plane=imagelists[studyN]->plane+1;
        sprintf(name,"petimage%dx%dof%d",frame,plane,studyN);
        imagelists[studyN]->image->widget=(PetimageWidget)XtVaCreateManagedWidget(name,
		petimageWidgetClass,matrix_widget,
		XmNwidth,xsize+6,
		XmNheight,ysize+6,
		XmNhighlightThickness,3,
  		petimageNimage,imagelists[studyN]->image->dimage,
		petimageNframe,frame,
		petimageNplane,plane,
		petimageNframeColor,extra_palette[RED].pixel,
		petimageNselected,curWidget==0,
		petimageNstudy,studyN,
		XmNtraversalOn,FALSE,
		XmNbackground,yait_palette[0][0].pixel,
		petimageNroiColor,extra_palette[GREEN].pixel,
		petimageNselRoiColor,extra_palette[RED].pixel,
		petimageNROI,panels.show_matrix_rois?&roi_list:NULL,
		NULL);
        argN=0;
        if(curWidget<widgetsPerRow) {
          XtSetArg(arg[argN],XmNtopAttachment,XmATTACH_FORM); argN++;
          if(curWidget==0) {
            XtSetArg(arg[argN],XmNleftAttachment,XmATTACH_FORM); argN++;
            selected_image=imagelists[studyN]->image;
          } else {
            XtSetArg(arg[argN],XmNleftAttachment,XmATTACH_WIDGET); argN++;
            XtSetArg(arg[argN],XmNleftWidget,last_image->prev->img->widget); argN++;
          }
        } else {
          ImageList *tmplist=last_image;
          int steps;
          for(steps=0;steps<=curWidget%widgetsPerRow;steps++) tmplist=tmplist->prev;
          XtSetArg(arg[argN],XmNtopAttachment,XmATTACH_WIDGET); argN++;
          XtSetArg(arg[argN],XmNtopWidget,tmplist->img->widget); argN++;
          if(curWidget%widgetsPerRow==0) {
            XtSetArg(arg[argN],XmNleftAttachment,XmATTACH_FORM); argN++;
          } else {
            XtSetArg(arg[argN],XmNleftAttachment,XmATTACH_WIDGET); argN++;
            XtSetArg(arg[argN],XmNleftWidget,last_image->prev->img->widget); argN++;
          } 
        }
        XtSetValues((Widget)imagelists[studyN]->image->widget,arg,argN);
        XtAddCallback((Widget)imagelists[studyN]->image->widget,petimageNselectedCb,(void*)select_image,NULL);
        curWidget++;
      }
      free(imagelists[studyN]);
      imagelists[studyN]=next;
    }
  }
}

/* Create a YaIT_Image from a matrix (and append it to TMP_ImageList) */
static void add_image(IMG *imgpack,ScaleType scale,TMP_ImageList **list,int frame,int plane,float *min,float *max,int curWidget,int widgetsPerRow,int studyN) {
  static Display *display=NULL;
  static Visual *visual=NULL;
  unsigned char *pdata;
  float *petPicture;
  XImage *ximage;
  YaIT_Image *img;
  int i,j,k;
  int framei,planei;
  int xsize,ysize;

  xsize=imgpack->dimx;
  ysize=imgpack->dimy;
  if(find_frame_plane_index(studies+studyN,frame,plane,&framei,&planei)) {
    fprintf(stderr,"add_image(): Error ! Couldn't find frame %d, plane %d from study %d !\n",frame,plane,studyN);
    fatal_error_dialog("Requested image was not found !");
  }
  if(!display) {
    display=XtDisplay(toplevel);
    visual=DefaultVisual(display,XDefaultScreen(display));
  }
  petPicture=(float*)malloc(xsize*ysize*sizeof(float));
  if(!petPicture) {
    fprintf(stderr,"add_image(): Cannot allocate %d bytes",xsize*ysize*sizeof(float));
    fatal_error_dialog("Out of memory !");
    return;
  }
  pdata=malloc(xsize*ysize);
  if(!pdata) {
    fprintf(stderr,"add_image(): Can't allocate %d bytes",xsize*ysize);
    fatal_error_dialog("Memory is full!\n");
    return;
  }
  /* Get PET image */
  k=0;
  if(scale==ScaleAutomatic) {*min=100000.0; *max=-100000.0;}
  for(i=0;i<ysize;i++) {
    for(j=0;j<xsize;j++) {
      petPicture[k]=imgpack->m[planei][i][j][framei];
      if(scale==ScaleAutomatic) {
        if(petPicture[k]<*min)
          *min=petPicture[k];
        if(petPicture[k]>*max)
          *max=petPicture[k];
      }
      k++;
    }
  }
  if(panels.no_neg_values){
    for(k=0; k<xsize*ysize; k++){ 
      if(petPicture[k] < 0) petPicture[k]=0.0;
    }
    *min=0.0;
  } else if(*min<0.0) {
    negative_values=1;
  }
  /* Scale values to the range of the palette */
  for(k=0;k<xsize*ysize; k++) { 
    double val;
    val=(petPicture[k]-*min)/(fabs(*min)+fabs(*max));
    pdata[k]=val*(PALETTE_COLORS-1);
  }
  /* Create XImage */
  ximage=XCreateImage(display,visual,8,ZPixmap,0,pdata,xsize,ysize,8,0);
  /* Create YaIT_Image */
  img=create_YaIT_Image(ximage);
  img->studyN=studyN;
  img->dimage=screen_format_ximage(ximage,studyN);
  if(img->dimage==NULL) return;
  if(*list==NULL) {
    *list=(TMP_ImageList*)malloc(sizeof(TMP_ImageList));
    memset(*list,0,sizeof(TMP_ImageList));
  } else {
    (*list)->next=(TMP_ImageList*)malloc(sizeof(TMP_ImageList));
    memset((*list)->next,0,sizeof(TMP_ImageList));
    (*list)->next->prev=*list;
    *list=(*list)->next;
  }
  (*list)->image=img;
  (*list)->frame=frame;
  (*list)->plane=plane;
  free(petPicture);
}

/* Find the smallest and largest values in the matrix */
void get_minmax(Study *study,float *min,float *max) {
  int b,a,i,j,frame,plane=-1;
  for(b=0; b<study->framelist.nr; b++){
    frame=study->framelist.i[b]-1;
    for(a = 0; a < study->planelist.nr; a++){
      for(j=0; j< study->imagepack.dimz; j++){
        if(study->planelist.i[a]==study->imagepack.planeNumber[j]){
	plane=j;
	break;
      }
    }
    if(plane==-1) {
      fprintf(stderr,"get_minmax(): Couldn't find plane !\n");
      break;
    }
    for(i = 0; i < study->imagepack.dimy; i++){
      for(j = 0; j < study->imagepack.dimx; j++){
        if(study->imagepack.m[plane][i][j][frame] < *min)
          *min =study->imagepack.m[plane][i][j][frame];
        if(study->imagepack.m[plane][i][j][frame] > *max)
          *max = study->imagepack.m[plane][i][j][frame];
	}
      }
    }
  }
}

/* Find an image. If study is <0, then all studies are included in the search.  */
YaIT_Image *find_image(int study,int frame,int plane) {
  ImageList *list=image_list;
  int f,p,s;
  while(list) {
    XtVaGetValues((Widget)list->img->widget,petimageNframe,&f,petimageNplane,&p,petimageNstudy,&s,NULL);
    if((study<0 || (study==s)) && f-1==frame && p-1==plane) {
      return list->img;
    }
    list=list->next;
  }
  return NULL;
}

/* Get the position of an image from the imagelist */
ImageList *find_image_pos(YaIT_Image *image) {
  ImageList *list=image_list;
  while(list) {
    if(list->img==image) return list;
    list=list->next;
  }
  return NULL;
}

/* Find the indexes for the given frame and plane */
int find_frame_plane_index(Study *study,int frame,int plane,int *iframe,int *iplane) {
  int a,j;
  /* Note. The frame number list is not stored in imagepack data structure, so we just have
     to trust that they are consecutive */
  //for(b=0;b<study->framelist.nr;b++) {
  //  if(frame==(study->framelist.i[b]-1)) {
  for(a = 0; a < study->planelist.nr; a++){
    for(j=0; j<study->imagepack.dimz; j++){
      if(study->planelist.i[a]==study->imagepack.planeNumber[j]){
        if(plane==study->planelist.i[a]-1) {
         *iframe=frame;
         *iplane=j;
         return 0;
       }
      }
    }
  }
  return 1;
}

/* Select an image when you know the frame and plane numbers */
int select_image_n(int study,int frame,int plane) {
  YaIT_Image *img;
  img=find_image(study,frame,plane);
  if(img) {
    select_image((Widget)img->widget,(XtPointer)1,NULL);
    return 0;
  }
  return 1;
}

/* Select an image when you know the petimage widget */
static void select_image(Widget widget, XtPointer client_data, XtPointer call_data) {
  ImageList *list=image_list;
  int frame,plane;
  if(selected_image && widget==(Widget)selected_image->widget) return;
  XtVaSetValues(widget,petimageNselected,TRUE,NULL);
  XtVaGetValues(widget,petimageNframe,&frame,petimageNplane,&plane,NULL);
  frame--;
  plane--;
  if(selected_image)
    XtVaSetValues((Widget)selected_image->widget,petimageNselected,FALSE,NULL);
  while(list) {
    if(list->img->widget==(PetimageWidget)widget) {
      selected_image=list->img;
    }
    list=list->next;
  }
  /* Update the spinboxes */
  if(client_data==NULL) {
    XmStringTable table;
    char tmp[5];
    int count;
    Arg getval[2];
    XtSetArg(getval[0],XmNvalues,&table);
    XtSetArg(getval[1],XmNnumValues,&count);
    XtGetValues(frame_spin,getval,2);
    sprintf(tmp,"%d",frame+1);
    XtVaSetValues (frame_spin, XmNposition, find_table_entry(table,count,tmp), NULL);
    XtGetValues(plane_spin,getval,2);
    sprintf(tmp,"%d",plane+1);
    XtVaSetValues (plane_spin,XmNposition,find_table_entry(table,count,tmp),NULL);
  }
  zoom_matrix.frame=frame;
  zoom_matrix.plane=plane;

  /* Update the message area */
  message_area_info();

}

void show_color_table(int d_limit, int u_limit,int cmapN)
{
  colormap_setscaleN(XtDisplay(color_bar),studies[cmapN].color,d_limit,u_limit,cmapN);
  ColorslideUpdate(color_bar);
  if(!haveRWpalette) {
    if(zoom_matrix.zoom) { 
      refresh_zoom(cmapN);
    } else {
      refresh_image_list(cmapN);
      update_matrix_view(cmapN);
    }
  }
}

/* Redraw the images in matrix widget */
/* TODO: Refresh only the images that are visible */
void update_matrix_view(int studyN) {
  ImageList *images=get_image_list();
  while(images) {
    if(studyN<0 || images->img->studyN==studyN)
      PetimageUpdate(images->img->widget);
    images=images->next;
  }
}

/* Show info in message area */
void message_area_info(void) {
  static const char *strframe[2]={"frame","frames"};
  static const char *strplane[2]={"plane","planes"};
  char msg[256];
  Study *study;
  if(zoom_matrix.zoom==0) {
    if(selected_image) {
      study=&studies[selected_image->studyN];
    } else {
      study=&studies[0];
    }
    sprintf(msg,"%s: contains: %d %s %d %s (%d,%d)",
   	study->filename,study->imagepack.dimt,strframe[study->imagepack.dimt>1],
	study->imagepack.dimz,strplane[study->imagepack.dimz>1],
	zoom_matrix.frame+1,zoom_matrix.plane+1);
  } else {
    if(strlen(studies[1].filename)>=4) {
      int frame1,frame2,plane1,plane2;
      ZoomImage *zi=get_zoom_image();
      XtVaGetValues((Widget)zi->widget[0],petimageNframe,&frame1,petimageNplane,&plane1,NULL);
      XtVaGetValues((Widget)zi->widget[1],petimageNframe,&frame2,petimageNplane,&plane2,NULL);
      sprintf(msg,"Zoom %d: %s (%d,%d) , %s (%d,%d)",
      	zoom_matrix.zoom,
	studies[0].filename,frame1,plane1,
	studies[1].filename,frame2,plane2);
    } else {
      sprintf(msg,"Zoom: %d %s (%d,%d)",
      	zoom_matrix.zoom,
	studies[0].filename,zoom_matrix.frame+1,zoom_matrix.plane+1);
    }
  }
  if(negative_values)
      strcat(msg," (there are negative pixel values)");
  XmTextFieldSetString(message_w, msg);
}

/* Show info about a certain study in message area */
void message_area_info_study(Study *study) {
  static const char *strframe[2]={"frame","frames"};
  static const char *strplane[2]={"plane","planes"};
  char msg[256];
  sprintf(msg,"%s: contains: %d %s %d %s",
  	study->filename,study->imagepack.dimt,strframe[study->imagepack.dimt>1],
	study->imagepack.dimz,strplane[study->imagepack.dimz>1]);
  XmTextFieldSetString(message_w, msg);
}

/* Get the YaIT image that will appear next to image in zoomed view. */
/* Returns NULL if the image doesn't have a neighbour */
YaIT_Image *get_2nd_zoom_image(YaIT_Image *image) {
  ImageList *list;
  if(strlen(studies[1].filename)<4) return NULL;
  list=find_image_pos(image);
  if(image->studyN>0) {
    if(list->prev && list->prev->img->studyN==0) {
      return list->prev->img;
    }
  } else {
    if(list->next && list->next->img->studyN==1) {
      return list->next->img;
    }
  }
  return NULL;
}

