#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <Xm/Xm.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <ecat63.h>

#include "petimageP.h"
#include "roiutils.h"

#define DrawGC(w) XmField(w,offsets,Petimage,draw_GC,GC)
#define XorGC(w) XmField(w,offsets,Petimage,xor_GC,GC)
#define Foreground(w)   XmField(w,offsets,XmPrimitive,foreground,Pixel)
#define Highlighted(w) XmField(w,offsets,XmPrimitive,highlighted,Boolean)
#define BackgroundPixel(w) XmField(w,offsets,Core,background_pixel,Pixel)
#define HighlightThickness(w) \
                XmField(w,offsets,XmPrimitive,highlight_thickness,Dimension)
#define Highlighted(w) XmField(w,offsets,XmPrimitive,highlighted,Boolean)

#define Width(w) XmField(w,offsets,Core,width,Dimension)
#define Height(w) XmField(w,offsets,Core,height,Dimension)

#define PetImage(w) XmField(w,offsets,Petimage,image,XImage*)
#define LegoRoi(w) XmField(w,offsets,Petimage,lego_roi,XImage*)

#define Frame(w) XmField(w,offsets,Petimage,frame,int)
#define Plane(w) XmField(w,offsets,Petimage,plane,int)
#define Study(w) XmField(w,offsets,Petimage,study,int)
#define Numcod(w) XmField(w,offsets,Petimage,numcod,int)
#define Selected(w) XmField(w,offsets,Petimage,selected,int)
#define FrameColor(w) XmField(w,offsets,Petimage,framecolor,int)
#define Roi(w) XmField(w,offsets,Petimage,rois,ROI_list*)
#define Zoom(w) XmField(w,offsets,Petimage,zoom,int)
#define RoiPreview(w) XmField(w,offsets,Petimage,roi_preview,PetimageRoiPoint*)
#define RoiMode(w) XmField(w,offsets,Petimage,roimode,int)
#define SelectedRoi(w) XmField(w,offsets,Petimage,selectedRoi,ROI*)
#define RequestRoi(w) XmField(w,offsets,Petimage,requestRoi,ROI*)
#define RoiColor(w) XmField(w,offsets,Petimage,roiColor,int)
#define SelRoiColor(w) XmField(w,offsets,Petimage,selRoiColor,int)
#define TmpX(w) XmField(w,offsets,Petimage,tmpX,int)
#define TmpY(w) XmField(w,offsets,Petimage,tmpY,int)
#define TmpD(w) XmField(w,offsets,Petimage,tmpD,double)
#define TmpD2(w) XmField(w,offsets,Petimage,tmpD2,double)


static void ClassInitialize(void);
static void Initialize(PetimageWidget request,PetimageWidget new);
static void Redisplay(PetimageWidget w,XEvent *event,Region region);
static void Resize(PetimageWidget w);
static void Destroy(PetimageWidget w);
static Boolean SetValues(PetimageWidget current,PetimageWidget request,PetimageWidget new);
static XtGeometryResult QueryGeometry(PetimageWidget w,XtWidgetGeometry *intended,XtWidgetGeometry *reply);

static void select_petimage(PetimageWidget w,XEvent *event);
static void click_petimage(PetimageWidget w,XEvent *event);
static void hover_petimage(PetimageWidget w,XEvent *event);
static void key_petimage(PetimageWidget w,XEvent *event);

static void draw_roi(PetimageWidget widget,ROI_list *rois,GC gc,Display *d,Drawable win,int x,int y,int w,int h);
static void draw_trace_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset);
static void draw_rectangle_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset);
static void draw_circle_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset);
static void draw_ellipse_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset);
static void draw_rotated_trace_roi(PetimageRoiPoint *rp,Display *d,Drawable win, GC gc,int oX,int oY,double a);

static char defaultTranslations[] =
  "<Btn1Down>: Select()\n \
   <Btn2Down>: Select()\n \
   <Btn2Down>: Click()\n \
   <Btn3Down>: Click()\n \
   <Motion>: MouseMovement()\n \
   <KeyDown>: Keyboard()\n \
   <KeyUp>: Keyboard()\n";


static XtActionsRec actionsList[] = {
	{ "Select", (XtActionProc) select_petimage},
	{ "Click", (XtActionProc) click_petimage},
	{ "MouseMovement",(XtActionProc) hover_petimage},
	{ "Keyboard",(XtActionProc) key_petimage}
};

static XmPartResource resources[] = {
	{petimageNselectedCb,petimageCselectedCb,XtRCallback,sizeof(XtPointer),
		XmPartOffset(Petimage,selected_callback),XtRCallback,NULL},
	{petimageNmouseCb,petimageCmouseCb,XtRCallback,sizeof(XtPointer),
		XmPartOffset(Petimage,mouse_callback),XtRCallback,NULL},
	{petimageNroiSelectedCb,petimageCroiSelectedCb,XtRCallback,sizeof(XtPointer),
		XmPartOffset(Petimage,roi_selected_callback),XtRCallback,NULL},
	{petimageNroiCommandCb,petimageCroiCommandCb,XtRCallback,sizeof(XtPointer),
		XmPartOffset(Petimage,roi_cmd_callback),XtRCallback,NULL},
	{petimageNhoverCb,petimageChoverCb,XtRCallback,sizeof(XtPointer),
		XmPartOffset(Petimage,hover_callback),XtRCallback,NULL},
	{petimageNimage,petimageCimage,XtRPointer,sizeof(XtPointer),
                XmPartOffset(Petimage,image),XmRImmediate,NULL},
	{petimageNframe,petimageCframe,XtRInt,sizeof(int),
		XmPartOffset(Petimage,frame),XmRImmediate,(XtPointer)-1},
	{petimageNplane,petimageCplane,XtRInt,sizeof(int),
		XmPartOffset(Petimage,plane),XmRImmediate,(XtPointer)-1},
	{petimageNstudy,petimageCstudy,XtRInt,sizeof(int),
		XmPartOffset(Petimage,study),XmRImmediate,(XtPointer)0},
	{petimageNselected,petimageCselected,XtRInt,sizeof(int),
		XmPartOffset(Petimage,selected),XmRImmediate,(XtPointer)0},
	{petimageNframeColor,petimageCframeColor,XtRInt,sizeof(int),
		XmPartOffset(Petimage,framecolor),XmRImmediate,(XtPointer)0},
	{petimageNROI,petimageCROI,XtRPointer,sizeof(XtPointer),
		XmPartOffset(Petimage,rois),XmRImmediate,NULL},
	{petimageNzoom,petimageCzoom,XtRInt,sizeof(int),
		XmPartOffset(Petimage,zoom),XmRImmediate,(XtPointer)1},
	{petimageNroiMode,petimageCroiMode,XtRInt,sizeof(int),
		XmPartOffset(Petimage,roimode),XmRImmediate,(XtPointer)-1},
	{petimageNselectedRoi,petimageCselectedRoi,XtRPointer,sizeof(XtPointer),
		XmPartOffset(Petimage,selectedRoi),XmRImmediate,NULL},
	{petimageNrequestRoi,petimageCrequestRoi,XtRPointer,sizeof(XtPointer),
		XmPartOffset(Petimage,requestRoi),XmRImmediate,NULL},
	{petimageNroiColor,petimageCroiColor,XtRInt,sizeof(int),
		XmPartOffset(Petimage,roiColor),XmRImmediate,(XtPointer)0},
	{petimageNselRoiColor,petimageCselRoiColor,XtRInt,sizeof(int),
		XmPartOffset(Petimage,selRoiColor),XmRImmediate,(XtPointer)0}

};

PetimageClassRec petimageClassRec = {
    {                                   /* core_class fields    */
    (WidgetClass) &xmPrimitiveClassRec, /* superclass           */
    "Petimage",                         /* class_name           */
    sizeof(PetimagePart),             /* widget_size          */
    ClassInitialize,                    /* class_initialize     */
    NULL,                               /* class_part_initialize*/
    False,                              /* class_inited         */
    (void*)Initialize,                  /* initialize           */
    NULL,                               /* initialize_notify    */
    XtInheritRealize,                   /* realize              */
    actionsList,                        /* actions              */
    XtNumber(actionsList),              /* num_actions          */
    (XtResourceList)resources,          /* resources            */
    XtNumber(resources),                /* num_resources        */
    NULLQUARK,                          /* xrm_class            */
    True,                               /* compress_motion      */
    True,                               /* compress_exposure    */
    True,                               /* compress_enterleave  */
    False,                              /* visible_interest     */
    (void*)Destroy,                            /* destroy              */
    (void*)Resize,                             /* resize               */
    (void*)Redisplay,                          /* expose               */
    (void*)SetValues,                          /* set_values           */
    NULL,                               /* set_values_hook      */
    XtInheritSetValuesAlmost,           /* set_values_almost    */
    NULL,                               /* get_values_hook      */
    NULL,                               /* accept_focus         */
    XtVersionDontCheck,                 /* version              */
    NULL,                               /* callback_private     */
    defaultTranslations,                /* tm_table             */
    (void*)QueryGeometry,               /* query_geometry       */
    NULL,                               /* disp accelerator     */
    NULL                                /* extension            */
    },
    {                                   /* primitive_class record */
    XmInheritWidgetProc,                /* border_highlight     */
    XmInheritWidgetProc,                /* border_unhighlight   */
    XtInheritTranslations,              /* translations         */
    (void*)select_petimage,             /* arm_and_activate     */
    NULL,                               /* syn resources        */
    0,                                  /* num syn_resources    */
    NULL,                               /* extension            */
    },
    {                                   /* petimage_class record  */
    0,                                  /* extension            */
    }
};



static XmOffsetPtr offsets; /* Part Offset table for XmResolvePartOffsets */
externaldef(petimagewidgetclass) WidgetClass petimageWidgetClass = (WidgetClass) &petimageClassRec;

/***********************************************\
| Class methods					|
\***********************************************/

static void create_GC(PetimageWidget w)
{
    XGCValues       values;
    XtGCMask        valueMask;
    Display	   *d=XtDisplay(w);

    valueMask = GCForeground | GCBackground | GCGraphicsExposures;
    values.foreground = Foreground(w);
    values.background = BackgroundPixel(w);
    values.graphics_exposures = False;
    DrawGC(w) = XtGetGC((Widget)w,valueMask,&values);

    valueMask = valueMask | GCFunction;
    values.function=GXxor;
    XorGC(w) = XtGetGC((Widget)w,valueMask,&values);
    if(DefaultDepth(XtDisplay(w),DefaultScreen(XtDisplay(w)))>8)
      XSetForeground(d,XorGC(w),WhitePixel(d,DefaultScreen(d)));
    else
      XSetForeground(d,XorGC(w),60);
    XSetBackground(d,XorGC(w),BlackPixel(d,DefaultScreen(d)));
}

static void ClassInitialize(void)
{
    XmResolvePartOffsets(petimageWidgetClass, &offsets);
}

static void Initialize(PetimageWidget request,PetimageWidget new)
{
  if (Width(request) == 0)
    Width(new) = 100;
  if (Height(request) == 0)
    Height(new) = 50;
  create_GC(new);
  RoiPreview(new)=NULL;
  LegoRoi(new)=NULL;
  Resize(new);
  Numcod(new)=mat_numcod(Frame(request), Plane(request), 1, 0, 0);
}

static void Destroy(PetimageWidget w)
{
    XtReleaseGC ((Widget)w, DrawGC(w));
    XtReleaseGC ((Widget)w, XorGC(w));
}

static void Resize(PetimageWidget w) {
}

static void Redisplay(PetimageWidget widget,XEvent *event,Region region) {
  XImage *image;
  int x,y,w,h;
  Drawable win;
  Display *d;
  ROI_list *rois;
  GC gc;
  int b;
  d=XtDisplay(widget);
  win=XtWindow(widget);
  gc=DrawGC(widget);
  b=HighlightThickness(widget);
  image=PetImage(widget);
  if(Selected(widget)) {/* Draw frame */
    int width,height,thick;
    width=Width(widget);
    height=Height(widget);
    thick=HighlightThickness(widget);
    XSetForeground(d,gc,FrameColor(widget));
    XFillRectangle(d,win,gc,0,0,width,thick);
    XFillRectangle(d,win,gc,0,thick,thick,height-thick);
    XFillRectangle(d,win,gc,width-thick,thick,thick,height-thick);
    XFillRectangle(d,win,gc,thick,height-thick,width-thick,thick);
  }
  if(!image) return;
  if(event) {
    x=event->xexpose.x;
    y=event->xexpose.y;
    if(x>b) x-=b; else x=0;
    if(y>b) y-=b; else y=0;
    w=event->xexpose.width;
    h=event->xexpose.height;
  } else {
    x=0;
    y=0;
    w=image->width;
    h=image->height;
  }
  XPutImage(d,win,gc,image,x,y,b+x,b+y,w,h);
  rois=Roi(widget);
  if(rois)
    draw_roi(widget,rois,gc,d,win,b+x,b+y,w,h);
}

static void draw_roi(PetimageWidget widget,ROI_list *rois,GC gc,Display *d,Drawable win,int x,int y,int w,int h) {
  RoiList *list=rois->rois;
  ROI *selected;
  int border;
  selected=SelectedRoi(widget);
  border=HighlightThickness(widget);
  while(list) {
    ROI *roi=list->roi;
    if(roi==selected)
      XSetForeground(d,gc,SelRoiColor(widget));
    else
      XSetForeground(d,gc,RoiColor(widget));
    if( roi->userdata && ((roi_userdata*)roi->userdata)->extra_study >= 0 ) { /* ROI has info on which study it was originally created in */
      if(Study(widget)==((roi_userdata*)roi->userdata)->extra_study) { /* This is not that study */
        if(Numcod(widget)!=((roi_userdata*)roi->userdata)->extra_matnum) {
	  list=list->next;	/* Wrong frame/plane */
	  continue;
	}
      } else {	/* This is that study */
        if(Numcod(widget)!=roi->matnum) {list=list->next; continue;}
      }
    } else { /* No information. Probably only one study available */
      if(Numcod(widget)!=roi->matnum) {list=list->next; continue;}
    }
    
    switch(roi->type) {
      case ROI_TRACE:
        draw_trace_roi(gc,d,win,roi,Zoom(widget),border);
	break;
      case ROI_RECTANGULAR:
        draw_rectangle_roi(gc,d,win,roi,Zoom(widget),border);
	break;
      case ROI_CIRCULAR:
        draw_circle_roi(gc,d,win,roi,Zoom(widget),border);
	break;
      case ROI_ELLIPSE:
        draw_ellipse_roi(gc,d,win,roi,Zoom(widget),border);
        break;
      default:
        fprintf(stderr,"PetimageWidget.draw_roi(): unknown ROI type %d !\n",roi->type);
	break;
    }
    list=list->next;
  }
}

static void draw_trace_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset) {
  int dx,dy,r;
  float f;
  dx=roi->pos_x;
  dy=-roi->pos_y;
  f=zoom/(float)roi->zoom;
  for(r=0;r<roi->point_nr;r++) {
    int x,y,x2,y2,n;
    if(r<roi->point_nr-1) n=r+1; else n=0;
    x=(roi->pos_x+roi->x[r])*f;
    y=(roi->pos_y+roi->y[r])*f;
    x2=(roi->pos_x+roi->x[n])*f;
    y2=(roi->pos_y+roi->y[n])*f;
    XDrawLine(d,win,gc,x+offset,y+offset,x2+offset,y2+offset);
  }
}

static void draw_rectangle_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset) {
  int x,y,w,h;
  float f;
  f=zoom/(float)roi->zoom;
  x=roi->pos_x*f;
  y=roi->pos_y*f;
  w=roi->w*f;
  h=roi->h*f;
  XDrawRectangle(d,win,gc,x+offset,y+offset,w,h);
}

static void draw_circle_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset) {
  float f;
  f=zoom/(float)roi->zoom;
  XDrawArc(d,win,gc,roi->pos_x*f+offset-roi->w*f,roi->pos_y*f+offset-roi->w*f,(roi->w*2)*f,(roi->w*2)*f,0,23040);
}

static void draw_ellipse_roi(GC gc,Display *d,Drawable win,ROI *roi,int zoom,int offset) {
  float f;
  f=zoom/(float)roi->zoom;
  XDrawArc(d,win,gc,roi->pos_x*f+offset-roi->w*f,roi->pos_y*f+offset-roi->h*f,(roi->w*2)*f,(roi->h*2)*f,0,23040);
}

static Boolean SetValues(PetimageWidget current,PetimageWidget request,PetimageWidget new)
{
  Boolean redraw=False;
  Boolean delete_lego=False;
  if(Selected(new) != Selected(current)) redraw=True;
  if(FrameColor(new) != FrameColor(current) && Selected(new)) redraw=True;
  if(Roi(new) != Roi(current)) {redraw=True;delete_lego=True;}
  if(RoiPreview(new) != RoiPreview(current)) redraw=True;
  if(Plane(new) != Plane(current) || Frame(new) != Frame(current)) {
    Numcod(new)=mat_numcod(Frame(new), Plane(new), 1, 0, 0);
  }
  if(SelectedRoi(new) != SelectedRoi(current)) {
    delete_lego=True;
    if(redraw==False && Roi(new)) {
      draw_roi(new,Roi(new),DrawGC(new),XtDisplay((Widget)new),XtWindow((Widget)new),0,0,0,0);
    }
  }
  if(RequestRoi(new) != RequestRoi(current)) {
    XtCallCallbacks((Widget)new,petimageNroiSelectedCb,(XtPointer)RequestRoi(new));
    RequestRoi(new)=NULL;
  }
  if(delete_lego) {
    if(LegoRoi(new)) {
      XDestroyImage(LegoRoi(new));
      LegoRoi(new)=NULL;
    }
  }
  return redraw;
}

static XtGeometryResult QueryGeometry (PetimageWidget w,XtWidgetGeometry *intended,XtWidgetGeometry *reply)
{
    reply->request_mode = 0;

    if ((intended->request_mode & (~(CWWidth | CWHeight))) != 0)
        return (XtGeometryNo);

    reply->request_mode = (CWWidth | CWHeight);
    reply->width = 128+2*HighlightThickness(w);
    reply->height = 128+2*HighlightThickness(w);

    if (reply->width != intended->width ||
        reply->height != intended->height ||
        intended->request_mode != reply->request_mode)
        return (XtGeometryAlmost);
    else {
        reply->request_mode = 0;
        return (XtGeometryYes);
    }
}

void PetimageUpdate(PetimageWidget w) {
  if(LegoRoi(w)) { /* Delete the lego roi buffer just to be safe */
    XDestroyImage(LegoRoi(w));
    LegoRoi(w)=NULL;
  }
  Redisplay(w,NULL,NULL);
}

/* Keyboard handler */
static void key_petimage(PetimageWidget w,XEvent *event) {
  Display *d;
  Drawable win;
  int b;
  d=XtDisplay(w);
  b=HighlightThickness(w);
  if(event->xkey.type!=KeyRelease) return;
  if(event->xkey.state&ControlMask) { /* Control: Show LEGO ROI */
    if(SelectedRoi(w)==NULL) return;
    d=XtDisplay(w);
    win=XtWindow(w);
    if(LegoRoi(w)==NULL) {
      LegoRoi(w)=lego_roi(d,DefaultVisual(d,0),
	SelectedRoi(w),Width(w)/Zoom(w),Height(w)/Zoom(w),Zoom(w));
    }
    XPutImage(d,win,XorGC(w),LegoRoi(w),0,0,b,b,Width(w),Height(w));
  } else {
    KeySym key;
    key=XKeycodeToKeysym(d,event->xkey.keycode,0);
    switch(key) {
      case XK_Escape: printf("ESC pressed !\n"); break;
      case XK_m:
      case XK_Delete:
        XtCallCallbacks((Widget)w,petimageNroiCommandCb,&event->xkey);
	break;
      default: break;
    }
  }
}

/* Left mouse button */
static void select_petimage(PetimageWidget w,XEvent *event) {
  ROI_list *rois=Roi(w);
  RoiList *list;
  int zoom=Zoom(w);
  XmProcessTraversal((Widget)w, XmTRAVERSE_CURRENT);
  /* Select ROI (Note that this doesn't actually select the ROI,  */
  /* it just calls the ROI selection callback and its up to the   */
  /* application to actually select the ROI.			  */
  if(RoiPreview(w)==NULL && rois && zoom>0) {
    int numcod,largest=900000,area;
    ROI *selected=NULL;
    double d;
    list=rois->rois;
    numcod=Numcod(w);
    while(list) {
      if(numcod!=list->roi->matnum) {list=list->next; continue;}
      d=Zoom(w)/list->roi->zoom;
      if(hit_roi(list->roi,event->xbutton.x/d,event->xbutton.y/d)) {
        area=roi_area(list->roi);
	if(area<largest) {
	  selected=list->roi;
	  largest=area;
	}
      }
      list=list->next;
    }
    if(selected) {
      /*printf("ROI %d selected\n",selected);*/
      XtCallCallbacks((Widget)w,petimageNroiSelectedCb,(XtPointer)selected);
    }
  }
  /* Call the mouse button callbacks */
  XtCallCallbacks((Widget)w,petimageNselectedCb,NULL);
  XtCallCallbacks((Widget)w,petimageNmouseCb,&event->xbutton);
}

/* Other mouse buttons */
static void click_petimage(PetimageWidget w,XEvent *event) {
  XtCallCallbacks((Widget)w,petimageNmouseCb,&event->xbutton);
}

/* Draw the roi preview */
static void hover_petimage(PetimageWidget w,XEvent *event) {
  int roimode;
  Display *d;
  Window win;
  int x,y;
  GC gc;
  PetimageRoiPoint *rp;
  rp=RoiPreview(w);
  XtCallCallbacks((Widget)w,petimageNhoverCb,&event->xmotion);
  if(rp) {
    gc=XorGC(w);
    d=XtDisplay(w);
    win=XtWindow(w);
    x=event->xmotion.x;
    y=event->xmotion.y;
    roimode=RoiMode(w);
    if(roimode>=100 && roimode<200 && TmpX(w)<0) {
      TmpX(w)=x;
      TmpY(w)=y;
    }
    switch(roimode) {
      case ROI_TRACE:		/* Trace ROI */
	XDrawLine(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
	rp->x=x;
	rp->y=y;
	XDrawLine(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
	break;
      case 100+ROI_TRACE: {	/* Move Trace ROI */
        int fx,fy,dx,dy;
	fx=rp->x;
	fy=rp->y;
        while(rp->next) {
	  XDrawLine(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
	  rp=rp->next;
	}
	XDrawLine(d,win,gc,rp->x,rp->y,fx,fy);
	dx=TmpX(w)-x;
	dy=TmpY(w)-y;
	TmpX(w)=x;
	TmpY(w)=y;
	rp=RoiPreview(w);
	rp->x-=dx;
	rp->y-=dy;
	fx=rp->x;
	fy=rp->y;
        while(rp->next) {
	  rp->next->x-=dx;
	  rp->next->y-=dy;
	  XDrawLine(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
	  rp=rp->next;
	}
	XDrawLine(d,win,gc,rp->x,rp->y,fx,fy);
      } break;
      case 200+ROI_TRACE: { /* Rotate Trace ROI */
        double a;
	if(TmpD(w)==31013) {
	  TmpD2(w)=atan2(TmpX(w)-x,TmpY(w)-y);
	  a=0;
	} else {
	  a=atan2(TmpX(w)-x,TmpY(w)-y)-TmpD2(w);
	  draw_rotated_trace_roi(RoiPreview(w),d,win,gc,TmpX(w),TmpY(w),TmpD(w));
	}
	TmpD(w)=a;
        draw_rotated_trace_roi(RoiPreview(w),d,win,gc,TmpX(w),TmpY(w),a);
	}
        break;
      case 100+ROI_RECTANGULAR:	/* Rectangle ROI */
      case ROI_RECTANGULAR:
        XDrawRectangle(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
	if(roimode>=100) {
	  int dx,dy;
	  dx=TmpX(w)-x;
	  dy=TmpY(w)-y;
	  rp->next->x-=dx;
	  rp->next->y-=dy;
	  TmpX(w)=x;
	  TmpY(w)=y;
	} else {
	  if(x<rp->next->next->x) {
	    rp->next->x=x;
  	    rp->x=rp->next->next->x-x;
	  } else {
	    rp->next->x=rp->next->next->x;
	    rp->x=x-rp->next->x;
	  }
	  if(y<rp->next->next->y) {
	    rp->next->y=y;
	    rp->y=rp->next->next->y-y;
	  } else {
	    rp->next->y=rp->next->next->y;
	    rp->y=y-rp->next->y;
	  }
	}
	XDrawRectangle(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y);
        break;
      case ROI_CIRCULAR:	/* Circle ROI */
      case 100+ROI_CIRCULAR:
        XDrawArc(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->x,0,23040);
	if(roimode>=100) {
	  int dx,dy;
	  dx=TmpX(w)-x;
	  dy=TmpY(w)-y;
	  rp->next->x-=dx;
	  rp->next->y-=dy;
	  TmpX(w)=x;
	  TmpY(w)=y;
	} else {
	  rp->x=hypot(x-rp->next->next->x,y-rp->next->next->y);
	  rp->next->x=rp->next->next->x-rp->x;
	  rp->next->y=rp->next->next->y-rp->x;
	  rp->x*=2;
	}
	XDrawArc(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->x,0,23040);
	break;
      case ROI_ELLIPSE:		/* Ellipse ROI */
      case 100+ROI_ELLIPSE:
        XDrawArc(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y,0,23040);
	if(roimode>=100) {
	  int dx,dy;
	  dx=TmpX(w)-x;
	  dy=TmpY(w)-y;
	  rp->next->x-=dx;
	  rp->next->y-=dy;
	  TmpX(w)=x;
	  TmpY(w)=y;
	} else {
	  rp->x=abs(x-rp->next->next->x);
	  rp->y=abs(y-rp->next->next->y);
	  rp->next->x=rp->next->next->x-rp->x;
	  rp->next->y=rp->next->next->y-rp->y;
	  rp->x*=2;
	  rp->y*=2;
	}
	XDrawArc(d,win,gc,rp->next->x,rp->next->y,rp->x,rp->y,0,23040);
        break;
    }
  }
}

/* Draw a rotated trace ROI */
static void draw_rotated_trace_roi(PetimageRoiPoint *rp,Display *d,Drawable win, GC gc,int oX,int oY,double a) {
  int fx,fy;
  fx=rp->x;
  fy=rp->y;
  while(rp) {
    int x1,y1,x2,y2;
    double b,c;
    b=atan2(rp->x-oX,rp->y-oY);
    b+=a; c=hypot(rp->x-oX,rp->y-oY);
    x1=sin(b)*c+oX;
    y1=cos(b)*c+oY;
    if(rp->next) {
      b=atan2(rp->next->x-oX,rp->next->y-oY);
      b+=a; c=hypot(rp->next->x-oX,rp->next->y-oY);
    } else {
      b=atan2(fx-oX,fy-oY);
      b+=a; c=hypot(fx-oX,fy-oY);
    }
    x2=sin(b)*c+oX;
    y2=cos(b)*c+oY;

    XDrawLine(d,win,gc,x1,y1,x2,y2);
    rp=rp->next;
  }
}

/* Set a point for the ROI preview */
void PetimageSetRoiDrawPoint(PetimageWidget w,int x,int y) {
  PetimageRoiPoint *newp,*rp=RoiPreview(w);
  newp=(PetimageRoiPoint*)malloc(sizeof(PetimageRoiPoint));
  newp->prev=NULL;
  if(RoiMode(w)==ROI_TRACE) {
    newp->x=x;
    newp->y=y;
  } else {
    newp->x=0;
    newp->y=0;
  }
  if(!rp) { /* This is the first entry. Create the origo point as well */
    rp=(PetimageRoiPoint*)malloc(sizeof(PetimageRoiPoint));
    rp->next=NULL;
    rp->x=x;
    rp->y=y;
  }
  newp->next=rp;
  rp->prev=newp;
  RoiPreview(w)=newp;
}

/* End ROI drawing and close the loop */
void PetimageEndRoiDraw(PetimageWidget w) {
  PetimageRoiPoint *rp,*next;
  int x1=-1,x2=-1,y1=-1,y2=-1;
  char tracemode;
  Drawable win;
  Display *d;
  GC gc;
  rp=RoiPreview(w);
  if(!rp) return;
  d=XtDisplay(w);
  win=XtWindow(w);
  gc=XorGC(w);
  tracemode=RoiMode(w)==ROI_TRACE;
  /* Clear the last line */
  if(rp->next && tracemode) {
    x1=rp->next->x;
    y1=rp->next->y;
    XDrawLine(d,win,gc,x1,y1,rp->x,rp->y);
    x2=x1;
    y2=y1;
  }
  /* Free the points */
  while(rp) {
    next=rp->next;
    if(!next) {	/* Remember the origo coordinates */
      x2=rp->x;
      y2=rp->y;
    }
    free(rp);
    rp=rp->next;
  }
  /* Connect the last fixed point with the origo */
  if(x1>=0 && tracemode)
    XDrawLine(d,win,gc,x1,y1,x2,y2);
  RoiPreview(w)=NULL;
}

/* Highlight a selected ROI from the widgets ROI list */
void PetimageHighlightRoi(PetimageWidget w,ROI *roi,int mode) {
  double d;
  int r;
  if(!w) {
    fprintf(stderr,"PetimageHighlightRoi(NULL,0x%x): No PetimageWidget !\n",(int)roi);
    return;
  }
  d=Zoom(w)/roi->zoom;
  switch(roi->type) {
    case ROI_TRACE:
      PetimageSetRoiDrawPoint(w,(roi->x[0]+roi->pos_x)*d,(roi->y[0]+roi->pos_y)*d);
      RoiPreview(w)->x=(roi->x[1]+roi->pos_x)*d;
      RoiPreview(w)->y=(roi->y[1]+roi->pos_y)*d;
      for(r=2;r<roi->point_nr;r++) {
        PetimageSetRoiDrawPoint(w,0,0);
	RoiPreview(w)->x=(roi->x[r]+roi->pos_x)*d;
	RoiPreview(w)->y=(roi->y[r]+roi->pos_y)*d;
      }
      break;
    case ROI_ELLIPSE:
    case ROI_CIRCULAR:
    case ROI_RECTANGULAR:
      PetimageSetRoiDrawPoint(w,0,0);
      PetimageSetRoiDrawPoint(w,0,0);
      if(roi->type!=ROI_CIRCULAR && roi->type != ROI_ELLIPSE) {
        RoiPreview(w)->next->x=roi->pos_x*d;
        RoiPreview(w)->next->y=roi->pos_y*d;
      } else {
        RoiPreview(w)->next->x=roi->pos_x*d-roi->w*d;
        RoiPreview(w)->next->y=roi->pos_y*d-((roi->type==ROI_ELLIPSE)?roi->h:roi->w)*d;
      }
      if(roi->type!=ROI_RECTANGULAR) {
        RoiPreview(w)->x=roi->w*2*d;
        RoiPreview(w)->y=roi->h*2*d;
      } else {
        RoiPreview(w)->x=roi->w*d;
        RoiPreview(w)->y=roi->h*d;
      }
      break;
    default:
      fprintf(stderr,"PetimageHighlightRoi(w,roi): Unknown ROI type '%d' !\n",roi->type);
      return;
  }
  switch(mode) {
    case ROI_MOVE:
      RoiMode(w)=100+roi->type;
      TmpX(w)=-1;
      break;
    case ROI_ROTATE: {
      PetimageRoiPoint *rp=RoiPreview(w);
      int minx=10000,maxx=-10000,miny=10000,maxy=-10000;
      RoiMode(w)=200+roi->type;
      TmpD(w)=31013;
      TmpD2(w)=0;
      while(rp) {
        if(rp->x<minx) minx=rp->x;
	if(rp->x>maxx) maxx=rp->x;
	if(rp->y<miny) miny=rp->y;
	if(rp->y>maxy) maxy=rp->y;
        rp=rp->next;
      }
      TmpX(w)=minx+(maxx-minx)/2;
      TmpY(w)=miny+(maxy-miny)/2;
      } break;
    default:
      fprintf(stderr,"PetimageHighlightRoi(): Unknown ROI mode '%d' !\n",mode);
      break;
  }
}

/* Delete Lego ROI from Petimage buffer */
void PetimageFlush(PetimageWidget w) {
  if(LegoRoi(w)) {
    XDestroyImage(LegoRoi(w));
    LegoRoi(w)=NULL;
  }
}

