#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>
#include "main.h"
#include "petimage.h"
#include "zoom_images.h"
#include "profile.h"

/* Initial size of the profile window */
#define PROFILE_DEFAULT_W 640
#define PROFILE_DEFAULT_H 480

#define PROFILE_BORDER	  10

/* A structure to hold profile data */
typedef struct {
  Pixmap buffer;
  int x1,y1,x2,y2;
  float *data;
  int len;
  float min,max;
} ProfileData;

static void profile_popup(Display *dpy,int study,float ***srcdata,int frame,int x1,int y1,int x2,int y2);
static void profile_da_callback(Widget w,XtPointer client_data,XtPointer call_data);
static void profile_destroy(Widget w,XtPointer client_data,XtPointer call_data);
static void profile_recalculate(Widget w,ProfileData *data,int width,int height);
static void calc_line_profile(float ***src,int frame,ProfileData *data);

/* Callback to draw the line. */
void profile_cb(Widget w,XtPointer client_data,XtPointer call_data) {
  XButtonEvent *event=call_data;
  static int x=-1,y;
  if(x<0) {
    x=event->x;
    y=event->y;
    PetimageSetRoiDrawPoint((PetimageWidget)w,x,y);
  } else {
    int x1,x2,y1,y2,z;
    int frame,plane,study;
    XtVaGetValues(w,petimageNzoom,&z,petimageNframe,&frame,petimageNplane,&plane,petimageNstudy,&study,NULL);
    frame--;
    plane--;
    PetimageEndRoiDraw((PetimageWidget)w);
    set_ROI_drawmode(NULL,-1);
    x1=x/z;
    x2=event->x/z;
    y1=y/z;
    y2=event->y/z;
    find_frame_plane_index(&studies[study],frame,plane,&frame,&plane);
    profile_popup(XtDisplay(w),study,studies[study].imagepack.m[plane],frame,x1,y1,x2,y2);
    x=-1;
  }
}

/* Create a profile window */
void profile_popup(Display *dpy,int study,float ***srcdata,int frame,int x1,int y1,int x2,int y2) {
  ProfileData *data;
  Widget profile_shell;
  Widget canvas;
  char title[256];
  int x,y;
  sprintf(title,"Line profile (%d,%d)->(%d,%d)",x1,y1,x2,y2);
  data=(ProfileData*)malloc(sizeof(ProfileData));
  memset(data,0,sizeof(ProfileData));
  profile_shell = XtVaAppCreateShell(NULL,"Class",
  	topLevelShellWidgetClass,dpy,
	XtNtitle,title,
	NULL);
  canvas=XtVaCreateManagedWidget("profile_canvas",xmDrawingAreaWidgetClass,profile_shell,
  	XmNwidth,PROFILE_DEFAULT_W,
	XmNheight,PROFILE_DEFAULT_H,
	XmNuserData,data,
  	NULL);
  XtAddCallback(canvas,XmNexposeCallback,profile_da_callback,NULL);	
  XtAddCallback(canvas,XmNresizeCallback,profile_da_callback,NULL);
  XtAddCallback(canvas,XmNdestroyCallback,profile_destroy,NULL);
  /* Init data */
  data->x1=x1; data->x2=x2;
  data->y1=y1; data->y2=y2;
  for(x=0;x<studies[study].imagepack.dimx;x++)
    for(y=0;y<studies[study].imagepack.dimy;y++) {
      if(srcdata[x][y][frame]>data->max) data->max=srcdata[x][y][frame];
      if(srcdata[x][y][frame]<data->min) data->min=srcdata[x][y][frame];
    }
  calc_line_profile(srcdata,frame,data);
  profile_recalculate(canvas,data,PROFILE_DEFAULT_W,PROFILE_DEFAULT_H);
  /* Popup */
  XtPopup(profile_shell,XtGrabNone);

}

/* Get the line data */
static void calc_line_profile(float ***src,int frame,ProfileData *data) {
  int dx,dy,ax,ay,sx,sy,x,y,d,pos=0;
  int startx,starty,endx,endy;
  data->len=hypot(data->x1-data->x2,data->y1-data->y2);
  data->data=(float*)malloc(sizeof(float)*data->len);
  startx=data->x1;
  endx=data->x2;
  starty=data->y1;
  endy=data->y2;
  dx=endx-startx;
  dy=endy-starty;
  ax=abs(dx)<<1;
  ay=abs(dy)<<1;
  sx=(dx>=0)?1:-1;
  sy=(dy>=0)?1:-1;
  x=startx; y=starty;
  if(ax>ay) {
    d=ay-(ax>>1);
    while(x!=endx) {
      data->data[pos]=src[y][x][frame];
      pos++;
      if(pos>data->len) {fprintf(stderr,"calc_line_profile(): bug: pos>len\n"); return;}
      if(d>0||(d==0&&sx==1)) {
        y+=sy;
	d-=ax;
      }
      x+=sx; d+=ay;
    }
  } else {
    d=ax-(ay>>1);
    while(y!=endy) {
      data->data[pos]=src[y][x][frame];
      pos++;
      if(pos>data->len) {fprintf(stderr,"calc_line_profile(): bug: pos>len\n"); return;}
      if(d>0||(d==0&&sy==1)) {
        x+=sx;
	d-=ay;
      }
      y+=sy;
      d+=ax;
    }
  }
  if(pos<data->len) {
    data->len=pos;
    data->data=realloc(data->data,pos*sizeof(float));
  }
}

/* Create a visual representation of the data */
static void profile_recalculate(Widget w,ProfileData *data,int width,int height) {
  Display *dpy;
  int p,pval,ytick,xtick;
  float linew,lineh;
  GC gc;
  dpy=XtDisplay(w);
  /* Calculate values */
  linew=(width-PROFILE_BORDER*2)/(float)data->len;
  lineh=(height-PROFILE_BORDER*2)/(data->max-data->min);
  ytick=((height-PROFILE_BORDER*2)/10.0)+0.5;
  xtick=linew*5;
  /* Initialize pixmap */
  if(data->buffer) XFreePixmap(dpy,data->buffer);
  data->buffer=XCreatePixmap(dpy,RootWindowOfScreen(XtScreen(w)),width,height,DefaultDepthOfScreen(XtScreen(w)));
  gc=DefaultGC(dpy,XDefaultScreen(dpy));
  XSetForeground(dpy,gc,WhitePixel(dpy,DefaultScreen(dpy)));
  XFillRectangle(dpy,data->buffer,gc,0,0,width,height);
  XSetForeground(dpy,gc,BlackPixel(dpy,DefaultScreen(dpy)));
  XDrawRectangle(dpy,data->buffer,gc,PROFILE_BORDER,PROFILE_BORDER,width-PROFILE_BORDER*2,height-PROFILE_BORDER*2);
  for(p=0;p<10;p++)
    XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER/2,PROFILE_BORDER+ytick*p,PROFILE_BORDER,PROFILE_BORDER+ytick*p);
  XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER/2,height-PROFILE_BORDER,PROFILE_BORDER,height-PROFILE_BORDER);
  for(p=0;p<width-PROFILE_BORDER/2/xtick;p++)
    XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER+p*xtick,height-PROFILE_BORDER,PROFILE_BORDER+p*xtick,height-PROFILE_BORDER/2);
  /* Draw */
  pval=(data->data[0]-data->min)*lineh;
  for(p=0;p<data->len;p++) {
    int val;
    val=(data->data[p]-data->min)*lineh;
    if(zoom_matrix.interpolation) {
      XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER+p*linew,height-PROFILE_BORDER-pval,PROFILE_BORDER+(p+1)*linew,height-PROFILE_BORDER-val);
    } else {
      XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER+p*linew,height-PROFILE_BORDER-pval,PROFILE_BORDER+p*linew,height-PROFILE_BORDER-val);
      XDrawLine(dpy,data->buffer,gc,PROFILE_BORDER+p*linew,height-PROFILE_BORDER-val,PROFILE_BORDER+(p+1)*linew,height-PROFILE_BORDER-val);
    }
    pval=val;
  }
}

/* Free memory associated with the profile window */
static void profile_destroy(Widget w,XtPointer client_data,XtPointer call_data) {
  ProfileData *data;
  XtVaGetValues(w,XmNuserData,&data,NULL);
  XFreePixmap(XtDisplay(w),data->buffer);
  free(data->data);
  free(data);
}

/* Profile window callbacks */
static void profile_da_callback(Widget w,XtPointer client_data,XtPointer call_data) {
  ProfileData *data;
  Display *dpy;
  GC gc;
  XmDrawingAreaCallbackStruct *cbs=call_data;
  dpy=XtDisplay(w);
  gc=DefaultGC(dpy,XDefaultScreen(dpy));
  if(cbs->reason==XmCR_RESIZE) {
    Dimension width,height;
    XtVaGetValues(w,XmNwidth,&width,XmNheight,&height,XmNuserData,&data,NULL);
    profile_recalculate(w,data,width,height);
    XCopyArea(dpy,data->buffer,XtWindow(w),gc,0,0,width,height,0,0);
  } else if(cbs->reason==XmCR_EXPOSE) {
    XExposeEvent ev = cbs->event->xexpose;
    XtVaGetValues(w,XmNuserData,&data,NULL);
    XCopyArea(dpy,data->buffer,XtWindow(w),gc,ev.x,ev.y,ev.width,ev.height,ev.x,ev.y);
  }
}

