/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// File containing curve
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public abstract class CurveFile : TPCFile
    {
        /// <summary>
        /// Filetype of file.
        /// </summary>
        public CurveFileType filetype;

        /// <summary>
        /// Curves in file.
        /// </summary>
        protected TACTable curves;
        /// <summary>
        /// Gets or sets curve data reference
        /// </summary>
        public virtual TACTable Curves
        {
            get
            {
                return curves;
            }
            set
            {
                curves = value;
            }
        }

        /// <summary>
        /// All the comment lines at the end of the file
        /// </summary>
        public List<String> Comments;

        /// <summary>
        /// Reads data from a file. Abstract function that will be overwritten
        /// </summary>
        public override void ReadFile()
        {
            throw new System.NotImplementedException();
        }

        /// <summary>
        /// Writes data into file.
        /// </summary>
        /// <exception cref="TPCException">if data was not written properly</exception>
        public override void WriteFile()
        {
            throw new System.NotImplementedException();
        }
        /// <summary>
        /// Resolves file format from available file formats in library.
        /// </summary>
        /// <param name="filename">full filename</param>
        /// <returns>curve filetype as enumerator</returns>
        public static CurveFileType resolveFileFormat(string filename)
        {
            if (DFTCurveFile.CheckFormat(filename)) return CurveFileType.DFT;
            if (IFCurveFile.checkFormat(filename)) return CurveFileType.IF;
            //if (CPTCurveFile.checkFormat(filename)) return CurveFileType.CPT;
            //if (XMLfile.checkFormat(filename)) return CurveFileType.XML;
            //if (IDWCCurveFile.checkFormat(filename)) return CurveFileType.IDWC;
            throw new TPCException("Failed to recognize file format.");
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] readWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else if (line[0] == '#')
                {
                    Comments.Add(line);
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

    }
}
