/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Curve
{
    /// <summary>
    /// Curve header used in DFT file format TAC.
    /// </summary>
    public class CurveHeader
    {
        /// <summary>
        /// Curve name
        /// </summary>
        public string curveName;
        /// <summary>
        /// Private curve name data
        /// </summary>
        public string secondaryName;
        /// <summary>
        /// Curve plane number (0-based)
        /// </summary>
        public string plane;
        /// <summary>
        /// ROI size in voxels
        /// </summary>
        public double ROIsize;

        /// <summary>
        /// Cosntructor with parameters
        /// </summary>
        /// <param name="curveName">curve name</param>
        /// <param name="secondaryCurveName">secondary curve name</param>
        /// <param name="plane">plane number (0-based)</param>
        /// <param name="ROIsize">ROI size in voxels</param>
        public CurveHeader(string curveName, string secondaryCurveName, string plane, double ROIsize)
        {
            this.curveName = curveName;
            this.secondaryName = secondaryCurveName;
            this.plane = plane;
            this.ROIsize = ROIsize;
        }
        /// <summary>
        /// Constructor that takes CurveHeader object
        /// </summary>
        /// <param name="chead">Curveheader object that contains initial data</param>
        public CurveHeader(CurveHeader chead)
        {
            this.curveName = chead.curveName;
            this.secondaryName = chead.secondaryName;
            this.plane = chead.plane;
            this.ROIsize = chead.ROIsize;
        }

        /// <summary>
        /// Default constructor
        /// </summary>
        public CurveHeader()
        {
            this.curveName = ".";
            this.plane = ".";
            this.ROIsize = Double.NaN;
            this.secondaryName = ".";
            //this.unit = Unit.Unknown;
        }
        /// <summary>
        /// Returns clone of this object
        /// </summary>
        /// <returns>Clone of the CurveHeader object</returns>
        public CurveHeader Clone()
        {
            return (CurveHeader)new CurveHeader( this );
        }
        /// <summary>
        /// Checks if the given CurveHeader contains same data.
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if curve headers are equal</returns>
        public bool Equals( CurveHeader obj )
        {
            

            CurveHeader first = (CurveHeader)obj;

            if (!first.curveName.Equals(curveName)) return false;
            if (!first.secondaryName.Equals(secondaryName)) return false;
            if (!first.plane.Equals(plane)) return false;

            if( !(Double.IsNaN( first.ROIsize ) && Double.IsNaN( ROIsize )) )
                if (first.ROIsize != ROIsize) return false;

            return true;
        }

        /// <summary>
        /// Returns string representation of this object.
        /// </summary>
        /// <returns>this object as string</returns>
        public override string ToString()
        {
            return this.curveName + "\n" + this.secondaryName + "\n" + this.plane + "\n" + this.ROIsize + "\n------\n";
        }
    }
}
