/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    #region exceptions
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCDFTCurveFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCDFTCurveFileException() : base("No message") {      
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCDFTCurveFileException(string message)
            : base(message)
        {
        }
    }
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCIFFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCIFFileException() : base("No message") {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCIFFileException(string message)
            : base(message)
        {
        }
    }
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCIDWCFileException : TPCCurveFileException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCIDWCFileException()
            : base("No message")
        {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCIDWCFileException(string message)
            : base(message)
        {
        }
    }  
    /// <summary>
    /// Exception class for exceptions occurring in class DFTCurveFile
    /// </summary>
    class TPCCurveFileException : TPCException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCCurveFileException()
            : base("No message")
        {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCCurveFileException(string message)
            : base(message)
        {
        }
    }
    #endregion

    /// <summary>
    /// DFT-file format.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class DFTCurveFile : CurveFile
    {
        /// <summary>
        /// Type of dft file
        /// </summary>
        public enum DFTFileType {
            /// <summary>
            /// Unknown DFT file
            /// </summary>
            Unknown = 0,
            /// <summary>
            /// Regional DFT file 
            /// </summary>
            Regional, 
            /// <summary>
            /// Plasma DFt file
            /// </summary>
            Plasma
        }
        /// <summary>
        /// Default constructor.
        /// </summary>
        public DFTCurveFile()
        {
            this.filetype = CurveFileType.DFT;
            Comments = new List<string>();
        }

        /// <summary>
        /// constructor with filename
        /// </summary>
        /// <param name="filename">name of the DFT file</param>
        public DFTCurveFile(String filename)
        {
            this.filetype = CurveFileType.DFT;
            Comments = new List<string>();
            this.filename = filename;
        }

        /// <summary>
        /// Reads the file. Filename must be given.
        /// </summary>
        /// <exception cref="TPCDFTCurveFileException">If filename is null.</exception>
        public override void ReadFile()
        {
            if (filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            
            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());

            string text = reader.ReadToEnd();
            StringReader strReader = new StringReader(text);
            filestream.Close();

            // we need new reader to emptyheader generation because we need to read one line. This line
            // would be lost in readCurves() if the reader would be same one
            StringReader strReader2 = new StringReader(text);
            generateEmptyHeader(ref strReader2);
            strReader2.Close();

            string[] lines = File.ReadAllLines(filename);
            TimeUnit timeunit = TimeUnit.minute;
            foreach (string line in lines)
            {
                if (line.StartsWith("# Activity units:"))
                {
                    string activity_units = line.Substring(line.LastIndexOf("units:") + 6).ToLower();
                    this.Curves.unit = UnitConverter.CreateDataUnit(activity_units.Trim());
                }
                if (line.StartsWith("# Time units:"))
                {
                    string time_unit = line.Substring(line.LastIndexOf("units:") + 6).Trim().ToLower();
                    if (time_unit.Contains("s")) timeunit = TimeUnit.second;
                }
                if (line.StartsWith("# unit :=")) 
                {
                    string activity_units = line.Substring(line.LastIndexOf("unit :=") + 7).ToLower();
                    this.Curves.unit = UnitConverter.CreateDataUnit(activity_units.Trim());
                }
                if (line.StartsWith("# timeunit :=")) 
                {
                    string time_unit = line.Substring(line.LastIndexOf("unit :=") + 7).Trim().ToLower();
                    if (time_unit.Contains("sec")) timeunit = TimeUnit.second;
                }
            }

            this.readCurves(ref strReader, timeunit);

            strReader.Close();
        }
        /// <summary>
        /// Resolves file format from available file formats in library.
        /// </summary>
        /// <param name="filename">full path to tested file(s)</param>
        /// <returns>resolved image file type</returns>
        public static DFTFileType ResolveFileFormat(string filename)
        {
            if (DFTPlasmaCurveFile.CheckFormat(filename)) return DFTFileType.Plasma;
            if (DFTRegionalCurveFile.CheckFormat(filename)) return DFTFileType.Regional;
            else return DFTFileType.Unknown;
        }
        /// <summary>
        /// Reads DFT file.
        /// </summary>
        /// <param name="filename">filename of DFT data</param>
        /// <returns>DFT file that has all data in image</returns>
        /// <exception cref="TPCException">if reading of file failed</exception>
        public static DFTCurveFile ReadFile(string filename)
        {
            DFTFileType filetype;
            try
            {
                filetype = DFTCurveFile.ResolveFileFormat(filename);
            }
            catch (Exception)
            {
                throw new TPCException("Failed to recognize file " + filename);
            }
            Console.WriteLine("filetype="+filetype);
            switch (filetype)
            {
                case DFTFileType.Plasma:
                    DFTPlasmaCurveFile plasma_curve_file = new DFTPlasmaCurveFile(filename);
                    plasma_curve_file.ReadFile();
                    return plasma_curve_file;
                case DFTFileType.Regional:
                    DFTRegionalCurveFile regional_curve_file = new DFTRegionalCurveFile(filename);
                    regional_curve_file.ReadFile();
                    return regional_curve_file;
                case DFTFileType.Unknown:
                default:
                    DFTCurveFile curve_file = new DFTCurveFile(filename);
                    curve_file.ReadFile();
                    return curve_file;
            }

        }

        /// <summary>
        /// Creates a DFT file from contents of DFTRegionalcurvefile. Filename must be specified before
        /// calling this method
        /// </summary>
        public override void WriteFile()
        {
            if (this.filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            else WriteFile(this.filename);
        }

        /// <summary>
        /// Creates normal DFTCurvefile with no header.
        /// </summary>
        /// <param name="filename">Name of file to create.</param>
        public virtual void WriteFile( String filename )
        {
            // Writing the text to file
            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;

            fileWriter.Write(GetDataAsString(TimeUnit.second));

            filestream.Close();
        }

        /// <summary>
        /// Returns a string representation of this object.
        /// </summary>
        /// <returns>This object as string</returns>
        public override string ToString()
        {
            return "DFTCurveFile[filename="+filename+" filetype="+filetype+"]";
        }

        /// <summary>
        /// Returns a String containing all information in form that would be written to file.
        /// </summary>
        /// <param name="time">Time format. Can be "(min)", "(sec)" or "."</param>
        /// <returns>Contents of file in string.</returns>
        protected string GetDataAsString(TimeUnit time)
        {
            StringBuilder sb = new StringBuilder();
            StringWriter writer = new StringWriter(sb);

            try
            {
                // Now we can write all the curve data;
                writeCurves(ref writer, time );

                // Comment lines
                foreach (String comment in Comments)
                {
                    writer.WriteLine(comment);
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("String writing failed: " + e + " -- " + e.StackTrace + "\r\n>>" + e.Source);
            }

            writer.Flush();
            writer.Close();

            return sb.ToString();
        }

        /// <summary>
        /// Checks file format based on contents of the file.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT file</returns>
        /// <exception cref="TPCDFTCurveFileException">if there was a read problem</exception>
        public static DFTFileType CheckFormatByContents(string filename)
        {
            try
            {
                // we open the file and put its contents to String which is
                // read with StringReader. We can close the file after String is in memory
                FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
                StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());

                string text = reader.ReadLine();
                filestream.Close();
                //Couldn't read single line
                if (text == null)
                    return DFTFileType.Unknown;

                // Next we must check what kind of file we are trying to read
                // We look into first 3 letters, which give a lot of information.
                // we ignore all the possible empty spaces                
                String StartString = text.Substring(0, 10);
                StartString.Trim();
                StartString = StartString.Substring(0, 3);

                // Case1: normal DFT file starts with string DFT
                if (StartString.Equals("DFT"))
                {
                    return DFTFileType.Regional;
                }

                // Case2: Typical plasma TAC starts with "# Injection"
                else if (StartString.Equals("# I"))
                {
                    return DFTFileType.Plasma;
                }
                else
                {
                    return DFTFileType.Unknown;
                }
            }
            catch (Exception e)
            {
                throw new TPCDFTCurveFileException("Failed to read file:" + e.Message);
            }
        }


        /// <summary>
        /// Checks file format based on filename extension.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT file</returns>
        /// <exception cref="TPCDFTCurveFileException">if there was a read problem</exception>
        public static bool CheckFormat( String filename )
        {
            FileInfo file = new FileInfo(filename);
            if (String.Compare(file.Extension, "if", true) == 0)
            {
                return false;
            }
            else if (String.Compare(file.Extension, "idwc", true) == 0 || String.Compare(file.Extension, "idw", true) == 0)
            {
                return false;
            }
            else if (String.Compare(file.Extension, "xml", true) == 0 || String.Compare(file.Extension, "html", true) == 0)
            {
                return false;
            }
            
            //try to read file with readFile()
            try
            {
                /*
                DFTCurveFile dftfile = new DFTCurveFile();
                dftfile.filename = filename;
                dftfile.read;*/
                return ( !CheckFormatByContents(filename).Equals( "unknown" ) );
            }
            catch (Exception e)
            {
                throw new TPCCurveFileException("Fileformat not recognized."+e);
            }
        }

        #region private members

        /// <summary>
        /// Reads all the curve data from file with no header
        /// </summary>
        /// <param name="reader">string reader stream</param>
        /// <param name="time">time format of file as string</param>
        protected void readCurves(ref StringReader reader, TimeUnit time)
        {
            String[] words = null;
            TableCell[] curvecells = null;
            FrameTimeCell timecell = null;

            // Secons are chanced to minutes if the original file have had minute values
            Double timefactor = 1.0000d;
            if (time == TimeUnit.minute) timefactor = 60.0000d;
            else timefactor = 1.0000d;

            int dataStartColumn = 1;
            if (Curves.timetype == FrameTimeCell.FrameTimetype.START_END) dataStartColumn = 2;

            do
            {
                words = this.readWordsFromLine(ref reader);

                if (words != null)
                {
                    // we need to convert the curve strings to curve doubles
                    int numcols = Curves.columns;

                    curvecells = new TableCell[numcols];

                    if( Curves.timetype == FrameTimeCell.FrameTimetype.START_END )
                    {
                        timecell = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                        timecell.start_time = StrToDouble(words[0]);
                        timecell.end_time = StrToDouble(words[1]);
                    }
                    else 
                    {
                        timecell = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                        timecell.mid_time = StrToDouble(words[0]);
                    }

                    // minutes are chanced to seconds
                    timecell *= timefactor;

                    for (int i = 0; i < numcols; i++)
                    {
                        curvecells[i] = new TableCell();
                        if (words[i+dataStartColumn].Equals(".")) curvecells[i].Value = Double.NaN;
                        else
                        {
                            curvecells[i].Value = StrToDouble(words[i + dataStartColumn]);
                        }
                    }

                    // we add the new row to TACTable
                    Curves.AddRow(timecell, curvecells);
                }
            }
            while (words != null);
        }

        /// <summary>
        /// Converts strings to double in scientific mode with dot
        /// </summary>
        /// <param name="str">input string</param>
        /// <returns>a double which is converted from string</returns>
        protected Double StrToDouble(String str)
        {
            return Convert.ToDouble(str.TrimEnd(','), new System.Globalization.CultureInfo("en-GB"));
        }

        /// <summary>
        /// Creates a fixed size String from input string and result length.
        /// All extra space is filled with spaces. If the input string is longer than result,
        /// the remainders are cut off
        /// </summary>
        /// <param name="input">The input string</param>
        /// <param name="length">length if result string</param>
        /// <returns>input string, but with fixed length</returns>
        protected String FixStr(String input, int length)
        {
            // null input is treated like empty string 
            if (input == null) return new String(' ',length);

            int difference = length - input.Length;

            // the input string is longer than the boundaries given
            if (difference < 0) { return input.Substring(0, length); }
            else if (difference == 0) { return input; }
            else return input.PadRight(input.Length + difference, ' ');
        }



        /// <summary>
        /// Writes all the curve data to the file with no headers.
        /// </summary>
        /// <param name="writer">Streamwriter reference to the DFT file</param>
        /// <param name="time">Unit of time. Can be "(min)", "(sec)", or "."</param>
        protected void writeCurves(ref StringWriter writer, TimeUnit time)
        {
            // Secons are chanced to minutes if the original file have had minute values
            Double timefactor=1.0000d;
            if (time == TimeUnit.minute) timefactor = 1.0000d / 60.0000d;
            else timefactor = 1.0000d;

            // header is now ready. Now we write the data
            for (int i = 0; i < Curves.rows; i++)
            {
                FrameTimeCell timecell = Curves.GetTimeCell(i);

                // we chance the seconds to minutes if we must
                timecell *= timefactor;

                // if there are two time columns, we print both values inside 22 characters
                if (timecell.type == FrameTimeCell.FrameTimetype.START_END)
                {
                    writer.Write(FixStr(
                    (timecell.start_time).ToString("F3", new System.Globalization.CultureInfo("en-GB")), 11));
                    writer.Write(FixStr(
                    (timecell.end_time).ToString("F3", new System.Globalization.CultureInfo("en-GB")), 11));
                }
                else // if there is only one time value, we write only start time
                    writer.Write(FixStr("" + timecell.start_time.ToString("F3", new System.Globalization.CultureInfo("en-GB")), 22));

                // next we print every datacells of the row
                TableCell[] tabcells = Curves.GetRow(i);

                foreach (TableCell cell in tabcells)
                {
                    if (cell==null || Double.IsNaN(cell.Value))
                    {
                        writer.Write(FixStr(".", 20));
                    }
                    else
                    {
                        String s = ((Double)cell).ToString("e11", new System.Globalization.CultureInfo("en-GB"));
                        writer.Write(FixStr(s, 20));
                    }
                }
                writer.Write("\r\n");
            }
        }

        /// <summary>
        /// If the there is no header data in the file the columns are generated according to
        /// curve data. This function assumes that the time data is written to one column
        /// </summary>
        /// <param name="reader">string reader stream</param>
        protected void generateEmptyHeader(ref StringReader reader)
        {
            String[] words = readWordsFromLine(ref reader);

            // the TACtable is created. The empty headercell list are needed because TACtable needs
            // to know how many columns there must be
            Curves = new TACTable(FrameTimeCell.FrameTimetype.MID, words.Length-1);
        }
        #endregion
    }
}
