/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// DFT plasma curve file.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class DFTPlasmaCurveFile : DFTCurveFile
    {
        /// <summary>
        /// This header struct contains all general information of DFT plasma file like injectiontime and isotope
        /// </summary>
        public struct DFTPlasmaFileHeader
        {
            /// <summary>
            /// Injection time
            /// </summary>
            public DateTime InjectionTime;
            /// <summary>
            /// Isotope of the tracer substance. Used in the plasma TAC
            /// </summary>
            public String Isotope;
            /// <summary>
            /// Value of calibration coefficient. Used in the plasma TAC
            /// TPClib.Curve.DFTPlasmaCurveFile.DFTPlasmaFileHeader     
            /// </summary>
            public Double CalibrationCoefficient;
            /// <summary>
            /// Activity unit type
            /// </summary>
            public Data_unit ActivityUnit;
            /// <summary>
            /// Unit of time. Can be (min) or (sec) or (s).
            /// </summary>
            public TimeUnit timeunit;
        }

        /// <summary>
        /// This header contains all general information of DFT plasma file like injectiontime and isotope
        /// </summary>
        public DFTPlasmaFileHeader dftFileheader;
        /// <summary>
        /// Default constructor.
        /// </summary>
        public DFTPlasmaCurveFile() : base() { init(); }
        /// <summary>
        /// Constructor with filename
        /// </summary>
        /// <param name="filenam">filename</param>
        public DFTPlasmaCurveFile(String filenam) : base(filenam) { init(); }
        /// <summary>
        /// Checks file format. Only DFT plasma file format is recognized.
        /// </summary>
        /// <param name="filename">full path to file</param>
        /// <returns>true if file is a DFT plasma file</returns>
        public new static bool CheckFormat(string filename)
        {
            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());
            string text = reader.ReadLine();
            StringReader strReader = new StringReader(text);
            filestream.Close();

            // if there is no header in the file we generate the columns manually
            return (text.Trim().Substring(0, 3).Equals("# I"));
        }
        /// <summary>
        /// Reads a DFTPlasmaCurveFile from disk. Filename must be given.
        /// </summary>
        public override void ReadFile()
        {
            if (filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            init();

            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());

            string text = reader.ReadToEnd();
            StringReader strReader = new StringReader(text);
            filestream.Close();

            // if there is no header in the file we generate the columns manually
            if (!(text.Trim().Substring(0, 3).Equals("# I")))
            {
                //generateEmptyHeader(ref strReader, text.Substring(0, 22) );
                throw new TPCDFTCurveFileException("No correct plasma header tag found in " + filename + ".");
            }
            else
            {
                // reads the header data of plasma DFT file
                readPlasmaHeader(ref strReader);
            }

            // reads all the data from file
            readCurves(ref strReader, this.dftFileheader.timeunit);

            strReader.Close();
        }

        /// <summary>
        /// Writes the DFT Plasmafile file into disk. Filename must be given.
        /// </summary>
        public override void WriteFile()
        {
            if (filename == null) throw new TPCDFTCurveFileException("You have not given a filename.");
            else WriteFile(filename);
        }
        /// <summary>
        /// Creates the DFT plasma file with given filename. Old file will be overwritten
        /// </summary>
        /// <param name="filename">name of the DFT plasma file</param>
        public override void WriteFile(String filename)
        {
            // checks if there is enough information to create a file. Throws an exception if not
            //isValid();
          
            StringBuilder sb = new StringBuilder();
            StringWriter writer = new StringWriter( sb );

            // if there is header data it will be written to file
            writer.WriteLine("# Injection time: " + this.dftFileheader.InjectionTime.ToString());
            writer.WriteLine("# Isotope: " + this.dftFileheader.Isotope);
            writer.WriteLine("# Calibration coefficient: " + this.dftFileheader.CalibrationCoefficient.ToString(new System.Globalization.CultureInfo("en-GB")));
            writer.WriteLine("# Activity units: " + UnitConverter.ConvertToString(this.dftFileheader.ActivityUnit));
            writer.WriteLine("# Time units: " + this.dftFileheader.timeunit);
    
            // Now we can write curve data
            writeCurves(ref writer, this.dftFileheader.timeunit );

            // Comment lines
            foreach (String comment in Comments)
            {
                writer.WriteLine(comment);
            }

            // Writing the text to file
            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;
            fileWriter.Write(writer.ToString());

            writer.Flush();
            writer.Close();
            filestream.Close();
        }

        #region private members

        /// <summary>
        /// Initializes all the objects
        /// </summary>
        private void init()
        {
            this.filetype = CurveFileType.DFT;
            this.dftFileheader.CalibrationCoefficient = Double.NaN;
            this.dftFileheader.InjectionTime = new DateTime();
            this.dftFileheader.Isotope = ".";
            this.dftFileheader.timeunit = TimeUnit.minute;

            this.Curves = new TACTable();
        }

        /// <summary>
        /// Reads all header data from DFT plasmafile
        /// </summary>
        /// <param name="reader">stringreader reference to the file</param>
        private void readPlasmaHeader(ref StringReader reader)
        {
            String line;
            String datetime = "";
            String isotope = "";
            String calibration = "";
            String activity_units = "";
            String time_units = "";
            for (int i = 0; i < 5; i++)
            {
                line = reader.ReadLine();
                if (line.StartsWith("# Injection time:")) datetime = line.Substring(line.LastIndexOf("time:") + 5);
                if (line.StartsWith("# Isotope:")) isotope = line.Substring(line.LastIndexOf("Isotope:") + 8);
                if (line.StartsWith("# Calibration coefficient:")) calibration = line.Substring(line.LastIndexOf("ficient:") + 8);
                if (line.StartsWith("# Activity units:")) activity_units = line.Substring(line.LastIndexOf("units:") + 6).ToLower();
                if (line.StartsWith("# Time units:")) time_units = line.Substring(line.LastIndexOf("units:") + 6);
                if (line.StartsWith("# unit :=")) activity_units = line.Substring(line.LastIndexOf("unit :=") + 7).ToLower();
                if (line.StartsWith("# timeunit :=")) time_units = line.Substring(line.LastIndexOf("unit :=") + 7);
            }
            
            try { this.dftFileheader.InjectionTime = System.Convert.ToDateTime(datetime.Trim()); }
            catch (Exception) { this.dftFileheader.InjectionTime = new DateTime(); }

            this.dftFileheader.Isotope = isotope.Trim();

            try { this.dftFileheader.CalibrationCoefficient = this.StrToDouble(calibration.Trim()); }
            catch (Exception) { this.dftFileheader.CalibrationCoefficient = Double.NaN; }

            this.dftFileheader.ActivityUnit = UnitConverter.CreateDataUnit(activity_units);

            time_units = time_units.Trim();
            if (time_units.Contains("min")) this.dftFileheader.timeunit = TimeUnit.minute;
            else if (time_units.Contains("sec")) this.dftFileheader.timeunit = TimeUnit.second;
            else if (time_units.Contains("s")) this.dftFileheader.timeunit = TimeUnit.second;
            else if (time_units.Contains(".")) this.dftFileheader.timeunit = TimeUnit.undefined;
            else this.dftFileheader.timeunit = TimeUnit.unknown;

            // we initialize the TACtable
            Curves = new TACTable(FrameTimeCell.FrameTimetype.START, 1 );
            Curves.SetUnit(activity_units.Trim());
        }
        #endregion
    }
}
