﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Curve {
    /// <summary>
    /// Table cell representing time information.
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class FrameTimeCell : TableCell, IComparable
    {
        /// <summary>
        /// Type of frame time in TAC.
        /// </summary>
        public enum FrameTimetype
        {
            /// <summary>
            /// Start of frame
            /// </summary>
            START,
            /// <summary>
            /// End of frame
            /// </summary>
            END,
            /// <summary>
            /// Middle of frame
            /// </summary>
            MID,
            /// <summary>
            /// Start and end of frame
            /// </summary>
            START_END
        }
        /// <summary>
        /// Type of frame time.
        /// </summary>
        public readonly FrameTimetype type = FrameTimetype.START;
        /// <summary>
        /// Time value (in seconds).
        /// </summary>
        new protected double Value = 0;
        /// <summary>
        /// End time value (in seconds).
        /// </summary>
        protected double EndValue = 0;
        /// <summary>
        /// Gets or set frame start time. Type must be FrameTimetype.START or FrameTimetype.START_END.
        /// </summary>
        public double start_time
        {
            get
            {
                if (type == FrameTimetype.START || type == FrameTimetype.START_END)
                {
                    return Value;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have start time");
                }
            }
            set
            {
                if (type == FrameTimetype.START || type == FrameTimetype.START_END)
                {
                    Value = value;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have start time");
                }
            }
        }
        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator ==(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if (((Object)cell2) == null)
                if (((Object)cell) == null) return true;
                else return false;
            if ((Object)cell == null) return false;

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END) { return (cell.start_time == cell2.start_time) && (cell.end_time == cell2.end_time); }
            else if (cell.type == FrameTimetype.START) { return cell.start_time == cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time == cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time == cell2.end_time; }

            return false;
        }
        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator !=(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if (((Object)cell2) == null)
                if (((Object)cell) == null) return false;
                else return true;
            if ((Object)cell == null) return true;

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END) { return (cell.start_time != cell2.start_time) || (cell.end_time != cell2.end_time); }
            else if (cell.type == FrameTimetype.START) { return cell.start_time != cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time != cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time != cell2.end_time; }

            return false;
        }

        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <=(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if ((Object)cell == null || (Object)cell2 == null) throw new TPCException("Cannot compare null values.");

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END)
            {
                if (cell.Value < cell2.Value)
                {
                    return true;
                }
                else if (cell.Value == cell2.Value)
                {
                    if (cell.EndValue <= cell2.EndValue) return true;
                }
                return false;
            }
            else if (cell.type == FrameTimetype.START) { return cell.start_time <= cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time <= cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time <= cell2.end_time; }

            return false;
        }

        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >=(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if ((Object)cell == null || (Object)cell2 == null) throw new TPCException("Cannot compare null values.");

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END)
            {
                if (cell.Value > cell2.Value)
                {
                    return true;
                }
                else if (cell.Value == cell2.Value)
                {
                    if (cell.EndValue >= cell2.EndValue) return true;
                }
                return false;
            }
            else if (cell.type == FrameTimetype.START) { return cell.start_time >= cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time >= cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time >= cell2.end_time; }

            return false;
        }

        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if ((Object)cell == null || (Object)cell2 == null) throw new TPCException("Cannot compare null values.");

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END)
            {
                if (cell.Value < cell2.Value)
                {
                    return true;
                }
                else if (cell.Value == cell2.Value)
                {
                    if (cell.EndValue < cell2.EndValue) return true;
                }
                return false;
            }
            else if (cell.type == FrameTimetype.START) { return cell.start_time < cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time < cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time < cell2.end_time; }

            return false;
        }

        /// <summary>
        /// Comparison operator for comparing against another FrameTimeCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >(FrameTimeCell cell, FrameTimeCell cell2)
        {
            // null check
            if ((Object)cell == null || (Object)cell2 == null) throw new TPCException("Cannot compare null values.");

            if (cell.type != cell2.type) throw new TPCCurveFileException("FrameTimeCell == operator: time type mismatch.");

            if (cell.type == FrameTimetype.START_END)
            {
                if (cell.Value > cell2.Value)
                {
                    return true;
                }
                else if (cell.Value == cell2.Value)
                {
                    if (cell.EndValue > cell2.EndValue) return true;
                }
                return false;
            }
            else if (cell.type == FrameTimetype.START) { return cell.start_time > cell2.start_time; }
            else if (cell.type == FrameTimetype.MID) { return cell.mid_time > cell2.mid_time; }
            else if (cell.type == FrameTimetype.END) { return cell.end_time > cell2.end_time; }

            return false;
        }

        /// <summary>
        /// Checks if two time values are overlapping
        /// </summary>
        /// <param name="cell">FrameTimeCell which is compared.</param>
        /// <returns>True if overlapping.</returns>
        public bool doOverlap(FrameTimeCell cell)
        {
            if (cell.type != this.type) throw new TPCException("FrameTimeCell: doOverlap: Time type mismach.");

            switch (this.type)
            {
                case FrameTimetype.START_END:
                    if (this.start_time >= cell.end_time && this.end_time > cell.end_time) return false;
                    if (this.start_time < cell.start_time && this.end_time <= cell.start_time) return false;
                    return true;
                case FrameTimetype.START:
                    if (this.start_time == cell.start_time) return true;
                    break;
                case FrameTimetype.MID:
                    if (this.mid_time == cell.mid_time) return true;
                    break;
                case FrameTimetype.END:
                    if (this.end_time == cell.end_time) return true;
                    break;
            }
            return false;
        }

        /// <summary>
        /// Multiplication operator for FrameTimeCell. All time values are multiplied
        /// </summary>
        /// <param name="cell">Object to multiply</param>
        /// <param name="value">Multiplication value</param>
        /// <returns>cell object with multiplied value</returns>
        public static FrameTimeCell operator *(FrameTimeCell cell, Double value)
        {
            if (cell.type == FrameTimetype.START_END)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.START_END);
                //Console.Write("start=" + cell.start_time + " end" + cell.end_time + " value=" + value + " -> ");
                ftc.start_time = cell.start_time * value;
                ftc.end_time = cell.end_time * value;
                //Console.WriteLine("start=" + ftc.start_time + " end" + ftc.end_time);
                return ftc;
            }
            else if (cell.type == FrameTimetype.START)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.START);
                ftc.start_time = cell.start_time * value;
                return ftc;
            }
            else if (cell.type == FrameTimetype.MID)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.MID);
                ftc.mid_time = cell.mid_time * value;
                return ftc;
            }
            else if (cell.type == FrameTimetype.END)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.END);
                ftc.end_time = cell.end_time * value;
                return ftc;
            }

            return null;
        }

        /// <summary>
        /// Division operator for FrameTimeCell. All time values are divided
        /// </summary>
        /// <param name="cell">Object to divide</param>
        /// <param name="value">Division value</param>
        /// <returns>cell object with divided value</returns>
        public static FrameTimeCell operator /(FrameTimeCell cell, Double value)
        {
            if (cell.type == FrameTimetype.START_END)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.START_END);
                ftc.start_time = cell.start_time / value;
                ftc.end_time = cell.end_time / value;
                return ftc;
            }
            else if (cell.type == FrameTimetype.START)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.START);
                ftc.start_time = cell.start_time / value;
                return ftc;
            }
            else if (cell.type == FrameTimetype.MID)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.MID);
                ftc.mid_time = cell.mid_time / value;
                return ftc;
            }
            else if (cell.type == FrameTimetype.END)
            {
                FrameTimeCell ftc = new FrameTimeCell(FrameTimetype.END);
                ftc.end_time = cell.end_time / value;
                return ftc;
            }

            return null;
        }

        /// <summary>
        /// Gets or set frame end time. Type must be FrameTimetype.END or FrameTimetype.START_END.
        /// </summary>
        public double end_time
        {
            get
            {
                if (type == FrameTimetype.END || type == FrameTimetype.START_END)
                {
                    return EndValue;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have end time");
                }
            }
            set
            {
                if (type == FrameTimetype.END || type == FrameTimetype.START_END)
                {
                    EndValue = value;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have end time");
                }
            }
        }
        /// <summary>
        /// Gets or set frame middle time. Type must be FrameTimetype.MID.
        /// </summary>
        public double mid_time
        {
            get
            {
                if (type == FrameTimetype.MID)
                {
                    return Value;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have mid time");
                }
            }
            set
            {
                if (type == FrameTimetype.MID)
                {
                    Value = value;
                }
                else
                {
                    throw new TPCTableCellException("Cell does not have mid time");
                }
            }
        }
        /// <summary>
        /// Constructor which defines time cell type.
        /// </summary>
        /// <param name="type">time information type</param>
        public FrameTimeCell(FrameTimetype type)
        {
            this.type = type;
        }
        /// <summary>
        /// Constructor which defines time cell type and times.
        /// </summary>
        /// <param name="type">time information type</param>
        /// <param name="start">frame start time in seconds</param>
        /// <param name="end">frame end time in seconds</param>
        public FrameTimeCell(FrameTimetype type, double start, double end)
        {
            this.type = type;
            if (end < start) throw new TPCInvalidArgumentsException("Start time "+start+" is after end time "+end);
            this.Value = start;
            this.EndValue = end;
        }

        /// <summary>
        /// Explicit conversion to double type. Returns only start time if the type of time data 
        /// is FrameTimetype.START_END.
        /// </summary>
        /// <param name="cell">cell that is converted</param>
        /// <returns>double value converted from private value field</returns>
        public static explicit operator Double(FrameTimeCell cell)
        {
            return (double)cell.Value;
        }
        /// <summary>
        /// Compares this frame time to another
        /// </summary>
        /// <param name="obj">other FrameTimeCell object</param>
        /// <returns>0: this equals obj, 1: this smaller than obj, -1: this greater than obj</returns>
        public override int CompareTo(object obj)
        {
            if (!(obj is FrameTimeCell)) throw new TPCTableCellException("Cannot compare FrameTimeCell to other than FrameTimeCell.");
            FrameTimeCell cell = (obj as FrameTimeCell);
            if (type != cell.type) throw new TPCTableCellException("Cannot compare between different frame time types.");
            int r = 0;
            switch (type)
            {
                case FrameTimeCell.FrameTimetype.START_END:
                    if (Value < cell.Value)
                    {
                        r = -1;
                    }
                    else if (Value > cell.Value)
                    {
                        r = 1;
                    }
                    else
                    {
                        if (EndValue < cell.EndValue) r = -1;
                        if (EndValue > cell.EndValue) r = 1;
                    }
                    break;
                case FrameTimeCell.FrameTimetype.START:
                case FrameTimeCell.FrameTimetype.MID:
                case FrameTimeCell.FrameTimetype.END:
                    //Console.WriteLine("compare: " + Value + "-" + cell.Value);
                    if (Value < cell.Value) r = -1;
                    if (Value > cell.Value) r = 1;
                    break;
                default:
                    break;
            }
            return r;
        }
        /// <summary>
        /// Gets string representation of this object.
        /// </summary>
        /// <returns>start, mid or end time depending of frame time type</returns>
        public override string ToString()
        {
            if (this.type == FrameTimetype.START_END)
            {
                return this.start_time + " - " + this.end_time;
            }
            else if (this.type == FrameTimetype.START)
            {
                return this.start_time.ToString();
            }
            else if (this.type == FrameTimetype.MID)
            {
                return this.mid_time.ToString();
            }
            else if (this.type == FrameTimetype.END)
            {
                return this.end_time.ToString();
            }
            else return "";
        }
        /// <summary>
        /// Gets hash code of this object.
        /// </summary>
        /// <returns>hash code</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        /// <summary>
        /// Evaluates if object is equal to this cell
        /// </summary>
        /// <param name="obj">evaluated object</param>
        /// <returns>true if objects are equal</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is FrameTimeCell)) return false;
            return (obj as FrameTimeCell) == this;
        }
    }
}