﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Curve
{
    /// <summary>
    /// Testbench class for TPClib.Curve.RegionalTACTable class
    /// </summary>
    [TestFixture]
    public class NUnitTestBench_RegionalTACTable
    {
        /// <summary>
        /// Testing of RegionalTACTables constructors
        /// </summary>
        [Test]
        public void test1_1_Constructor()
        {
            Console.WriteLine("TEST CASE 1.1: CONSTRUCTING:");
            RegionalTACTable tab = new RegionalTACTable(TestHelpers.generateTimes());
            TestHelpers.fillstaticNumbers( tab );

            RegionalTACTable tab2 = new RegionalTACTable();
            tab2.SetTimeCells( TestHelpers.generateTimes() );

            TestHelpers.fillstaticNumbers(tab2);

            Assert.IsTrue( RegionalTACTable.Equals( tab, tab2 ) );
        }


        /// <summary>
        /// AddColumn
        /// </summary>
        [Test]
        public void test2_1_AddColumn()
        {
            String original = "P:\\data\\dft\\regional\\test2_1_original.dft";

            RegionalTACTable tab = new RegionalTACTable( TestHelpers.generateTimes() );
            tab.AddColumn();
            tab.AddColumn(TestHelpers.generateTableCells(2));
            tab.AddColumn(new TAC(new CurveHeader(), TestHelpers.generateTableCells(2)));
            tab.AddColumnAt(0, new TAC(new CurveHeader("huu", "hoo", "plane", 8), TestHelpers.generateTableCells(10)));
            tab.AddColumnAt(4, new TAC(new CurveHeader(), TestHelpers.generateTableCells(22)));

            //tab.ReplaceColumn(1, new TAC( new DFTCurveHeader("huu","hoo","plane",8) , TestHelpers.generateTableCells(61))) ;

            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file.Curves = tab;
            file.WriteFile("regional\\test2_1.dft");

            Assert.IsTrue( TestHelpers.getStdOut("P:\\bin\\windows\\dftmatch.exe", original + " regional\\test2_1.dft") == 0 );
        }

        /// <summary>
        /// DeleteColumn
        /// </summary>
        [Test]
        public void test2_2_DeleteColumn()
        {
            String original = "P:\\data\\dft\\regional\\test2_1_original.dft";
            String original2 = "P:\\data\\dft\\regional\\test2_2_original.dft";
            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            file.Curves.DeleteColumn(4);
            file.Curves.DeleteColumn(0);
            file.Curves.DeleteColumn(1);

            file.WriteFile("regional\\test2_2.dft");
            Assert.IsTrue(TestHelpers.getStdOut("P:\\bin\\windows\\dftmatch.exe", original2+" regional\\test2_2.dft") == 0);
        }

        /// <summary>
        /// GetColumn
        /// </summary>
        [Test]
        public void test2_3_GetColumn()
        {
            String original = "P:\\data\\dft\\regional\\test2_1_original.dft";

            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);
            RegionalTACTable table = new RegionalTACTable(file.Curves);

            table.AddColumn(file.Curves.GetColumn(0));
            table.AddColumn(file.Curves.GetColumn(1));
            table.AddColumn(file.Curves.GetColumn(2));
            table.AddColumn(file.Curves.GetColumn(3));
            table.AddColumn(file.Curves.GetColumn(4));

            file.Curves = table;

            file.WriteFile("regional\\test2_3.dft");
            Assert.IsTrue(TestHelpers.getStdOut("P:\\bin\\windows\\dftmatch.exe", original + " regional\\test2_3.dft") == 0);
        }
        /// <summary>
        /// Test case for sorting TAC columns
        /// </summary>
        [Test]
        public void test3_4_Sorting_TACs()
        {
            Console.WriteLine("\n\n\nTEST CASE 3.4: SORTING TACs BY NAME");
            String original = "P:\\data\\dft\\regional\\test3_4_original.dft";

            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            //TAC ct = file.compareField = CompareFields.CurveName;
            //file.Curves.ReplaceColumn(2, new TAC(null, ((TACTable)file.Curves).GetColumn(0)));

            file.Curves.SortColumns( new DFTTACComparer( DFTTACComparer.CompareField.CurveName ) );

            file.WriteFile("regional\\test3_4.dft");

            System.IO.File.Copy(original, "C:\\trunk\\bin\\debug\\regional\\test3_4_osort.dft", true);
            TestHelpers.getStdOut(@"P:\\bin\\windows\\dftsort_0_2_0.exe", "regional\\test3_4_osort.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "regional\\test3_4_osort.dft regional\\test3_4.dft") == 0);
        }

        /// <summary>
        /// 6.5 Find TAC:s by name	Select VOIs (TAC:s), whose names are matching specified string.
        /// </summary>
        [Test]
        public void test3_3_Getting_TACS_matching_search_String()
        {
            Console.WriteLine("TEST CASE 3.3: FINDING TACS BY STRING.");
            String str = "amy";
            String original = "P:\\data\\dft\\Imadeus.dft";

            // first we make the original one
            System.IO.File.Delete("regional\\test3_3_correct.dft");
            TestHelpers.getStdOut("P:\\bin\\windows\\dftadd.exe", "regional\\test3_3_correct.dft " + original + " " + str);


            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            List<TAC> result = file.Curves.Find(str, new DFTTACComparer(DFTTACComparer.CompareField.CurveName));

            // we do new table
            //RegionalTACTable table = new RegionalTACTable(file.Curves.GetTimeCells());
            RegionalTACTable table = new RegionalTACTable(file.Curves);

            foreach (TAC t in result)
            {
                table.AddColumn(t);
            }

            file.Curves = table;
            file.WriteFile("regional\\test3_3.dft");

            // comparison
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "-roughly regional\\test3_3.dft regional\\test3_3_correct.dft") == 0);
        }


        /// <summary>
        /// Equals
        /// </summary>
        [Test]
        public void test3_1_Equals()
        {
            Console.WriteLine("TEST CASE 3.1: EQUALS");
            String original = "P:\\data\\dft\\regional\\test2_1_original.dft";

            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            DFTRegionalCurveFile file2 = new DFTRegionalCurveFile();
            file2 = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            // First same file should be equal
            Assert.IsTrue( RegionalTACTable.Equals( file.Curves, file2.Curves ) );

            // Next we try different CurveName
            ((CurveHeader)file.Curves.curveheader[0]).curveName = "hui";
            Assert.IsFalse(RegionalTACTable.Equals(file.Curves, file2.Curves));
            ((CurveHeader)file.Curves.curveheader[0]).curveName = "huu";
            Assert.IsTrue(RegionalTACTable.Equals(file.Curves, file2.Curves));

            // Next we try different SecondaryCurveName
            ((CurveHeader)file.Curves.curveheader[0]).secondaryName = "hoi";
            Assert.IsFalse(RegionalTACTable.Equals(file.Curves, file2.Curves));
            ((CurveHeader)file.Curves.curveheader[0]).secondaryName = "hoo";
            Assert.IsTrue(RegionalTACTable.Equals(file.Curves, file2.Curves));

            // Next we try change one value
            file.Curves[0,0] = new TableCell(4.0d);
            Assert.IsFalse(RegionalTACTable.Equals(file.Curves, file2.Curves));
            file.Curves[0, 0] = new TableCell(10.0d);
            Assert.IsTrue(RegionalTACTable.Equals(file.Curves, file2.Curves));
        }

        /// <summary>
        /// Clone
        /// </summary>
        [Test]
        public void test3_2_Clone()
        {
            String original = "P:\\data\\dft\\regional\\test3_4_original.dft";

            DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(original);

            RegionalTACTable tab = file.Curves.Clone();
            Assert.IsTrue( RegionalTACTable.Equals(tab, file.Curves) );
        }

        /// <summary>
        /// Index out of bounds exceptions
        /// </summary>
        [Test]
        public void test4_1_Index_out_of_bounds()
        {
            Console.WriteLine("Testing Index out of bound exceptions...");
            RegionalTACTable tab = new RegionalTACTable(TestHelpers.generateTimes());
            TestHelpers.fillstaticNumbers(tab);
            bool pass = false;

            pass = false; Console.WriteLine("Testing ReplaceColumn(-1, TAC)");
            try { tab.ReplaceColumn(-1, new TableCell[1] ); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing ReplaceColumn(columns, TAC)");
            try { tab.ReplaceColumn(tab.columns, new TableCell[1]); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing DeleteColumn(-1)");
            try { tab.DeleteColumn(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing DeleteColumn(columns)");
            try { tab.DeleteColumn(tab.columns); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing GetColumn(-1)");
            try { tab.GetColumn(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing GetColumn(columns)");
            try { tab.GetColumn(tab.columns); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);
        }

        /// <summary>
        /// Index out of bounds exceptions
        /// </summary>
        [Test]
        public void test4_2_null_tests()
        {
            Console.WriteLine("TESTING NULL EXCEPTIONS...\n");
            RegionalTACTable tab = new RegionalTACTable(TestHelpers.generateTimes());
            TestHelpers.fillstaticNumbers(tab);
            bool pass = false;

            pass = false; Console.WriteLine("Testing AddColumn(null)");
            try { tab.AddColumn((TableCell[])null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddColumn(null) - TAC");
            try { tab.AddColumn((TAC)null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddColumn( TAC(null, null) )");
            try { tab.AddColumn( new TAC( null, null ) ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing ReplaceColumn( 0, (TAC)null )");
            try { tab.ReplaceColumn(0, (TAC)null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddColumnAt( 0, (TAC)null )");
            try { tab.AddColumnAt(0, (TAC)null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing SortColumns( (TACComparer)null )");
            try { tab.SortColumns( (TACComparer)null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing Find( null, null )");
            try { tab.Find( null, null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing Find( null, new DFTTACComparer() )");
            try { tab.Find(null, new DFTTACComparer() ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

        }
    }

}
