/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using TPClib.Curve;

namespace TPClib.Curve
{

    #region DFTCurvefile_regional

    /// <summary>
    /// NUnit Testbench class for DFT curvefile
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_DFTCurveFile
    {
        /// <summary>
        /// Tolerance of one frametime value. Conversions from minutes to seconds and vice versa affect
        /// some rounding errors.
        /// </summary>
        public Double tolerance = 0.001;

        /// <summary>
        /// Reads one file Imadeus.dft and writes it.
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void test1_0_read_write()
        {
            Console.WriteLine("\n\n\nTEST CASE 1.0 and 2.0: READING AND WRITING A FILE");

            //testFile("P:\\data\\dft\\test7.kbq");
            testFile("P:\\data\\dft\\Imadeus.dft");
        }

        /// <summary>
        /// Reads many different files and writes those and eventually compares with dftmatch. 
        /// </summary>
        [Test]
        public void test1_1_read_write_FileVariations()
        {
            Console.WriteLine("\n\n\nTEST CASE 1.1 and 2.1: READING AND WRITING DIFFERENT FILE VARIATIONS");

            testFile("P:\\data\\dft\\Imadeus.dft");
            testFile("P:\\data\\dft\\Imadeus_weights.dft");
            testFile("P:\\data\\dft\\test2.dft"); // This one wont work
            testFile("P:\\data\\dft\\test3.dft");
            testFile("P:\\data\\dft\\test4.dft");
            testFile("P:\\data\\dft\\typical.dft");
            testFile("P:\\data\\dft\\test7.kbq");  // this is a plasma file
            testFile("P:\\data\\dft\\typical_weights.dft");

            testFile("P:\\data\\dft\\test3_noheader.dft");
        }

        /// <summary>
        /// Writes frames to file and reads frames from file
        /// </summary>
        [Test]
        public void test3_0_Reading_writing_Frames()
        {
            Console.WriteLine("\n\n\nTEST CASE 3.0: READING AND WRITING FRAMEDATA FROM/TO FILE");
            
            DFTRegionalCurveFile file = new DFTRegionalCurveFile("P:\\data\\dft\\Imadeus_weights.dft");
            file.ReadFile();

           
            FrameTimeCell ftc = file.Curves.GetTimeCell(2);

            // new value
            FrameTimeCell tcell = new FrameTimeCell( ftc.type );
            tcell.start_time = 31.000000d;
            tcell.end_time = 60.00000d;

            // tcell must be different value
            Assert.IsFalse( ftc == tcell );

            file.Curves.SetTimeCell( 2, tcell );

            file.WriteFile("S:\\temp\\testi3_0.dft");
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("S:\\temp\\testi3_0.dft");

            Console.WriteLine( tcell.ToString() );
            Console.WriteLine( file.Curves.GetTimeCell(2).ToString() );
            //maximum error tolerance due to 3 decimals in time values in dft file
            //is 0.005*60=0.3 ( = maximum rounding error * conversion from seconds to minutes when writing file)
            Assert.AreEqual(tcell.start_time, file.Curves.GetTimeCell(2).start_time, 0.3);
        }

        /// <summary>
        /// Reads TACS(colums) from file and writes those
        /// </summary>
        [Test]
        public void test2_5_and_2_6_Reading_writing_TACS()
        {
            Console.WriteLine("\n\n\nTEST CASE 2.5 and 2.6: READING AND WRITING TACDATA FROM/TO FILE");

            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("P:\\data\\dft\\Imadeus_weights.dft");
            
            // testing the addTACAt -method
            TAC tac = file.Curves.GetColumn(0);

            TAC oldTAC = file.Curves.GetColumn(3);
            file.Curves.AddColumnAt(3, tac);

            file.WriteFile("S:\\temp\\testi2_5.dft");

            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("S:\\temp\\testi2_5.dft");
            TAC tac2 = file.Curves.GetColumn(3);

            // tests if the tacs are not pointing to same reference
            Assert.AreNotSame(tac, tac2);

            // Contents should be identical
            Assert.IsTrue( TAC.Equals( tac,tac2 ) );

            // The old column should be preserved
            Assert.IsTrue(TAC.Equals( oldTAC, file.Curves.GetColumn(4)) );

            // testing the replaceTAC -method (TEST 2.6)

            file.Curves.ReplaceColumn(2, tac2);
            file.WriteFile("S:\\temp\\testi2_5.dft");
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("S:\\temp\\testi2_5.dft");

            TAC tac3 = file.Curves.GetColumn(2);

            Assert.AreNotSame(tac2,tac3); // TACs must be different instances
            Assert.IsTrue( TAC.Equals(tac2,tac3) );
        }

        /// <summary>
        /// Tests the radioactivity values
        /// </summary>
        [Test]
        public void test5_1_Reading_writing_radioactivity()
        {
            Console.WriteLine("\n\n\nTEST CASE 5.1: READING AND WRITING RADIOACTIVITY UNIT");

            DFTRegionalCurveFile file = new DFTRegionalCurveFile("P:\\data\\dft\\Imadeus_weights.dft");
            file.ReadFile();

            file.Curves.UnitConversion( Data_unit.BQ_per_ML );

            file.WriteFile("S:\\temp\\testi5_1.dft");

            // we copy and change the original file with dftunit.exe to bq/cc
            System.IO.File.Copy("P:\\data\\dft\\Imadeus_weights.dft", "S:\\temp\\testi5_1_original.dft", true);
            TestHelpers.getStdOut(@"P:\\bin\\windows\\dftunit.exe", "-uc=bq/cc S:\\temp\\testi5_1_original.dft");

            // we compare our own file to dftunits file:
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "S:\\temp\\testi5_1_original.dft S:\\temp\\testi5_1.dft") == 0);
        }

        /// <summary>
        /// Test the time unit conversion
        /// </summary>
        [Test]
        public void test5_2_Reading_writing_timeunit()
        {
            String testfile = "P:\\data\\dft\\Imadeus_weights.dft";
            Console.WriteLine("\n\n\nTEST CASE 5.2: READING AND WRITING TIME UNIT");

            DFTRegionalCurveFile file = new DFTRegionalCurveFile(testfile);

            file.ReadFile();
            file.TimeConversion( TimeUnit.second );
            file.WriteFile("S:\\temp\\testi5_2.dft");

            Console.WriteLine("Changing frames from minutes to seconds.");
            System.IO.File.Copy(testfile, "S:\\temp\\testi5_2_original.dft", true);
            TestHelpers.getStdOut(@"P:\\bin\\windows\\min2sec.exe", "S:\\temp\\testi5_2_original.dft S:\\temp\\testi5_2_original.dft");
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "S:\\temp\\testi5_2_original.dft S:\\temp\\testi5_2.dft") == 0);

            Console.WriteLine("Changing frames from seconds to minutes.");
            TestHelpers.getStdOut(@"P:\\bin\\windows\\sec2min.exe", "S:\\temp\\testi5_2_original.dft S:\\temp\\testi5_2_original.dft");
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("S:\\temp\\testi5_2.dft");
            file.TimeConversion( TimeUnit.minute );
            file.WriteFile("S:\\temp\\testi5_2.dft");
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "S:\\temp\\testi5_2_original.dft S:\\temp\\testi5_2.dft") == 0);
        }

        /// <summary>
        /// Reads/Write test of single column in plasma file
        /// </summary>
        [Test]
        public void test1_2_read_write_SingleColumn_PLASMA()
        {
            Console.WriteLine("\n\n\nTEST CASE 1.2 and 2.2: READING SINGLE COLUMN FROM FILE");

            String filename = "P:\\data\\dft\\test7.kbq";
            DFTPlasmaCurveFile file = new DFTPlasmaCurveFile(filename);
            file.ReadFile();

            TableCell[] ct = file.Curves.GetColumn(0);

            // These are the right values in test7.kbq
            Double[] fixedValues = new double[] { 10.272, 9.082, 8.281, 7.706, 7.344, 6.951, 6.748, 6.467 };

            for( int i=0; i<file.Curves.rows; i++ ) Assert.IsTrue (ct[i] == fixedValues[i] );
        }

        /// <summary>
        /// Test of one column reading/Writing
        /// </summary>
        [Test]
        public void test1_2_read_write_SingleColumn()
        {
            Console.WriteLine("\n\n\nTEST CASE 1.2 and 2.2: READING SINGLE COLUMN FROM FILE");

            String filename = "P:\\data\\dft\\Imadeus_weights.dft";
            DFTRegionalCurveFile file = new DFTRegionalCurveFile(filename);
            file.ReadFile();

            TAC ct = file.Curves.GetColumn(2);

            // file2 is generated with same header information as file
            DFTRegionalCurveFile file2 = new DFTRegionalCurveFile(file);
            
            file2.Curves.AddColumn(ct);

            file2.WriteFile("S:\\temp\\testi1_2.dft");

            // tm ei vain toimi.
            System.IO.File.Delete("S:\\temp\\testi1_2_original.dft");
            
            TestHelpers.addSingleColumnToFile(filename, "S:\\temp\\testi1_2_original.dft", 2);

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "S:\\temp\\testi1_2.dft S:\\temp\\testi1_2_original.dft") == 0);
        }

        /// <summary>
        /// Prints the file output to stdout
        /// </summary>
        [Test]
        public void test2_9_print_output_to_stdout()
        {
            Console.WriteLine("\n\n\nTEST CASE 2.9: READING FILE TO STDOUT");
            DFTRegionalCurveFile file = new DFTRegionalCurveFile("P:\\data\\dft\\Imadeus_noheader.dft");
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("P:\\data\\dft\\Imadeus.dft");

            //DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            //TAC customTAC = this.generateCustomTAC();
            //file.AddColumn(customTAC);

            TAC t = file.Curves.GetColumn(1);
            //Console.WriteLine( t.ToString() );

            file.writeToStdOut();
        }

        /// <summary>
        /// Test of removing one column from file
        /// </summary>
        [Test]
        public void test2_4_removeTACfromFile()
        {
            Console.WriteLine("\n\n\nTEST CASE 2.4: REMOVING A TAC FROM FILE");
            DFTRegionalCurveFile file = new DFTRegionalCurveFile("P:\\data\\dft\\Imadeus_weights.dft");
            file.ReadFile();

            file.Curves.DeleteColumn(2);
            //Console.WriteLine(t.ToString());
            file.WriteFile("S:\\temp\\testi2_4.dft");

            System.IO.File.Copy("P:\\data\\dft\\Imadeus_weights.dft", "S:\\temp\\test2_4_original.dft", true);

            // lets remove one TAC
            TestHelpers.getStdOut("P:\\bin\\windows\\dftdel_1_3_0.exe", "S:\\temp\\test2_4_original.dft 3");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "S:\\temp\\test2_4_original.dft S:\\temp\\testi2_4.dft") == 0);
        }

        /// <summary>
        /// Test of writing one column from file to file
        /// </summary>
        [Test]
        public void Test2_3_write_read_one_column_fileToFile()
        {
            
            Console.WriteLine("\n\n\nTEST CASE 2.3: READING AND WRITING ONE COLUMN FROM FILE TO FILE");
            DFTRegionalCurveFile file = new DFTRegionalCurveFile("P:\\data\\dft\\Imadeus.dft");
            file.ReadFile();
            TAC tac = file.Curves.GetColumn(2);

            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("P:\\data\\dft\\Imadeus_weights.dft");
            file.Curves.AddColumnAt(5,tac);
            file.WriteFile("S:\\temp\\testi2_3.dft");

            // weread the file again and take the column 5 and compare it to original
            file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("S:\\temp\\testi2_3.dft");

            TAC tac2 = file.Curves.GetColumn(5);

            Console.WriteLine( tac.ToString() + tac2.ToString() );

            Assert.IsTrue(TAC.Equals(tac, tac2) );

        }

        #region private members

        private void testFile(String filename)
        {
            Console.WriteLine("\n\n------\nTesting file: " + filename);

            DFTCurveFile.DFTFileType filetype = DFTCurveFile.CheckFormatByContents(filename);
            DFTCurveFile file;

            switch(filetype) { 
                case DFTCurveFile.DFTFileType.Regional:
                Console.WriteLine("File contents has been recognized to DEF_regional.");
                file = new DFTRegionalCurveFile(filename);

                ((DFTRegionalCurveFile)file).ReadFile();
                ((DFTRegionalCurveFile)file).WriteFile("testi.dft");
                break;
                case DFTCurveFile.DFTFileType.Plasma:
                Console.WriteLine("File contents has been recognized to DEF_plasma.");
                file = new DFTPlasmaCurveFile(filename);

                ((DFTPlasmaCurveFile)file).ReadFile();
                ((DFTPlasmaCurveFile)file).WriteFile("testi.dft");
                break;
                case DFTCurveFile.DFTFileType.Unknown:
                default:
                Console.WriteLine("Unknown header. Possibly missing header data. Trying Parent format.");
                file = new DFTCurveFile(filename);

                file.ReadFile();
                file.WriteFile("testi.dft");
                break;
            }

            //getStdOut("P:\\bin\\windows\\dftlist.exe", "testi.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "testi.dft " + filename) == 0);


        }


        /// <summary>
        /// generates a custom TAC for testing purposes
        /// </summary>
        /// <returns>TAC</returns>
        private TAC generateCustomTAC()
        {
            CurveHeader ch = new CurveHeader("head", "juu", "plane", 100.0f);
            //List<FrameTimeCell> frames = new List<FrameTimeCell>();
            List<TableCell> tablecells = new List<TableCell>();

            for (int i = 0; i < 10; i++)
            {
                /*FrameTimeCell c = new FrameTimeCell(FrameTimeCell.FrameTimetype.START);
                c.start_time = (Double)i;
                frames.Add(c);
                */
                TableCell tablecell = new TableCell();
                tablecell.Value = (double)i + (Double)100.05;
                tablecells.Add(tablecell);
            }

            return new TAC(ch, tablecells.ToArray() );
        }

        private void deleteFile(String filename)
        {
            if (System.IO.File.Exists("filename"))
            {
                System.IO.File.Delete("filename");
                Console.WriteLine("The file: " + filename + " is deleted...");
            }
            else
            {
                Console.WriteLine("no such file: "+filename );
            }
        }

        #endregion
    }

    #endregion

    /// <summary>
    /// Helper functions for test cases
    /// </summary>
    public class TestHelpers
    {
        /// <summary>
        /// This method runs a console program and returns its return value. Output of the program is put into Stdout
        /// </summary>
        /// <param name="procname">Name of program to run.</param>
        /// <param name="parameters">Parameters of the program.</param>
        /// <returns>Return value of the program.</returns>
        public static int getStdOut(string procname, string parameters)
        {
            Console.WriteLine(procname+" "+parameters);

            System.Diagnostics.Process proc = new System.Diagnostics.Process();
            proc.StartInfo.FileName = procname;
            proc.StartInfo.Arguments = parameters;
            proc.StartInfo.UseShellExecute = false;
            proc.StartInfo.RedirectStandardOutput = true;
            proc.Start();
            proc.WaitForExit(10000);
            string r = proc.StandardOutput.ReadToEnd();
            int code = proc.ExitCode;

            Console.WriteLine(procname + "printed " + r);
            Console.WriteLine(procname + "returned " + code);

            return code;
        }

        /// <summary>
        /// Adds one column from file to file using dftadd.exe.
        /// </summary>
        /// <param name="fromwhere">Filename where the column is retrieved.</param>
        /// <param name="towhere">Filename where the column is added</param>
        /// <param name="index">Index of column in source file</param>
        /// <returns>The code returned from dftadd.exe</returns>
        public static int addSingleColumnToFile(String fromwhere, String towhere, int index)
        {
            index++;
            System.Diagnostics.Process proc = new System.Diagnostics.Process();
            proc.StartInfo.FileName = "P:\\bin\\windows\\dftadd.exe";
            proc.StartInfo.Arguments = towhere + " " + fromwhere + " " + index;
            proc.StartInfo.UseShellExecute = false;
            proc.StartInfo.RedirectStandardOutput = true;
            proc.Start();
            proc.WaitForExit(10000);
            string r = proc.StandardOutput.ReadToEnd();
            int code = proc.ExitCode;

            Console.WriteLine("dftadd printed " + r);
            Console.WriteLine("dftadd returned " + code);

            return code;
        }

        /// <summary>
        /// Calculates the difference between two doubles
        /// </summary>
        /// <param name="a">First double</param>
        /// <param name="b">Second double</param>
        /// <returns>The difference between two doubles</returns>
        public static Double Difference( Double a, Double b )
        {
            return Math.Abs(a - b);
        }

        /// <summary>
        /// Generates 10 START_END frametimes as 0-1  1-2  2-3  3-4 ... etc 
        /// </summary>
        /// <returns>An array of FrameTimeCells</returns>
        public static FrameTimeCell[] generateTimes()
        {
            int num=10;

            FrameTimeCell[] cells = new FrameTimeCell[num];
            for (int i = 0; i < num; i++)
            {
                cells[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                cells[i].start_time = i;
                cells[i].end_time = i + 1;
            }

            return cells;
        }

        /// <summary>
        /// Fills TACTable with random values from 1 to 1000. 10 columns are created
        /// </summary>
        /// <param name="table">TACTable containing the added values</param>
        public static void fillNumbers( TACTable table )
        {
            Random rnd = new Random();

            // we fill the table with numbers 1-1000
            for( int w=0; w<10; w++ )
            {
                TableCell[] cells = new TableCell[table.rows];
                for (int i = 0; i < table.rows; i++)
                {
                    cells[i] = new TableCell();
                    cells[i].Value = rnd.Next(3, 1000);
                }
                table.AddColumn(cells);
            }
        }

        /// <summary>
        /// Generates an 10 cell array of TableCells starting from value baseint 
        /// </summary>
        /// <param name="baseint">The base value where the cell values are created.</param>
        /// <returns>An array of TableCells</returns>
        public static TableCell[] generateTableCells(int baseint)
        {
            TableCell[] cells = new TableCell[10];

            for( int i= 0; i<10; i++ ) cells[i] = new TableCell( baseint+i );

            return cells;
        }

        /// <summary>
        /// Fills TACTable with static values. 10 Columns will be created
        /// </summary>
        /// <param name="table">The TACTable containing the added values</param>
        public static void fillstaticNumbers(TACTable table)
        {
            int value=2;
            // we fill the table with numbers 1-1000
            for (int w = 0; w < 10; w++)
            {
                TableCell[] cells = new TableCell[table.rows];
                for (int i = 0; i < table.rows; i++)
                {
                    cells[i] = new TableCell();
                    cells[i].Value = value++;
                }
                table.AddColumn(cells);
            }
        }
    }
}
