﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.Data;
using System.IO;
using System.Text.RegularExpressions;

namespace TPClib.Curve
{
    /// <summary>
    /// Class for FITfile unit tests.
    /// (Classic Assert Model)
    /// </summary>
    [TestFixture]
    [Category("ShortRunning")]
    public class NUnitTestbench_FITfile
    {
        /// <summary>
        /// The path where is all the test data.
        /// </summary>
        private const string path = @"P:\data\fit\";

        /// <summary>
        /// 1.0 Test contstructor.
        /// </summary>
        [Test]
        public void Test1_0_Constructor()
        {
            string str = path + @"test1_0.fit";
            FITFile f = new FITFile(str);
            Assert.AreEqual(str, f.filename);
        }

        /// <summary>
        /// 1.1 Test that parameter table is formed correctly.
        /// </summary>
        [Test]
        public void Test1_1_CorrectParameterDataStructure()
        {
            FITFile file = new FITFile(path + @"test1_0.fit");
            ParameterTable pt = file.DataTable;
            Assert.AreEqual(7, pt.Parameters.Count);
        }

        /// <summary>
        /// 1.2 Read file test1_0.fit into FITHeader. 
        /// </summary>
        [Test]
        public void Test1_2_ReadHeader1()
        {
            FITFile f = new FITFile(path + @"test1_0.fit");
            f.ReadFile();
            Assert.AreEqual("fit_exp 2.6.0  (c) 1994-2006 by Turku PET Centre", f.Header.Program);
            Assert.AreEqual(DateTime.Parse("2008-05-20"), f.Header.DateTime);
            Assert.AreEqual("ua4131dy1.dft", f.Header.DataFile);
            Assert.AreEqual(UnitConverter.CreateActivityUnit("kBq/ml"), f.Header.DataUnit);
            Assert.AreEqual(TimeUnit.minute, f.Header.TimeUnit);
            Assert.AreEqual(9, f.Header.NumberOfVOIs);
        }

        /// <summary>
        /// 1.3 Read file test1_1.fit into FITHeader.
        /// </summary>
        [Test]
        public void Test1_3_ReadHeader2()
        {
            FITFile f = new FITFile(path + @"test1_1.fit");
            f.ReadFile();
            Assert.AreEqual("fit_exp 2.6.0  (c) 1994-2006 by Turku PET Centre", f.Header.Program);
            Assert.AreEqual(DateTime.Parse("2008-05-20"), f.Header.DateTime);
            Assert.AreEqual("ua4131dy1.dft", f.Header.DataFile);
            Assert.AreEqual(UnitConverter.CreateActivityUnit("kBq/ml"), f.Header.DataUnit);
            Assert.AreEqual(TimeUnit.minute, f.Header.TimeUnit);
            Assert.AreEqual(9, f.Header.NumberOfVOIs);
        }

        /// <summary>
        /// 1.4 Write the file test1_0.fit to the disk
        /// </summary>
        [Test]
        public void Test1_4_WriteFile1()
        {
            string testFile = path + @"test1_0.fit";
            string resultFile = path + "temp_result.fit";

            FITFile g = new FITFile(testFile);
            g.ReadFile();

            FITFile f = new FITFile(resultFile);
            f.Header = g.Header;
            f.DataTable = g.DataTable;
            f.WriteFile();

            FileInfo nfo = new FileInfo(resultFile);
            Assert.AreEqual(true, nfo.Exists);

            FileAssert.AreEqual(testFile, resultFile);

            File.Delete(resultFile);
        }

        /// <summary>
        /// 1.5 Write the file test1_1.fit to the disk
        /// </summary>
        [Test]
        public void Test1_5_WriteFile2()
        {
            FITFile g = new FITFile(path + @"test1_1.fit");
            g.ReadFile();

            FITFile f = new FITFile(path + "temp_result.fit");
            f.Header = g.Header;
            f.DataTable = g.DataTable;
            f.WriteFile();

            FileInfo nfo = new FileInfo(path + "temp_result.fit");
            Assert.AreEqual(true, nfo.Exists);

            FileAssert.AreEqual(path + @"test1_1.fit", path + "temp_result.fit");

            File.Delete(path + "temp_result.fit");
        }

        /// <summary>
        /// 1.6 Write a file with different time unit.
        /// </summary>
        [Test]
        public void Test1_6_Write()
        {
            FITHeader h = new FITHeader();
            h.Program = "Abstract FIT skit program";
            h.DataFile = "abcdefg.dat";
            h.Version = "1";
            h.DataUnit = Data_unit.ML_per_DLtimesMIN;
            h.NumberOfVOIs = 0;
            h.TimeUnit = TimeUnit.nanosecond;

            FITFile f = new FITFile(path + "temp.fit");
            f.Header = h;
            f.WriteFile();

            f.ReadFile();
            Console.WriteLine(f.Header.HeaderToString());
            File.Delete(path + "temp.fit");
            File.Delete(path + "temp.fit.bak");
        }

        /// <summary>
        /// 1.8 A Header is missing a line.
        /// </summary>
        [Test]
        public void Test1_8_MissingHeaderLine()
        {
            try
            {
                // the missing line is commented

                FITHeader h = new FITHeader();
                string line = "FIT1 SomeProgram 10 000 BC.\n";
                line += "Date:       2008-05-20\n";
                line += "Data file:  ua4131dy1.dft\n";
                //line += "Data unit:  kBq/ml\n";  
                line += "Time unit:  min\n";
                line += "Nr of VOIs: 2\n";
                line += "Extra Line\n";
                h.Parse(line);
                Assert.Fail();
            }
            catch(TPCFITfileException e)
            {
                StringAssert.StartsWith("Header Format Error", e.Message);
            }
        }

        /// <summary>
        /// 2.0 Try to read null file.
        /// </summary>
        [Test]
        public void Test2_0_TryToReadNull()
        {
            try
            {
                FITFile f = new FITFile(null);
                f.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("Null pointer", e.Message);
            }
        }

        /// <summary>
        /// 2.1 Read incorrect binary file test2_1.fit
        /// </summary>
        [Test]
        public void Test2_1_Binary()
        {
            try
            {
                FITFile fff = new FITFile(path + @"test2_1.fit");
                fff.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("Wrong File Format", e.Message);
            }
        }


        /// <summary>
        /// 2.2 Read file test_2_2.fit with wrong data with ASCII encoding
        /// </summary>
        [Test]
        public void Test2_2_WrongASCIIFile()
        {
            try
            {
                FITFile file = new FITFile(path + "test2_2.fit");
                file.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("Wrong File Format", e.Message);
            }
        }

        /// <summary>
        /// 2.3 Try read file with wrong extension ua4131dy1.dft
        /// </summary>
        [Test]
        public void Test2_3_WrongExtension()
        {
            try
            {
                FITFile fff = new FITFile(path + @"ua4131dy1.dft");
                fff.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("Wrong File Extension", e.Message);
            }
        }

        /// <summary>
        /// 2.4 Try to read empty file
        /// </summary>
        [Test]
        public void Test2_4_EmptyFile()
        {
            FITFile f = new FITFile(path + "test2_4.fit");
            try
            {
                f.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("Wrong File Format", e.Message);
            }
        }
        
        /// <summary>
        /// 2.5 The header ROI number doesn't match with the real
        /// ROI number
        /// </summary>
        [Test]
        public void Test2_5_WrongROInumber()
        {
            try
            {
                FITFile file = new FITFile(path + @"test2_0.fit");
                file.ReadFile();
                Assert.Fail();
            }
            catch (Exception e)
            {
                StringAssert.StartsWith("TEST1", e.Message);
            }

        }

        /// <summary>
        /// 2.6 The file doesn't exist.
        /// </summary>
        [Test]
        public void Test2_6_FileNotExist()
        {
            string filename = "kasjasdfasdf.fit";

            try
            {
                FITFile ff = new FITFile(filename);
                ff.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("File "+filename+" doesn't exist", e.Message);
            }
        }

        /// <summary>
        /// 2.7 Tests a file which has more spaces than needed or
        /// less than needed.
        /// </summary>
        [Test]
        public void Test2_7_IncorrectFile2()
        {
            string tmp = path + "temp.fit";
            FITFile file = new FITFile();
            file.filename = path + "test2_3.fit";
            file.ReadFile();
            file.filename = tmp;
            file.WriteFile();

            // tiedoston on oltava merkilleen sama
            FileAssert.AreEqual(path + @"test1_0.fit", tmp);
            File.Delete(tmp);

            FileInfo nfo = new FileInfo(tmp);
            Assert.IsFalse(nfo.Exists);
        }

        /// <summary>
        /// 3.1 Read file test3_1.fit with  parameters of variable length
        /// </summary>
        [Test]
        public void Test3_1_DifferentParameters()
        {
            string _name = path + "test3_1.fit";
            string result = path + "result.fit";
            FITFile g = new FITFile(_name);
            g.ReadFile();

            FITFile f = new FITFile(result);
            f.Header = g.Header;
            f.DataTable = g.DataTable;
            f.WriteFile();

            FileInfo nfo = new FileInfo(result);
            Assert.AreEqual(true, nfo.Exists);

            FileAssert.AreEqual(_name, result);

            File.Delete(result);            
        }

        /// <summary>
        /// 3.2 The file has more parameters
        /// than class variable MAX_FITPARAMS defines. 
        /// </summary>
        /// <remarks>
        /// In here 8 > 6.
        /// </remarks>
        [Test]
        public void Test3_2_TooManyParameters()
        {
            try
            {
                FITFile g = new FITFile(path + "test3_2.fit");
                g.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                StringAssert.EndsWith("too many parameters", e.Message);
            }
        }

        /// <summary>
        /// 3.3 The parameter data has incorrect number of parameters
        /// </summary>
        [Test]
        public void Test3_3_IncorrectParamNumber()
        {
            try
            {
                FITFile g = new FITFile(path + "test3_3.fit");
                g.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                StringAssert.StartsWith("parNr", e.Message);
            }
        }

        /// <summary>
        /// 3.4 The parameter data has too few parameters
        /// </summary>
        [Test]
        public void Test3_4_ParameterNumberNotMatch()
        {
            try {
                FITFile f = new FITFile(path + "test3_4.fit");
                f.ReadFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.AreEqual("parNr doesn't match", e.Message);
            }
        }

        /// <summary>
        /// 3.5 File contains one incorrect double value
        /// </summary>
        [Test]
        [ExpectedException( typeof(System.FormatException))]
        public void Test3_5_FormatException()
        {
            FITFile f = new FITFile(path + "test3_5.fit");
            f.ReadFile();
        }

        /// <summary>
        /// 3.6 Read file test3_6.fit and test3_7.fit  with wrong number of columns.
        /// File lacks one column.
        /// </summary>
        [Test]
        [ExpectedException(typeof(System.FormatException))]
        public void Test3_6_FormatException()
        {
            FITFile f = new FITFile(path + "test3_6.fit");
            f.ReadFile();
        }

        /// <summary>
        /// Read file test3_6.fit and test3_7.fit  with wrong number of columns
        /// File lacks one column
        /// </summary>
        [Test]
        [ExpectedException(typeof(System.FormatException))]
        public void Test3_7_FormatException()
        {
            FITFile f = new FITFile(path + "test3_7.fit");
            f.ReadFile();
        }


        /// <summary>
        /// 4.0 Make a backup and write to existing file. 
        /// </summary>
        [Test]
        public void Test4_0_FileIsReadOnly()
        {
            string fname = path + "test4_0.fit";
            string fbak = fname + ".bak";
            FITFile f = new FITFile(fname);
            File.SetAttributes(fname, FileAttributes.ReadOnly);
            
            f.ReadFile();

            try
            {
                f.Header.Program = "dummy.exe";
                f.WriteFile();
                Assert.Fail();
            }
            catch (UnauthorizedAccessException e)
            {
                string s = e.Message;
                Assert.IsTrue(s.StartsWith("Access to the path"));
                Assert.IsTrue(s.EndsWith("is denied."));
            }

            Assert.IsTrue(File.Exists(fbak));
            FileAssert.AreEqual(fbak, fname);
            File.SetAttributes(fbak, FileAttributes.Normal);
            File.Delete(fbak);
        }

        /// <summary>
        /// 4.2 Write to existing file and make backup.
        /// Backup file doesn't exist.
        /// </summary>
        [Test]
        public void Test4_2_Backup()
        {
            string fname = path + "test4_2.fit";
            string forg = path + "original.fit";
            string fbak = path + "test4_2.fit.bak";
            
            FITFile f = new FITFile(fname);
            File.SetAttributes(fname, FileAttributes.Normal);
            f.ReadFile();

            if (File.Exists(fbak))
            {
                File.SetAttributes(fbak,FileAttributes.Normal);
                File.Delete(fbak);
            }

            if (File.Exists(forg))
            {
                File.SetAttributes(forg, FileAttributes.Normal);
                File.Delete(forg);
            }

            File.Copy(fname, forg);

            Assert.IsTrue( File.Exists(forg) );
            Assert.IsFalse( File.Exists(fbak) );
            Assert.IsTrue( File.Exists(fname) );
            
            // edit
            f.Header.Program = "dummy.exe";
            f.WriteFile();

            Assert.IsTrue(File.Exists(fbak));

            FileAssert.AreEqual(fbak, forg);

            // delete test file
            File.Delete(fbak);
            File.Delete(forg);
        }

        /// <summary>
        /// 4.3 Backup a file that exists already, but
        /// is not read-only. It will be overwritten.
        /// </summary>
        [Test]
        public void Test4_3_BackupReadOnly()
        {
            string fname = path + "test4_2.fit";
            string fbak = path + "test4_2.fit.bak";

            FITFile f = new FITFile(fname);
            f.ReadFile();

            if (!File.Exists(fbak))
                File.Copy(fname, fbak);
            
            Assert.IsTrue(File.Exists(fbak));
            Assert.IsTrue(File.Exists(fname));

            File.SetAttributes(fbak, FileAttributes.Normal);

            f.WriteFile();

            Assert.IsTrue(File.Exists(fbak));

            // delete test file
            File.Delete(fbak);
        }

        /// <summary>
        /// 4.4 The Backup file already exists and is readonly.
        /// The original file is read-only.
        /// </summary>
        [Test]
        public void Test4_4_BackupAndFileReadOnly()
        {
            string fname = path + "test4_2.fit";
            string fbak = path + "test4_2.fit.bak";

            FITFile f = new FITFile(fname);
            f.ReadFile();

            if (!File.Exists(fbak))
                File.Copy(fname, fbak);

            Assert.IsTrue(File.Exists(fbak));
            Assert.IsTrue(File.Exists(fname));

            File.SetAttributes(fbak,
                File.GetAttributes(fbak) |
                FileAttributes.ReadOnly);

            File.SetAttributes(fname,
                File.GetAttributes(fname) |
                FileAttributes.ReadOnly);

            try
            {
                f.WriteFile();
                Assert.Fail();
            }
            catch (TPCFITfileException e)
            {
                Assert.IsTrue(
                    e.Message.StartsWith("Read-only"));
            }

            Assert.IsTrue(File.Exists(fbak));

            // restore
            File.SetAttributes(fname, FileAttributes.Normal);
        }

    }
}

