﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System;
using System.IO;
using TPClib.Curve;
using System.Collections.Generic;

namespace TPClib.Curve {
    /// <summary>
    /// 
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_IDWCCurveFile {

        // Path to test files
        private const string path = @"P:\data\idwc\";
        //private const string path = @"d:\pet\";

        // Some datavalues to check against
        private double[] expected_times   = {  30, 90, 240, 480 };
        private double[] expected_weights = {  1.86300000000000, 0.18430000000000, 0.07599000000000,  0.05491000000000 };
        private double[] expected_tac1    = {  0.24100000000000, 2.88800000000000, 7.60500000000000, 10.22000000000000 };
        private double[] expected_tac2    = {  0.25760000000000, 2.55800000000000, 6.30600000000000,  9.18200000000000 };
        private double[] expected_tac3    = { -0.09263000000000, 3.00001200000000, 5.89620000000000, 11.00420000000000 };

        /// <summary>
        /// 1.0 Data is read from file IDWCTest1_0.idwc into TACtable.
        /// </summary>
        [Test]
        public void Test1_0()
        {
            string filename = path + @"IDWCTest1_0.idwc";

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();

            FrameTimeCell[] times = idwc.Curves.GetTimeCells();
            TableCell[] tac1 = idwc.Curves.GetColumn(0);
            TableCell[] tac2 = idwc.Curves.GetColumn(1);

            for(int i = 0; i < times.Length; i++)
            {
                Assert.AreEqual(expected_times[i], times[i].Value);
                Assert.AreEqual(expected_tac1[i], tac1[i].Value);
                Assert.AreEqual(expected_tac2[i], tac2[i].Value);
                Assert.AreEqual(expected_weights[i], (tac1[i] as WeightedTableCell).Weight);
                Assert.AreEqual(expected_weights[i], (tac2[i] as WeightedTableCell).Weight);
            }
        }

        /// <summary>
        /// 1.1 Read file IDWCTest1_1.idwc with binary data
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test1_1()
        {
            string filename = path + @"IDWCTest1_1.bin";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.2 Read file IDWCTest1_2.idwc with TACs of variable length
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test1_2()
        {
            string filename = path + @"IDWCTest1_2.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }
        /// <summary>
        /// 1.3 Read file IDWCTest1_3.idwc missing the first line
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test1_3()
        {
            string filename = path + @"IDWCTest1_3.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.4 Read file IDWCTest1_4.idwc with wrong number of columns
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test1_4()
        {
            string filename = path + @"IDWCTest1_4.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.5 Read empty file
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test1_5()
        {
            string filename = path + @"IDWCTest1_5.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.6 Read file with 1 TACs
        /// </summary>
        [Test]
        public void Test1_6()
        {
            string filename = path + @"IDWCTest1_6.idwc";
            string before = File.ReadAllText(filename, System.Text.Encoding.ASCII);

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.7 Read file with 2 TACs
        /// </summary>
        [Test]
        public void Test1_7()
        {
            string filename = path + @"IDWCTest1_7.idwc";
            string before = File.ReadAllText(filename, System.Text.Encoding.ASCII);

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
        }

        /// <summary>
        /// 1.8 Check file format of valid IDWC file
        /// </summary>
        [Test]
        public void Test1_8()
        {
            string filename = path + @"IDWCTest1_8.idwc";
            Assert.IsTrue(IDWCCurveFile.CheckFormat(filename));
        }
        
        /// <summary>
        /// 1.9 Check file not in valid IDWC format
        /// </summary>
        [Test]
        public void Test1_9()
        {
            string filename = path + @"IDWCTest1_9.idwc";
            Assert.IsFalse(IDWCCurveFile.CheckFormat(filename));
        }

        /// <summary>
        /// 2.0 Write data that is read from file
        /// </summary>
        [Test]
        public void Test2_0()
        {
            string filename = path + @"IDWCTest2_0.idwc";
            string before = File.ReadAllText(filename, System.Text.Encoding.ASCII);

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
            idwc.WriteFile();
            string after = idwc.FileToString();

            // Possible trailing whitespace is not reproduced
            Assert.IsTrue(String.Equals(before.Trim(), after));
        }

        /// <summary>
        /// 2.1 Write into file that has no write permission
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test2_1()
        {
            string filename = path + @"IDWCTest2_1.idwc";

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            idwc.ReadFile();
            idwc.WriteFile();
        }

        /// <summary>
        /// 2.2 Write data with no TACs
        /// </summary>
        [Test]
        [ExpectedException(typeof(TPCIDWCFileException))]
        public void Test2_2()
        {
            string filename = path + @"IDWCTest2_2.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);

            // Insert data directly into TACTable
            FrameTimeCell[] frames = new FrameTimeCell[expected_times.Length];
            for(int i=0; i<frames.Length; i++)
            {
                frames[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                frames[i].Value = expected_times[i];
            }

            TACTable ttable = new TACTable(frames);

            // Write data to file
            idwc.Curves = ttable;
            Assert.AreEqual(0, idwc.Curves.columns);
            idwc.WriteFile();

            // Read data from file
            idwc.ReadFile();

            /* Comparison below doesn't work! TACTable compares objects, not values...
            // Compare data from file to the original TACTable
            Assert.IsTrue(TACTable.Equals(idwc.Curves, ttable));
            */
        }

        /// <summary>
        /// 2.3 Write data with 1 TAC
        /// </summary>
        [Test]
        public void Test2_3()
        {
            string filename = path + @"IDWCTest2_3.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);

            // Insert data directly into TACTable
            FrameTimeCell[] frames = new FrameTimeCell[expected_times.Length];
            WeightedTableCell[] samples = new WeightedTableCell[expected_tac1.Length];
            for (int i = 0; i < frames.Length; i++)
            {
                frames[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                frames[i].Value = expected_times[i];
                samples[i] = new WeightedTableCell(expected_tac1[i], expected_weights[i]);
            }

            TACTable ttable = new TACTable(frames);
            ttable.AddColumn(samples);

            // Write data to file
            idwc.Curves = ttable;
            Assert.AreEqual(1, idwc.Curves.columns);
            idwc.WriteFile();

            // Read data from file
            idwc.ReadFile();

            // Compare data from file to the original TACTable
            Assert.IsTrue(TACTable.Equals(idwc.Curves, ttable));
        }

        /// <summary>
        /// 2.4 Write data with 3 TACs
        /// </summary>
        [Test]
        public void Test2_4()
        {
            string filename = path + @"IDWCTest2_4.idwc";
            IDWCCurveFile idwc = new IDWCCurveFile(filename);

            // Insert data directly into TACTable
            FrameTimeCell[] frames = new FrameTimeCell[expected_times.Length];
            WeightedTableCell[] samples1 = new WeightedTableCell[expected_tac1.Length];
            WeightedTableCell[] samples2 = new WeightedTableCell[expected_tac2.Length];
            WeightedTableCell[] samples3 = new WeightedTableCell[expected_tac3.Length];

            for (int i = 0; i < frames.Length; i++)
            {
                frames[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                frames[i].Value = expected_times[i];
                samples1[i] = new WeightedTableCell(expected_tac1[i], expected_weights[i]);
                samples2[i] = new WeightedTableCell(expected_tac2[i], expected_weights[i]);
                samples3[i] = new WeightedTableCell(expected_tac3[i], expected_weights[i]);
            }

            TACTable ttable = new TACTable(frames);
            ttable.AddColumn(samples1);
            ttable.AddColumn(samples2);
            ttable.AddColumn(samples3);

            // Write data to file
            idwc.Curves = ttable;
            Assert.AreEqual(3, idwc.Curves.columns);
            idwc.WriteFile();

            // Read data from file
            idwc.ReadFile();

            // Compare data from file to the original TACTable
            Assert.IsTrue(TACTable.Equals(idwc.Curves, ttable));
        }

        /// <summary>
        /// 2.5 Write empty dataset
        /// </summary>
        [Test]
        public void Test2_5()
        {
            string filename = path + @"IDWCTest2_5.idwc";

            IDWCCurveFile idwc = new IDWCCurveFile(filename);
            Assert.AreEqual(0, idwc.Curves.columns);
            idwc.WriteFile();

            // Read data from file
            idwc.ReadFile();

            TACTable ttable = new TACTable();

            // Compare data from file to the original TACTable
            Assert.IsTrue(TACTable.Equals(idwc.Curves,ttable)); 
        }

    }
}
