﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using NUnit.Framework;
using System.Data;
using System.IO;


//DataTable.AddColumn( colName[i], colType[i] );

namespace TPClib.Curve
{
    /// <summary>
    /// Class for FITfile unit tests
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_ParameterTable
    {
        private ParameterTable table;

#region General operations

        /// <summary>
        /// Inits the testbench for parametertable
        /// </summary>
        [SetUp]
        public void Init()
        {
            /*
            String[] h = new string[] { "region", "value" };
            Type[] t = new Type[] { typeof(String), typeof(Double) };

            table = new ParameterTable(h, t);

            Assert.AreEqual(2, table.Columns);*/
        }
 
        /// <summary>
        /// Test case for adding duplicate column into parameter table
        /// </summary>
        [Test]
        [ExpectedException(typeof(DuplicateNameException))]
        public void Test4_1_AddDuplicateParameter()
        {
            table = new ParameterTable();
            table.Parameters.Add("Region");
        }

        /// <summary>
        /// Test case for adding parameters into parameter table
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test2_1_AddParameters()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell( new RegionCell.RegionStruct("Reg1") ) );
            table.Regions.Add(new RegionCell( new RegionCell.RegionStruct("Reg2") ) );
            table.Regions.Add(new RegionCell( new RegionCell.RegionStruct("Reg3") ) );

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            Assert.IsTrue(table.Parameters[0][0].Value == new RESCell(1.0d).Value);
            Assert.IsFalse(table.Parameters[0][0].Value == new RESCell(5.0d).Value);

            Assert.IsTrue(table.Parameters[1][2].Value == new RESCell(7.0d).Value);
            Assert.IsFalse(table.Parameters[1][2].Value == new RESCell(5.0d).Value);

            // Testing also name indexes
            Assert.IsTrue(table.Parameters["MyOwn"][0].Value == new RESCell(1.0d).Value);
            Assert.IsFalse(table.Parameters["myown"][0].Value == new RESCell(5.0d).Value);

            Assert.IsTrue(table.Parameters["second"][2].Value == new RESCell(7.0d).Value);
            Assert.IsFalse(table.Parameters["Second"][2].Value == new RESCell(5.0d).Value);


            Console.WriteLine( table.ToString() );
        }


        /// <summary>
        /// Test case for getting index of region by name
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test1_1_Get_region_index()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d, 5.0d ),
                new RESCell( 3.0d )}
                );

            Assert.IsTrue( table.Regions.GetIndex("reg1") == 0 );
            Assert.IsTrue( table.Regions.GetIndex("reg2") == 1 );
            Assert.IsTrue( table.Regions.GetIndex("reg3") == 2 );
        }

        /// <summary>
        /// Test case for constructors of ParameterTable
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_1_Constructors()
        {
            // First table
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d, 5.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            // Second table
            ParameterTable table2 = new ParameterTable(
                new RegionCell(new RegionCell.RegionStruct("Reg1")),
                new RegionCell(new RegionCell.RegionStruct("Reg2")),
                new RegionCell(new RegionCell.RegionStruct("Reg3"))
                );
            table2.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d, 5.0d ),
                new RESCell( 3.0d )}
                );

            table2.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            // third table
            ParameterTable table3 = new ParameterTable(table);
            table3.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d, 5.0d ),
                new RESCell( 3.0d )}
                );

            table3.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            // fourth table
            ParameterTable table4 = new ParameterTable("MyOwn", "Second");
            table4.Regions.Add(
                new RegionCell(new RegionCell.RegionStruct("Reg1")),
                new RESCell(1.0d),
                new RESCell(5.0d)
                );
            table4.Regions.Add(
                new RegionCell(new RegionCell.RegionStruct("Reg2")),
                new RESCell(2.0d, 5.0d),
                new RESCell(6.5d)
                );
            table4.Regions.Add(
                new RegionCell(new RegionCell.RegionStruct("Reg3")),
                new RESCell(3.0d),
                new RESCell(7.0d)
                );

            Assert.IsTrue( table.Equals( table2 ) );
            Assert.IsTrue( table.Equals( table3 ) );
            Assert.IsTrue( table.Equals( table4 ) );
        }

        /// <summary>
        /// Test case for removing regions/parameters from ParameterTable
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_2_Remove_regions_parameters()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d, 5.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            // Lets test remove() for parameters:
            Assert.IsTrue(table.Parameters.Count == 2);
            table.Parameters.Remove(1);
            Assert.IsTrue(table.Parameters.Count == 1);
            Assert.IsTrue(table.Parameters["MyOwn"][0].Value == 1.0d);
            Assert.IsTrue(table.Parameters["MyOwn"][1].Value == 2.0d);
            Assert.IsTrue(((RESCell)table.Parameters["MyOwn"][1]).SD == 5.0d); // Test for not TableCell

            // Lets add the parameter column again
            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            // Lets test remove for regions
            Assert.IsTrue(table.Regions.Count == 3);
            table.Regions.Remove(2);
            Assert.IsTrue(table.Regions.Count == 2);
            Assert.IsTrue(table.Regions[1][0].Value == 2.0d);
            Assert.IsTrue(((RESCell)table.Regions[1][0]).SD == 5.0d);
            table.Regions.Remove(0);
            Assert.IsTrue(table.Regions.Count == 1);
            Assert.IsTrue(table.Regions[0][0].Value == 2.0d);
            Assert.IsTrue(((RESCell)table.Regions[0][0]).SD == 5.0d);
            Assert.IsTrue(table.Regions[0][1].Value == 6.5d);
        }

        /// <summary>
        /// Test case for getting and setting parameters
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test2_2_Get_Set_Parameters()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            Assert.IsTrue(table.Parameters[0][0].Value == new RESCell(1.0d).Value);
            Assert.IsFalse(table.Parameters[0][0].Value == new RESCell(5.0d).Value);

            Assert.IsTrue(table.Parameters[1][2].Value == new RESCell(7.0d).Value);
            Assert.IsFalse(table.Parameters[1][2].Value == new RESCell(5.0d).Value);

            table.Parameters[0] = new RESCell[] {
                new RESCell( 11.0d ), 
                new RESCell( 12.0d ),
                new RESCell( 13.0d )
            };

            table.Parameters["second"] = new RESCell[] {
                new RESCell( 16.0d ), 
                new RESCell( 17.0d ),
                new RESCell( 18.0d )
            };

            Assert.IsTrue(table.Parameters[0][0].Value == new RESCell(11.0d).Value);
            Assert.IsFalse(table.Parameters[0][0].Value == new RESCell(5.0d).Value);

            Assert.IsTrue(table.Parameters[1][2].Value == new RESCell(18.0d).Value);
            Assert.IsFalse(table.Parameters[1][2].Value == new RESCell(5.0d).Value);

            TableCell[] cells = table.Parameters.Get("MyOwn");
            Assert.IsTrue(cells[1].Value == 12.0d);

           

            table.Parameters["MyOwn", 1] = new RESCell( 2.0d );
            Assert.IsTrue(table.Parameters[ "MyOwn", 1 ].Value == 2.0d);
            Assert.IsTrue(table.Parameters.Get( 0, 1 ).Value == 2.0d);

            /*
            table.Parameters.Set("MyOwn", new RESCell(6.6d), new RESCell(7.7d), new RESCell(8.8d));
            Assert.IsTrue(table.Parameters["MyOwn", 0 ].Value == 6.6d );
            Assert.IsTrue(table.Parameters["MyOwn", 1 ].Value == 7.7d);
            Assert.IsTrue(table.Parameters["MyOwn", 2 ].Value == 8.8d);*/
        }


        /// <summary>
        /// Test case for adding regions into parameter table
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test1_2_Add_Regions()
        {
            ParameterTable table = new ParameterTable();
            table.Parameters.Add("param1");
            table.Parameters.Add("param2");
            table.Parameters.Add("param3", "param4");

            table.Regions.Add(
                new RegionCell( new RegionCell.RegionStruct("Region1", "Jee", "plane")),
                new RESCell[] {
                    new RESCell( 1.0d ),
                    new RESCell( 2.0d ),
                    new RESCell( 3.0d ),
                    new RESCell( 4.0d )
                });
            table.Regions.Add(
                new RegionCell(new RegionCell.RegionStruct("Region2", "Jee", "plane2")),
                new RESCell( 11.0d ),
                new RESCell( 12.0d ),
                new RESCell( 13.0d ),
                new RESCell( 14.0d )
                );

            Assert.IsTrue( table.Regions[0][0].Value == 1.0d );
            Assert.IsTrue(table.Regions[1][0].Value == 11.0d);
            Assert.IsTrue(table.Regions[1][3].Value == 14.0d);
            Assert.IsTrue(table.Regions.GetParameters(1)[3].Value == 14.0d);

            table.Regions.AddAt(0,
                new RegionCell(new RegionCell.RegionStruct("AddAt", "huu", "plane")),
                new RESCell(21.0d),
                new RESCell(22.0d),
                new RESCell(23.0d),
                new RESCell(24.0d));

            Assert.IsTrue(table.Regions[0][0].Value == 21.0d);
            Assert.IsTrue(table.Regions[2][3].Value == 14.0d);
        }

        /// <summary>
        /// Test case for getting/setting regions into parameter table
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test1_3_Get_Set_Regions()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            Assert.IsTrue(table.Parameters[0][0].Value == new RESCell(1.0d).Value);
            Assert.IsFalse(table.Parameters[0][0].Value == new RESCell(5.0d).Value);

            // Test of full parameter line
            table.Regions.SetParameters(0, new TableCell(40.0d), new TableCell(41.0d));
            Assert.IsTrue(table[0, 0].Value == 40.0d); // test of tables main indexer: [int,int]
            Assert.IsTrue(table[1, 0].Value == 41.0d); // test of tables main indexer: [int,int]

            TableCell[] cells = table.Regions.GetParameters(0); // reading
            Assert.IsTrue(cells[0].Value == 40.0d);
            Assert.IsTrue(cells[1].Value == 41.0d);

            // Test of one parameter
            Assert.IsTrue(table.Regions.Get(0, 0) == 40.0d);
            Assert.IsTrue(table.Regions.Get(0, 1) == 41.0d);

            table.Regions.Set(1,1, new TableCell(55.0d) ); // test of setting one cell
            Assert.IsTrue(table.Regions.Get(1, 1) == 55.0d);
        }


        /// <summary>
        /// Test case for index out of bounds exception
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test4_2_Boundary_Checks()
        {
            Console.WriteLine("TESTING INDEX BOUNDARIES:");

            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            for( int i=0; i <= 40; i++ )
            {
                try
                {
                    if (i == 0) { table.Parameters.Get(-1); }
                    if (i == 1) { table.Parameters.Get(2); }
                    if (i == 2) { table.Parameters.GetName(-1);  }
                    if (i == 3) { table.Parameters.GetName(2); }
                    if (i == 4) { table.Parameters.Remove(-1); }
                    if (i == 5) { table.Parameters.Remove(2); }
                    if (i == 6) { table.Parameters.Set(-1, new TableCell( 1.0d), new TableCell(2.0d), new TableCell(3.0d)); }
                    if (i == 7) { table.Parameters.Set(2, new TableCell(1.0d), new TableCell(2.0d), new TableCell(3.0d)); }
                    if (i == 8) { table.Parameters.SetName(-1, "g"); }
                    if (i == 9) { table.Parameters.SetName(2, "g"); }
                    if (i == 10) { table.Parameters.Sort(-1, false); }
                    if (i == 11) { table.Parameters.Sort(2, false); }
                    if (i == 12) { table.Parameters.Set(-1,2, new TableCell(2.0d) ); }
                    if (i == 13) { table.Parameters.Set(2, 2, new TableCell(2.0d)); }
                    if (i == 14) { table.Parameters.Set(0, 3, new TableCell(2.0d)); }
                    if (i == 15) { table.Parameters.Set(0, -1, new TableCell(2.0d)); }
                    if (i == 16) { table[-1, -1].Value = 2.0d; }
                    if (i == 17) { table[-1, 0].Value = 2.0d; }
                    if (i == 18) { table[ 2, 0].Value = 2.0d; }
                    if (i == 19) { table[ 0, -1].Value = 2.0d; }
                    if (i == 20) { table[ 0, 3].Value = 2.0d; }

                    if (i == 21) { table.Regions.GetHeader(-1); }
                    if (i == 22) { table.Regions.GetHeader( 3); }
                    if (i == 23) { table.Regions.SetHeader(-1, new RegionCell() ); }
                    if (i == 24) { table.Regions.SetHeader(-3, new RegionCell() ); }
                    if (i == 25) { table.Regions.AddAt(-1, new RegionCell(), new TableCell(1.0d), new TableCell(1.0d)); }
                    if (i == 26) { table.Regions.AddAt( 4, new RegionCell(), new TableCell(1.0d), new TableCell(1.0d)); }
                    if (i == 27) { table.Regions.Get(-1,0); }
                    if (i == 28) { table.Regions.Get( 3,0); }
                    if (i == 29) { table.Regions.Get(0, -1); }
                    if (i == 30) { table.Regions.Get(0, 2); }
                    if (i == 31) { table.Regions.GetParameters(-1); }
                    if (i == 32) { table.Regions.GetParameters( 3); }
                    if (i == 33) { table.Regions.SetParameters(-1, new TableCell(1.0d), new TableCell(1.0d)); }
                    if (i == 34) { table.Regions.SetParameters( 3, new TableCell(1.0d), new TableCell(1.0d)); }
                    if (i == 35) { table.Regions.Set(-1, 0, new TableCell(1.0d)); }
                    if (i == 36) { table.Regions.Set( 3, 0, new TableCell(1.0d)); }
                    if (i == 37) { table.Regions.Set( 0,-1, new TableCell(1.0d)); }
                    if (i == 38) { table.Regions.Set( 0, 2, new TableCell(1.0d)); }
                    if (i == 39) { table.Regions.Remove(-1); }
                    if (i == 40) { table.Regions.Remove(3); }

                    Assert.Fail();
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.Message);
                    Assert.IsTrue(e is TPCParameterTableException);
                    Assert.IsTrue(e.Message.Contains("out of bounds"));
                    Console.WriteLine("i value "+i+" OK.");
                }
            }
        }       

        /// <summary>
        /// Test case for comparing match of two tables
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_3_Equals()
        {

            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            ParameterTable table2 = new ParameterTable();
            table2.Parameters.Add("MyOwn","Second");

            table2.Regions.Add( 
                new RegionCell(new RegionCell.RegionStruct("Reg1")),
                new RESCell( 1.0d ),
                new RESCell( 5.0d ));
            table2.Regions.Add( 
                new RegionCell(new RegionCell.RegionStruct("Reg2")),
                new RESCell( 2.0d ),
                new RESCell( 6.5d ));
            table2.Regions.Add( 
                new RegionCell(new RegionCell.RegionStruct("Reg3")),
                new RESCell( 3.0d ),
                new RESCell( 7.0d ));

            Console.WriteLine(table.ToString());
            Console.WriteLine(table2.ToString());

            Assert.IsTrue( table.Equals( table2 ) );

            // Index operators testing
            table2.Regions[0,1] = new RESCell( 100.0d );
            Assert.IsFalse( table.Equals( table2 ) );
            table2.Regions.Set(0, 1, new RESCell(5.0d) );
            Assert.IsTrue(table.Equals(table2));

            table2.Regions[0,0] = new RESCell( 100.0d );
            table2[0, 0] = new RESCell(1.0d);
            Assert.IsTrue(table.Equals(table2));
        }

        /// <summary>
        /// Test case for changing region names
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_7_Changing_Region_Parameter_Names()
        {
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );


            ParameterTable table2 = new ParameterTable();

            table2.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table2.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table2.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Lohi")));

            table2.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 1.0d ), 
                new RESCell( 2.0d ),
                new RESCell( 3.0d )}
                );

            table2.Parameters.Add("Kukko", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );

            Assert.IsFalse( table.Equals(table2) );

            RegionCell head = (RegionCell)table.Regions.GetHeader( 2 );
            head.Region.Name = "Lohi";
            table.Regions.SetHeader(2, head);

            Assert.IsTrue( table.Parameters.GetName(1).Equals("Second") );
            table.Parameters.SetName(1, "Kukko");

            Console.WriteLine( table.ToString() );

            Assert.IsTrue( table.Equals(table2) );
        }


        /// <summary>
        /// Test case for printing contents to stdout
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_4_PrintToStdOut()
        {
            String filename = "P:\\data\\res\\testi.res";

            RESFile file = new RESFile();
            file.ReadFile(filename);

            file.Data.WriteToSTDOut();            
        }

#endregion

#region Finding and sorting

        /// <summary>
        /// Test case for sorting columns into parameter table
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test2_3_SortByParameter()
        {
            String filename = "P:\\data\\res\\testi.res";

            RESFile file = new RESFile();
            file.ReadFile(filename);
            file.Data.Parameters.Sort("vb", false);

            file.WriteFile("RES.res");

            Console.WriteLine( file.Data.ToString() );
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\resmatch.exe", "RES.res " + "res\\Params_Sorted.res") == 0);

        }



        /// <summary>
        /// Test case for finding parameters from parametertable
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test2_4_FindParameters()
        {
            Console.WriteLine("TEST: FINDING PARAMETERS");
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 2.0d ), 
                new RESCell( 1.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );
            table.Parameters.Add("See", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.6d ),
                new RESCell( 7.0d )}
                );
            table.Parameters.Add("Soo", new TableCell[] {
                new RESCell( 15.0d ), 
                new RESCell( 16.5d ),
                new RESCell( 17.0d )}
                );

            // There should be 3 match
            ParameterTable result = table.Parameters.Find("o");

            Assert.IsTrue( result.Parameters.Count == 3 );
            Assert.IsTrue(result.Parameters.GetName(0).Equals("MyOwn"));
            Assert.IsTrue(result.Parameters.GetName(1).Equals("Second"));
            Assert.IsTrue(result.Parameters.GetName(2).Equals("Soo"));
            Assert.IsTrue(result.Parameters[1][1].Value == 6.5d);

            Console.WriteLine( result.ToString() );
        }

        /// <summary>
        /// Test case for parameter enumerator
        /// </summary>
        [Test]
        //[Ignore("Disabled")]
        public void Test3_5_Enumerators()
        {
            Console.WriteLine("TESTING ENUMERATORS:");
            Console.WriteLine("Parameters:");
            ParameterTable table = new ParameterTable();

            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg1")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg2")));
            table.Regions.Add(new RegionCell(new RegionCell.RegionStruct("Reg3")));

            table.Parameters.Add("MyOwn", new TableCell[] {
                new RESCell( 2.0d ), 
                new RESCell( 1.0d ),
                new RESCell( 3.0d )}
                );

            table.Parameters.Add("Second", new TableCell[] {
                new RESCell( 5.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 7.0d )}
                );
            table.Parameters.Add("third", new TableCell[] {
                new RESCell( 4.0d ), 
                new RESCell( 6.5d ),
                new RESCell( 8.0d )}
                );

            int index = 0;
            foreach (TableCell[] cells in table.Parameters)
            {
                if (index == 0)
                {
                    Assert.IsTrue(cells[0].Value == 2.0d);
                    Assert.IsTrue(cells[2].Value == 3.0d);
                }
                else if (index == 1)
                {
                    Assert.IsTrue(cells[0].Value == 5.0d);
                    Assert.IsTrue(cells[2].Value == 7.0d);
                }
                else if (index == 2)
                {
                    Assert.IsTrue(cells[0].Value == 4.0d);
                    Assert.IsTrue(cells[2].Value == 8.0d);
                }
                else throw new Exception();

                Console.WriteLine("Index "+index+" OK");
                index++;
            }

            Console.WriteLine("Regions:");

            index = 0;
            foreach (TableCell[] cells in table.Regions)
            {
                if (index == 0)
                {
                    Assert.IsTrue(cells[0].Value == 2.0d);
                    Assert.IsTrue(cells[1].Value == 5.0d);
                }
                else if (index == 1)
                {
                    Assert.IsTrue(cells[0].Value == 1.0d);
                    Assert.IsTrue(cells[2].Value == 6.5d);
                }
                else if (index == 2)
                {
                    Assert.IsTrue(cells[0].Value == 3.0d);
                    Assert.IsTrue(cells[2].Value == 8.0d);
                }
                else throw new Exception();

                Console.WriteLine("Index " + index + " OK");
                index++;
            }

        }



        /// <summary>
        /// Test case for Sorting by another table
        /// </summary>
        [Test]
        [Ignore("Disabled")]
        public void Test3_6_SortByTable()
        {
        }
# endregion

    }
}