﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using NUnit.Framework;
using System;
using System.Collections.Generic;
using System.Text;
using TPClib.Curve;

namespace TPClib.Curve
{
    /// <summary>
    /// Testbench class for TPClib.Curve.TACTable class
    /// </summary>
    [TestFixture]
    public class NUnitTestbench_TACTable
    {
        /// <summary>
        /// Tests min and max activity 
        /// </summary>
        [Test]
        public void test6_1_and_6_2_find_MaxMin_value()
        {
            Console.WriteLine("TEST CASE 6.1: Finding the maximum value from TACTable.");

            // generates times 0-1, 1-2, 2-3....
            FrameTimeCell[] test_times = TestHelpers.generateTimes();
            TACTable table = new TACTable(test_times);

            TestHelpers.fillNumbers(table);

            // next we put the greatest value
            TableCell great = new TableCell(1001.0);
            table[2, 2] = great;

            // next we put the minimal value
            TableCell min = new TableCell(1.0);
            table[table.columns - 1, table.rows - 1] = min;

            Console.WriteLine(table.ToString());

            Assert.IsTrue(table.getMax() == 1001);
            Assert.IsTrue(table.GetMin() == 1);

            // testing of MinMax function
            double[] d = table.GetMinMax();

            Console.WriteLine("min=" + d[0]);
            Console.WriteLine("max=" + d[1]);

            Assert.IsTrue(d[0] == 1);
            Assert.IsTrue(d[1] == 1001);
        }

        /// <summary>
        /// 6.3 number of NA	Finds number of NaN (“.” In DFT file)
        /// </summary>
        [Test]
        public void test6_3_findNumberOf_NaNs()
        {
            FrameTimeCell[] test_times = TestHelpers.generateTimes();
            TACTable table = new TACTable(test_times);
            TestHelpers.fillNumbers(table);

            // There should be none NaNs
            Assert.IsTrue(table.getNrOfNA() == 0);
            Console.WriteLine("Nollalla onnistui");

            ((TableCell)table[5, 0]).Value = Double.NaN;
            ((TableCell)table[1, 2]).Value = Double.NaN;

            // There should be 2
            Assert.IsTrue(table.getNrOfNA() == 2);
            Console.WriteLine("Kahdella onnistui");

            // We add NaN also to time data
            FrameTimeCell ftc = table.GetTimeCell(2);
            ftc.end_time = Double.NaN;
            table.SetTimeCell(2, ftc);

            ((TableCell)table[0, 2]).Value = Double.NaN;

            // There should be 4
            Assert.IsTrue(table.getNrOfNA() == 4);
            Console.WriteLine("neljällä onnistui");


        }

        /// <summary>
        /// 6.4 replace NA:s	Replace NaN values (“.” in DFT file) with interpolated values
        /// </summary>
        [Test]
        [Ignore("Currently disabled")]
        public void test_replace_NAs()
        {
        }


        /// <summary>
        /// 7.1 null frame	Add time frame with time 0
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test7_1_TimeFrame_zero()
        {
            FrameTimeCell[] test_times = TestHelpers.generateTimes();
            RegionalTACTable table = new RegionalTACTable(test_times);
            TestHelpers.fillstaticNumbers(table);

            //table.SetTimeCell(2, new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END));
            try
            {
                table.AddRowAt(2, null, new TableCell[table.columns]);
            }
            catch (TPCTACTableException e)
            { 
                if ( e.Message.Contains("null") ) return;
            }
            Assert.Fail();

            // There should be 2 NaNs
            //Assert.IsTrue(table.getNrOfNA() == 2);

            //DFTRegionalCurveFile file = new DFTRegionalCurveFile();
            //file.Curves = table;
            //file.WriteFile("tactable\\test7_1.dft");

            // comparison
            //Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test7_1.dft tactable\\test7_1_original.dft") == 0);
        }

        /// <summary>
        /// 7.2 Del frame overlap	Delete frames that overlap each other.
        /// </summary>
        [Test]
        [Ignore("Currently disabled")]
        public void test7_2_delete_frame_overlap()
        {
            Console.WriteLine("TEST CASE 7.2: DELETING FRAME TIME OVERLAP.");

            FrameTimeCell[] test_times = TestHelpers.generateTimes();
            RegionalTACTable table = new RegionalTACTable(test_times);
            TestHelpers.fillNumbers(table);

            FrameTimeCell c = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
            c.start_time = 1.0;
            c.end_time = 1.6;
            table.SetTimeCell(2, c);

            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("tactable\\test7_2_overlapping.dft");
            file.Curves = table;

            System.IO.File.Copy("tactable\\test7_2_overlapping.dft", "tactable\\test7_2_original.dft", true);

            // we remove the time overlap rows
            file.Curves.DelFrameOverlap();

            // we write a dft file to disk without overlapping information
            file.WriteFile("tactable\\test7_2.dft");

            // we remove the overlapping information with dftrmovl
            Console.WriteLine(TestHelpers.getStdOut("P:\\bin\\windows\\dftrmovl.exe", "tactable\\test7_2_original.dft"));

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test7_2.dft tactable\\testi7_2_original.dft") == 0);

            Console.WriteLine(table.ToString());
        }

        /// <summary>
        /// 7.3	Calculate frame times
        /// </summary>
        [Test]
        [Ignore("Currently disabled")]
        public void test7_3_calculate_frame_times()
        {
            Console.WriteLine("TEST CASE 7.3: CALCULATING FRAME TIME LENGTHS:");

            Console.WriteLine("Start/End value:");
            String filename = "P:\\data\\dft\\Imadeus.dft";

            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(filename);
            
            file.Curves.UpdateFrameTimes(file.Curves.timetype);

            System.IO.File.Delete("tactable\\test7_3_framefile.dft");
            TestHelpers.getStdOut(@"P:\\bin\\windows\\dftframe.exe", filename + " tactable\\test7_3_framefile.dft");

            DFTCurveFile origConverted = DFTCurveFile.ReadFile("tactable\\test7_3_framefile.dft");
            origConverted.Curves.timetype = FrameTimeCell.FrameTimetype.START;
            
            // This framefile has minute values, so it must be changed
            origConverted.Curves.MultiplyFrames( 60.000d );
            // In this case also activities (frame lengths) must be multiplied
            origConverted.Curves.MultiplyActivities(60.000d);

            Console.WriteLine("Originally converted:\n"+origConverted.Curves.ToString());
            Console.WriteLine("Library converted:\n" + file.Curves.ToString());

            // Comparing:
            Assert.IsTrue(TACTable.Equals(origConverted.Curves, file.Curves));

            return; // not yet implemented
            /*
            Console.WriteLine("Mid value:");
            filename = "P:\\data\\dft\\test2.dft";

            file = new DFTRegionalCurveFile();
            file.ReadFile(filename);

            tab = file.Curves.CalcFrameTimes();

            System.IO.File.Delete("tactable\\test7_3_2_framefile.dft");
            TestHelpers.getStdOut(@"P:\\bin\\windows\\dftframe.exe", filename + " tactable\\test7_3_2_framefile.dft");

            origConverted = new DFTCurveFile();
            origConverted.Curves.timetype = FrameTimeCell.FrameTimetype.START;
            origConverted.ReadFile("tactable\\test7_3_2_framefile.dft");

            // This framefile has minute values, so it must be changed
            origConverted.Curves.MultiplyFrames(60.000d);
            // In this case also activities (frame lengths) must be multiplied
            origConverted.Curves.MultiplyActivities(60.000d);

            Console.WriteLine("Originally converted:\n" + origConverted.Curves.ToString());
            Console.WriteLine("Library converted:\n" + tab.ToString());

            // Comparing:
            Assert.IsTrue(TACTable.Equals(origConverted.Curves, tab));

            */
            //Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "-roughly tactable\\test6_7.dft tactable\\test6_7_original.dft") == 0);
        }

        /// <summary>
        /// 6.7 Sort	Sort by frames
        /// </summary>
        [Test]
        public void test6_7_Sort_table_by_frames()
        {
            String filename = "P:\\data\\dft\\tactable\\test6_7_wrongorder.dft";

            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(filename);
            
            file.Curves.SortRows(false);
            file.WriteFile("tactable\\test6_7.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test6_7.dft P:\\data\\dft\\tactable\\test6_7_original.dft") == 0);
        }


        /// <summary>
        /// 7.4	Change frame number
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test7_4_Change_frame_times()
        {
            String filename = "P:\\data\\dft\\tactable\\test7_4_original.dft";

            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile(filename);
           
            FrameTimeCell ftc = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
            ftc.start_time=1.0d;
            ftc.end_time=1.5d;
            file.Curves.SetTimeCell(1, ftc);

            file.WriteFile("tactable\\test7_4.dft");
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test7_4.dft P:\\data\\dft\\tactable\\test7_4_correct.dft") == 0);
        }

        /// <summary>
        /// 8.1	Multiply frames
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test8_1_Multiply_frames()
        {
            DFTCurveFile file = DFTCurveFile.ReadFile("P:\\data\\dft\\tactable\\test8_1_original.dft");
            file.Curves.MultiplyFrames(2.0d);
            file.WriteFile("tactable\\test8_1.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test8_1.dft P:\\data\\dft\\tactable\\test8_1_right.dft") == 0);
        }

        /// <summary>
        /// 8.2	Multiply frames
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test8_2_Multiply_activities()
        {
            DFTCurveFile file = DFTCurveFile.ReadFile("P:\\data\\dft\\tactable\\test8_2_original.dft");
            file.Curves.MultiplyActivities(2.0d);
            file.WriteFile("tactable\\test8_2.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test8_2.dft P:\\data\\dft\\tactable\\test8_2_right.dft") == 0);
        }

        /// <summary>
        /// 9.1	Catenation
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test9_1_Table_Catenation()
        {
            DFTCurveFile file = DFTCurveFile.ReadFile("P:\\data\\dft\\tactable\\test9_1_original1.dft");
            TACTable tab = file.Curves;
            file = DFTCurveFile.ReadFile("P:\\data\\dft\\tactable\\test9_1_original2.dft");
            TACTable tab2 = file.Curves;

            tab.Catenate( tab2 );
            file.Curves = tab;

            file.WriteFile("tactable\\test9_1.dft");
            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test9_1.dft P:\\data\\dft\\tactable\\test9_1_right.dft") == 0);
        }

        /// <summary>
        /// 9.2	Catenation
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test9_2_Table_Clone()
        {
            Console.WriteLine("Clone tests:");
            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("p:\\data\\dft\\test3.dft");
            TACTable tab = (TACTable)file.Curves;

            TACTable tab2 = tab.Clone();

            DFTCurveFile file2 = new DFTCurveFile();
            file2.Curves = tab2;
            file2.WriteFile("tactable\\test9_2.dft");

            Assert.IsTrue(TestHelpers.getStdOut(@"P:\\bin\\windows\\dftmatch.exe", "tactable\\test9_2.dft p:\\data\\dft\\test3.dft") == 0);

            // Equals part
            Console.WriteLine("Equals tests:");
            //Assert.IsTrue(TACTable.Equals(tab, tab2));
        }

        /// <summary>
        /// 9.3	Equals
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test9_3_Table_Equals()
        {
            Console.WriteLine("Equals tests:");
            DFTRegionalCurveFile file = (DFTRegionalCurveFile)DFTCurveFile.ReadFile("p:\\data\\dft\\test3.dft");
            TACTable tab = (TACTable)file.Curves;

            TACTable tab2 = tab.Clone();

            // This should be true
            Assert.IsTrue(TACTable.Equals(tab, tab2));

            // now we change one value and now result should be false
            TableCell temp = new TableCell(tab[0, 0]);
            tab[0, 0] = new TableCell(0.0d);
            Assert.IsFalse(TACTable.Equals(tab, tab2));
            tab[0, 0] = temp;

            // lets also test unit.
            Assert.IsTrue(TACTable.Equals(tab, tab2));
            tab.unit = Data_unit.NCI_per_ML;
            Assert.IsFalse(TACTable.Equals(tab, tab2));
        }

        /// <summary>
        /// 9.5	Comparison
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test9_5_FrameTimeCell_Comparison()
        {
            FrameTimeCell c1 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START); c1.start_time = 1.0d;
            FrameTimeCell c2 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START); c2.start_time = 3.0d;
            FrameTimeCell c3 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START); c3.start_time = 1.0d;
            FrameTimeCell c4 = null;

            Console.WriteLine("FrameTimeCell comparison START TIME:");
            Console.WriteLine("FrameTimeCell comparison against null");
            Assert.IsTrue(c4 == null);
            Assert.IsFalse(c1 == null);

            Console.WriteLine("FrameTimeCell comparison: ==");
            Assert.IsTrue(c1 == c3);
            Assert.IsFalse(c1 == c2);

            Console.WriteLine("FrameTimeCell comparison: !=");
            Assert.IsFalse(c1 != c3);
            Assert.IsTrue(c1 != c2);

            Console.WriteLine("FrameTimeCell comparison: <");
            Assert.IsFalse(c1 < c3);
            Assert.IsTrue(c1 < c2);

            Console.WriteLine("FrameTimeCell comparison: >");
            Assert.IsFalse(c1 > c3);
            Assert.IsTrue(c2 > c1);

            Console.WriteLine("FrameTimeCell comparison: >=");
            Assert.IsTrue(c1 >= c3);
            Assert.IsTrue(c2 >= c1);
            Assert.IsFalse(c1 >= c2);

            Console.WriteLine("FrameTimeCell comparison: <=");
            Assert.IsTrue(c1 <= c3);
            Assert.IsFalse(c2 <= c1);
            Assert.IsTrue(c1 <= c2);


            c1 = new FrameTimeCell(FrameTimeCell.FrameTimetype.END); c1.end_time = 1.0d;
            c2 = new FrameTimeCell(FrameTimeCell.FrameTimetype.END); c2.end_time = 3.0d;
            c3 = new FrameTimeCell(FrameTimeCell.FrameTimetype.END); c3.end_time = 1.0d;
            c4 = null;

            Console.WriteLine("\nFrameTimeCell comparison END TIME:");
            Console.WriteLine("FrameTimeCell comparison against null");
            Assert.IsTrue(c4 == null);
            Assert.IsFalse(c1 == null);

            Console.WriteLine("FrameTimeCell comparison: ==");
            Assert.IsTrue(c1 == c3);
            Assert.IsFalse(c1 == c2);

            Console.WriteLine("FrameTimeCell comparison: !=");
            Assert.IsFalse(c1 != c3);
            Assert.IsTrue(c1 != c2);

            Console.WriteLine("FrameTimeCell comparison: <");
            Assert.IsFalse(c1 < c3);
            Assert.IsTrue(c1 < c2);

            Console.WriteLine("FrameTimeCell comparison: >");
            Assert.IsFalse(c1 > c3);
            Assert.IsTrue(c2 > c1);

            Console.WriteLine("FrameTimeCell comparison: >=");
            Assert.IsTrue(c1 >= c3);
            Assert.IsTrue(c2 >= c1);
            Assert.IsFalse(c1 >= c2);

            Console.WriteLine("FrameTimeCell comparison: <=");
            Assert.IsTrue(c1 <= c3);
            Assert.IsFalse(c2 <= c1);
            Assert.IsTrue(c1 <= c2);

            c1 = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID); c1.mid_time = 1.0d;
            c2 = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID); c2.mid_time = 3.0d;
            c3 = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID); c3.mid_time = 1.0d;
            c4 = null;

            Console.WriteLine("\nFrameTimeCell comparison MID TIME:");
            Console.WriteLine("FrameTimeCell comparison against null");
            Assert.IsTrue(c4 == null);
            Assert.IsFalse(c1 == null);

            Console.WriteLine("FrameTimeCell comparison: ==");
            Assert.IsTrue(c1 == c3);
            Assert.IsFalse(c1 == c2);

            Console.WriteLine("FrameTimeCell comparison: !=");
            Assert.IsFalse(c1 != c3);
            Assert.IsTrue(c1 != c2);

            Console.WriteLine("FrameTimeCell comparison: <");
            Assert.IsFalse(c1 < c3);
            Assert.IsTrue(c1 < c2);

            Console.WriteLine("FrameTimeCell comparison: >");
            Assert.IsFalse(c1 > c3);
            Assert.IsTrue(c2 > c1);

            Console.WriteLine("FrameTimeCell comparison: >=");
            Assert.IsTrue(c1 >= c3);
            Assert.IsTrue(c2 >= c1);
            Assert.IsFalse(c1 >= c2);

            Console.WriteLine("FrameTimeCell comparison: <=");
            Assert.IsTrue(c1 <= c3);
            Assert.IsFalse(c2 <= c1);
            Assert.IsTrue(c1 <= c2);

            c1 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END); c1.start_time = 1.0d; c1.end_time = 2.0d;
            c2 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END); c2.start_time = 3.0d; c2.end_time = 4.0d;
            c3 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END); c3.start_time = 1.0d; c3.end_time = 2.0d;
            c4 = null;
            FrameTimeCell c5 = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END); c5.start_time = 1.0d; c5.end_time = 2.1d;

            Console.WriteLine("\nFrameTimeCell comparison START/END TIME:");
            Console.WriteLine("FrameTimeCell comparison against null");
            Assert.IsTrue(c4 == null);
            Assert.IsFalse(c1 == null);

            Console.WriteLine("FrameTimeCell comparison: ==");
            Assert.IsTrue(c1 == c3);
            Assert.IsFalse(c1 == c2);
            Assert.IsFalse(c1 == c5);

            Console.WriteLine("FrameTimeCell comparison: !=");
            Assert.IsFalse(c1 != c3);
            Assert.IsTrue(c1 != c2);

            Console.WriteLine("FrameTimeCell comparison: <");
            Assert.IsFalse(c1 < c3);
            Assert.IsTrue(c1 < c2);
            Assert.IsTrue(c1 < c5);

            Console.WriteLine("FrameTimeCell comparison: >");
            Assert.IsFalse(c1 > c3);
            Assert.IsTrue(c2 > c1);
            Assert.IsFalse(c1 > c5);

            Console.WriteLine("FrameTimeCell comparison: >=");
            Assert.IsTrue(c1 >= c3);
            Assert.IsTrue(c2 >= c1);
            Assert.IsFalse(c1 >= c2);
            Assert.IsFalse(c1 >= c5);

            Console.WriteLine("FrameTimeCell comparison: <=");
            Assert.IsTrue(c1 <= c3);
            Assert.IsFalse(c2 <= c1);
            Assert.IsTrue(c1 <= c2);
            Assert.IsTrue(c1 <= c5);
        }

        /// <summary>
        /// 9.4	Equals
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test9_4_TableCell_Comparison()
        {
            TableCell c1 = new TableCell(1.0d);
            TableCell c2 = new TableCell(3.0d);
            TableCell c3 = new TableCell(1.0d);
            TableCell c4 = null;

            Console.WriteLine("TableCell comparison against null");
            Assert.IsTrue( c4 == null );
            Assert.IsFalse( c1 == null );

            Console.WriteLine("TableCell comparison: ==");
            Assert.IsTrue(c1 == c3);
            Assert.IsFalse( c1 == c2 );
            Assert.IsTrue(c1 == 1.0d);

            Console.WriteLine("TableCell comparison: !=");
            Assert.IsFalse(c1 != c3);
            Assert.IsTrue(c1 != c2);
            Assert.IsTrue(c1 != 2.0d); // comparison against double

            Console.WriteLine("TableCell comparison: <");
            Assert.IsFalse(c1 < c3);
            Assert.IsTrue(c1 < c2);
            Assert.IsTrue(c1 < 2.0d); // comparison against double

            Console.WriteLine("TableCell comparison: >");
            Assert.IsFalse(c1 > c3);
            Assert.IsTrue(c2 > c1);
            Assert.IsTrue(c1 > 0.5d); // comparison against double

            Console.WriteLine("TableCell comparison: >=");
            Assert.IsTrue(c1 >= c3);
            Assert.IsTrue(c2 >= c1);
            Assert.IsFalse(c1 >= c2);
            Assert.IsTrue(c1 >= 0.5d); // comparison against double

            Console.WriteLine("TableCell comparison: <=");
            Assert.IsTrue(c1 <= c3);
            Assert.IsFalse(c2 <= c1);
            Assert.IsTrue(c1 <= c2); 
            Assert.IsFalse(c1 <= 0.5d); // comparison against double
        }

        /// <summary>
        /// 10.1	Exceptions
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test10_1_Index_out_of_Bounds()
        {
            Console.WriteLine("Testing Index out of bound exceptions...");
            TACTable tab = new TACTable( TestHelpers.generateTimes() );
            TestHelpers.fillstaticNumbers(tab);
            bool pass = false;

            pass = false; Console.WriteLine("Testing getrow(-1)");
            try {   tab.GetRow(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass=true; }
            Assert.IsTrue( pass );

            pass = false; Console.WriteLine("Testing getrow(rows)");
            try { tab.GetRow(tab.rows); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing getColumn(-1)");
            try { tab.GetColumn(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing getColumn(columns)");
            try { tab.GetColumn(tab.columns); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing getTimecell(-1)");
            try { tab.GetTimeCell(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing getTimecell(rows)");
            try { tab.GetTimeCell(tab.rows); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing setTimecell(-1)");
            try { tab.SetTimeCell(-1, new FrameTimeCell(FrameTimeCell.FrameTimetype.START) ); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing setTimecell(rows)");
            try { tab.SetTimeCell(tab.rows, new FrameTimeCell(FrameTimeCell.FrameTimetype.START)); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing TACTable[-1, int]:");
            try { Object o = tab[-1,0]; }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing TACTable[columns, int]:");
            try { Object o = tab[tab.columns, 0]; }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing TACTable[int, -1]:");
            try { Object o = tab[0, -1]; }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing TACTable[int, rows]:");
            try { Object o = tab[0, tab.rows]; }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRowAt(-1):");
            try { tab.AddRowAt(-1, new FrameTimeCell(FrameTimeCell.FrameTimetype.START),new TableCell[1]); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRowAt(rows+1):");
            try { tab.AddRowAt(tab.rows+1, new FrameTimeCell(FrameTimeCell.FrameTimetype.START), new TableCell[1]); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing DeleteColumn(-1):");
            try { tab.DeleteColumn(-1); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing DeleteColumn(columns):");
            try { tab.DeleteColumn(tab.columns); }
            catch (Exception e)
            { if (e.ToString().Contains("Index out of bounds")) pass = true; }
            Assert.IsTrue(pass);
        }

        /// <summary>
        /// 10.2	Exceptions
        /// </summary>
        [Test]
        //[Ignore("Currently disabled")]
        public void test10_2_null_tests()
        {
            Console.WriteLine("TESTING NULL EXCEPTIONS...\n");
            TACTable tab = new TACTable(TestHelpers.generateTimes());
            TestHelpers.fillstaticNumbers(tab);
            bool pass = false;

            pass = false; Console.WriteLine("Testing AddColumn(null)");
            try { tab.AddColumn( (TableCell[])null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);


            pass = false; Console.WriteLine("Testing AddColumn(time,null)");
            try { tab.AddColumn( new FrameTimeCell[1], (TableCell[])null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddColumn(null,data)");
            try { tab.AddColumn( (FrameTimeCell[])(null), new TableCell[1]); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRow(null) - float ");
            try { tab.AddRow( (float[])null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRow(null) - double");
            try { tab.AddRow((double[])null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRow(time, null) - TableCell");
            try { tab.AddRow( new FrameTimeCell(FrameTimeCell.FrameTimetype.START) ,(TableCell[])null); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRow(null, data) - TableCell");
            try { tab.AddRow( null, new TableCell[1] ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRowAt(1, null, data) - TableCell");
            try { tab.AddRowAt(1, null, new TableCell[1]); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing AddRowAt(1, time, null) - TableCell");
            try { tab.AddRowAt(1, new FrameTimeCell(FrameTimeCell.FrameTimetype.START), (TableCell[])null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("Testing Catenate( null ).");
            try { tab.Catenate( (TACTable)null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("ReplaceColumn(1,null).");
            try { tab.ReplaceColumn(1, (TableCell[])null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

            pass = false; Console.WriteLine("UnitConversion(null).");
            try { tab.UnitConversion( (UnitConverter)null ); }
            catch (Exception e)
            { if ((e is TPCTACTableException) && e.Message.ToString().Contains("null")) { pass = true; Console.WriteLine(e.Message); } }
            Assert.IsTrue(pass);

        }

    }
}
