﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// Result data file
    /// </summary>
    /// 
    /// RES file header is more flexible than FIT file header.
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class RESFile : TPCFile
    {
        /// <summary>
        /// All the comment lines at the end of the file
        /// </summary>
        public List<String> Comments;

        /// <summary>
        /// RES-files Header information 
        /// </summary>
        public struct RHeader
        {
            /// <summary>
            /// Name of program
            /// </summary>
            public String ProgramName;
            /// <summary>
            /// Version of program
            /// </summary>
            public String ProgramVersion;
            /// <summary>
            /// Files copyright information
            /// </summary>
            public String CopyrightInformation;
            /// <summary>
            /// Date of file
            /// </summary>
            public DateTime Date;
            /// <summary>
            /// Study information
            /// </summary>
            public String Study;
            /// <summary>
            /// Data file and path
            /// </summary>
            public String DataFile;
            /// <summary>
            /// Plasma file and path
            /// </summary>
            public String PlasmaFile;
            /// <summary>
            /// Blood file and path
            /// </summary>
            public String BloodFile;
            /// <summary>
            /// ROI file and path
            /// </summary>
            public String ROIFile;
            /// <summary>
            /// Reference file and path
            /// </summary>
            public String ReferenceFile;
            /// <summary>
            /// Reference region
            /// </summary>
            public String ReferenceRegion;
            /// <summary>
            /// Density of tissue
            /// </summary>
            public String TissueDensity;

            /// <summary>
            /// Beta value
            /// </summary>
            public String Beta;
            /// <summary>
            /// Vb Blood plasma ratio (percentage)???
            /// </summary>
            public String Vb;

            /// <summary>
            /// Range of data
            /// </summary>
            public String DataRange;
            /// <summary>
            /// Data number
            /// </summary>
            public Double DataNr;
            /// <summary>
            /// Lumped constant
            /// </summary>
            public String LumpedConstant;
            /// <summary>
            /// Fit time
            /// </summary>
            public String FitTime;
            /// <summary>
            /// Fit method
            /// </summary>
            public String FitMethod;
            /// <summary>
            /// Information indicating whether the data was weighted or not 
            /// </summary>
            public bool Weighted;

            /// <summary>
            /// Concentration
            /// </summary>
            public String Concentration;
        }
        
        /// <summary>
        /// All RES-file header information. 
        /// </summary>
        public RHeader Header;

        /// <summary>
        /// Gets and sets the ParameterTable value
        /// </summary>
        public ParameterTable Data
        {
            get
            {
                return table;
            }
            set
            {
                table = value;
            }
        }

        /// <summary>
        /// Result data in file.
        /// </summary>
        private ParameterTable table;

        /// <summary>
        /// Default constructor
        /// </summary>
        public RESFile()
        {
            Init();
        }

        /// <summary>
        /// Constructs RESFile class with filename
        /// </summary>
        /// <param name="filename">Name of file.</param>
        public RESFile(String filename)
        {
            Init();
            base.filename = filename;
        }

        /// <summary>
        /// Reads RES files. Filename must be given before calling this method.
        /// </summary>
        public override void ReadFile()
        {
            if (filename == null) throw new TPCRESFileException("You have not given a filename.");
            ReadFile(filename);
        }

        /// <summary>
        /// Reads a RESfile with given filename.
        /// </summary>
        /// <param name="filename">Name of RES file to read.</param>
        public void ReadFile(String filename)
        {
            // we open the file and put its contents to String which is
            // read with StringReader. We can close the file after String is in memory
            FileStream filestream = new FileStream(filename, FileMode.Open, FileAccess.Read);
            StreamReader reader = new StreamReader(filestream, new ASCIIEncoding());

            string text = reader.ReadToEnd();
            //StringReader strReader = new StringReader(text);

            filestream.Close();
            String HeaderPart = "";
            String DataPart = "";

            try
            {
                HeaderPart = text.Substring(0, text.LastIndexOf("\nRegion"));
                DataPart = text.Substring(text.LastIndexOf("\nRegion"));
            }
            catch (Exception)
            {
                throw new TPCRESFileException("ReadFile: The file " + filename + " was not in correct RES format.");
            }

            Console.WriteLine(HeaderPart);

            if ( !(HeaderPart.ToLower().Contains("date:") && HeaderPart.ToLower().Contains("weight")) )
                throw new TPCRESFileException("ReadFile: The file " + filename + " was not in correct RES format.");

            // If we have got this far, we can assume that the file is valid file and we empty
            // all the old values. 
            Init();

            // Reads all the RES header information
            ReadHeader(HeaderPart);

            // Reads all the RES data information
            ReadData(DataPart);

            string[] u = new string[0];
            
            foreach (string line in Comments)
            {
                if (line.StartsWith("# Units :"))
                {
                    string l = line.Trim();
                    u = l.Split(new char[] {' '}, StringSplitOptions.RemoveEmptyEntries);
                }
            }
            List<string> units = new List<string>();
            for (int i = 3; i < u.Length; i++)
                units.Add(u[i]);

            this.Data.Parameter_Units = units.ToArray();
            
        }

        /// <summary>
        /// Writes data to RES file. Filename must be given before calling this method.
        /// </summary>
        public override void WriteFile()
        {
            if (filename == null) throw new TPCRESFileException("You have not given a filename.");
            WriteFile(filename);
        }

        /// <summary>
        /// Writes data to RES file with given filename. Old file will be overwritten.
        /// </summary>
        /// <param name="filename">Name of file.</param>
        public void WriteFile(String filename)
        {
            //this.Data.Find("Vb");
            //Console.WriteLine( WriteHeader() + WriteData() );
            // Writing the text to file
            
            FileStream filestream = new FileStream(filename, FileMode.Create, FileAccess.Write);
            StreamWriter fileWriter = new StreamWriter(filestream, new ASCIIEncoding());

            fileWriter.AutoFlush = true;

            fileWriter.Write( WriteHeader() + WriteData() );

            filestream.Close();
        }

        /// <summary>
        /// Returns the contents of RESFile as string.
        /// </summary>
        /// <returns>Contents of the RES file as string.</returns>
        public override String ToString()
        {
            String str = "RESfile contains:\n"+table.Regions.Count+" regions and "+table.Parameters.Count+ " parameters.\n";

            return table.ToString();
        }

        #region private_members

        /// <summary>
        /// Inits all the resfiles fields.
        /// </summary>
        private void Init()
        {
            this.Comments = new List<string>();
            this.filename = null;

            // Initial header information
            this.Header.Beta = ".";
            this.Header.BloodFile = ".";
            this.Header.CopyrightInformation = ".";
            this.Header.DataFile = ".";
            this.Header.DataRange = ".";
            this.Header.Date = new DateTime();
            this.Header.FitTime = ".";
            this.Header.LumpedConstant = ".";
            this.Header.PlasmaFile = ".";
            this.Header.ProgramName = ".";
            this.Header.ProgramVersion = ".";
            this.Header.ReferenceFile = ".";
            this.Header.ReferenceRegion = ".";
            this.Header.ROIFile = ".";
            this.Header.Study = ".";
            this.Header.TissueDensity = ".";
            this.Header.Vb = ".";
            this.Header.DataNr = Double.NaN;
            this.Header.FitMethod = ".";
            this.Header.Weighted = false;
            this.Header.Concentration = ".";
        }

        /// <summary>
        /// Reads header information from RESfile.
        /// </summary>
        private void ReadHeader( String text )
        {
            StringReader sr = new StringReader(text);

            String[] words = readWordsFromLine( ref sr );

            // Line1: machineName, version, other information
            this.Header.ProgramName = words[0];
            this.Header.ProgramVersion = words[1];

            // Copyright information can contain spaces, so it must be handled differently
            if( words.Length >= 2 )
            {
                this.Header.CopyrightInformation = words[2];
                for (int i = 3; i < words.Length; i++) this.Header.CopyrightInformation += (" " + words[i]);
            }

            // Getting the Date
            String date_str = getString("Date:", ref text);
            try
            {
                Header.Date = System.Convert.ToDateTime(date_str);
            }
            catch (Exception)
            {
                throw new TPCRESFileException("ReadFile(): Cannot convert datetime. Datetime was in invalid format.");
            }

            Header.DataFile = getString("Data file:", ref text);
            Header.BloodFile = getString("Blood file:", ref text);
            Header.PlasmaFile = getString("Plasma file:", ref text);
            Header.ReferenceFile = getString("Reference file:", ref text);
            Header.ROIFile = getString("ROI file:", ref text);

            Header.ReferenceRegion = getString("Reference region:", ref text);
            Header.DataRange = getString("Data range:", ref text);
            Header.Study = getString("Study:", ref text);
            Header.TissueDensity = getString("Tissue density:", ref text);
            Header.Vb = getString("Vb:", ref text );
            Header.Beta = getString("Beta:", ref text);
            Header.FitTime = getString("Fit time:", ref text);
            Header.LumpedConstant = getString("Lumped constant:", ref text);
            Header.FitMethod = getString("Fit method:", ref text);

            String data_nr = getString("Data nr:", ref text);
            if (data_nr.Equals(".")) Header.DataNr = Double.NaN;
            else Header.DataNr = StrToDouble(data_nr);

            // Weighting information can be told two ways
            if (text.Contains("Data was not weighted")) Header.Weighted = false;
            else if (text.Contains("Data was weighted")) Header.Weighted = true;
            else
            {
                String weight_str = getString("Weighting:", ref text);
                if (weight_str.Equals("."))
                {
                    // There is no weighting information
                    Header.Weighted = false;
                }
                else
                {
                    // There is line Weighting: Yes
                    if (weight_str.Equals("Yes")) Header.Weighted = true;
                    else Header.Weighted = false;
                }
            }
        }

        /// <summary>
        /// Data reading states
        /// </summary>
        private enum ReadState
        {
            RowFinished,
            Name,
            StandardDeviation,
            CL_lower,
            CL_upper,
            Identify,
            END
        };

        /// <summary>
        /// Reads RESfiles data part
        /// </summary>
        /// <param name="text">Text containing all the data in the RES file.</param>
        private void ReadData(String text)
        {
            StringReader reader = new StringReader(text);

            String[] words = readWordsFromLine( ref reader );

            // Next we add parameters to parametertable
            table = new ParameterTable();
            for( int i=1; i<words.Length; i++ )
            {
                table.Parameters.Add(words[i]);
            }

            words = readWordsFromLine(ref reader);
            if (words == null || words.Length <= 3) return;

            // header and cells for one row
            RegionCell resHeader = new RegionCell();
            RESCell[] cells = new RESCell[words.Length - 3];

            ReadState state = new ReadState();
            state = ReadState.Name;

            // These bools are used for providing ability to read rows in any order
            bool SD_given = false;
            bool CL_up_given = false;
            bool CL_low_given = false;

            do
            {
                switch (state)
                {
                    // In this state we read the words as region names and parameter values.
                    case ReadState.Name:
                        // Row1: Region, values
                        resHeader.Region.Name = words[0];
                        resHeader.Region.SecondaryName = words[1];
                        resHeader.Region.Plane = words[2];

                        for (int i = 3; i < words.Length; i++)
                        {
                            cells[i - 3] = new RESCell(StrToDouble(words[i]));
                        }
                        state = ReadState.Identify;

                        words = readWordsFromLine(ref reader);
                        break;

                    // This state looks to the next words and sets the state according to it
                    case ReadState.Identify:
                        if (words == null) { state = ReadState.RowFinished; break; }
                        if (SD_given && CL_up_given && CL_low_given) { state = ReadState.RowFinished; break; }
                        if (words[0].Equals("SD") && !SD_given) { state = ReadState.StandardDeviation; break; }
                        if (words[0].Equals("CL") && words[2].ToLowerInvariant().Equals("upper") && !CL_up_given) 
                            { state = ReadState.CL_upper; break; }
                        if (words[0].Equals("CL") && words[2].ToLowerInvariant().Equals("lower") && !CL_low_given) 
                            { state = ReadState.CL_lower; break; }

                        state = ReadState.RowFinished;
                        break;

                    // In this state we try to read standard deviation information. If the word
                    // "SD" can be found, then the standard deviation information exists in the
                    // file. If not, the state is changed to the RowFinished and the words will
                    // be considered as nameinformation of next row.
                    case ReadState.StandardDeviation:
                        // Row1: Region, values
                        SD_given = true;
                        resHeader.SD = true;

                        for (int i = 3; i < words.Length; i++)
                        {
                            cells[i - 3].SD = StrToDouble(words[i]);
                        }

                        state = ReadState.Identify;
                        words = readWordsFromLine(ref reader);
                        
                        //else state = ReadState.RowFinished;
                        break;

                    // In this state we try to read lower confidence limit row. If the string
                    // "CL" cannot be found, the row we are reading must be the name row of another
                    // line, which means that CL values are missing from file. State is changed to
                    // RowFinished in that case and reading of new row is started
                    case ReadState.CL_lower:
                        // Row1: Region, values
                        // last char (%) must be removed before string2double conversion
                        CL_low_given = true;
                        resHeader.CL_lower = StrToDouble(words[1].Substring(0, words[1].Length-1));

                        for (int i = 3; i < words.Length; i++)
                        {
                            cells[i - 3].CL_lower = StrToDouble(words[i]);
                        }

                        state = ReadState.Identify;
                        words = readWordsFromLine(ref reader);
                        break;

                    // In this state we try to read upper confidence limit row. If the string
                    // "CL" cannot be found, the row we are reading must be a row of another line and state 
                    // is changed to Rowfinished.
                    case ReadState.CL_upper:
                        // Row1: Region, values
                        // last char (%) must be removed before string2double conversion
                        CL_up_given = true;
                        resHeader.CL_upper = StrToDouble(words[1].Substring(0, words[1].Length-1));

                        for (int i = 3; i < words.Length; i++)
                        {
                            cells[i - 3].CL_upper = StrToDouble(words[i]);
                        }

                        state = ReadState.Identify;
                        words = readWordsFromLine(ref reader);
                        break;

                    // This state means that one row has been read to the end and all the 
                    // row data will put into the parametertable.
                    case ReadState.RowFinished:
                        // all states are set to false for the next line
                        SD_given = false;
                        CL_low_given = false;
                        CL_up_given = false;
                        /*
                        Object[] row = new Object[cells.Length + 1];
                        row[0] = resHeader;
                        for (int i = 0; i < cells.Length; i++)
                        {
                            row[i + 1] = cells[i];
                        }
                        table.AddRow(row);*/

                        table.Regions.Add(resHeader, cells );

                        // next we allocate memory for new row
                        resHeader = new RegionCell();
                        cells = new RESCell[cells.Length];

                        // If file has ended, state is set to end. Else new row will be read in
                        // name state
                        if (words == null) state = ReadState.END;
                        else state = ReadState.Name;
                        break;
                }

                // If the words list is null, it means that file has been read to the end. In this
                // case we must store the row to parametertable in the RowFinished state or else
                // the last row would be missing.
                /*
                if (words == null && state != ReadState.END)
                {
                    state = ReadState.RowFinished;
                }*/
            }
            while (state != ReadState.END);
        }

        /// <summary>
        /// Converts strings to double in scientific mode with dot
        /// </summary>
        /// <param name="str"></param>
        /// <returns>a double which is converted from string</returns>
        protected Double StrToDouble(String str)
        {
            if (str.Equals(".")) return Double.NaN;
            else return Convert.ToDouble(str, new System.Globalization.CultureInfo("en-GB"));
        }

        /// <summary>
        /// Writes header information to string such as it will be in the RES file
        /// </summary>
        /// <returns>String containing the header information.</returns>
        private String WriteHeader()
        {
            StringBuilder text = new StringBuilder();
            text.Append( FixStr( Header.ProgramName + " " + Header.ProgramVersion, 18 ) + Header.CopyrightInformation + "\r\n" );
            text.Append( FixStr( "Date:", 18 ) + Header.Date.ToString() + "\r\n" );

            WriteHeaderLine(ref text, "Data file:", Header.DataFile);
            WriteHeaderLine(ref text, "Blood file:", Header.BloodFile);
            WriteHeaderLine(ref text, "Plasma file:", Header.PlasmaFile);
            WriteHeaderLine(ref text, "Reference file:", Header.ReferenceFile);
            WriteHeaderLine(ref text, "ROI file:", Header.ROIFile);
            WriteHeaderLine(ref text, "Study:", Header.Study);
            WriteHeaderLine(ref text, "Vb:", Header.Vb);
            WriteHeaderLine(ref text, "Reference region:", Header.ReferenceRegion);
            
            WriteHeaderLine(ref text, "Fit time:", Header.FitTime);
            WriteHeaderLine(ref text, "Data range:", Header.DataRange);
            WriteHeaderLine(ref text, "Beta:", Header.Beta);
            
            if( !Double.IsNaN(Header.DataNr) ) text.Append( FixStr( "Data nr:",18 ) + ((int)Header.DataNr).ToString() + "\r\n" );
            WriteHeaderLine(ref text, "Fit method:", Header.FitMethod);
            WriteHeaderLine(ref text, "Tissue density:", Header.TissueDensity);
            WriteHeaderLine(ref text, "Lumped constant:", Header.LumpedConstant);
            WriteHeaderLine(ref text, "Concentration:", Header.Concentration);
            if (Header.Weighted) text.Append("Data was weighted.\r\n");
            else text.Append("Data was not weighted.\r\n\r\n");

            return text.ToString();
        }

        /// <summary>
        /// Writes all the data inside the parametertable to string as it would be in file
        /// </summary>
        /// <returns></returns>
        private String WriteData()
        {
            StringBuilder text = new StringBuilder();

            // We write the first columns name that should be "Region"
            text.Append(FixStr(( "Region"), 22) );

            // Next we write names of parameter columns
            for (int i = 0; i < table.Parameters.Count; i++)
            {
                text.Append( FixStr(table.Parameters.GetName(i),17) );
            }
            text.Append("\r\n");

            // If there is unit information, it is placed next
            if (table.Parameter_Units != null)
            {
                text.Append(FixStr("# Units :", 22));
                foreach (String s in table.Parameter_Units) text.Append(FixStr(s, 17));
                text.Append("\r\n");
            }

            // Now we must add all the data
            for (int i = 0; i < table.Regions.Count; i++)
            {
                //System.Data.DataRow row = table.GetRow(i);

                RegionCell header = table.Regions.GetHeader(i); // (row[0] as RegionCell);
                TableCell[] rescells = table.Regions.GetParameters( i );
                    //new RESCell[ table.Parameters.Count ];


                text.Append( FixStr(header.Region.Name + " " + header.Region.SecondaryName + " " + header.Region.Plane, 24) );

                
                for (int param_i = 0; param_i < table.Parameters.Count; param_i++)
                {
                    //rescells[param_i] = table.Reg // (row[param_i+1] as RESCell);
                    text.Append( FixStr( DoubleToStr(((RESCell)rescells[param_i]).Value) , 17 ) );
                }
                text.Append("\r\n");

                // If there is standard deviation values present, it will be written into file
                if (header.SD)
                {
                    text.Append( FixStr( "  SD    .     .", 24) );
                    for (int param_i = 0; param_i < table.Parameters.Count; param_i++)
                    {
                        text.Append(FixStr(DoubleToStr(((RESCell)rescells[param_i]).SD), 17));
                    }
                    text.Append("\r\n");
                }

                // If there is lower confidence limit values present, it will be written into file
                if ( !Double.IsNaN( header.CL_lower ) )
                {
                    text.Append(FixStr("  CL    "+header.CL_lower+"%   lower" ,24));
                    for (int param_i = 0; param_i < table.Parameters.Count; param_i++)
                    {
                        text.Append(FixStr(DoubleToStr(((RESCell)rescells[param_i]).CL_lower), 17));
                    }
                    text.Append("\r\n");
                }

                // If there is upper confidence limit values present, it will be written into file
                if (!Double.IsNaN(header.CL_upper))
                {
                    text.Append(FixStr("  CL    " + header.CL_upper + "%   upper", 24));
                    for (int param_i = 0; param_i < table.Parameters.Count; param_i++)
                    {
                        text.Append(FixStr(DoubleToStr(((RESCell)rescells[param_i]).CL_upper), 17));
                    }
                    text.Append("\r\n");
                }
            }

            // Next we add all the comments to the end of the file
            foreach (String comment in Comments)
            {
                text.Append( comment+"\r\n" );
            }

            return text.ToString();
        }

        /// <summary>
        /// Converts double value to string. NaN value is returned as "."
        /// </summary>
        /// <param name="number">Double value that is to be converted</param>
        /// <returns>Double value as string</returns>
        private String DoubleToStr( Double number )
        {
            if (Double.IsNaN(number)) return ".";
            else
            {
                // If the number is very small/big, it is better to print the double value in scientific format
                //if (Math.Abs(number) > 100000 || Math.Abs(number) < 0.00001) 
                    //return number.ToString("e8", new System.Globalization.CultureInfo("en-GB"));
                string str = number.ToString("F8", new System.Globalization.CultureInfo("en-GB"));
                if (str.Length > 16) str = str.Remove(16);
                return str;
            }
        }

        private void WriteHeaderLine(ref StringBuilder str, String info, String value)
        {
            if (!value.Equals(".")) str.Append(FixStr(info, 18) + value.ToString() + "\r\n" );
        }

        /// <summary>
        /// Gets one string information from RES header
        /// </summary>
        /// <param name="str">Name of information, for example "Study:"</param>
        /// <param name="text">Reference to string object containing all header text.</param>
        /// <returns></returns>
        private String getString(String str, ref String text)
        {
            if (text.Contains(str))
            {
                int start_index = text.IndexOf(str) + str.Length;
                int end_index = text.IndexOf("\n", start_index);
                String result = text.Substring(start_index, end_index-start_index );

                result = result.Trim(new char[] {' ', '\t', '\r', '\n' } );
                return result;
            }
            else
            {
                return ".";
            }
        }

        /// <summary>
        /// This function reads all the words from next line of the file and returns then as String list.
        /// All possible Comment lines are added to Comments list and empty lines are ignored. Returns null if
        /// file has been read to the end.
        /// </summary>
        /// <param name="reader">Reader object that reads the string.</param>
        /// <returns>List of all words in the next line of reader. Null if the string has been readed to the end.</returns>
        protected String[] readWordsFromLine(ref System.IO.StringReader reader)
        {
            string line = "";
            bool correct_line = true;

            // space and tab will divide strings
            char[] separators = new char[] { ' ', '\t' };

            // every loop reads one line from file. Empty lines are ignored.
            do
            {
                line = reader.ReadLine();

                // null line means that the string has been read to the end
                if (line == null) return null;

                line.Trim();

                // All comments are added to Comments list
                if (line.Length <= 1) // empty lines are ignored
                {
                    correct_line = false;
                }
                else if (line[0] == '#')
                {
                    Comments.Add(line);
                    correct_line = false;
                }
                else correct_line = true;
            }
            while (!correct_line);

            //Console.WriteLine(line);

            return line.Split(separators, StringSplitOptions.RemoveEmptyEntries);
        }

        /// <summary>
        /// Creates a fixed size String from input string and result length.
        /// All extra space is filled with spaces. If the input string is longer than result,
        /// the remainders are cut off
        /// </summary>
        /// <param name="input">The input string</param>
        /// <param name="length">length if result string</param>
        /// <returns></returns>
        protected String FixStr(String input, int length)
        {
            // null input is treated like empty string 
            if (input == null) return new String(' ', length);

            int difference = length - input.Length;

            // the input string is longer than the boundaries given
            if (difference < 0) { return input.Substring(0, length); }
            else if (difference == 0) { return input; }
            else return input.PadRight(input.Length + difference, ' ');
        }

        #endregion;
    }
}
