﻿/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// This class tells what kind of information is stored to one RES parameter. 
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class RegionCell : TableCell
    {
        /// <summary>
        /// All parts of Region name
        /// </summary>
        public struct RegionStruct
        {
            /// <summary>
            /// Region name
            /// </summary>
            public String Name;
            /// <summary>
            /// Secondary region name (hemisphere).
            /// </summary>
            public String SecondaryName;
            /// <summary>
            /// Plane of region
            /// </summary>
            public String Plane;

            /// <summary>
            /// Constructs RegionStruct struct
            /// </summary>
            /// <param name="name">Name of region</param>
            /// <param name="Secondary_name">Secondary name of region</param>
            /// <param name="plane">Plane of region</param>
            public RegionStruct(String name, String Secondary_name, String plane)
            {
                this.Name = name;
                this.SecondaryName = Secondary_name;
                this.Plane = plane;
            }
            /// <summary>
            /// Constructs RegionStruct struct with name.
            /// </summary>
            /// <param name="name">Name of region</param>
            public RegionStruct(String name)
            {
                this.Name = name;
                this.SecondaryName = ".";
                this.Plane = ".";
            }
        };

        /// <summary>
        /// All parts of region name. Unknown parts can be set "."
        /// </summary>
        public RegionStruct Region;
        /// <summary>
        /// If true, the parameters are containing standard deviation values.
        /// </summary>
        public bool SD;
        /// <summary>
        /// Upper confidence limit in per cents.
        /// </summary>
        public Double CL_upper;
        /// <summary>
        /// Lower confidence limit in per cents
        /// </summary>
        public Double CL_lower;

        /// <summary>
        /// Constructor that creates a header containing only region name. The header will contain
        /// no standard deviation or confidence limits. 
        /// </summary>
        /// <param name="reg">Region name</param>
        public RegionCell(RegionStruct reg)
        {
            Region = reg;
            SD = false;
            CL_upper = Double.NaN;
            CL_lower = Double.NaN;
        }

        /// <summary>
        /// Default constructor
        /// </summary>
        public RegionCell()
        {
            Region.Name = ".";
            Region.Plane = ".";
            Region.SecondaryName = ".";
            SD = false;
            CL_upper = Double.NaN;
            CL_lower = Double.NaN;
        }

        /// <summary>
        /// Writes the Region name to string
        /// </summary>
        /// <returns>String containing the region name.</returns>
        public override String ToString()
        {
            return this.Region.Name;
        }

        /// <summary>
        /// Constructor that creates a header containing regiona name and standard deviation.
        /// Header will contain no confidence limits.
        /// </summary>
        /// <param name="reg">Region name</param>
        /// <param name="sd">Standard deviation</param>
        public RegionCell(RegionStruct reg, bool sd)
        {
            Region = reg;
            SD = sd;
            CL_upper = Double.NaN;
            CL_lower = Double.NaN;
        }

        /// <summary>
        /// Constructs header with region name, standard deviation and confidence limits.
        /// </summary>
        /// <param name="reg">Region name</param>
        /// <param name="sd">Standard deviation</param>
        /// <param name="cl_upper">Upper confidence limit</param>
        /// <param name="cl_lower">Lower confidence limit</param>
        public RegionCell(RegionStruct reg, bool sd, Double cl_upper, Double cl_lower)
        {
            Region = reg;
            SD = sd;
            CL_upper = cl_upper;
            CL_lower = cl_lower;
        }

        /// <summary>
        /// Returns copy of the object
        /// </summary>
        /// <returns>Copy of the object.</returns>
        public override TableCell Clone()
        {
            RegionCell cell = new RegionCell();
            cell.Region.Name = this.Region.Name;
            cell.Region.SecondaryName = this.Region.SecondaryName;
            cell.Region.Plane = this.Region.Plane;

            cell.SD = this.SD;
            cell.Value = this.Value;
            cell.CL_lower = this.CL_lower;
            cell.CL_upper = this.CL_upper;

            return cell;
        }

        /// <summary>
        /// Checks if two REScells are identical by contents
        /// </summary>
        /// <param name="obj">Another rescell</param>
        /// <returns>True if the cells are identical.</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is RegionCell)) return false;

            RegionCell rc = (obj as RegionCell);

            if (!DoubleCheck(this.CL_lower, rc.CL_lower)) return false;
            if (!DoubleCheck(this.CL_upper, rc.CL_upper)) return false;
            if (this.SD != rc.SD) return false;
            if (!DoubleCheck(this.Value, rc.Value)) return false;

            if (!this.Region.Name.Equals(rc.Region.Name)) return false;
            if (!this.Region.SecondaryName.Equals(rc.Region.SecondaryName)) return false;
            if (!this.Region.Plane.Equals(rc.Region.Plane)) return false;

            return true;
        }


        /// <summary>
        /// Compares this region_cell to another
        /// </summary>
        /// <param name="obj">other RESCell object</param>
        /// <returns>0: this equals obj, 1: this smaller than obj, -1: this greater than obj</returns>
        public override int CompareTo(object obj)
        {
            if (!(obj is RegionCell)) throw new TPCTableCellException("Cannot compare region_cell to other than region_cell.");

            return String.Compare(this.Region.Name, ((RegionCell)obj).Region.Name);
        }
        /// <summary>
        /// Gets hash code of this object. Default bahaviour.
        /// </summary>
        /// <returns>returns hash code with default implementation</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }
}