/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// This class is like TACTable but its columns have curveheaders
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class RegionalTACTable : TACTable
    {
        //public CompareFields compareField;
        /// <summary>
        /// Every column have its own header.
        /// </summary>
        public List<CurveHeader> curveheader;

        /// <summary>
        /// Default constructor
        /// </summary>
        public RegionalTACTable()
        {
            // if this is first time we must allocate the curveheader list;
            this.curveheader = new List<CurveHeader>();
        }

        /// <summary>
        /// Constructs from superclass
        /// </summary>
        /// <param name="t">Table data</param>
        public RegionalTACTable(TACTable t) : base(t.GetTimeCells())
        {
            this.unit = t.unit;
            this.timetype = t.timetype;
            this.curveheader = new List<CurveHeader>();
            for (int i = 0; i < t.columns; i++) {
                this.AddColumn(t.GetColumn(i));
            }
        }

        /// <summary>
        /// Creates a TACTable and fills it with frame(time) data
        /// </summary>
        /// <param name="times">Frame time data</param>
        public RegionalTACTable(FrameTimeCell[] times) : base(times)
        {
            curveheader = new List<CurveHeader>();
        }

        /// <summary>
        /// Constructor which creates a RegionalTactable with same time information and unit as input RegionalTACtable
        /// </summary>
        /// <param name="table_source">RegionalTACTable containing the initial time and unit information.</param>
        public RegionalTACTable(RegionalTACTable table_source) : base(table_source.GetTimeCells())
        {
            if (table_source == null) throw new TPCTACTableException("Constructor: Source table can't be null.");

            curveheader = new List<CurveHeader>();
            this.unit = table_source.unit;
            this.timetype = table_source.timetype;
        }

        /// <summary>
        /// Adds a TAC to RegionalCurveFile.
        /// </summary>
        /// <param name="curvecolumn">column data to add</param>
        public void AddColumn(TAC curvecolumn)
        {
            if (curvecolumn == null) throw new TPCTACTableException("AddColumn: Input TAC can't be null.");

            if (curvecolumn.Header != null) this.curveheader.Add(curvecolumn.Header);
            else throw new TPCTACTableException("AddColumn: CurveHeader cant be null.");

            base.AddColumn(curvecolumn.ToArray());
        }

        /// <summary>
        /// Replaces one TAC column.
        /// </summary>
        /// <param name="index">Index of TAC</param>
        /// <param name="curveTac">TAC</param>
        public void ReplaceColumn(int index, TAC curveTac)
        {
            if (index < 0 || index >= columns) throw new TPCTACTableException("ReplaceColumn: Index out of bounds.");
            if (curveTac == null) throw new TPCTACTableException("ReplaceColumn: Input TAC can't be null.");

            base.ReplaceColumn(index, curveTac.ToArray());

            if (curveTac.Header is CurveHeader) this.curveheader[index] = new CurveHeader((CurveHeader)curveTac.Header);
            else this.curveheader[index] = new CurveHeader();
        }

        /// <summary>
        /// Adds a TAC to given index in RegionalCurveFile.
        /// </summary>
        /// <param name="index">index number of the column (0-based)</param>
        /// <param name="curvecolumn">column data to add</param>
        public void AddColumnAt(int index, TAC curvecolumn)
        {
            if (index < 0 || index > columns) throw new TPCTACTableException("AddColumnAt: Index out of bounds.");
            if (curvecolumn == null) throw new TPCTACTableException("AddColumnAt: Input TAC can't be null.");

            List<TAC> tacs = new List<TAC>();

            // first we put all the data to TAC list
            for (int i = 0; i < columns; i++)
            {
                TAC t = new TAC(this.curveheader[i], base.GetColumn(i));
                tacs.Add(t);
            }

            // next we insert the new TAC
            tacs.Insert(index, curvecolumn);

            // finally we replace all columns
            for (int i = 0; i < columns; i++)
            {
                // lets replace the old data
                curveheader[i] = tacs[i].Header.Clone();
                base.ReplaceColumn(i, tacs[i].ToArray());
            }
            
            // and finally the last column
            int last = columns;
            base.AddColumn( tacs[last].ToArray() );
            curveheader.Add(tacs[last].Header);
        }

        /// <summary>
        /// Adds one empty column to RegionalTACTable
        /// </summary>
        public override void AddColumn()
        {
            base.AddColumn();
            if (base.columns-1 == this.curveheader.Count) this.curveheader.Add(new CurveHeader());
        }

        /// <summary>
        /// Adds one column to RegionalTACTable with no header
        /// </summary>
        /// <param name="times">Time cells</param>
        /// <param name="data">TableCell data</param>
        public override void AddColumn(FrameTimeCell[] times, TableCell[] data)
        {
            base.AddColumn(times, data);

            // We must make sure that if someone uses base classes addColumn method, curveheader list
            // is also updated
            if (base.columns - 1 == this.curveheader.Count) this.curveheader.Add(new CurveHeader());
        }

        /// <summary>
        /// Adds one column to RegionalTACTable. Header is left blanc ( dots )
        /// </summary>
        /// <param name="data">Data which is put to the column</param>
        public override void AddColumn(params double[] data)
        {
            base.AddColumn(data);

            // We must make sure that if someone uses base classes addColumn method, curveheader list
            // is also updated
            if (base.columns - 1 == this.curveheader.Count) this.curveheader.Add(new CurveHeader());
        }

        /// <summary>
        /// Adds one column to RegionalTACTable. Header is left blanc ( dots )
        /// </summary>
        /// <param name="data">Data which is put to the column</param>
        public override void AddColumn(TableCell[] data)
        {
            base.AddColumn(data);

            // We must make sure that if someone uses base classes addColumn method, curveheader list
            // is also updated
            if (base.columns - 1 == this.curveheader.Count) this.curveheader.Add(new CurveHeader());
        }

        /// <summary>
        /// Replaces one column in RegionalTACTable.
        /// </summary>
        /// <param name="index">Index of column</param>
        /// <param name="data">Replacing data</param>
        public override void ReplaceColumn(int index, TableCell[] data)
        {
            base.ReplaceColumn(index, data);

            // We must make sure that if someone uses base classes addColumn method, curveheader list
            // is also updated
            this.curveheader[index] = new CurveHeader();
        }

        /// <summary>
        /// Sorts all the TACs in new order accordiong to compareField element.
        /// </summary>
        /// <param name="Comparer">Comparer class. May be for example DFTTACComparer or CPTTACComparer.</param>
        public void SortColumns(TACComparer Comparer)
        {
            if (Comparer == null) throw new TPCTACTableException("SortColumns: TACComparer object can't be null.");
            if (this.curveheader == null) throw new TPCTACTableException("Cannot sort TACS: There is no header.");

            // Nothing to sort
            if (columns == 0) return;

            List<TAC> tacs = new List<TAC>();

            // first we put all the data to TAC list
            for (int i = 0; i < columns; i++)
            {
                TAC t = new TAC(this.curveheader[i], base.GetColumn(i));
                tacs.Add(t);
            }

            // next we can sort all the columns
            // tacs.Sort(new TACComparer(this.compareField));
            tacs.Sort( Comparer );

            // finally we replace all columns
            for (int i = 0; i < columns; i++)
            {
                // lets replace the old data
                base.ReplaceColumn(i, tacs[i].ToArray());
                curveheader[i] = tacs[i].Header.Clone();
            }
        }

        /// <summary>
        /// Removes one regional TAC from table
        /// </summary>
        /// <param name="index">Index of TAC to remove.</param>
        new public void DeleteColumn(int index)
        {
            if (index < 0 || index >= columns) throw new TPCTACTableException("removeTAC: Index out of bounds.");

            this.curveheader.RemoveAt(index);
            base.DeleteColumn(index);
        }

        /// <summary>
        /// Gets one TAC (column) from DFT file
        /// </summary>
        /// <param name="index">Index of the TAC</param>
        /// <returns>TAC</returns>
        new public TAC GetColumn(int index)
        {
            if (index < 0 || index >= columns) throw new TPCTACTableException("getTAC: Index out of bounds.");

            // curve header data
            CurveHeader ch = null;
            if (this.curveheader != null) ch = this.curveheader[index].Clone();

            TAC dftct = new TAC(ch, base.GetColumn(index));

            return dftct;
        }

        /// <summary>
        /// Searchs elements from RegionalTACTable. Matching TACS will be returned in List.
        /// Searching is done by element object which is usually string.
        /// Example for finding TACs from DFT: Find( "someword", new DFTTACComparer(DFTTACComparer.CompareField.CurveName ))
        /// </summary>
        /// <param name="element">searched element</param>
        /// <param name="comparer">match comparison operator</param>
        /// <returns>all matched TACs</returns>
        public List<TAC> Find(Object element, TACComparer comparer)
        { 
            if (comparer == null) throw new TPCTACTableException("Find: TACComparer object cant be null.");
            if (element == null) throw new TPCTACTableException("Find: Comparing object cant be null.");
            if (this.curveheader == null) throw new TPCTACTableException("Find: There is no header.");

            List<TAC> tacs = new List<TAC>();
            if (columns == 0) return tacs;

            // first we put all the data to TAC list
            for (int i = 0; i < columns; i++)
            {
                TAC t = new TAC(this.curveheader[i], base.GetColumn(i));
                tacs.Add(t);
            }

            // We forward the comparising element to comparer. It can be for example some string for
            // CurveName or plane, or Double for ROISize
            comparer.FindValue = element;
            return tacs.FindAll(comparer.GetFindFunction());
        }

        /// <summary>
        /// Returns new identical object of RegionalTACTable
        /// </summary>
        /// <returns>New identical copy of RegionalTACTable</returns>
        new public RegionalTACTable Clone()
        {
            if (this == null) return null;

            RegionalTACTable tab = new RegionalTACTable(this);
            for( int i=0; i<columns; i++ )
            {
                tab.AddColumn( this.GetColumn(i) );
            }
            return tab;
        }

        /// <summary>
        /// Compares two RegionalTACTables. Returns true if them are identical.
        /// </summary>
        /// <param name="first">First RegionalTACTable object</param>
        /// <param name="second">Second RegionalTACTable object</param>
        /// <returns>True if the tables are identical.</returns>
        public static bool Equals(RegionalTACTable first, RegionalTACTable second)
        {
            if (!TACTable.Equals(first, second)) return false;

            // null check
            if((     first.curveheader == null && second.curveheader != null ) 
                || ( first.curveheader != null && second.curveheader == null) ) return false;
            if (first.curveheader != null)
            {
                //if curveheaders are not null we can compare headers
                if (first.curveheader.Count != second.curveheader.Count) return false;

                for (int i = 0; i < first.columns; i++)
                {
                    if (!first.curveheader[i].Equals( second.curveheader[i] ) ) return false;
                }
            }

            return true; 
        }

        /// <summary>
        /// Set user-defined fit time range to comply with RegionalTACTable data.
        /// </summary>
        /// <param name="s_time">user-defined start time</param>
        /// <param name="e_time">user-defined end time</param>
        /// <returns></returns>
        public void FitTime(double s_time, double e_time)
        {
            TPClib.Curve.FrameTimeCell t;
            int start_index = -1;
            int end_index = -1;
            int i = 0;
            bool cont = true;

            if (this.timetype == FrameTimeCell.FrameTimetype.START_END)
            {

                while (cont)
                {
                    if (s_time < this.GetTimeCell(0).start_time) throw new TPCException("Start time " + s_time + "s is out of tissue file's time range, which starts at value " + this.GetTimeCell(0).start_time);
                    t = this.GetTimeCell(i);
                    if (t.start_time <= s_time) start_index = i;
                    if (t.start_time >= s_time) cont = false;
                    if (cont) i++;
                }
                cont = true;
                i = 0;
                while (cont)
                {
                    if (i >= this.rows) throw new TPCException("End time " + e_time + "s is out of tissue file's time range, which ends at time " + this.GetTimeCell(this.rows - 1).end_time);
                    t = this.GetTimeCell(i);
                    if (t.end_time <= e_time) end_index = i;
                    if (t.end_time >= e_time) cont = false;
                    if (cont) i++;
                }

                i = end_index;
                while (this.rows > i && this.rows - 1 != end_index)
                {
                    this.RemoveRow(i + 1);
                }
                i = start_index;
                while (i > 0)
                {
                    this.RemoveRow(i - 1);
                    i--;
                }

            }
            else
            {
                while (cont)
                {
                    if (s_time < this.GetTimeCell(0).mid_time) throw new TPCException("Start time " + s_time + "s is out of tissue file's time range, which starts at value " + this.GetTimeCell(0).mid_time);
                    t = this.GetTimeCell(i);
                    if (t.mid_time <= s_time) start_index = i;
                    if (t.mid_time >= s_time) cont = false;
                    if (cont) i++;
                }
                cont = true;
                i = 0;
                while (cont)
                {
                    if (i >= this.rows) throw new TPCException("End time " + e_time + "s is out of tissue file's time range, which ends at time " + this.GetTimeCell(this.rows - 1).mid_time);
                    t = this.GetTimeCell(i);
                    if (t.mid_time <= e_time) end_index = i;
                    if (t.mid_time >= e_time) cont = false;
                    if (cont) i++;
                }

                i = end_index;
                while (this.rows > i && this.rows - 1 != end_index)
                {
                    this.RemoveRow(i + 1);
                }
                i = start_index;
                while (i > 0)
                {
                    this.RemoveRow(i - 1);
                    i--;
                }


            }
        }

        /// <summary>
        /// Method for getting region's indexes with given name from RegionalTACTable
        /// </summary>
        /// <param name="region_name">The region's name, which is wanted to search</param>
        /// <returns>Array, that contains parameter's region numbers</returns>
        public int[] DftSelectRegions(string region_name)
        {
            /* Check the input */
            if (this.columns < 1 || region_name.Length < 1) return null;

            List<int> result = new List<int>();

            /* Check each VOI */
            for (int ri = 0; ri < this.columns; ri++)
            {
                if (this.GetColumn(ri).Header.curveName.Equals(region_name))
                {
                    result.Add(ri);
                }
            }
            return result.ToArray();
        }



    }
}
