/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Data;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{
    /// <summary>
    /// Exception class for exceptions occurring in class TACFrame
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class TPCParameterTableException : TPCException
    {
        /// <summary>
        /// Constructor with no error message
        /// </summary>
        public TPCParameterTableException()
            : base("No message")
        {
        }
        /// <summary>
        /// Constructor with error message
        /// </summary>
        /// <param name="message">message</param>
        public TPCParameterTableException(string message)
            : base(message)
        {
        }
    }

    /// <summary>
    /// Time Activity Curve (TAC) table. Table contains activity data in rows. 
    /// Each row has time value(s). The name of the 1st column is 'FrameTimeCells' and it exists allways. 
    /// Activity data has unit. Times are in seconds.
    /// </summary>
    public class TACTable
    {
        /// <summary>
        /// Table of data inside table.
        /// </summary>
        protected DataTable table;
        /// <summary>
        /// Time type of this TAC table. (default == FrameTimetype.START)
        /// </summary>
        public FrameTimeCell.FrameTimetype timetype = FrameTimeCell.FrameTimetype.START;
        /// <summary>
        /// Unit of data in table.
        /// </summary>
        public Data_unit unit;
        /// <summary>
        /// Number of rows in the table.
        /// </summary>
        public int rows
        {
            get { return table.Rows.Count; }
        }
        /// <summary>
        /// Number of regions (curves)
        /// </summary>
        public int columns
        {
            get
            {
                //return number of columns - 1 column for frames
                return (table.Columns.Count - 1);
            }
        }
        /// <summary>
        /// Number of frames = number of rows in the table. 
        /// </summary>
        public int frames
        {
            get
            {
                return table.Rows.Count;
            }
        }

        /// <summary>
        /// Multiplication for TACTables activity values. All curve elements are multiplied (NOT FRAME DATA)
        /// </summary>
        /// <param name="value">Multiplication value</param>
        public void MultiplyActivities(Double value)
        {
            for (int y = 0; y < rows; y++)
            {
                for (int x = 0; x < columns; x++)
                {
                    this[x, y] *= value;
                }
            }
        }

        /// <summary>
        /// Multiplication for TACTables Frame(time) values. All time elements are multiplied (NOT ACTIVITY DATA)
        /// </summary>
        /// <param name="value">Multiplication value</param>
        public void MultiplyFrames(Double value)
        {
            for (int y = 0; y < rows; y++)
            {
                table.Rows[y]["time"] = (table.Rows[y]["time"] as FrameTimeCell) * value;
            }
        }

        /// <summary>
        /// Gets a column from TACTable.  
        /// </summary>
        /// <seealso cref="GetTimeCells()"/>
        /// <param name="index">zero-based index of data column in TACTable</param>
        /// <returns>An array of TableCells</returns>
        public TableCell[] GetColumn(int index)
        {
            if (index < 0 || index >= columns)
                throw new TPCTACTableException("GetColumn: Index out of bounds: "+ index);
            TableCell[] tablecells = new TableCell[rows];
            for (int i = 0; i < rows; i++)
            {
                tablecells[i] = (table.Rows[i][index+1] as TableCell);
            }

            return tablecells;
        }
        /// <summary>
        /// Comparison class for TAC data row sorting 
        /// </summary>
        protected class RowFrameComparer : IComparer<DataRow>
        {
            /// <summary>
            /// Sorting according to frame times
            /// </summary>
            /// <param name="x">first evaluated row</param>
            /// <param name="y">second evaluated row</param>
            /// <returns>0:x equals y, 1:x smaller than y, -1:x greater than y</returns>
            public int Compare(DataRow x, DataRow y)
            {
                return ((FrameTimeCell)x[0]).CompareTo((FrameTimeCell)y[0]);
            }
        }
        /// <summary>
        /// Constructor with number of frames and curves. Frame time type is set
        /// to FrameTimetype.START by default.
        /// </summary>
        public TACTable()
        {
            table = new DataTable();
            table.Columns.Add(new DataColumn("time", Type.GetType("TPClib.Curve.FrameTimeCell")));
            unit = Data_unit.Unknown;
        }
        /// <summary>
        /// Creates a TACTale with time type and number of columns
        /// </summary>
        /// <param name="type">frame time type</param>
        /// <param name="numColumns">Number of TACs(columns).</param>
        public TACTable(FrameTimeCell.FrameTimetype type, int numColumns )
        {
            table = new DataTable();
            timetype = type;
            //add frame time column
            DataColumn col = new DataColumn("time", Type.GetType("TPClib.Curve.FrameTimeCell"));
            table.Columns.Add(col);
            //add columns to match the number of headers
            for( int i=0; i<numColumns; i++ )
            {
                table.Columns.Add("",Type.GetType("TPClib.Curve.TableCell"));
            }
            this.unit = Data_unit.Unknown;
        }

        /// <summary>
        /// Creates a TACTable and fills it with frame(time) data
        /// </summary>
        /// <param name="frames">Frame data</param>
        public TACTable(FrameTimeCell[] frames)
        {
            if (frames == null) throw new TPCTACTableException("Constructor: Cannot init with null array.");

            table = new DataTable();

            // If argument is an empty array, timetype is the default type
            if (frames.Length > 0)
            {
                timetype = frames[0].type;
            }
            DataColumn col = new DataColumn("time", Type.GetType("TPClib.Curve.FrameTimeCell"));
            table.Columns.Add(col);
            this.unit = Data_unit.Unknown;

            if( frames != null || frames.Length>0) foreach (FrameTimeCell ftc in frames)
            {
                DataRow row = table.NewRow();
                row["time"] = ftc;

                table.Rows.Add(row);
            }
        }

        /// <summary>
        /// Creates an ampty TACTale
        /// </summary>
        /// <param name="numColumns">Number of TACs(columns).</param>
        public TACTable(int numColumns)
        {
            table = new DataTable();

            //add frame time column
            DataColumn col = new DataColumn("time", Type.GetType("TPClib.Curve.FrameTimeCell"));
            table.Columns.Add(col);
            //add columns to match the number of headers
            for (int i = 0; i < numColumns; i++)
            {
                table.Columns.Add("", Type.GetType("TPClib.Curve.TableCell"));
            }
            this.unit = Data_unit.Unknown;
        }

        /// <summary>
        /// Adds float array to the end of the table (row). Leaves timecell empty
        /// </summary>
        /// <param name="curvedata">data of new row</param>
        public void AddRow(float[] curvedata)
        {
            if (curvedata == null) throw new TPCTACTableException("AddRow: Curvedata parameter can't be null.");

            if (curvedata.Length != this.columns)
                throw new TPCTACTableException("Number of columns does not match new row to be added.");
            //INFINITE LOOP???
            AddRow(curvedata);
        }
        /// <summary>
        /// Adds decimal data to the end of the table (row). Leaves timecell empty.
        /// </summary>
        /// <param name="curvedata">data of new row</param>
        public void AddRow(params Double[] curvedata)
        {
            if (curvedata == null) throw new TPCTACTableException("AddRow: Curvedata parameter can't be null.");

            if (curvedata.Length != this.columns)
                throw new TPCTACTableException("Number of columns does not match new row to be added.");
            
            //Create new row
            DataRow row = table.NewRow();

            int index = 1;
            //add data into row
            foreach (double curve in curvedata)
            {
                TableCell tabl = new TableCell();
                tabl.Value = curve;
                row[index] = tabl;
                index++;
            }
            //add row into table
            table.Rows.Add(row);
        }

        /// <summary>
        /// Adds a row with given index.
        /// </summary>
        /// <param name="position">Index number</param>
        /// <param name="ftc">Frame time.</param>
        /// <param name="curvedata"></param>
        public void AddRowAt(int position, FrameTimeCell ftc, params TableCell[] curvedata)
        {
            if( position<0 || position > rows) throw new TPCTACTableException("AddRowAt: Index out of bounds: "+position);
            if (curvedata == null)
                throw new TPCTACTableException("AddRowAt: Curve data can't be null.");
            if (ftc == null)
                throw new TPCTACTableException("AddRowAt: Frame(time) can't be null.");
            if (curvedata.Length != this.columns)
                throw new TPCTACTableException("AddRowAt: Number of columns does not match new row to be added.");

            //Create new row
            DataRow row = table.NewRow();
            row[0] = ftc;
            int index = 1;
            //add data into row
            foreach (TableCell curve in curvedata)
            {
                if (curve == null) row[index] = new TableCell(Double.NaN);
                else row[index] = curve;
                index++;
            }
            //add row into table
            table.Rows.InsertAt(row, position);
        }

        /// <summary>
        /// Adds decimal data(rows) to the table.
        /// </summary>
        /// <param name="ftc">frame time cell object</param>
        /// <param name="curvedata">table cells of the new row</param>
        public void AddRow(FrameTimeCell ftc, params TableCell[] curvedata)
        {
            if (curvedata==null)
                throw new TPCTACTableException("AddRow: Curve data can't be null.");
            if (ftc == null)
                throw new TPCTACTableException("AddRow: Time cell can't be null.");
            if (curvedata.Length != this.columns)
                throw new TPCTACTableException("AddRow: Number of columns does not match new row to be added.");

            //Create new row
            DataRow row = table.NewRow();
            row[0] = ftc;
            int index = 1;
            //add data into row
            foreach (TableCell curve in curvedata)
            {
                if (curve == null) row[index] = new TableCell(Double.NaN);
                else row[index] = curve;
                index++;
            }
            //add row into table
            table.Rows.Add(row);
        }

        /// <summary>
        /// gets one row of curve data.
        /// </summary>
        /// <param name="rowindex">index of row</param>
        /// <returns>An array of TableCells</returns>
        public TableCell[] GetRow(int rowindex)
        {
            if (rowindex < 0 || rowindex >= rows)
                throw new TPCTACTableException("GetRow: Index out of bounds: "+rowindex);

            TableCell[] result = new TableCell[this.columns];

            // start index 1 because the index 0 column is for timedata
            for (int i = 1; i < table.Columns.Count; i++)
            {
                result[i - 1] = (table.Rows[rowindex][i] as TableCell);
            }
            return result;
        }
        
        /// <summary>
        /// gets a frametimecell
        /// </summary>
        /// <param name="rowindex">row of the data</param>
        /// <returns>FrameTimeCell containing the time information of the row</returns>
        public FrameTimeCell GetTimeCell(int rowindex)
        {
            if (rowindex < 0 || rowindex >= rows) throw new TPCTACTableException("GetTimeCell: Index out of bounds: "+rowindex);

            return (table.Rows[rowindex][0]) as FrameTimeCell;
        }

        /// <summary>
        /// gets all frametimecells
        /// </summary>
        /// <returns>FrameTimeCells containing the time information</returns>
        public FrameTimeCell[] GetTimeCells()
        {
            FrameTimeCell[] cells = new FrameTimeCell[rows];
            
            for (int i = 0; i < rows; i++)
            {
                cells[i] = GetTimeCell(i);
            }

            return cells;
        }

        /// <summary>
        /// Fills the FrameTimeCell values by new ones. 
        /// </summary>
        /// <param name="times">a FrameTimeCell array containing the time information.</param>
        public void SetTimeCells( FrameTimeCell[] times )
        {
            if (times == null) throw new TPCTACTableException("SetTimeCells: Time data can't be null.");
            if (rows > 0 && rows != times.Length) throw new TPCTACTableException("setTimeCells: Array lengths do not match.");

            if (times.Length==0 ) return;

            // If the table is empty, we init it
            if (rows == 0) for (int i = 0; i < times.Length; i++)
                {
                    DataRow row = table.NewRow();
                    row["time"] = times[i];

                    table.Rows.Add(row);
                }

            this.timetype = times[0].type;

            for (int i = 0; i < rows; i++)
            {
                SetTimeCell(i, times[i]);
            }
        }

        /// <summary>
        /// Changes one frame cell
        /// </summary>
        /// <param name="index">index of the frame cell</param>
        /// <param name="NewValue">new FrameTimeCell object</param>
        public void SetTimeCell( int index, FrameTimeCell NewValue )
        {
            if (NewValue == null) throw new TPCTACTableException("SetTimeCell: Time data can't be null.");

            if (index < 0 || index >= rows) throw new TPCTACTableException("setTimeCell: Index out of bounds: " + index);
            if ( table.Rows[index][0] != null && NewValue.type != timetype )
                throw new TPCCurveFileException("setTimeCell: Cannot set new value. New type differs from other cells.");

            table.Rows[index][0] = NewValue;
        }

        /// <summary>
        /// Removes the specified datarow from table.
        /// </summary>
        /// <param name="i">The index of the row to remove</param>
        public void RemoveRow(int i)
        {
            if (i < 0 || i >= table.Rows.Count)
                throw new TPCTACTableException("RemoveRow: Index out of bounds: " + i + " [0.." + (table.Rows.Count - 1) + "]");
            table.Rows.RemoveAt(i);
        }

        /// <summary>
        /// Removes the specified datarow from table.
        /// </summary>
        /// <param name="row">row to be removed</param>
        public void RemoveRow(DataRow row) {
            table.Rows.Remove(row);
        }

        /// <summary>
        /// Removes one column from table
        /// </summary>
        /// <param name="index">index of the column to remove</param>
        public void DeleteColumn( int index )
        {
            if (index < 0 || index >= columns) throw new TPCTACTableException("DeleteColumn: Index out of bounds: "+index);

            table.Columns.RemoveAt(index+1);
        }


        /// <summary>
        /// Gets the index of the specified row.
        /// </summary>
        /// <param name="row">The row to search for.</param>
        public int IndexOf(DataRow row) {
            return table.Rows.IndexOf(row);
        }


        /// <summary>
        /// The zero-based indexes for TAC table activity items.
        /// </summary>
        /// <param name="x">zero-based column number</param>
        /// <param name="y">zero-based row number</param>
        /// <returns></returns>
        public TableCell this[int x, int y]
        {
            get
            {
                if (x < 0 || x >= columns) throw new TPCTACTableException("column: Index "+x+" out of bounds [0.." + (table.Columns.Count - 1) + "]");
                if (y < 0 || y >= rows) throw new TPCTACTableException("row: Index " + y + " out of bounds [0.." + (table.Rows.Count - 1) + "]");
                return (TableCell)table.Rows[y][x+1];
            }
            set
            {
                if (x < 0 || x >= columns) throw new TPCTACTableException("column: Index " + x + " out of bounds [0.." + (table.Columns.Count - 1) + "]");
                if (y < 0 || y >= rows) throw new TPCTACTableException("row: Index " + y + " out of bounds [0.." + (table.Rows.Count - 1) + "]");
                //if (y == 0 && !(value is FrameTimeCell)) throw new TPCTACTableException("can only set object of type FrameTimeCell into first column");
                if (!(value is TableCell)) throw new TPCTACTableException("can only set object of type TableCell into table data cells");
                table.Rows[y][x+1] = value;
            }
        }

        /// <summary>
        /// Returns the lowest value in Curve
        /// </summary>
        /// <returns>the lowest activity value in Curve</returns>
        public double GetMin()
        {
            double min = double.MaxValue;
            for (int i = 0; i < rows; i++)
            {
                for (int j = 1; j <= columns; j++)
                {
                    if (((TableCell)table.Rows[i][j]).Value == double.NaN) continue;
                    if (((TableCell)table.Rows[i][j]).Value < min) min = ((TableCell)table.Rows[i][j]).Value;
                }
            }
            return min;
        }
        /// <summary>
        /// Returns the highest value in Curve
        /// </summary>
        /// <returns>the lowest activity value in Curve</returns>
        public double getMax()
        {
            double max = double.MinValue;
            for (int i = 0; i < rows; i++)
            {
                for (int j = 1; j <= columns; j++)
                {
                    if (((TableCell)table.Rows[i][j]).Value == double.NaN) continue;
                    if (((TableCell)table.Rows[i][j]).Value > max) max = ((TableCell)table.Rows[i][j]).Value;
                }
            }
            return max;
        }

        /// <summary>
        /// Check Curve for NA's in sample times and values.
        /// </summary>
        /// <remarks></remarks>
        /// <returns>Nr of NA's in sample times and values</returns>
        public int getNrOfNA()
        {
            int na_nr = 0;
            foreach(DataRow row in table.Rows) {
                FrameTimeCell timecell = (FrameTimeCell)row[0];
                if( row[0] != null ) switch(timecell.type)
                {
                    case FrameTimeCell.FrameTimetype.START_END:
                        if ( Double.IsNaN( timecell.start_time ) ) na_nr++;
                        if ( Double.IsNaN( timecell.end_time )) na_nr++;
                        break;
                    case FrameTimeCell.FrameTimetype.MID:
                        if ( Double.IsNaN( timecell.mid_time )) na_nr++;
                        break;
                    case FrameTimeCell.FrameTimetype.START:
                        if ( Double.IsNaN( timecell.start_time )) na_nr++;
                        break;
                    case FrameTimeCell.FrameTimetype.END:
                        if ( Double.IsNaN( timecell.end_time ) ) na_nr++;
                        break;
                }
            
                for (int ri = 1; ri < table.Columns.Count; ri++)
                {
                    if (row[ri] == null) continue; 
                    if (Double.IsNaN( ((TableCell)row[ri]).Value)) na_nr++;
                }
            }
            return (na_nr);
        }


        /// <summary>
        /// Correct frame start and end times if frames are slightly overlapping or have small gaps in between. 
        /// Large gap is not corrected and it does not lead to an error
        /// </summary>
        public void DelFrameOverlap()
        {
            // If there is no rows, there is nothing to do
            if (rows <= 0) return;

            // list of all rows which will be removed.
            bool[] remove_list = new bool[rows]; 

            // checks through all the overlaps
            for (int row = 0; row < table.Rows.Count; row++ )
            {
                FrameTimeCell ftc = ((FrameTimeCell)table.Rows[row][0]);
                remove_list[row] = false;

                for (int i = row+1; i < table.Rows.Count; i++)
                    if (ftc.doOverlap(((FrameTimeCell)table.Rows[i][0])))
                    {
                        remove_list[row] = true;
                        Console.WriteLine("row "+row+" overlaps with row "+i );
                        Console.WriteLine( ((FrameTimeCell)table.Rows[row][0]).ToString() );
                        Console.WriteLine(((FrameTimeCell)table.Rows[i][0]).ToString());
                    }
            }

            // next we remove all overlapping rows. The first instance is always removed.
            int listindex = 0;
            int rowcount = rows;
            for (int i = 0; i < rowcount; i++)
            {
                //Console.WriteLine("Line "+i+" removed because of overlap? "+remove_list[i] );
                if (remove_list[i] == true) { table.Rows.RemoveAt(listindex); }
                else { listindex++; }
            }
        }

        /// <summary>
        /// Calculate frame mid or start and end times and change frametype to use them. (dftFrameTimes)
        /// </summary>
        /// <param name="type">desired timetype</param>
        /// <returns>2-column TACTable rows x [modified times, frame lengths]</returns>
        public void UpdateFrameTimes(FrameTimeCell.FrameTimetype type)
        {
                
                FrameTimeCell[] fcells = new FrameTimeCell[rows];

                switch(this.timetype)
                {
                    case FrameTimeCell.FrameTimetype.START:
                        switch(type)
                        {
                            case FrameTimeCell.FrameTimetype.START:
                                return;
                            case FrameTimeCell.FrameTimetype.START_END:
                                if (this.rows <= 1) throw new TPCTACTableException("Cannot convert to start_end because there is only one start time.");
                                for (int i = 0; i < rows-1; i++)
                                {
                                    fcells[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                                    fcells[i].start_time = this.GetTimeCell(i).start_time;
                                    fcells[i].end_time = this.GetTimeCell(i + 1).start_time;
                                }
                                //use length of previous frame to estimate end time of the last frame
                                fcells[rows - 1] = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                                fcells[rows - 1].start_time = this.GetTimeCell(rows - 1).start_time;
                                fcells[rows - 1].end_time = this.GetTimeCell(rows - 1).start_time + (this.GetTimeCell(rows - 1).start_time - this.GetTimeCell(rows - 2).start_time);
                                break;
                            default:
                                throw new TPCTACTableException("Conversion from "+this.timetype+" "+type+" is not implemented.");
                        }
                        break;                    
                    case FrameTimeCell.FrameTimetype.START_END:
                        switch (type)
                        {
                            case FrameTimeCell.FrameTimetype.START_END:
                                return;
                            case FrameTimeCell.FrameTimetype.MID:
                                for (int i = 0; i < rows; i++)
                                {
                                    fcells[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.MID);
                                    fcells[i].mid_time = this.GetTimeCell(i).start_time + (this.GetTimeCell(i).end_time - this.GetTimeCell(i).start_time)/2;
                                }
                                break;
                            default:
                                throw new TPCTACTableException("Conversion from " + this.timetype + " " + type + " is not implemented.");
                        }
                        break;
                    case FrameTimeCell.FrameTimetype.MID:
                        switch (type)
                        {
                            case FrameTimeCell.FrameTimetype.MID:
                                return;
                            case FrameTimeCell.FrameTimetype.START_END:
                                if (this.rows < 1) throw new TPCTACTableException("Cannot convert to start_end because there is no rows.");
                                //expect first frame to start from zero
                                fcells[0] = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                                fcells[0].start_time = 0.0f;
                                fcells[0].end_time = 2*this.GetTimeCell(0).mid_time;
                                for (int i = 1; i < rows; i++)
                                {
                                    fcells[i] = new FrameTimeCell(FrameTimeCell.FrameTimetype.START_END);
                                    fcells[i].start_time = fcells[i-1].end_time;
                                    fcells[i].end_time = fcells[i].start_time + 2*(fcells[i].start_time-this.GetTimeCell(i).mid_time);
                                }
                                break;
                            default:
                                throw new TPCTACTableException("Conversion from " + this.timetype + " " + type + " is not implemented.");
                        }
                        break;
                    default:
                        throw new TPCTACTableException("Conversion from " + this.timetype + " " + type + " is not implemented.");
                }
                this.SetTimeCells(fcells);
                this.timetype = type;
        }

        /// <summary>
        /// Search the min and max values of Curve data. Data may contain NA's. (see dftMinMAx)
        /// </summary>
        /// <returns>the min (index 0) and max (index 1) values of Curve data</returns>
        public double[] GetMinMax()
        {
            double min = double.MaxValue;
            double max = double.MinValue;
            for(int i = 0; i < rows; i++) {
                for(int j = 1; j <= columns; j++) {
                    if (((TableCell)table.Rows[i][j]).Value == double.NaN) continue;
                    if (((TableCell)table.Rows[i][j]).Value < min) min = ((TableCell)table.Rows[i][j]).Value;
                    if (((TableCell)table.Rows[i][j]).Value > max) max = ((TableCell)table.Rows[i][j]).Value;
                }
            }
            return new double[] { min, max };
        }

        /// <summary>
        /// Replace NA's in Curve data with interpolated values. If extrapolation is necessary, 
        /// then the values (0,0) and (Infinity,last measured) are assumed (see dftNAfill)
        /// </summary>
        public void FillNA()
        {
            throw new NotImplementedException();
            /*
            CODE FROM C-library
            int ri, fi, fj;
            double x1, x2, y1, y2, x, y;

            if (dft->voiNr < 1 || dft->frameNr < 1) return (1);
            for (ri = 0; ri < dft->voiNr; ri++) for (fi = 0; fi < dft->frameNr; fi++)
            {
                {
                    if (dft->x[fi] == NA) return (2);
                    if (dft->voi[ri].y[fi] == NA)
                    {
                        // NA's before zero time are always replaced with 0 
                        if (dft->x[fi] < 0.0) { dft->voi[ri].y[fi] = 0.0; continue; }
                        x = dft->x[fi];
                        // Get the previous data that is not NA 
                        for (x1 = y1 = NA, fj = fi - 1; fj >= 0; fj--) if (dft->voi[ri].y[fj] != NA)
                            {
                                x1 = dft->x[fj]; y1 = dft->voi[ri].y[fj]; break;
                            }
                        if (x1 == NA || y1 == NA) x1 = y1 = 0.0;
                        // Get the following data that is not NA 
                        for (x2 = y2 = NA, fj = fi + 1; fj < dft->frameNr; fj++) if (dft->voi[ri].y[fj] != NA)
                            {
                                x2 = dft->x[fj]; y2 = dft->voi[ri].y[fj]; break;
                            }
                        if (x2 == NA || y2 == NA) for (fj = fi - 1; fj >= 0; fj--) if (dft->voi[ri].y[fj] != NA)
                                {
                                    x2 = dft->x[fj]; y2 = dft->voi[ri].y[fj]; break;
                                }
                        if (x2 == NA || y2 == NA) return (2);
                        // Calculate new value 
                        if (x2 == x1) y = 0.5 * (y1 + y2); else y = y2 - (x2 - x) * (y2 - y1) / (x2 - x1);
                        dft->voi[ri].y[fi] = y;
                    }
                }
            }
            */
        }


        /// <summary>
        /// Adds a column to the TACTable and fills it with data.
        /// </summary>
        /// <param name="data">Data to fill the table</param>
        public virtual void AddColumn(TableCell[] data)
        {
            if (data == null) throw new TPCTACTableException("AddColumn: data parameter can't be null.");
            if (columns > 0 && data.Length != rows)
                throw new TPCTACTableException("Number of items does not match table number of rows.");
            table.Columns.Add("", Type.GetType("TPClib.Curve.TableCell"));

            for (int i = 0; i < data.Length; i++)
            {
                table.Rows[i][table.Columns.Count - 1] = data[i];
            }
        }

        /// <summary>
        /// Adds a column to TACtable with time information. The time values are compared and data values
        /// are placed to time corresponding rows. Use this if the rows are sorted to some unusual order
        /// </summary>
        /// <param name="times">FrameTimeCell array containing tablecell time information.</param>
        /// <param name="data">The data</param>
        public virtual void AddColumn( FrameTimeCell[] times, TableCell[] data )
        {
            if (data == null) throw new TPCTACTableException("AddColumn: data parameter can't be null.");
            if (times == null) throw new TPCTACTableException("AddColumn: time parameter can't be null.");

            if (columns > 0 && data.Length != rows)
                throw new TPCTACTableException("Number of items does not match table number of rows.");
            table.Columns.Add("", Type.GetType("TPClib.Curve.TableCell"));

            int added = 0;

            for (int i = 0; i < times.Length; i++)
            {
                for (int w = 0; w < rows; w++)
                {
                    if (((FrameTimeCell)table.Rows[w][0]) == times[i])
                    {
                        // Null values are storad as NaN
                        if (data[i] == null) data[i] = new TableCell( Double.NaN );

                        table.Rows[w][table.Columns.Count - 1] = data[i];
                        added++;
                    }
                }
            }
            // If some value is missing, we raise an Exception
            if (added != rows) throw new TPCTACTableException("addColumn: Some objects of data cannot find its time from table.");
        }


        /// <summary>
        /// Adds one column at the end of table.
        /// </summary>
        /// <param name="data">The column data to be added to the end of the table.</param>
        public virtual void AddColumn(params double[] data)
        {
            if (data == null) throw new TPCTACTableException("AddColumn: data parameter can't be null.");

            if (columns > 0 && data.Length != rows)
                throw new TPCTACTableException("Number of items does not match table number of rows.");
            table.Columns.Add("", Type.GetType("TPClib.Curve.TableCell"));
            for (int i = 0; i < data.Length; i++)
            {
                table.Rows[i][table.Columns.Count - 1] = new TableCell( data[i] );
            }
        }
        /// <summary>
        /// Adds one column at the end of table.
        /// </summary>
        public virtual void AddColumn()
        {
            table.Columns.Add("", Type.GetType("TPClib.Curve.TableCell"));

            // if there is already rows, them are filled with nans
            if (rows > 0) for (int i = 0; i < rows; i++) this[columns - 1, i] = new TableCell(Double.NaN);
        }

        /// <summary>
        /// Replaces one column in TACtable
        /// </summary>
        /// <param name="index">Index of column to replace</param>
        /// <param name="data">The new data</param>
        public virtual void ReplaceColumn(int index, TableCell[] data)
        {
            if (data == null) throw new TPCTACTableException("ReplaceColumn: data parameter can't be null.");

            if (index < 0 || index >= columns)
                throw new TPCTACTableException("ReplaceColumn: Index out of bounds.");
            if (columns > 0 && data.Length != rows)
                throw new TPCTACTableException("Number of items does not match table number of rows.");

            for (int i = 0; i < data.Length; i++)
            {
                table.Rows[i][index + 1] = data[i];
            }
        }

        /// <summary>
        /// Sets table value unit. If unit accroding to string is not 
        /// recognized, Unit.Undefined is used insted.
        /// Values inside table are not modified.
        /// <remarks>Time unit cannot be set, since it is allways second inside TACTable</remarks>
        /// </summary>
        /// <param name="unit">new unit as string</param>
        public void SetUnit(string unit) {
            this.unit = UnitConverter.CreateDataUnit(unit);
        }

        /// <summary>
        /// Changes unit in header AND recalculates activity values.
        /// </summary>
        /// <remarks>Currently available conversions are:between MBq/cc and kBq/cc, kBq/cc and Bq/cc, MBq/cc and Bq/cc, nCi/cc and  kBq/cc</remarks>
        /// <param name="u">new data unit</param>
        public void UnitConversion(Data_unit u)
        {
            Double factor = UnitConverter.GetConversionFactor(this.unit, u);
            for (int y = 0; y < rows; y++)
            {
                for (int x = 0; x < columns; x++)
                {
                    (table.Rows[y][x + 1] as TableCell).Value *= factor;
                }
            }
            this.unit = u;
        }

        /// <summary>
        /// Sorts all rows by frame times.
        /// </summary>
        /// <param name="descending">False if the result is increasing order. True if decreasing.</param>
        public void SortRows( bool descending )
        {
            DataView view = new DataView(table);
            if (descending) view.Sort = "Time DESC";
            else view.Sort = "time";

            table = view.ToTable();
        }

        /// <summary>
        /// Compares two TACTable objects and if the contents are identical, true will be returned
        /// </summary>
        /// <param name="first">first TACTable object.</param>
        /// <param name="second">second TACTable object.</param>
        /// <returns>True if tables are identical.</returns>
        public static bool Equals( TACTable first, TACTable second )
        {
            if (first == null || second == null) throw new TPCTACTableException("TACTable-Equals: Cannot compare null values.");

            if (first.columns != second.columns) return false;
            if (first.rows != second.rows) return false;
            if (first.unit != second.unit) return false;
            if (first.timetype != second.timetype) return false;

            for (int y=0; y<first.rows;y++)
            {
                for (int x = 0; x < first.columns; x++)
                {
                    if ( !(first[x, y] as TableCell).Equals( (second[x, y] as TableCell) ) ) return false;
                }
            }

            for( int i=0; i<first.rows; i++ )
            {
                if (first.GetTimeCell(i) != second.GetTimeCell(i)) return false;
            }

            return true;
        }

        /// <summary>
        /// Catenates tables. Frame data must be identical
        /// </summary>
        /// <param name="tab">Another table to catenate.</param>
        public void Catenate(TACTable tab)
        {
            if (tab == null) throw new TPCTACTableException("Catenate: Cannot catenate null table.");

            if (tab.rows != rows) throw new TPCTACTableException("Catenate: Both objects must have same amount of rows.");
            // time check

            for (int i = 0; i < rows; i++) if (tab.GetTimeCell(i) != this.GetTimeCell(i)) 
                throw new TPCTACTableException("Catenate: Frame(time) data is not identical.");

            for( int i=0; i<tab.columns; i++ )
            {
                this.AddColumn( tab.GetColumn(i) );
            }            
        }

        /// <summary>
        /// Returns identical TACTable object.
        /// </summary>
        /// <returns>An identical TACTable object</returns>
        public TACTable Clone()
        {
            TACTable tab = new TACTable( this.GetTimeCells() );
            tab.timetype = this.timetype;
            tab.unit = this.unit;

            for (int i = 0; i < columns; i++)
            {
                tab.AddColumn(this.GetColumn(i));
            }
            return tab;
        }

        /// <summary>
        /// Writes the TACTables contents to string
        /// </summary>
        /// <returns>String containing all the time and data information.</returns>
        public override string ToString()
        {
            if (table == null) return "Table is empty\n";
            StringBuilder str = new StringBuilder();

            foreach (DataRow row in table.Rows)
            {
                str.Append(row[0].ToString() + "  |  ");
                if (columns > 0) for (int i = 1; i <= columns; i++)
                    {
                        str.Append(((TableCell)row[i]).Value + "  ");
                    }
                str.Append("\n");
            }

            return str.ToString();
        }

        

    }
}
