/******************************************************************************
 *
 * Copyright (c) 2008 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Runtime.InteropServices;

namespace TPClib.Curve
{   
    /// <summary>
    /// Cell of data table
    /// </summary>
    [ClassInterface(ClassInterfaceType.AutoDual), ComSourceInterfacesAttribute(typeof(Ifile))]
    public class TableCell : IComparable
    {
        /// <summary>
        /// Value inside cell
        /// </summary>
        public Double Value;

        /// <summary>
        /// Creates a TableCell object and inits its value to num.
        /// </summary>
        /// <param name="value">Initial value.</param>
        public TableCell(double value)
        {
            this.Value = value;
        }

        /// <summary>
        /// Creates a TableCell object and inits its value.
        /// </summary>
        /// <param name="tablecell">Tablecell object where the initial value is retrieved.</param>
        public TableCell(TableCell tablecell)
        {
            this.Value = tablecell.Value;
        }

        /// <summary>
        /// General information
        /// </summary>
        public TableCell()
        {
            Value = double.NaN;
        }

        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator !=(TableCell cell, double doubleValue) {
            return cell.Value != doubleValue;
        }
        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator ==(TableCell cell, TableCell cell2)
        {
            if (((Object)cell2) == null)
                if (((Object)cell) == null) return true;
                else return false;
            return cell.Value == cell2.Value;
        }
        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator !=(TableCell cell, TableCell cell2)
        {
            if (((Object)cell2) == null)
                if (((Object)cell) == null) return false;
                else return true;
            return cell.Value != cell2.Value;
        }
        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator ==(TableCell cell, double doubleValue)
        {
            return cell.Value == doubleValue;
        }

        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <(TableCell cell, double doubleValue)
        {
            return cell.Value < doubleValue;
        }
        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >(TableCell cell, double doubleValue)
        {
            return cell.Value > doubleValue;
        }

        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <=(TableCell cell, double doubleValue)
        {
            return cell.Value <= doubleValue;
        }

        /// <summary>
        /// Comparison operator for comparing against double value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="doubleValue">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >=(TableCell cell, double doubleValue)
        {
            return cell.Value >= doubleValue;
        }

        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <=(TableCell cell, TableCell cell2)
        {
            return cell.Value <= cell2.Value;
        }

        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >=(TableCell cell, TableCell cell2)
        {
            return cell.Value >= cell2.Value;
        }

        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator <(TableCell cell, TableCell cell2)
        {
            return cell.Value < cell2.Value;
        }
        /// <summary>
        /// Comparison operator for comparing against TableCell value
        /// </summary>
        /// <param name="cell">cell</param>
        /// <param name="cell2">double value</param>
        /// <returns>logical value accordin to comparison</returns>
        public static bool operator >(TableCell cell, TableCell cell2)
        {
            return cell.Value > cell2.Value;
        }

        /// <summary>
        /// Explicit conversion to double type.
        /// </summary>
        /// <param name="cell">cell that is converted</param>
        /// <returns>double value converted from private value field</returns>
        public static explicit operator Double(TableCell cell)
        {
            return (double)cell.Value;
        }
        /// <summary>
        /// Compares if this table cell is equal to other one
        /// </summary>
        /// <param name="obj">other table cell</param>
        /// <returns>true if values inside cells are equal</returns>
        public override bool Equals(object obj)
        {
            if (!(obj is TableCell)) return false;
            //if (Double.IsNaN((obj as TableCell).Value) && Double.IsNaN(Value)) return true;

            return DoubleCheck( Value , (obj as TableCell).Value );
        }
        /// <summary>
        /// Gets hash code of TableCell. Default behaviour. 
        /// </summary>
        /// <returns>hash code value</returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
        /// <summary>
        /// Checks equality of two double values. Values are equal if both 
        /// are NaN, or numerical valueas are identical.
        /// </summary>
        /// <param name="first">1st tested value</param>
        /// <param name="second">2nd tested value</param>
        /// <returns>true if both values are true</returns>
        protected bool DoubleCheck(double first, double second)
        {
            if (Double.IsNaN(first) && Double.IsNaN(second)) return true;
            if (Double.IsNaN(first) || Double.IsNaN(second)) return false;
            return (first == second);
        }

        /// <summary>
        /// Multiplication operator for TableCell.
        /// </summary>
        /// <param name="cell">Object to multiply</param>
        /// <param name="value">Multiplication value</param>
        /// <returns></returns>
        public static TableCell operator *(TableCell cell, Double value)
        {
            return new TableCell(cell.Value * value);
        }

        /// <summary>
        /// Compares two TableCells together.
        /// </summary>
        /// <param name="obj">Another TableCell</param>
        /// <returns>-1 if object is lesser, 0 if both are identical, 1 if object is greater.</returns>
        public virtual int CompareTo(object obj)
        {
            
            if (!(obj is TableCell)) throw new TPCTableCellException("Cannot compare TableCell to other than TableCell.");
            /*
            if ( Double.IsNaN( ((RESCell)obj).Value ) )
            {
                if (Double.IsNaN(this.Value)) return 0; // NaNs are threated as identical
                else return -1; // NaN is threated always as smaller value than normal value
            }

            // if the object is not NaN and this.value is, then this value is threated as smaller
            if (Double.IsNaN(this.Value)) return 1;
            */

            return this.Value.CompareTo(((TableCell)obj).Value); ;
        }

        /// <summary>
        /// Returns copy of the object
        /// </summary>
        /// <returns>Copy of the object.</returns>
        public virtual TableCell Clone()
        {
            Console.WriteLine("tablecell clone");
            return new TableCell(this.Value);
        }

    }
}
