﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace TPClib.Forms
{

    /// <summary>
    /// Class for one Tissue active curve ( TAC )
    /// </summary>
    public class Curve
    {
        /// <summary>
        /// Curve name
        /// </summary>
        public string name = "";
        /// <summary>
        /// Data unit
        /// </summary>
        public TPClib.Data_unit unit;
        /// <summary>
        /// Default constructor
        /// </summary>
        public Curve()
        {
            name = "None";
            unit = Data_unit.Unknown;
            comments = "<NO COMMENTS>";
            data = new List<Point>();
            lineStyle = new LineStyle();
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="name">Name of TAC</param>
        /// <param name="xdata">x-axis data</param>
        /// <param name="ydata">y-axis data</param>
        /// <param name="lineStyle">Style of line when visualized</param>
        public Curve(string name, double[] xdata, double[] ydata, LineStyle lineStyle)
        {
            this.name = name;
            unit = Data_unit.Undefined;
            comments = "<NO COMMENTS>";
            this.lineStyle = lineStyle;
            data = new List<Point>();
            for (int i = 0; i < xdata.Length && i < ydata.Length; i++) {
                Point p;
                p.X = xdata[i];
                p.Y = ydata[i];
                data.Add(p);
            }
        }

        /// <summary>
        /// Returns name of the curve as string
        /// </summary>
        /// <returns>Curve Name</returns>
        public override string ToString()
        {
            return "Curve[name="+name+" data.Count="+data.Count+" lineStyle="+lineStyle.ToString()+"]";
        }
        /// <summary>
        /// Data of curve
        /// </summary>
        protected List<Point> data;

        /// <summary>
        /// Style of line when visualized
        /// </summary>
        public LineStyle lineStyle = new LineStyle();
        /// <summary>
        /// Gets length of curve
        /// </summary>
        public int Length {
            get {
                return data.Count;
            }
        }
        /// <summary>
        /// Curve point
        /// </summary>
        public struct Point {
            public double X;
            public double Y;
        }
        /// <summary>
        /// Gets data from curve
        /// </summary>
        /// <param name="i">zero-based index or data</param>
        /// <returns>data (x,y) pair at index</returns>
        public double this[int i, int j]
        {
            get
            {
                if (j == 0) return data[i].X;
                else return data[i].Y;
            }
            set
            {
                Point p = data[i];
                if (j == 0)
                    p.X = value;
                else 
                    p.Y = value;
                data[i] = p;
            }
        }
        /// <summary>
        /// Gets data from curve
        /// </summary>
        /// <param name="i">zero-based index or data</param>
        /// <returns>data (x,y) pair at index</returns>
        public Point this[int i] {
            get {
                return data[i];
            }
            set {
                data[i] = value;
            }
        }
        /// <summary>
        /// Returns y-value of data
        /// </summary>
        /// <param name="i">zero-based index for value</param>
        /// <returns>y-value at index</returns>
        public double GetY(int i) {
            return data[i].Y;
        }
        /// <summary>
        /// Returns x-value of data
        /// </summary>
        /// <param name="i">zero-based index for value</param>
        /// <returns>x-value at index</returns>
        public double GetX(int i)
        {
            return data[i].X;
        }

        /// <summary>
        /// Comments for the curve
        /// </summary>
        public string comments;
    }
}
