﻿/******************************************************************************
 *
 * Copyright (c) 2009 Turku PET Centre
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 * Turku PET Centre hereby disclaims all copyright interest in the program.
 * Juhani Knuuti
 * Director, Professor
 * 
 * Turku PET Centre, Turku, Finland, http://www.turkupetcentre.fi/
 * 
 ******************************************************************************/
using System;
using System.Collections.Generic;
using System.Text;

namespace TPClib.Forms
{
    /// <summary>
    /// Axis class for floating axis. Handles both linear and locarithmic scales
    /// </summary>
    public class FloatAxis : PlotAxis
    {
        /// <summary>
        /// Axis scale type
        /// </summary>
        private ScaleType scale = ScaleType.Fit_To_Screen;
        /// <summary>
        /// Gets or set scaling type. 
        /// </summary>
        public ScaleType Scale
        {
            get { return scale; }
            set
            {
                scale = value;
                CalculateScale(this.value_min / factor, this.value_max / factor);
            }
        }

        private double exp_max;
        private double exp_min;
        private double interval_zeros;
        public double static_min = -10000.0d;
        public double static_max = 10000.0d;

        private double factor = 1.0d;

        /// <summary>
        /// Gives multiplication factor to axis. With this we can view the values in
        /// different scales min, sec, km, mm
        /// </summary>
        public double Factor { get { return factor; } set { factor = value; } }

        private int interval_return;
        //public int zeros;
        /// <summary>
        /// The interval of one tick. (displayed on the screen)
        /// </summary>
        private double interval;
        private double interval_max;
        private double interval_min;
        private int stepnum;
        private double currentvalue;
        private int min_zeros;
        private double minpoint = 0.0d;

        /// <summary>
        /// Tells how many tics will be displayed withot text. For example, if
        /// text_interval = 5, there will be 5 lines and every fifth ticks value
        /// will be displayed as string
        /// </summary>
        public int text_interval;
        /// <summary>
        /// Returns starting graphics location 
        /// </summary>
        /// <returns>location in axis</returns>
        public override double ResetInterval()
        {
            if (scale == ScaleType.Log)
            {
                interval_return = 0;
                interval_zeros = min_zeros;
                currentvalue = minpoint;
            }
            else
            {
                currentvalue = interval_min + minpoint;
            }

            stepnum = 0;
            return currentvalue;
        }
        /// <summary>
        /// Steps to the next interval
        /// </summary>
        /// <returns>value of interval</returns>
        public override double NextInterval()
        {
            stepnum++;

            if (scale == ScaleType.Log)
            {
                if (interval_return == 0)
                {
                    interval_return = 1;
                    currentvalue = 2.0 * Math.Pow(10, interval_zeros);
                }
                else if (interval_return == 1)
                {
                    interval_return = 2;
                    currentvalue = 5.0 * Math.Pow(10, interval_zeros);
                }
                else
                {
                    interval_return = 0;
                    interval_zeros++;
                    currentvalue = 1.0 * Math.Pow(10, interval_zeros);
                }
            }
            else
            {
                currentvalue += interval;
            }

            return currentvalue;
        }
        /// <summary>
        /// Returns current axis value as string
        /// </summary>
        /// <returns>string representation of current axis value</returns>
        public override string GetTextOfCurrentInterval()
        {
            if (stepnum % text_interval == 0) return (currentvalue).ToString("0.#");
            else return null;
        }

        /// <summary>
        /// Calculates scales and intervals for axis
        /// </summary>
        /// <param name="value_min">minimum value on axis</param>
        /// <param name="value_max">maximum value on axis</param>
        /// <param name="log">is the axis displayed in logarithmic scale</param>
        public void CalculateScale(double valuemin, double valuemax)
        {

            if (this.Scale == ScaleType.Static)
            {
                this.value_min = static_min * factor;
                this.value_max = static_max * factor;
            }
            else
            {
                value_min = valuemin * factor;
                value_max = valuemax * factor;
            }

            // finding the next 10 step above the max
            exp_max = Double.MinValue;

            // Finding the next base 10 value below the min value
            if (value_min == 0) exp_min = 0;
            else if (value_min > 0)
            {
                exp_min = Double.MaxValue;
                for (int i = 16; i >= -16; i--)
                {
                    if (value_min >= exp_min) { break; }
                    exp_min = Math.Pow(10, i);
                    min_zeros = i;
                }
            }
            else if (value_min < 0)
            {
                exp_min = Double.MinValue;
                for (int i = 16; i >= -16; i--)
                {
                    if (value_min <= exp_min) { exp_min = -Math.Pow(10, i + 2); min_zeros = i + 2; break; }
                    exp_min = -Math.Pow(10, i);
                    min_zeros = i;
                }
            }


            this.minpoint = exp_min;
            this.value_min = value_min - minpoint;
            this.value_max = value_max - minpoint;


            // Finding the next base 10 value above the max value
            for (int i = -16; i <= 16; i++)
            {
                if (value_max <= exp_max) { break; }
                exp_max = Math.Pow(10, i);
            }
            exp_min = 0.0d;

            //double val = value_max / exp_max; //Math.Log10(exp_max / exp_min);
            double val = (value_max) / (exp_max); //Math.Log10(exp_max / exp_min);

            //zeros = i;
            if (val > 0.2)
            {
                interval = ((exp_max) / 2.0) / 10.0 / 5.0;
                text_interval = 5;
            }
            else
            {
                interval = ((exp_max) / 10.0) / 10 / 2.0;
                text_interval = 2;
            }

            interval_max = ((int)(value_max / interval) + 1) * interval;
            interval_min = ((int)(value_min / interval)) * interval;
            if (value_min < 0) interval_min -= interval;

            // Now the intervals are calculated

            if (scale == ScaleType.Log)
            {
                min = exp_min;
                max = exp_max;

                // We find the correct min/max interval
                for (double time = ResetInterval(); time <= exp_max; time = NextInterval())
                {
                    if (time < value_min) min = time;
                    if (time > value_max) { max = time; break; }
                }

                text_interval = 1;
            }
            else
            {
                min = interval_min + minpoint;
                max = interval_max + minpoint;
            }
        }

        /// <summary>
        /// Calculates axis point place between (0-1). 0 is first interval and 1 is last interval.
        /// </summary>
        /// <param name="x">the time value</param>
        /// <returns>x value from 0 to 1</returns>
        public override double GetPlace(double px)
        {
            double x = px * factor;

            if (Scale == ScaleType.Log)
            {
                // logarithmic
                if (x <= 0) return 0;
                if (min == max) return 0;

                double log_max = Math.Log10(max + 1.0);
                double log_min = Math.Log10(min + 1.0);
                return (Math.Log10(x - min)) / (log_max - log_min);

            }
            else
            {
                return ((x - minpoint) - interval_min) / (interval_max - interval_min);
            }

        }
    }
}
